/*
 * Copyright (C) 2007, 2008  Miriam Ruiz <little_miry@yahoo.es>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _games_filter_h
#define _games_filter_h

#include "taghandler.h"

#include <set>
#include <iostream>

class PackageFilter
{
public:
	PackageFilter();
	~PackageFilter();

	typedef enum {
		Unknown = 1,   // The calification for the tag/package is unknown
		Green, // Green light, the tag/package is safe
		Yellow,    // Yellow light, handle with care
		Red,       // Green light, the tag/package is definitely unsafe
		Black,     // Mayday, mayday, the tag/package might be really dangerous!
	} Type;

	typedef std::string Tag;
	typedef std::set<std::string> TagSet;

	int TagValue(const Tag &tag);
	int TagsValue(const TagSet &tags);

protected:
	class ResultList : public FilterTagHandler::Result
	{
		public:
			inline ResultList(const std::string str, Type t) {
				name = str;
				type = t;
				next= NULL;
			}

			inline PackageFilter::ResultList *GetLast() {
				PackageFilter::ResultList *l = this, *t;
				while ( (t = l->next) != NULL) l = t;
				return l;
			}

			std::string name;
			Type type;
			PackageFilter::ResultList *next;
	};

	FilterTagHandler tagdata;
	PackageFilter::ResultList *list;

	inline PackageFilter::ResultList *GetFirst() {
		return list;
	}

	inline PackageFilter::ResultList *GetLast() {
		return list ? list->GetLast() : NULL;
	}

	inline void AddFirst(PackageFilter::ResultList *new_list) {
		if (!new_list) return;
		PackageFilter::ResultList *old_list = list;
		list = new_list;
		list->GetLast()->next = old_list;
	}

	inline void AddLast(PackageFilter::ResultList *new_list) {
		if (!list) { list = new_list; }
		else { list->GetLast()->next = new_list; }
	}

	inline void DeleteList() {
		while (list) {
			PackageFilter::ResultList *rest = list->next;
			list->next = NULL;
			delete list;
			list = rest;
		}
	}

	public:
		inline void Print(std::ostream &out) const {
			PackageFilter::ResultList *item = list;
			while (item) {
				out << item->name << ": ";
				tagdata.Print(out, item);
				item = item->next;
			}
		}
};

inline std::ostream &operator << (std::ostream &out, const PackageFilter *filter)
{
	if (filter != NULL)
		filter->Print(out);
	return out;
}

extern PackageFilter pkgfilter;

#endif
