CMAKE_MINIMUM_REQUIRED(VERSION 3.1.0)

###########################################
#### Usage: cmake -DCMD= -P cmake/cmd.cmake
###########################################

FUNCTION(MESSAGE type)
	IF(ARGV0 STREQUAL "STDOUT")
		EXECUTE_PROCESS(COMMAND ${CMAKE_COMMAND} -E echo "${ARGN}")
	ELSE()
		_MESSAGE(${type} "${ARGN}")
	ENDIF()
ENDFUNCTION()


FUNCTION(CREATE_HASH)
	IF(NOT FILES)
		MESSAGE(FATAL_ERROR "You need to specify 'FILES'")
	ENDIF()

	IF(NOT ALGORITHM)
		SET(ALGORITHM SHA256)
	ENDIF()
	STRING(TOLOWER "${ALGORITHM}" HASHFILE_ENDING)

	FILE(GLOB GLOBBED_FILES RELATIVE "${CMAKE_CURRENT_BINARY_DIR}" "${FILES}")

	FOREACH(f ${GLOBBED_FILES})
		FILE(${ALGORITHM} ${f} fHash)
		SET(OUTPUT "${fHash}  ${f}")
		MESSAGE(STDOUT ${OUTPUT})
		IF(CREATE_FILE)
			FILE(WRITE ${f}.${HASHFILE_ENDING} "${OUTPUT}\n")
		ENDIF()
	ENDFOREACH()
ENDFUNCTION()

FUNCTION(CHECK_WIX_WARNING)
	FILE(STRINGS "${FILE}" WIX_WARNINGS REGEX "warning")
	LIST(LENGTH WIX_WARNINGS WARNING_COUNT)
	IF(WARNING_COUNT GREATER 0)
		FOREACH(m ${WIX_WARNINGS})
			MESSAGE(STATUS "${m}\n")
		ENDFOREACH()
		MESSAGE(FATAL_ERROR "Found ${WARNING_COUNT} new WIX warnings")
	ELSE()
		MESSAGE(STATUS "No WIX warnings found")
	ENDIF()
ENDFUNCTION()


FUNCTION(IMPORT_PATCH)
	MESSAGE(STATUS "Import patch...")
	IF(CMAKE_VERSION VERSION_LESS "3.12")
		FIND_PACKAGE(PythonInterp REQUIRED)
		SET(Python_EXECUTABLE ${PYTHON_EXECUTABLE})
	ELSE()
		FIND_PACKAGE(Python REQUIRED)
	ENDIF()

	EXECUTE_PROCESS(COMMAND ${Python_EXECUTABLE} "${CMAKE_BINARY_DIR}/resources/jenkins/import.py")
ENDFUNCTION()


FUNCTION(DEPLOY_NEXUS)
	IF(NOT DEFINED ENV{NEXUS_USERNAME} OR NOT DEFINED ENV{NEXUS_PSW})
		MESSAGE(FATAL_ERROR "Please provide environment variable NEXUS_USERNAME and NEXUS_PSW")
	ENDIF()

	FIND_PROGRAM(MVN_BIN mvn)
	IF(NOT MVN_BIN)
		MESSAGE(FATAL_ERROR "Cannot find mvn")
	ENDIF()

	SET(SETTINGS_XML "<settings><servers><server>
					<id>nexus</id>
					<username>\${env.NEXUS_USERNAME}</username>
					<password>\${env.NEXUS_PSW}</password>
				</server></servers></settings>")
	FILE(WRITE settings.xml "${SETTINGS_XML}")

	FILE(GLOB FILE_AAR RELATIVE ${CMAKE_BINARY_DIR} *.aar)
	FILE(GLOB FILE_POM RELATIVE ${CMAKE_BINARY_DIR} *.pom)
	FILE(GLOB FILE_JAR RELATIVE ${CMAKE_BINARY_DIR} *-sources.jar)

	FILE(STRINGS "${FILE_POM}" is_snapshot REGEX "<version>.+-SNAPSHOT</version>")
	IF(is_snapshot)
		SET(NEXUS_URL https://repo.govkg.de/repository/ausweisapp-snapshots)
	ELSE()
		SET(NEXUS_URL https://repo.govkg.de/repository/ausweisapp-releases)
	ENDIF()

	EXECUTE_PROCESS(COMMAND ${MVN_BIN} deploy:deploy-file -Dfile=${FILE_AAR} -DpomFile=${FILE_POM} -Dsources=${FILE_JAR} -DrepositoryId=nexus -Durl=${NEXUS_URL} --settings settings.xml)
ENDFUNCTION()



IF(NOT CMD)
	MESSAGE(FATAL_ERROR "You need to specify 'CMD'")
ENDIF()

IF(CMD STREQUAL "HASH")
	CREATE_HASH()
ELSEIF(CMD STREQUAL "CHECK_WIX_WARNING")
	CHECK_WIX_WARNING()
ELSEIF(CMD STREQUAL "IMPORT_PATCH")
	IMPORT_PATCH()
ELSEIF(CMD STREQUAL "DEPLOY_NEXUS")
	DEPLOY_NEXUS()
ELSE()
	MESSAGE(FATAL_ERROR "Unknown CMD: ${CMD}")
ENDIF()
