/***************************************************************************
 $RCSfile$
                             -------------------
    cvs         : $Id$
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/


#ifdef HAVE_CONFIG_H
# include <config.h>
#endif


#include "calcpayee.h"
#include "kbanking.h"
#include "payee.h"

#include <qdatetime.h>
#include <qdatetimeedit.h>
#include <qmessagebox.h>
#include <qlineedit.h>
#include <qprogressdialog.h>
#include <qapplication.h>
#include <qpushbutton.h>

#include <gwenhywfar/debug.h>




CalculatePayee::CalculatePayee(KBanking *kb,
                               Payee *p,
                               AB_VALUE *inVal,
                               AB_VALUE *outVal,
                               QWidget* parent,
                               const char* name,
                               WFlags fl)
:CheckDuplicatesUI(0, name, fl)
,_app(kb)
,_aborted(false)
,_payee(p)
,_inVal(inVal)
,_outVal(outVal) {
  QObject::connect(abortButton, SIGNAL(clicked()),
                   this, SLOT(slotAbortClicked()));

  accountsProgress->setTotalSteps(-1);
  accountsProgress->setProgress(-1);
  transactionsProgress->setTotalSteps(-1);
  transactionsProgress->setProgress(-1);
}



CalculatePayee::~CalculatePayee(){
}



bool CalculatePayee::calculate() {
  std::list<Account*>::const_iterator it;
  int accnt;
  bool setRemoteAccount;

  setCaption(tr("Calculating for payee"));

  setRemoteAccount=(_payee->bankCode().empty() ||
                    _payee->accountNumber().empty());

  accountsProgress->setTotalSteps(_app->getAppAccounts().size());
  accountsProgress->setProgress(0);
  accnt=0;
  for (it=_app->getAppAccounts().begin();
       it!=_app->getAppAccounts().end();
       it++) {
    AH_STORAGE *st;
    int year, month, day;
    int rv;
    GWEN_IDLIST *idl;

    DBG_DEBUG(0, "Handling account %s / %s",
              (*it)->getBankCode().c_str(),
              (*it)->getAccountNumber().c_str());

    st=(*it)->getStorage();
    assert(st);

    idl=AH_Storage_GetAvailableDays(st);
    assert(idl);
    rv=AH_Storage_GetFirstDay(st, idl, &year, &month, &day);
    if (!rv) {
      GWEN_TIME *ct;
      GWEN_TIME *ft;
      int totalDays;

      ct=GWEN_CurrentTime();
      assert(ct);
      ft=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
      assert(ft);

      totalDays=(int)((GWEN_Time_Seconds(ct)-GWEN_Time_Seconds(ft))/
                      (60*60*24));
      transactionsProgress->setTotalSteps(totalDays);
      transactionsProgress->setProgress(0);
      GWEN_Time_free(ct);

      while(!rv) {
        uint32_t id = 0;
        GWEN_DB_NODE *db;
        GWEN_TIME *ti;

        ti=0;
        DBG_DEBUG(0, "Loading day %04d/%02d/%02d", year, month, day);
  
        qApp->processEvents();
        if (_aborted) {
          DBG_ERROR(0, "User aborted");
          if (id)
            AH_Storage_AbandonDay(st, id);
          GWEN_IdList_free(idl);
          return false;
        }
  
        id=AH_Storage_OpenDay(st, year, month, day, 1);
        if (!id) {
          DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                    year, month, day);
          GWEN_IdList_free(idl);
          return false;
        }
    
        db=AH_Storage_GetFirstTransaction(st, id);
        if (!db) {
          DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                   year, month, day);
        }
        while(db) {
          const char *pid;

          DBG_DEBUG(0, "Got a transaction");
          pid=GWEN_DB_GetCharValue(db, "payee", 0, 0);
          if (pid) {
            if (strcasecmp(_payee->id().c_str(), pid)==0) {
              GWEN_DB_NODE *dbValue;

              /* update account settings if possible. Since version
               * 0.9.28.1 QBankManager is able to gather remote account
               * information from the purpose lines, but some payees which
               * have been created before this upgrade may still have no
               * account data. */
              if (setRemoteAccount) {
                const char *s;

                s=GWEN_DB_GetCharValue(db, "remoteBankCode", 0, 0);
                if (s) {
                  _payee->setBankCode(s);
                  s=GWEN_DB_GetCharValue(db, "remoteAccountNumber", 0, 0);
                  if (s) {
                    _payee->setAccountNumber(s);
                    setRemoteAccount=false;
                  }
                }
              }

              dbValue=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                                       "value");
              if (dbValue) {
                AB_VALUE *v;

                v=AB_Value_fromDb(dbValue);
                if (v) {
                  if (AB_Value_IsNegative(v)) {
                    if (_outVal) {
                      AB_Value_Negate(v);
                      AB_Value_AddValue(_outVal, v);
                    }
                  }
                  else {
                    if (_inVal)
                      AB_Value_AddValue(_inVal, v);
                  }
                }
                AB_Value_free(v);
              }
            }
          }
          db=AH_Storage_GetNextTransaction(st, id);
        } // for transaction

        if (AH_Storage_CloseDay(st, id)) {
          DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                    year, month, day);
          GWEN_IdList_free(idl);
          return false;
        }

        ti=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
        transactionsProgress->setProgress((int)((GWEN_Time_Seconds(ti)-
                                                 GWEN_Time_Seconds(ft)
                                                )/
                                                (60*60*24)));
        GWEN_Time_free(ti);

        rv=AH_Storage_GetNextDay(st, idl, &year, &month, &day);
      } /* while !rv */
      GWEN_Time_free(ft);
    } /* if first day */
    GWEN_IdList_free(idl);

    accnt++;
    accountsProgress->setProgress(accnt);
  } // for account

  accountsProgress->setTotalSteps(1);
  accountsProgress->setProgress(1);
  transactionsProgress->setTotalSteps(1);
  transactionsProgress->setProgress(1);

  return true;
}



bool CalculatePayee::calculateAll() {
  std::list<Account*>::const_iterator it;
  std::list<Payee*>::const_iterator pit;
  int accnt;
  bool doneSomething=false;

  setCaption(tr("Calculating for payees"));
  accountsProgress->setTotalSteps(_app->getAppAccounts().size());
  accountsProgress->setProgress(0);
  accnt=0;

  // reset all payees' values
  for (pit=_app->getPayees().begin();
       pit!=_app->getPayees().end();
       pit++)
    (*pit)->resetValues();

  // calculate new ones
  for (it=_app->getAppAccounts().begin();
       it!=_app->getAppAccounts().end();
       it++) {
    AH_STORAGE *st;
    int year, month, day;
    int rv;
    GWEN_IDLIST *idl;

    DBG_DEBUG(0, "Handling account %s / %s",
              (*it)->getBankCode().c_str(),
              (*it)->getAccountNumber().c_str());

    st=(*it)->getStorage();
    assert(st);

    idl=AH_Storage_GetAvailableDays(st);
    assert(idl);
    rv=AH_Storage_GetFirstDay(st, idl, &year, &month, &day);
    if (!rv) {
      GWEN_TIME *ct;
      GWEN_TIME *ft;
      int totalDays;

      ct=GWEN_CurrentTime();
      assert(ct);
      ft=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
      assert(ft);

      totalDays=(int)((GWEN_Time_Seconds(ct)-GWEN_Time_Seconds(ft))/
                      (60*60*24));
      transactionsProgress->setTotalSteps(totalDays);
      transactionsProgress->setProgress(0);
      GWEN_Time_free(ct);

      while(!rv) {
        uint32_t id = 0;
        GWEN_DB_NODE *db;
        std::list<RefPointer<Transaction> > tl;
        std::list<RefPointer<Transaction> >::iterator xait;
        GWEN_TIME *ti;

        ti=0;
        DBG_DEBUG(0, "Loading day %04d/%02d/%02d", year, month, day);
  
        qApp->processEvents();
        if (_aborted) {
          DBG_ERROR(0, "User aborted");
          if (id)
            AH_Storage_AbandonDay(st, id);
          GWEN_IdList_free(idl);
          return false;
        }
  
        id=AH_Storage_OpenDay(st, year, month, day, 1);
        if (!id) {
          DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                    year, month, day);
          GWEN_IdList_free(idl);
          return false;
        }
    
        db=AH_Storage_GetFirstTransaction(st, id);
        if (!db) {
          DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                   year, month, day);
        }
        while(db) {
          const char *pid;

          DBG_DEBUG(0, "Got a transaction");
          pid=GWEN_DB_GetCharValue(db, "payee", 0, 0);
          if (pid && *pid) {
            Payee *payee;

            payee=_app->findPayeeById(pid);
            if (payee) {
              GWEN_DB_NODE *dbValue;

              /* update account settings if possible. Since version
               * 0.9.28.1 QBankManager is able to gather remote account
               * information from the purpose lines, but some payees which
               * have been created before this upgrade may still have no
               * account data. */
              if (payee->bankCode().empty() ||
                  payee->accountNumber().empty()) {
                const char *s;

                s=GWEN_DB_GetCharValue(db, "remoteBankCode", 0, 0);
                if (s) {
                  payee->setBankCode(s);
                  s=GWEN_DB_GetCharValue(db, "remoteAccountNumber", 0, 0);
                  if (s) {
                    payee->setAccountNumber(s);
                  }
                }
              }

              dbValue=GWEN_DB_GetGroup(db, GWEN_PATH_FLAGS_NAMEMUSTEXIST,
                                       "value");
              if (dbValue) {
                AB_VALUE *v;

                v=AB_Value_fromDb(dbValue);
                if (v) {
                  if (AB_Value_IsNegative(v)) {
                    AB_Value_Negate(v);
                    payee->addOutValue(v);
                  }
                  else
                    payee->addInValue(v);
                }
                AB_Value_free(v);
              }
            }
          }
          db=AH_Storage_GetNextTransaction(st, id);
        } // for transaction

        if (AH_Storage_CloseDay(st, id)) {
          DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                    year, month, day);
          if (doneSomething) {
            _app->flagStaff()->payeesUpdated();
          }
          GWEN_IdList_free(idl);
          return false;
        }

        ti=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
        transactionsProgress->setProgress((int)((GWEN_Time_Seconds(ti)-
                                                 GWEN_Time_Seconds(ft)
                                                )/
                                                (60*60*24)));
        GWEN_Time_free(ti);

        rv=AH_Storage_GetNextDay(st, idl, &year, &month, &day);
      } /* while !rv */
      GWEN_Time_free(ft);
    } /* if first day */
    GWEN_IdList_free(idl);

    accnt++;
    accountsProgress->setProgress(accnt);
  } // for account

  accountsProgress->setTotalSteps(1);
  accountsProgress->setProgress(1);
  transactionsProgress->setTotalSteps(1);
  transactionsProgress->setProgress(1);

  if (doneSomething) {
    _app->flagStaff()->payeesUpdated();
  }
  return true;
}



void CalculatePayee::slotAbortClicked(){
  _aborted=true;
  abortButton->setEnabled(false);
}



bool CalculatePayee::getTransactions(std::list<RefPointer<Transaction> > &l) {
  std::list<Account*>::const_iterator it;
  int accnt;

  setCaption(tr("Getting transactions for payee"));

  accountsProgress->setTotalSteps(_app->getAppAccounts().size());
  accountsProgress->setProgress(0);
  accnt=0;
  for (it=_app->getAppAccounts().begin();
       it!=_app->getAppAccounts().end();
       it++) {
    AH_STORAGE *st;
    int year, month, day;
    int rv;
    GWEN_IDLIST *idl;

    DBG_DEBUG(0, "Handling account %s / %s",
              (*it)->getBankCode().c_str(),
              (*it)->getAccountNumber().c_str());

    st=(*it)->getStorage();
    assert(st);

    idl=AH_Storage_GetAvailableDays(st);
    assert(idl);
    rv=AH_Storage_GetFirstDay(st, idl, &year, &month, &day);
    if (!rv) {
      GWEN_TIME *ct;
      GWEN_TIME *ft;
      int totalDays;

      ct=GWEN_CurrentTime();
      assert(ct);
      ft=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
      assert(ft);

      totalDays=(int)((GWEN_Time_Seconds(ct)-GWEN_Time_Seconds(ft))/
                      (60*60*24));
      transactionsProgress->setTotalSteps(totalDays);
      transactionsProgress->setProgress(0);
      GWEN_Time_free(ct);

      while(!rv) {
        uint32_t id = 0;
        GWEN_DB_NODE *db;
        GWEN_TIME *ti;

        ti=0;
        DBG_DEBUG(0, "Loading day %04d/%02d/%02d", year, month, day);
  
        qApp->processEvents();
        if (_aborted) {
          DBG_ERROR(0, "User aborted");
          if (id)
            AH_Storage_AbandonDay(st, id);
          GWEN_IdList_free(idl);
          return false;
        }
  
        id=AH_Storage_OpenDay(st, year, month, day, 1);
        if (!id) {
          DBG_ERROR(0, "Error loading day %04d/%02d/%02d",
                    year, month, day);
          GWEN_IdList_free(idl);
          return false;
        }
    
        db=AH_Storage_GetFirstTransaction(st, id);
        if (!db) {
          DBG_WARN(0, "No transactions in day %04d/%02d/%02d",
                   year, month, day);
        }
        while(db) {
          const char *pid;

          DBG_DEBUG(0, "Got a transaction");
          pid=GWEN_DB_GetCharValue(db, "payee", 0, 0);
          if (pid) {
            if (strcasecmp(_payee->id().c_str(), pid)==0) {
              Transaction *t;

              t=new Transaction();
              if (!t->fromDb(db)) {
                DBG_ERROR(0, "Bad transaction, ignoring");
                GWEN_DB_Dump(db, stderr, 2);
              }
              else
                l.push_back(t);
            }
          }
          db=AH_Storage_GetNextTransaction(st, id);
        } // for transaction

        if (AH_Storage_CloseDay(st, id)) {
          DBG_ERROR(0, "Error closing day %04d/%02d/%02d",
                    year, month, day);
          GWEN_IdList_free(idl);
          return false;
        }

        ti=GWEN_Time_new(year, month-1, day, 0, 0, 0, 1);
        transactionsProgress->setProgress((int)((GWEN_Time_Seconds(ti)-
                                                 GWEN_Time_Seconds(ft)
                                                )/
                                                (60*60*24)));
        GWEN_Time_free(ti);

        rv=AH_Storage_GetNextDay(st, idl, &year, &month, &day);
      } /* while !rv */
      GWEN_Time_free(ft);
    } /* if first day */
    GWEN_IdList_free(idl);

    accnt++;
    accountsProgress->setProgress(accnt);
  } // for account

  accountsProgress->setTotalSteps(1);
  accountsProgress->setProgress(1);
  transactionsProgress->setTotalSteps(1);
  transactionsProgress->setProgress(1);

  return true;
}


