#!/usr/bin/env python
# arch-tag: 3fe5d578-07de-406e-950e-6bd26dc93af0
# Copyright (C) 2004--2005 Canonical Limited
#                    Author: David Allouche <david@allouche.net>
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

"""Test cases for output parsing.

Unit tests for output classifiers and integration tests with actual
output.
"""

import os
import stat
import unittest
import pybaz as arch
import framework
from framework import TestCase
import framework_output


class ChangeHelper(unittest.TestCase):

    def _make_expected_change(self, type_, nametype, name, **kwargs):
        assert nametype is [arch.FileName, arch.DirName][kwargs['directory']]
        make_expected = framework_output.ExpectedChange
        expected = make_expected(type_, name, kwargs['directory'])
        if 'binary' in kwargs:
            expected.is_binary(kwargs['binary'])
        if 'oldname' in kwargs:
            expected.has_oldname(kwargs['oldname'])
        return expected

    def expect_one_change(self, type_, name, directory=False):
        return framework_output.ExpectedChange(type_, name, directory)

    def expect_composite_change(self):
        return framework_output.ExpectedCompositeChange()


class Classifiers(ChangeHelper, unittest.TestCase):

    tests = []

    def _classify_chatter_helper(self, classify):
        lines = ['* chatty', 'witty']
        classifier = classify(lines)
        chatter = classifier.next()
        self.assertEqual('chatty', chatter.text)
        self.assertEqual('* chatty', str(chatter))
        self.assertEqual(arch.Chatter, type(chatter))
        non_chatter = classifier.next()
        self.assert_(non_chatter is lines[1])
        self.assertRaises(StopIteration, classifier.next)

    def classify_chatter(self):
        """classify_chatter works"""
        self._classify_chatter_helper(arch.classify_chatter)
    tests.append('classify_chatter')

    def classify_changes(self):
        """classify_changes classifies chatter correctly"""
        self._classify_chatter_helper(arch.classify_changeset_creation)
    tests.append('classify_changes')

    def _classify_tree_change_helper(self, line, type_,
                                     nametype, name, **kwargs):
        classifier = arch.classify_changeset_creation([line])
        self.assert_(hasattr(classifier, 'next'))
        changes = list(classifier)
        expected = self._make_expected_change(type_, nametype, name, **kwargs)
        expected.string_value(line)
        expected.check(self, changes)

    def _classify_merge_outcome_helper(self, line, type_,
                                       nametype, name, **kwargs):
        classifier = arch.classify_changeset_application([line])
        self.assert_(hasattr(classifier, 'next'))
        changes = list(classifier)
        expected = self._make_expected_change(type_, nametype, name, **kwargs)
        expected.string_value(line)
        expected.check(self, changes)


class ClassifierIntegration(ChangeHelper, TestCase):

    def extraSetup(self):
        self.params.create_working_tree()
        self.params.create_other_tree()
        self.orig = self.params.working_tree
        self.mod = self.params.other_tree
        self.cset = self.params.arch_dir/'cset'

    def _check_diff_patch(self, expected):
        def filter_treechange(obj):
            return isinstance(obj, arch.TreeChange)
        expected.filter = filter_treechange
        gen = arch.iter_delta(self.orig, self.mod, self.cset)
        gen_out = list(gen)
        expected.check(self, gen_out)
        app = gen.changeset.iter_apply(self.orig)
        app_out = list(app)
        expected.check(self, app_out)

    def set_tagging_method(self, method):
        self.orig.tagging_method = method
        self.mod.tagging_method = method


class ClassifyFileAddition(Classifiers):
    tests = []
    def classify_changes_addition(self):
         """classify_changes classifies FileAddition"""
         type_ = arch.FileAddition
         self._classify_tree_change_helper(
             'A  foo', type_, arch.FileName, 'foo', directory=False)
         self._classify_tree_change_helper(
             'A/ foo', type_, arch.DirName, 'foo', directory=True)
         self._classify_tree_change_helper(
             'A  fo\(sp)o', type_, arch.FileName, 'fo o', directory=False)
         self._classify_tree_change_helper(
             'A/ fo\(sp)o', type_, arch.DirName, 'fo o', directory=True)
    tests.append('classify_changes_addition')


class IntegrateFileAddition(ClassifierIntegration):

    tests = []

    def _file_addition_helper(self, name, directory):
        self.set_tagging_method('names')
        if not directory:
            open(self.mod/name, 'w').close()
        else:
            os.mkdir(self.mod/name)
        expected = self.expect_one_change(arch.FileAddition, name, directory)
        self._check_diff_patch(expected)

    def file_addition(self):
        """integration of FileAddition of a file"""
        self._file_addition_helper('foo', False)
    tests.append('file_addition')

    def file_addition_space(self):
        """integration of FileAddition of a file with a space"""
        self._file_addition_helper('foo bar', False)
    tests.append('file_addition_space')

    def dir_addition(self):
        """integration of FileAddition of a directory"""
        self._file_addition_helper('foo', True)
    tests.append('dir_addition')

    def dir_addition_space(self):
        """integration of FileAddition of a directory with a space"""
        self._file_addition_helper('foo bar', True)
    tests.append('dir_addition_space')


class ClassifyFileDeletion(Classifiers):
    tests = []
    def classify_changes_deletion(self):
         """classify_changes classifies FileDeletion"""
         type_ = arch.FileDeletion
         self._classify_tree_change_helper(
             'D  foo', type_, arch.FileName, 'foo', directory=False)
         self._classify_tree_change_helper(
             'D/ foo', type_, arch.DirName, 'foo', directory=True)
         self._classify_tree_change_helper(
             'D  fo\(sp)o', type_, arch.FileName, 'fo o', directory=False)
         self._classify_tree_change_helper(
             'D/ fo\(sp)o', type_, arch.DirName, 'fo o', directory=True)
    tests.append('classify_changes_deletion')


class IntegrateFileDeletion(ClassifierIntegration):

    tests = []

    def _file_deletion_helper(self, name, directory):
        self.set_tagging_method('names')
        if not directory:
            open(self.orig/name, 'w').close()
        else:
            os.mkdir(self.orig/name)
        expected = self.expect_one_change(arch.FileDeletion, name, directory)
        self._check_diff_patch(expected)

    def file_deletion(self):
        """integration of FileDeletion of a file"""
        self._file_deletion_helper('foo', False)
    tests.append('file_deletion')

    def file_deletion_space(self):
        """integration of FileDeletion of a file with a space"""
        self._file_deletion_helper('foo bar', False)
    tests.append('file_deletion_space')

    def dir_deletion(self):
        """integration of FileDeletion of a directory"""
        self._file_deletion_helper('foo', True)
    tests.append('dir_deletion')

    def dir_deletion_space(self):
        """integration of FileDeletion of a directory with a space"""
        self._file_deletion_helper('foo bar', True)
    tests.append('dir_deletion_space')


class ClassifyFileModification(Classifiers):
    tests = []
    def classify_changes_modification(self):
         """classify_changes classifies FileModification"""
         type_ = arch.FileModification
         self._classify_tree_change_helper(
             'M  foo', type_, arch.FileName, 'foo', binary=False,
             directory=False)
         self._classify_tree_change_helper(
             'Mb foo', type_, arch.FileName, 'foo', binary=True,
             directory=False)
         self._classify_tree_change_helper(
             'M  fo\(sp)o', type_, arch.FileName, 'fo o', binary=False,
             directory=False)
         self._classify_tree_change_helper(
             'Mb fo\(sp)o', type_, arch.FileName, 'fo o', binary=True,
             directory=False)
    tests.append('classify_changes_modification')


class IntegrateFileModification(ClassifierIntegration):

    tests = []

    def _file_modification_helper(self, name, binary):
        self.set_tagging_method('names')
        if not binary:
            open(self.orig/name, 'w').write('hello world')
            open(self.mod/name, 'w').write('Hello, World!')
        else:
            open(self.orig/name, 'w').write('\0')
            open(self.mod/name, 'w').write('\1')
        expected = self.expect_one_change(arch.FileModification, name)
        expected.is_binary(binary)
        self._check_diff_patch(expected)

    def file_modification(self):
        """integration of FileModification of a text file"""
        self._file_modification_helper('foo', False)
    tests.append('file_modification')

    def file_modification_space(self):
        """integration of FileModification of a text file with a space"""
        self._file_modification_helper('foo bar', False)
    tests.append('file_modification_space')

    def file_modification_binary(self):
        """integration of FileModification of a binary file"""
        self._file_modification_helper('foo', True)
    tests.append('file_modification_binary')

    def file_modification_binary_space(self):
        """integration of FileModification of a binary file with a space"""
        self._file_modification_helper('foo bar', True)
    tests.append('file_modification_binary_space')


class ClassifyFilePermissionsChange(Classifiers):
    tests = []
    def classify_changes_permissions(self):
        """classify_changes classifies FilePermissionsChange"""
        type_ = arch.FilePermissionsChange
        self._classify_tree_change_helper(
            '-- foo', type_, arch.FileName, 'foo', directory=False)
        self._classify_tree_change_helper(
            '-/ foo', type_, arch.DirName, 'foo', directory=True)
        self._classify_merge_outcome_helper(
            '--/ foo', type_, arch.DirName, 'foo', directory=True)
        self._classify_tree_change_helper(
            '-- fo\(sp)o', type_, arch.FileName, 'fo o', directory=False)
        self._classify_tree_change_helper(
            '-/ fo\(sp)o', type_, arch.DirName, 'fo o', directory=True)
        self._classify_merge_outcome_helper(
            '--/ foo\(sp)bar', type_, arch.DirName, 'foo bar', directory=True)
    tests.append('classify_changes_permissions')


class IntegrateFilePermissionsChange(ClassifierIntegration):

    tests = []

    def _file_permissions_helper(self, name, directory):
        self.set_tagging_method('names')
        if not directory:
            open(self.orig/name, 'w').close()
            os.chmod(self.orig/name, stat.S_IRWXU)
            open(self.mod/name, 'w').close()
            os.chmod(self.orig/name, stat.S_IRWXU | stat.S_IRWXG)
        else:
            os.mkdir(self.orig/name)
            os.chmod(self.orig/name, stat.S_IRWXU)
            os.mkdir(self.mod/name)
            os.chmod(self.orig/name, stat.S_IRWXU | stat.S_IRWXG)
        type_ = arch.FilePermissionsChange
        expected = self.expect_one_change(type_, name, directory)
        self._check_diff_patch(expected)

    def file_permissions(self):
        """integration of FilePermissionsChange of a file"""
        self._file_permissions_helper('foo', False)
    tests.append('file_permissions')

    def file_permissions_space(self):
        """integration of FilePermissionsChange of a file with a space"""
        self._file_permissions_helper('foo bar', False)
    tests.append('file_permissions_space')

    def dir_permissions(self):
        """integration of FilePermissionsChange of a directory"""
        self._file_permissions_helper('foo', True)
    tests.append('dir_permissions')

    def dir_permissions_space(self):
        """integration of FilePermissionsChange of a directory with a space"""
        self._file_permissions_helper('foo bar', True)
    tests.append('dir_permissions_space')


class ClassifyFileRename(Classifiers):
    tests = []
    def classify_changes_rename(self):
        """classify_changes classifies FileRename"""
        type_ = arch.FileRename
        self._classify_tree_change_helper(
            '=> bar\tfoo', type_, arch.FileName, 'foo',
            directory=False, oldname='bar')
        self._classify_tree_change_helper(
            '/> bar\tfoo', type_, arch.DirName, 'foo',
            directory=True, oldname='bar')
        self._classify_tree_change_helper(
            '=> b\(sp)ar\tfo\(sp)o', type_, arch.FileName, 'fo o',
            directory=False, oldname='b ar')
        self._classify_tree_change_helper(
            '/> b\(sp)ar\tfo\(sp)o', type_, arch.DirName, 'fo o',
            directory=True, oldname='b ar')
    tests.append('classify_changes_rename')


class IntegrateFileRename(ClassifierIntegration):

    tests = []

    def _explicit_id_name(self, name):
        dirname, basename = name.splitname()
        return arch.FileName(dirname/'.arch-ids'/basename+'.id')

    def _set_explicit_id(self, name, xid):
        xid_name = self._explicit_id_name(name)
        os.mkdir(xid_name.dirname())
        open(xid_name, 'w').write(xid)

    def _file_rename_helper(self, oldname, newname):
        oldname, newname = map(arch.FileName, (oldname, newname))
        self.set_tagging_method('explicit')
        open(self.orig/oldname, 'w').close()
        self._set_explicit_id(self.orig/oldname, 'foo')
        open(self.mod/newname, 'w').close()
        self._set_explicit_id(self.mod/newname, 'foo')
        expected = self.expect_composite_change()
        expected_source = self.expect_one_change(arch.FileRename, newname)
        expected_source.has_oldname(oldname)
        expected.add(expected_source)
        old_xid, new_xid = map(self._explicit_id_name, (oldname, newname))
        expected_id = self.expect_one_change(arch.FileRename, new_xid)
        expected_id.has_oldname(old_xid)
        expected.add(expected_id)
        self._check_diff_patch(expected)

    def file_rename(self):
        """Integration of FileRename for a file."""
        self._file_rename_helper('foo', 'bar')
    tests.append('file_rename')

    def file_rename_space(self):
        """Integration of FileRename for a file with a space."""
        self._file_rename_helper('foo bar', 'foo gam')
    tests.append('file_rename_space')


class IntegrateDirRename(ClassifierIntegration):

    tests = []

    def _dir_rename_helper(self, oldname, newname):
        nametype = arch.DirName
        args = [arch.FileRename, nametype, newname]
        kwargs = {'directory': True, 'oldname': oldname}
        self.orig.tagging_method = 'explicit'
        self.mod.tagging_method = 'explicit'
        os.mkdir(self.orig/oldname)
        os.mkdir(self.orig/oldname/'.arch-ids')
        open(self.orig/oldname/'.arch-ids'/'=id', 'w').write('foo')
        os.mkdir(self.mod/newname)
        os.mkdir(self.mod/newname/'.arch-ids')
        open(self.mod/newname/'.arch-ids'/'=id', 'w').write('foo')
        self._check_diff_patch(*args, **kwargs)

    # FIXME: write integration tests


class ClassifySymLinkModification(Classifiers):
    tests = []
    def classify_changes_symlink(self):
        """classify_changes classifies SymlinkModification"""
        self._classify_tree_change_helper(
            '-> foo', arch.SymlinkModification, arch.FileName, 'foo',
            directory=False)
        self._classify_tree_change_helper(
            '-> fo\(sp)o', arch.SymlinkModification, arch.FileName, 'fo o',
            directory=False)
    tests.append('classify_changes_symlink')

    # FIXME: write integration tests


class ClassifyPatchConflict(Classifiers):
    tests = []
    def classify_changes_conflict(self):
        """classify_changes classifies PatchConflict"""
        self._classify_merge_outcome_helper(
            'C   foo', arch.PatchConflict, arch.FileName, 'foo',
            directory=False)
        self._classify_merge_outcome_helper(
            'C   fo\(sp)o', arch.PatchConflict, arch.FileName, 'fo o',
            directory=False)
    tests.append('classify_changes_conflict')

    # FIXME: write integration tests


framework.register(__name__)
