(* File: make_pcre.ml
   Time-stamp: <2007-07-17 20:41:18 trch>
   Copyright (C) 2007, Christophe Troestler
*)


(** Run "ocaml make_pcre.ml" in the pcre source directory in order to
    create a make.bat file for compiling and installing pcre.  Then do
    the same in the lib/ directory of this ocaml-pcre tarball.

    To run the .bat file, you need MSVC and the MS SDK to installed
    and in your path.  For that, follow the instructions in the
    README.win32 in the OCaml distribution.  Scripts are provided by
    the MS packages to put the executables in your PATH; typically you
    create a .bat file containing:

    CALL "C:\Program Files\Microsoft Visual Studio 8\Common7\Tools\vsvars32.bat"
    CALL "c:\Program Files\Microsoft Platform SDK"\SetEnv.Cmd /SRV32
    @cmd
*)

#load "str.cma";;
#load "unix.cma";;
open Printf


(** The defaults below should be OK ------------------------------ *)

(* Names of the C-pcre library; names are chosen so they do not clash with
   OCaml ones. *)
let pcre_dll = "dllcpcre.dll"
let pcre_lib = "libcpcre.lib"
let pcre_def = "pcre.DEF"
let pcreposix_lib = "pcreposix.lib"

let ocamllib =
  let re = Str.regexp "\n\\|\r" in
  let fh = Unix.open_process_in "ocamlc -where" in
  let l = input_line fh in
  match Unix.close_process_in fh with
  | Unix.WEXITED st when st = 0 -> Str.global_replace re "" l
  | _ -> printf "ocamlc -where failed"; Unix.getenv "OCAMLLIB"

(* C pcre installation directories *)
let install_c = ocamllib
let install_lib = Filename.concat install_c "pcre"
let install_dll = Filename.concat install_c "stublibs"
let install_h = Filename.concat install_c "pcre"

let d = "/DSUPPORT_UTF8 /DSUPPORT_UCP /DPOSIX_MALLOC_THRESHOLD=10"

let cc_static fh ?(flags="") ~ext source =
  fprintf fh "cl /nologo /Ox /MT /DPCRE_STATIC %s %s /I . \
	/c %s.c /Fo%s%s\n" d flags source source ext

let mklib fh lib objects =
  fprintf fh "lib /nologo /out:%s %s\n" lib (String.concat " " objects)

let cc_dynamic fh ?(flags="") ~ext source =
  fprintf fh "cl /nologo /Ox /MD /DCAML_DLL %s %s /I . \
	/c %s.c /Fo%s%s\n" d flags source source ext

let mkdll fh dll def objects =
  let objects = String.concat " " (List.map Filename.quote objects) in
  fprintf fh "link /nologo /dll /out:%s /DEF:%s %s\n" dll def objects

(* Whether to compile the C-pcre lib or the pcre-ocaml lib *)
let make =
  if (Sys.file_exists "pcre.h.generic" || Sys.file_exists "pcre.h")
    && (Sys.file_exists "config.h.generic" || Sys.file_exists "config.h")
  then `Pcre
  else if Sys.file_exists "pcre.ml" then `OCaml
  else (
    eprintf "\n**ERROR**: You must run this script in the C pcre \
	source or in the lib\\\n           directory of ocaml-pcre.\n";
    exit 1
  )


(**************************************************************************)

(* Helpers *)

let install fh files dir =
  let dir = Filename.quote dir in
  fprintf fh "mkdir %s \t>NUL\n" dir;
  List.iter (fun f -> fprintf fh "copy \"%s\" %s\n" f dir) files

let make_bat ?(fname="make.bat") body =
  let fh = open_out fname in
  fprintf fh "@REM Generated by OCaml script %s\n" Sys.argv.(0);
  fprintf fh "@prompt $G$S\n\n";
  body fh;
  fprintf fh "\n\n@REM Restore a decent prompt\n";
  fprintf fh "@prompt $P$G$S\n";
  close_out fh;
  printf "Run \"%s\".\n" fname


(**************************************************************************)
(*
 * C pcre lib
 *)

(* Create pcre.DEF
   See http://msdn2.microsoft.com/en-us/library/d91k01sh(VS.80).aspx for
   the structure of DEF files *)
let make_pcre_def pcre_def =
  let re = Str.regexp "PCRE_EXP_DECL .*[* ]+\\([a-z_0-9]+\\)(.*" in
  let exports = ref [] in
  begin
    let n = ref 0 in
    let fh = open_in "pcre.h" in
    try
      while true do
        let l = input_line fh in
        if Str.string_match re l 0 then (
          incr n;
          let name = Str.matched_group 1 l ^ " @" ^ string_of_int !n in
          exports := name :: !exports
        )
      done
    with End_of_file -> ()
  end;
  let exports = List.rev !exports in
  let fh = open_out pcre_def in
  fprintf fh "LIBRARY %s\n" pcre_dll;
  fprintf fh "EXPORTS\n  %s\n" (String.concat "\n  " exports);
  close_out fh

let () =
  if make = `Pcre then begin
    (* Check the existence of some headers/impl *)
    let mv f0 f1 =
      if not(Sys.file_exists f1) then Sys.rename f0 f1 in
    mv "config.h.generic" "config.h";
    mv "pcre.h.generic" "pcre.h";
    make_pcre_def pcre_def;
    let sources =
      let re = Str.regexp "^pcre_.*\\.c$" in
      let dir = Array.to_list(Sys.readdir ".") in
      let sources = List.filter (fun f -> Str.string_match re f 0) dir in
      let sources = List.map (fun f -> Filename.chop_extension f) sources in
      (* "pcre_chartables" will only be created when make.bat is run *)
      "pcre_chartables" :: sources
    in
    make_bat begin fun fh ->
      (* Create pcre_chartables.c *)
      output_string fh "\n@REM Make chartables\n";
      fprintf fh "cl /nologo /DSUPPORT_UTF8 /DSUPPORT_UCP \
	/D_CRT_SECURE_NO_DEPRECATE /I . dftables.c\n";
      fprintf fh "dftables pcre_chartables.c\n";

      output_string fh "\n@REM Static library\n";
      List.iter (cc_static ~ext:".s.obj" fh) sources;
      mklib fh pcre_lib (List.map (fun f -> f ^ ".s.obj") sources);

      output_string fh "@REM Dynamic library\n";
      List.iter (cc_dynamic ~ext:".d.obj" fh) sources;
      mkdll fh pcre_dll pcre_def (List.map (fun f -> f ^ ".d.obj") sources);

      output_string fh "@REM Installation\n";
      install fh [pcre_lib] install_lib;
      install fh [pcre_dll] install_dll;
      install fh ["pcre.h"] install_h;

      (* pcreposix *)
      cc_static fh ~flags:"/DPCREPOSIX_STATIC /D_CRT_SECURE_NO_DEPRECATE"
        ~ext:".s.obj" "pcreposix";
      mklib fh pcreposix_lib ["pcreposix.s.obj"];
      install fh [pcreposix_lib] install_lib;

      (* pcretest *)
      fprintf fh "cl /nologo /F0x400000 /Ox /MT /I . \
	/D_CRT_SECURE_NO_DEPRECATE /DPCRE_STATIC /DPCREPOSIX_STATIC \
	pcretest.c %s pcreposix.lib\n" pcre_lib;
    end
  end

(**************************************************************************)
(*
 * OCaml pcre lib
 *)

let () =
  if make = `OCaml then begin
    (* Make pcre_stubs.DEF *)
    let fh = open_out "pcre_stubs.DEF" in
    fprintf fh "\
LIBRARY dllpcre_stubs.dll

EXPORTS
  pcre_ocaml_init @1
  pcre_version_stub @2
  pcre_compile_stub @3
  pcre_study_stub @4
  pcre_set_imp_match_limit_stub @5
  pcre_get_match_limit_stub @6
  pcre_options_stub @7
  pcre_size_stub @8
  pcre_studysize_stub @9
  pcre_capturecount_stub @10
  pcre_backrefmax_stub @11
  pcre_namecount_stub @12
  pcre_nameentrysize_stub @13
  pcre_firstbyte_stub @14
  pcre_firsttable_stub @15
  pcre_lastliteral_stub @16
  pcre_study_stat_stub @17
  pcre_exec_stub @18
  pcre_exec_stub_bc @19
  pcre_maketables_stub @20
  pcre_isspace_stub @21
  pcre_get_stringnumber_stub @22
  pcre_config_utf8_stub @23
  pcre_config_newline_stub @24
  pcre_config_link_size_stub @25
  pcre_config_match_limit_stub @26
  pcre_config_stackrecurse_stub @27\n";
    close_out fh;
    (*
     * Compile the OCaml library (and stubs)
     *)
    make_bat begin fun fh ->
      let inc =
        sprintf "/I \"%s\" /I %s" install_h (Filename.quote ocamllib) in
      let pcre_lib = Filename.concat install_lib pcre_lib in
(*       let pcre_dll = Filename.concat install_dll pcre_dll in *)

      cc_static fh ~ext:".s.obj" "pcre_stubs" ~flags:inc;
      mklib fh "libpcre_stubs.lib" ["pcre_stubs.s.obj"];

(*       cc_dynamic fh ~ext:".d.obj" "pcre_stubs" ~flags:inc; *)
(*       mkdll fh "dllpcre_stubs.dll" "pcre_stubs.DEF" *)
(*         ["pcre_stubs.d.obj"; Filename.concat ocamllib "ocamlrun.lib"; *)
(*          pcre_dll]; *)

      let caml_sources = "pcre.ml" in
      let pcre_lib = Filename.quote(Filename.quote pcre_lib) in
      fprintf fh "ocamlc -c pcre.mli\n";
      fprintf fh "ocamlc -custom -a -o pcre.cma %s -ccopt %s \
	-cclib -lpcre_stubs\n" caml_sources pcre_lib;
      fprintf fh "ocamlopt -a -o pcre.cmxa %s -ccopt %s \
	-cclib -lpcre_stubs\n" caml_sources pcre_lib;

      install fh ["libpcre_stubs.lib";
                  "pcre.mli"; "pcre.cmi"; "pcre.cma"; "pcre.cmxa"; "pcre.lib"]
        (Filename.concat ocamllib "pcre");
(*       install fh ["dllpcre_stubs.dll"] *)
(*         (Filename.concat ocamllib "stublibs"); *)
    end;
    (*
     * Examples compilation scripts
     *)
    let is_ml = Str.regexp "^.*\\.ml$" in
    let examples = "..\\examples" in
    let ex = Array.map (Filename.concat examples) (Sys.readdir examples) in
    Array.iter (fun d ->
      let bat = Filename.concat d "make.bat" in
      make_bat ~fname:bat  begin fun fh ->
        let dir = Array.to_list(Sys.readdir d) in
        match List.filter (fun f -> Str.string_match is_ml f 0) dir with
        | [] -> fprintf fh "@REM No .ml files in this directory!!!\n"
        | (name :: _) as ml ->
            let name = Filename.chop_extension name in
            let ml = String.concat " " (List.map Filename.quote ml) in
            fprintf fh "ocamlc -o %s.exe -I +pcre pcre.cma %s\n"
              name ml;
            fprintf fh "ocamlopt -o %s.com -I +pcre pcre.cmxa %s\n"
              name ml;
      end
    ) ex;
  end

