/***************************************************************************
 *   Copyright (C) 2005 - 2006 by Christian Muehlhaeuser, Last.fm Ltd.     *
 *   chris@last.fm                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include <QtGui>

#include "playback.h"
#include "playlist.h"
#include "searchresultdialog.h"
#include "settings.h"
#include "stationdialog.h"
#include "webserviceconnector.h"


StationDialog::StationDialog( QWidget *parent )
    : QDialog( parent )
{
    ui.setupUi( this );
    ui.customSearchBox->setImage( QApplication::applicationDirPath() + "/data/watermarks/radio.png" );
    ui.searchBox->setImage( QApplication::applicationDirPath() + "/data/watermarks/search.png" );

    menu = new QMenu( this );
    deleteAction = new QAction( tr( "&Delete item" ), this );
    resetAction  = new QAction( tr( "&Reset counter" ), this );
    clearAction  = new QAction( tr( "&Clear history" ), this );

    menu->addAction( deleteAction );
    menu->addAction( resetAction );
    menu->addSeparator();
    menu->addAction( clearAction );

    QStringList headerLabels;
    headerLabels.append( "Station" );
    headerLabels.append( "Listens" );

    ui.historyTree->setHeaderLabels( headerLabels );
    ui.historyTree->setRootIsDecorated( true );
    ui.historyTree->header()->resizeSection( 0, width() - 132 );
    ui.historyTree->headerItem()->setTextAlignment( 1, Qt::AlignHCenter );

    connect( deleteAction, SIGNAL( triggered() ), this, SLOT( deleteItem() ) );
    connect( resetAction, SIGNAL( triggered() ), this, SLOT( resetItem() ) );
    connect( clearAction, SIGNAL( triggered() ), this, SLOT( clearItems() ) );

    connect( ui.personalButton, SIGNAL( clicked() ), this, SLOT( personalStation() ) );
    connect( ui.neighbourButton, SIGNAL( clicked() ), this, SLOT( neighbourStation() ) );
    connect( ui.lovedButton, SIGNAL( clicked() ), this, SLOT( lovedStation() ) );

    connect( ui.tagsList, SIGNAL( itemDoubleClicked( QListWidgetItem* ) ), this, SLOT( changeStationThreaded( QListWidgetItem* ) ) );
    connect( ui.historyTree, SIGNAL( itemDoubleClicked( QTreeWidgetItem*, int ) ), this, SLOT( changeStationThreaded( QTreeWidgetItem*, int ) ) );
    connect( ui.historyTree, SIGNAL( customContextMenuRequested ( QPoint ) ), this, SLOT( historyMenu( QPoint ) ) );
    connect( ui.playlistRemoveButton, SIGNAL( clicked() ), this, SLOT( playlistRemove() ) );
    connect( ui.playlistClearButton, SIGNAL( clicked() ), this, SLOT( playlistClear() ) );

    connect( ui.searchCombo, SIGNAL( activated( int ) ), this, SLOT( searchModeChanged( int ) ) );
    connect( ui.searchButton, SIGNAL( clicked() ), this, SLOT( search() ) );
    connect( ui.customSearchEdit, SIGNAL( returnPressed() ), this, SLOT( customStation() ) );
    connect( ui.customListenButton, SIGNAL( clicked() ), this, SLOT( customStation() ) );

    connect( WebserviceConnector::instance(), SIGNAL( changeStationPrepared() ), this, SLOT( changeStationPrepared() ) );
    connect( WebserviceConnector::instance(), SIGNAL( tagsForUserResult( QStringList ) ), this, SLOT( tagsForUserResult( QStringList ) ) );

    updateInformation();
}


StationDialog::~StationDialog()
{
    delete menu;
    delete deleteAction;
    delete resetAction;
    delete clearAction;
}


void
StationDialog::resizeEvent ( QResizeEvent *event )
{
    ui.historyTree->header()->resizeSection( 0, width() - 132 );
}


void
StationDialog::updateInformation()
{
    updateHistory();
    updatePlaylist();

    ui.personalButton->setEnabled( WebserviceConnector::instance()->isSubscriber() );
    ui.lovedButton->setEnabled( WebserviceConnector::instance()->isSubscriber() );
    ui.tagsList->clear();

    if ( WebserviceConnector::instance()->isSubscriber() )
        WebserviceConnector::instance()->tagsForUser( Settings::instance()->username() );
}


void
StationDialog::updateHistory()
{
    ui.historyTree->clear();

    QTreeWidgetItem *personal = new QTreeWidgetItem( ui.historyTree, 0 );
    personal->setText( 0, "Personal Stations" );

    QTreeWidgetItem *profile = new QTreeWidgetItem( ui.historyTree, 0 );
    profile->setText( 0, "Neighbour Stations" );

    QTreeWidgetItem *loved = new QTreeWidgetItem( ui.historyTree, 0 );
    loved->setText( 0, "Loved Tracks Stations" );

    QTreeWidgetItem *groups = new QTreeWidgetItem( ui.historyTree, 0 );
    groups->setText( 0, "Group Stations" );

    QTreeWidgetItem *similar = new QTreeWidgetItem( ui.historyTree, 0 );
    similar->setText( 0, "Similar Artist Stations" );

    QTreeWidgetItem *tags = new QTreeWidgetItem( ui.historyTree, 0 );
    tags->setText( 0, "Global Tag Stations" );

    QTreeWidgetItem *usertags = new QTreeWidgetItem( ui.historyTree, 0 );
    usertags->setText( 0, "Personal Tag Stations" );

    QTreeWidgetItem *custom = new QTreeWidgetItem( ui.historyTree, 0 );
    custom->setText( 0, "Custom Stations" );

    QStringList stations = Settings::instance()->stationItems();
    for ( int i = 0; i < stations.size(); i++ )
    {
        QTreeWidgetItem *parent = 0;
        QStringList parts = stations.at( i ).split( "/", QString::SkipEmptyParts );
        QString group = parts.at( 0 );

        if ( parts.size() > 2 && group != "usertags" )
            group = parts.at( 2 );

        if ( group == "similarartists" ) parent = similar;
        if ( group == "personal" )       parent = personal;
        if ( group == "neighbours" )     parent = profile;
        if ( group == "group" )          parent = groups;
        if ( group == "globaltags" )     parent = tags;
        if ( group == "usertags" )       parent = usertags;
        if ( group == "loved" )          parent = loved;
        if ( group == "artists" )        parent = custom;

        if ( parent )
        {
            QString stationName = Settings::instance()->stationName( stations.at( i ) );
            if ( parent != custom || !stationName.isEmpty() )
            {
                if ( stationName.isEmpty() )
                {
                    if ( group == "usertags" )
                        stationName = QUrl::fromPercentEncoding( parts.at( 2 ).toUtf8() );
                    else
                        stationName = QUrl::fromPercentEncoding( parts.at( 1 ).toUtf8() );
                }

                QTreeWidgetItem *subitem = new QTreeWidgetItem( parent, 0 );

                subitem->setText( 0, stationName );
                subitem->setText( 1, QString().setNum( Settings::instance()->stationCounter( stations.at( i ) ) ) );
                subitem->setText( 2, stations.at( i ) );
                subitem->setTextAlignment( 1, Qt::AlignHCenter );
            }
        }
    }

    ui.historyTree->setItemExpanded( similar, true );
    ui.historyTree->setItemExpanded( personal, true );
    ui.historyTree->setItemExpanded( profile, true );
    ui.historyTree->setItemExpanded( groups, true );
    ui.historyTree->setItemExpanded( tags, true );
    ui.historyTree->setItemExpanded( usertags, true );
    ui.historyTree->setItemExpanded( loved, true );
    ui.historyTree->setItemExpanded( custom, true );

    if ( similar->childCount() == 0 )  delete similar;
    if ( personal->childCount() == 0 ) delete personal;
    if ( profile->childCount() == 0 )  delete profile;
    if ( groups->childCount() == 0 )   delete groups;
    if ( tags->childCount() == 0 )     delete tags;
    if ( usertags->childCount() == 0 ) delete usertags;
    if ( loved->childCount() == 0 )    delete loved;
    if ( custom->childCount() == 0 )   delete custom;
}


void
StationDialog::tagsForUserResult( const QStringList &result )
{
    ui.tagsList->clear();

    for ( int i = 0; i < result.count(); i++ )
    {
        ui.tagsList->addItem( result[i] );
    }
}


void
StationDialog::websiteSearch()
{
    Settings::instance()->startBrowser( "http://www.last.fm/radio/" );
}


void
StationDialog::searchModeChanged( int index )
{
    switch ( index )
    {
        case 0:
            ui.infoLabel->setText( "Last.fm can play radio stations by finding music similar to a given artist. Find a station by typing an artist name above." );
            ui.searchLabel->setText( "Artist:" );
            break;

        case 1:
            ui.infoLabel->setText( "Last.fm can play radio stations by finding music tagged by our users. Find a station by typing a tag name above." );
            ui.searchLabel->setText( "Tag:" );
            break;
    }
}


void
StationDialog::search()
{
    SearchResultDialog *d = new SearchResultDialog( this, ui.searchCombo->currentIndex(), ui.searchEdit->text() );
    d->show();
}


void
StationDialog::personalStation()
{
    WebserviceConnector::instance()->changeStation( "user/" + Settings::instance()->username() + "/personal" );
}


void
StationDialog::neighbourStation()
{
    WebserviceConnector::instance()->changeStation( "user/" + Settings::instance()->username() + "/neighbours" );
}


void
StationDialog::lovedStation()
{
    WebserviceConnector::instance()->changeStation( "user/" + Settings::instance()->username() + "/loved" );
}


void
StationDialog::customStation()
{
    QString token;
    QStringList artists = ui.customSearchEdit->toPlainText().split( "," );

    for ( int i = 0; i < artists.count(); i++ )
        token += ( i > 0 ? "," : "" ) + artists.at( i ).simplified();

    WebserviceConnector::instance()->changeStation( "artistnames/" + QUrl( token ).toEncoded() );
}


void
StationDialog::updatePlaylist()
{
    ui.playList->clear();
    playlistData.first.clear();
    playlistData.second.clear();

    Playlist *playlist = Playback::instance()->playlist();
    if ( !playlist )
        return;

    if ( playlist->count() )
    {
        ui.playlistRemoveButton->setEnabled( true );
        ui.playlistClearButton->setEnabled( true );
    }
    else
    {
        ui.playlistRemoveButton->setEnabled( false );
        ui.playlistClearButton->setEnabled( false );
    }

    for ( int i = 0; i < playlist->count(); i++ )
    {
        QPair<QPair<QString, QString>, QString> track = playlist->trackAt( i );

        playlistData.first.append( track.first.second );
        playlistData.second.append( track.second );
        ui.playList->addItem( track.first.first );
    }
}


void
StationDialog::playlistRemove()
{
    if ( ui.playList->currentItem() )
    {
        QString id = playlistData.first.at( ui.playList->currentRow() );

        WebserviceConnector::instance()->playlistRemove( id );
        ui.playList->takeItem( ui.playList->currentRow() );
    }

    if ( ui.playList->count() == 0 )
    {
        ui.playlistRemoveButton->setEnabled( false );
        ui.playlistClearButton->setEnabled( false );
    }
}


void
StationDialog::playlistClear()
{
    ui.playList->clear();
    WebserviceConnector::instance()->playlistClear();

    ui.playlistRemoveButton->setEnabled( false );
    ui.playlistClearButton->setEnabled( false );
}


void
StationDialog::changeStationHistory()
{
    WebserviceConnector::instance()->changeStation( ui.historyTree->currentItem()->text( 2 ) );
}


void
StationDialog::changeStationTags()
{
    WebserviceConnector::instance()->changeStation( QString( "lastfm://usertags/%1/%2" )
                                                                                   .arg( QString( QUrl( Settings::instance()->username() ).toEncoded() ) )
                                                                                   .arg( QString( QUrl( ui.tagsList->currentItem()->text() ).toEncoded() ) ) );
}


void
StationDialog::changeStationThreaded( QTreeWidgetItem *item, int column )
{
    QTimer::singleShot( 0, this, SLOT( changeStationHistory() ) );
}


void
StationDialog::changeStationThreaded( QListWidgetItem *item )
{
    QTimer::singleShot( 0, this, SLOT( changeStationTags() ) );
}


void
StationDialog::changeStationPrepared()
{
    close();
    updateHistory();
}


void
StationDialog::historyMenu( const QPoint & pos )
{
    if ( ui.historyTree->currentItem() && !ui.historyTree->currentItem()->childCount() )
        menu->exec( QCursor::pos() );
}


void
StationDialog::deleteItem()
{
    if ( ui.historyTree->currentItem() )
    {
        Settings::instance()->deleteStationCounter( ui.historyTree->currentItem()->text( 2 ) );
        delete ui.historyTree->currentItem();
    }
}


void
StationDialog::resetItem()
{
    if ( ui.historyTree->currentItem() )
    {
        Settings::instance()->resetStationCounter( ui.historyTree->currentItem()->text( 2 ) );
        ui.historyTree->currentItem()->setText( 1, "0" );
    }
}


void
StationDialog::clearItems()
{
    if ( QMessageBox::question( this, "Clear history?", "This will clear your entire station history! Are you sure?",
                                      QMessageBox::Yes, QMessageBox::No ) )
    {
        Settings::instance()->clearStationCounters();
        ui.historyTree->clear();
    }
}

