#ifndef POWERKADU_H
#define POWERKADU_H

/*!
 * \mainpage PowerKadu
 * This is API documentation is linked to Qt and Kadu documentation to ease
 * browsing. See "Classes Hierarchy"  in "Classes" tab for details.
 */

#include <qobject.h>
#include <qcolor.h>

#define MODULE_POWERKADU_VERSION "0.3-svn"

class ConfigFile;
class WordFix;
class Translator;
class AnonymousCheck;
class Infos;
class CmdLine;
class Chat;
class UserGroup;
class EKGCmds;
class Antistring;
class QPopupMenu;
class IHideFromU;
class Autostatus;
class Split;
class MimeTeX;
class AutoHide;
class Cenzor;
class ParserExtender;

/*!
 * Contains all subclasses for handling precise problems.
 * Also manages PowerKadu tab name in configuration dialog
 * and config file name.
 * \class PowerKadu
 * \brief Main PowerKadu class.
 */
class PowerKadu : public QObject
{
	Q_OBJECT

	private:
		/*!
		 * \var ConfigFile* config
		 * Configuration file object pointer.
		 */
		ConfigFile* config;

		/*!
		 * \var WordFix *wordFix
		 * Pointer to object handling words fixing.
		 */
		WordFix *wordFix;

		/*!
		 * \var Translator *morse
		 * A Morse code translator
		 */
		Translator *morse;

		/*!
		 * \var Translator *basic1337
		 * A basic 1337 code translator
		 */
		Translator *basic1337;

		/*!
		 * \var Translator *alt1337
		 * An alternative 1337 code translator
		 */
		Translator *alt1337;
		
		/*!
		 * \var AnonymousCheck *anonymousCheck
		 * An instance of AnonymousCheck
		 * \see AnonymousCheck
		 */
		AnonymousCheck *anonymousCheck;

		/*!
		 * \var Infos *infos
		 * An instance of Infos class. Handles "Last seen, IP, DNS".
		 */
		Infos *infos;
		
		/*!
		 * \var CmdLine *cmdLine
		 * Command line handler for chat windows.
		 */
		CmdLine *cmdLine;

		/*!
		 * \var EKGCmds *ekgCmds;
		 * EKG commands.
		 */
		EKGCmds *ekgCmds;
		
		/*!
		 * \var IHideFromU *iHideFromU
		 * An instance of IHideFromU.
		 * \see IHideFromU
		 */
		IHideFromU *iHideFromU;
		
		/*!
		 * \var AutoHide *autoHide
		 * An instance of AutoHide
		 * \see AutoHide
		 */
		AutoHide *autoHide;
		/*!
		 * \fn QString HTMLColorStr(const QColor &color)
		 * This function returns the HTML color value, e. g.: "#0EB367", quotation marks are included.
		 * \return Returns the string containing the color value.
		 */
		QString HTMLColorStr(const QColor &color);
		
		/*!
		 * \fn void createMainMenu()
		 * Creates main PowerKadu menu.
		 */
		void createMainMenu();

		Antistring *antistring;
		
		/*!
		 * \var QPopupMenu *menu
		 * holds a pointer to PowerKadu main menu.
		 * \see mainMenu()
		 */
		QPopupMenu *menu;
		
		/*!
		
		
		
		 */
		Autostatus *autostatus;
		
		/*!
		 * \var Split *split
		 * An nstance of Split class. Splits messages longer than 2000 characters.
		 */
		Split *split;

		/*!
		 * \var MimeTeX *mimeTeX
		 * An instance of MimeTeX class. Handles the TeX formula creation action.
		 */
		MimeTeX *mimeTeX;

		/*!
		 * \var Cenzor *cenzor
		 * An instance of Cenzor class. To look up swearwords.
		 */
		Cenzor *cenzor;
		
		/*!
		 * \var ParserExtender *parserExtender
		 * An instance of ParserExtender class. Extends KaduParser.
		 */
		ParserExtender *parserExtender;

	public:
		/*!
		 * \fn PowerKadu()
		 * Default constructor. Initializes whole internal data and settings.
		 */
		PowerKadu();

		/*!
		 * \fn ~PowerKadu()
		 * Default destructor.
		 */
		~PowerKadu();

		/*!
		 * \fn ConfigFile* cfg()
		 * Gives direct access to configuration file handler.
		 * \return Pointer to configuration file handler object.
		 */
		ConfigFile* cfg();

		/*!
		 * \fn CmdLine* cmds()
		 * Gives direct access to command line handler
		 * to allow a commands adding.
		 * \return Pointer to command line handler object.
		 */
		CmdLine* cmds();

		/*!
		 * \fn void showPkMsg(Chat* chat, QString msg)
		 * Adds given message to given chat window, marked
		 * as PK message.
		 * \param chat Chat window to show message in.
		 * \param msg Message content to add.
		 */
		void showPkMsg(Chat* chat, QString msg);

		/*!
		 * \overload void showPkMsg(UserGroup *users, QString msg)
		 * \param users Users which chat window is opened for.
		 * \param msg Message content to add.
		 */
		void showPkMsg(UserGroup *users, QString msg);

		/*!
		 * \fn static QString PK()
		 * String generated by this method should be prepended
		 * to all string displayed in chat window messages by PK.
		 * \return HTML-formatted string which identifies chat window messages
		 * that comes from PowerKadu. Its contents are: "&lt;b&gt;[PK]&lt;/b&gt;".
		 */
		static QString PK();

		/*!
		 * \fn void init()
		 * Initializes module after main PK object has been created.
		 * It calls functions that requires powerKadu object to already exists.
		 */
		void init();
		
		/*!
		 * \fn QPopupMenu *mainMenu()
		 * Lets you access the PowerKadu's main menu.
		 * \return pointer to main menu.
		 */
		QPopupMenu *mainMenu();

	private slots:
		/*!
		 * \fn void onAboutPowerKadu()
		 * Shoows "About PowerKadu..." dialog
		 */
		void onAboutPowerKadu();
	
	public slots:
		/*!
		 * \fn void onCreateConfig()
		 * Increments module usage of config dialog.
		 */
		void onCreateConfig();

		/*!
		 * \fn void onDestroyConfig()
		 * Decrements module usage of config dialog and saves changes.
		 */
		void onDestroyConfig();

		/*!
		 * \fn void onApplyConfig()
		 * Saves changes in config dialog.
		 */
		void onApplyConfig();
};

/*!
 * \var extern PowerKadu* powerKadu
 * The main PowerKadu object pointer.
 */
extern PowerKadu* powerKadu;

#endif
