/*
 * client.h - declaration of class client which represents a client and displays the X-screen of a client and allows remote control
 *
 * iTALC
 * Copyright (c) 2004-2005 Tobias Doerffel <tobias@doerffel.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program (see COPYING); if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 */


#ifndef _CLIENT_H
#define _CLIENT_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qwidget.h>
#include <qimage.h>
#include <qmutex.h>

#include "system_environment.h"



/*!
  \brief The Client

  The client is the real interesting thing in iTALC, so the client-class contains all code for actual actions, like
  command-execution or SSH-Login and so on...

  \sa   clientManager for detailed description of management of clients
*/




class QPopupMenu;

class rfbConnection;

class classRoom;
class classRoomItem;
class client;


typedef void( client:: * execCmd )( const QString & );

const QString CONFIRM_NO = "n";
const QString CONFIRM_YES = "y";



class client : public QWidget
{
	Q_OBJECT
public:
	client( const QString & _ip, const QString & _mac, const QString & _name, classRoom * _class_room, int _id = -1 );
	~client();

	enum clientCmds
	{
		NONE, RELOAD, START_FULLSCREEN_DEMO, START_DEMO, STOP_DEMO, CLIENT_DEMO, WORK_ON_CLIENT, SEND_MESSAGE,
		DISTRIBUTE_FILE, COLLECT_FILES,
		LOCK_X, UNLOCK_X,
#ifdef COMPILE_LINUX
		RESTART_X,
#endif
		KILL_GAMES, KILL_BROWSERS, SCREENSHOT, EXEC_CMDS, EXEC_CMDS_IRFB,
#ifdef COMPILE_LINUX
		RUN_X_APP,
#endif
		SSH_LOGIN, POWER_ON, REBOOT, POWER_OFF,
		CMD_COUNT	// last enum-constant specifies number of actual used constants
	} ;

	int id( void ) const;

	inline QString fullName( void ) const
	{
		return( m_name + " (" + m_ip + ")" );
	}
	inline QString name( void ) const
	{
		return( m_name );
	}
	inline const QString & ip( void ) const
	{
		return( m_ip );
	}
	inline const QString & mac( void ) const
	{
		return( m_mac );
	}
	inline const QString & user( void ) const
	{
		return( m_user );
	}
	inline bool demoRunning( void ) const
	{
		return( m_demoRunning );
	}
	inline classRoomItem * getClassRoomItem( void )
	{
		return( m_classRoomItem );
	}

	inline void setIP( const QString & _ip )
	{
		m_ip = _ip;
	}
	inline void setMac( const QString & _mac )
	{
		m_mac = _mac;
	}
	inline void setName( const QString & _name )
	{
		m_name = _name;
	}
	inline void setClassRoomItem( classRoomItem * _i )
	{
		m_classRoomItem = _i;
	}
	void setClassRoom( classRoom * _cr );

	bool inviteForSupportingLocalUser( void );

	void resetConnection( void );

	virtual void update( void );

	void createActionMenu( QPopupMenu * _m );


	void processCmd( clientCmds _cmd, const QString & _u_data = "" );

	static inline bool reloadScreenshotList( void )
	{
		return( s_reloadScreenshotList );
	}
	static inline void resetReloadOfScreenshotList( void )
	{
		s_reloadScreenshotList = FALSE;
	}
	static client * clientFromID( int _id );


	float m_rasterX;
	float m_rasterY;


	struct clientCommand
	{
		// stuff for cmd-exec
		clientCmds	m_cmd;
		execCmd 	m_exec;
		// stuff for UI
		QString		m_name;
		QString		m_icon;
		bool		m_insertSep;
	} ;


	static const clientCommand s_commands[CMD_COUNT];



public slots:
	void processCmdSlot( int _cmd );


private:
	bool userLoggedIn( void );

	void contextMenuEvent( QContextMenuEvent * _cme );
	void mouseDoubleClickEvent( QMouseEvent * _me );
	void paintEvent( QPaintEvent * _pe );
	void closeEvent( QCloseEvent * _ce );
	void hideEvent( QHideEvent * _he );
	void showEvent( QShowEvent * _se );
	void resizeEvent( QResizeEvent * _re );

	// action-handlers
	inline void doNothing( const QString & )
	{
	}
	void reload( const QString & _update = CONFIRM_YES );
	void startFullScreenDemo( const QString & );
	void startDemo( const QString & );
	void stopDemo( const QString & );
	void clientDemo( const QString & );
	void workOnClient( const QString & );
	void sendMessage( const QString & _msg );
	void distributeFile( const QString & _file );
	void collectFiles( const QString & _name_filter );
	void lockX( const QString & _confirm );
	void unlockX( const QString & _confirm );
#ifdef COMPILE_LINUX
	void restartX( const QString & _confirm );
#endif
	void killGames( const QString & );
	void killBrowsers( const QString & _confirm );
	void screenShot( const QString & );
	void execCmds( const QString & _cmds );
	void execCmdsIRFB( const QString & _cmds );
#ifdef COMPILE_LINUX
	void runXApp( const QString & _app );
#endif
	void SSHLogin( const QString & _user );
	void powerOn( const QString & );
	void reboot( const QString & _confirm );
	void powerOff( const QString & _confirm );


	rfbConnection * m_connection;

	QString m_ip;
	QString m_mac;
	QString m_name;
	QString m_user;
	bool m_demoRunning;
	QImage m_msgImg;
	volatile bool m_makeScreenshot;

	QMutex m_syncMutex;
	QMutex m_msgImgMutex;

	classRoomItem * m_classRoomItem;

	static int freeID( void );


	static QImage * s_computerPowerOffImg;
	static QImage * s_noUserLoggedInImg;
	static QImage * s_demoRunningImg;

	static bool s_reloadScreenshotList;

	static QMap<int, client *> s_idMap;


	friend void systemEnvironment::distributeFile( const QString & _file, client * _client );
	friend void systemEnvironment::collectFiles( const QString & _files, const QString & _user );

} ;


#endif
