/**
* Copyright 2005-2007 ECMWF
*
* Licensed under the GNU Lesser General Public License which
* incorporates the terms and conditions of version 3 of the GNU
* General Public License.
* See LICENSE and gpl-3.0.txt for details.
*/

#include "grib_api_internal.h"
/*
  This is used by make_class.pl

  START_CLASS_DEF
  CLASS      = accessor
  SUPER      = grib_accessor_class_long
  IMPLEMENTS = unpack_long;pack_long
  IMPLEMENTS = init;dump
  MEMBERS = const char* codedStep
  MEMBERS = const char* codedUnits
  MEMBERS = const char* stepUnits

  END_CLASS_DEF

 */

/* START_CLASS_IMP */

/*

Don't edit anything between START_CLASS_IMP and END_CLASS_IMP
Instead edit values between START_CLASS_DEF and END_CLASS_DEF
or edit "accessor.class" and rerun ./make_class.pl

*/

static int pack_long(grib_accessor*, const long* val,size_t *len);
static int unpack_long(grib_accessor*, long* val,size_t *len);
static void dump(grib_accessor*, grib_dumper*);
static void init(grib_accessor*,const long, grib_arguments* );
static void init_class(grib_accessor_class*);

typedef struct grib_accessor_step_in_units {
    grib_accessor          att;
/* Members defined in gen */
/* Members defined in long */
/* Members defined in step_in_units */
	const char* codedStep;
	const char* codedUnits;
	const char* stepUnits;
} grib_accessor_step_in_units;

extern grib_accessor_class* grib_accessor_class_long;

static grib_accessor_class _grib_accessor_class_step_in_units = {
    &grib_accessor_class_long,                      /* super                     */
    "step_in_units",                      /* name                      */
    sizeof(grib_accessor_step_in_units),  /* size                      */
    0,                           /* inited */
    &init_class,                 /* init_class */
    &init,                       /* init                      */
    0,                  /* post_init                      */
    0,                    /* free mem                       */
    &dump,                       /* describes himself         */
    0,                /* get length of section     */
    0,                /* get number of values      */
    0,                 /* get number of bytes      */
    0,                /* get offset to bytes           */
    0,            /* get native type               */
    0,                /* get sub_section                */
    0,               /* grib_pack procedures long      */
    0,               /* grib_pack procedures long      */
    &pack_long,                  /* grib_pack procedures long      */
    &unpack_long,                /* grib_unpack procedures long    */
    0,                /* grib_pack procedures double    */
    0,              /* grib_unpack procedures double  */
    0,                /* grib_pack procedures string    */
    0,              /* grib_unpack procedures string  */
    0,                 /* grib_pack procedures bytes     */
    0,               /* grib_unpack procedures bytes   */
    0,            /* pack_expression */
    0,              /* notify_change   */
    0,                /* update_size   */
    0,            /* preferred_size   */
    0,                    /* resize   */
    0,      /* nearest_smaller_value */
    0,                       /* next accessor    */
    0,                    /* compare vs. another accessor   */
    0,     /* unpack only ith value          */
    0,             		/* clear          */
};


grib_accessor_class* grib_accessor_class_step_in_units = &_grib_accessor_class_step_in_units;


static void init_class(grib_accessor_class* c)
{
	c->next_offset	=	(*(c->super))->next_offset;
	c->value_count	=	(*(c->super))->value_count;
	c->byte_count	=	(*(c->super))->byte_count;
	c->byte_offset	=	(*(c->super))->byte_offset;
	c->get_native_type	=	(*(c->super))->get_native_type;
	c->sub_section	=	(*(c->super))->sub_section;
	c->pack_missing	=	(*(c->super))->pack_missing;
	c->is_missing	=	(*(c->super))->is_missing;
	c->pack_double	=	(*(c->super))->pack_double;
	c->unpack_double	=	(*(c->super))->unpack_double;
	c->pack_string	=	(*(c->super))->pack_string;
	c->unpack_string	=	(*(c->super))->unpack_string;
	c->pack_bytes	=	(*(c->super))->pack_bytes;
	c->unpack_bytes	=	(*(c->super))->unpack_bytes;
	c->pack_expression	=	(*(c->super))->pack_expression;
	c->notify_change	=	(*(c->super))->notify_change;
	c->update_size	=	(*(c->super))->update_size;
	c->preferred_size	=	(*(c->super))->preferred_size;
	c->resize	=	(*(c->super))->resize;
	c->nearest_smaller_value	=	(*(c->super))->nearest_smaller_value;
	c->next	=	(*(c->super))->next;
	c->compare	=	(*(c->super))->compare;
	c->unpack_double_element	=	(*(c->super))->unpack_double_element;
	c->clear	=	(*(c->super))->clear;
}
/* END_CLASS_IMP */

static void init(grib_accessor* a,const long l, grib_arguments* c)
{
  grib_accessor_step_in_units* self = (grib_accessor_step_in_units*)a;
  int n = 0;

  self->codedStep = grib_arguments_get_name(a->parent->h,c,n++);
  self->codedUnits  = grib_arguments_get_name(a->parent->h,c,n++);
  self->stepUnits  = grib_arguments_get_name(a->parent->h,c,n++);

}

static void dump(grib_accessor* a, grib_dumper* dumper)
{
  grib_dump_double(dumper,a,NULL);

}
static int u2s[] =  {
  60,      /* (0) minutes */
  3600,    /* (1) hour    */
  86400,   /* (2) day     */
  -1, -1, -1, -1, -1, -1,-1,
  10800,   /* (10) 3 hours */
  21600,   /* (11) 6 hours */
  43200,   /* (12) 12 hours */
  1        /* (13) seconds  */
};

static int unpack_long(grib_accessor* a, long* val, size_t *len)
{
  grib_accessor_step_in_units* self = (grib_accessor_step_in_units*)a;
  int err = 0;
  long codedStep,codedUnits,stepUnits;
  grib_handle* h=a->parent->h;

  
  if((err = grib_get_long_internal(h,self->codedUnits,&codedUnits))) return err;
  if((err = grib_get_long_internal(h,self->stepUnits,&stepUnits))) return err;
  if((err = grib_get_long_internal(h,self->codedStep,&codedStep))) return err;

  if (stepUnits!=codedUnits) {
    *val=codedStep*u2s[codedUnits];
    if (*val%u2s[stepUnits]!=0) {
      err = grib_set_long_internal(h,self->stepUnits,codedUnits);
	  *val=codedStep;
	  return err;
    }
    *val = *val / u2s[stepUnits];
  } else *val=codedStep;

  return GRIB_SUCCESS;
}

static int pack_long(grib_accessor* a, const long* val, size_t *len)
{
  grib_accessor_step_in_units* self = (grib_accessor_step_in_units*)a;
  grib_handle* h=a->parent->h;
  int err = 0;
  long codedStep,codedUnits,stepUnits;
  char codedUnitsString[20]={0,};
  char stepUnitsString[20]={0,};
  size_t codedUnitsStringLength=sizeof(codedUnitsString);
  size_t stepUnitsStringLength=sizeof(stepUnitsString);
  
  if((err = grib_get_long_internal(h,self->codedUnits,&codedUnits))) return err;
  if((err = grib_get_long_internal(h,self->stepUnits,&stepUnits))) return err;

  if (stepUnits!=codedUnits) {
    codedStep=*val * u2s[stepUnits];
    if (codedStep%u2s[codedUnits]!=0) {
      if((err = grib_get_string_internal(h,self->codedUnits,
          codedUnitsString,&codedUnitsStringLength))) return err;
      if((err = grib_get_string_internal(h,self->stepUnits,
          stepUnitsString,&stepUnitsStringLength))) return err;
      grib_context_log(h->context,GRIB_LOG_ERROR,
                       "unable to convert %s=%ld %s in %s",
                       a->name,*val,
                       stepUnitsString,codedUnitsString);
      return GRIB_WRONG_STEP_UNIT;
    }
    codedStep = codedStep / u2s[codedUnits];
  } else codedStep=*val;
  
  return grib_set_long_internal(a->parent->h,self->codedStep,codedStep);
}

