
/**
 * \file execute.c
 *
 * \brief Low level SQLite sql execute.
 *
 * This program is free software under the GNU General Public License
 * (>=v2). Read the file COPYING that comes with GRASS for details.
 *
 * \author Radim Blazek
 * \author Antonio Galea
 *
 * \date 2005-2007
 */

#include <stdlib.h>
#include <grass/gis.h>
#include <grass/dbmi.h>
#include "globals.h"
#include "proto.h"


/**
 * \fn int db__driver_execute_immediate (dbString *sql)
 *
 * \brief Low level SQLite execute sql text.
 *
 * \param[in] sql SQL statement
 * \return int DB_FAILED on error; DB_OK on success
 */

int db__driver_execute_immediate(dbString * sql)
{
    char *s;
    int ret;
    sqlite3_stmt *stmt;
    const char *rest;

    s = db_get_string(sql);

    G_debug(3, "execute: %s", s);

    ret = sqlite3_prepare(sqlite, s, -1, &stmt, &rest);

    if (ret != SQLITE_OK) {
	append_error("Error in sqlite3_prepare():\n");
	append_error((char *)sqlite3_errmsg(sqlite));
	report_error();
	return DB_FAILED;
    }

    ret = sqlite3_step(stmt);

    if (ret != SQLITE_DONE) {
	append_error("Error in sqlite3_step():\n");
	append_error((char *)sqlite3_errmsg(sqlite));
	report_error();
	return DB_FAILED;
    }

    ret = sqlite3_finalize(stmt);

    if (ret != SQLITE_OK) {
	append_error("Error in sqlite3_finalize():\n");
	append_error((char *)sqlite3_errmsg(sqlite));
	report_error();
	return DB_FAILED;
    }

    /*
       if ( rest )
       G_free ( rest );
     */

    return DB_OK;
}


/**
 * \fn int db__driver_begin_transaction (void)
 *
 * \brief Low level SQLite begin SQL transaction.
 *
 * \return int DB_FAILED on error; DB_OK on success
 */

int db__driver_begin_transaction(void)
{
    int ret;

    G_debug(3, "execute: BEGIN");

    ret = sqlite3_exec(sqlite, "BEGIN", NULL, NULL, NULL);
    if (ret != SQLITE_OK) {
	append_error("Cannot 'BEGIN' transaction:\n");
	append_error((char *)sqlite3_errmsg(sqlite));
	report_error();
	return DB_FAILED;
    }

    return DB_OK;
}


/**
 * \fn int db__driver_commit_transaction (void)
 *
 * \brief Low level SQLite commit transaction.
 *
 * \return int DB_FAILED on error; DB_OK on success
 */

int db__driver_commit_transaction(void)
{
    int ret;

    G_debug(3, "execute: COMMIT");

    ret = sqlite3_exec(sqlite, "COMMIT", NULL, NULL, NULL);
    if (ret != SQLITE_OK) {
	append_error("Cannot 'COMMIT' transaction:\n");
	append_error((char *)sqlite3_errmsg(sqlite));
	report_error();
	return DB_FAILED;
    }

    return DB_OK;
}
