#!/usr/bin/env python
# -*- coding: utf-8 -*-
#
#  Epiphany extension: Universal Edit Button
#  Version 1 (2008-06-27)
#  Copyright (C) 2006 Jaime Frutos Morales <acidborg@gmail.com>
#  Copyright (C) 2008 Johannes Berg <johannes@sipsolutions.net>
#
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 2, or (at your option)
#  any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program; if not, write to the Free Software
#  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#

import epiphany
import pygtk
pygtk.require('2.0')
import gtk
from gettext import gettext as _
import os.path
import re
import urllib
from htmlentitydefs import name2codepoint as n2cp

# Create UEB icon from code generated using "gdk-pixbuf-csource --raw cc.png"
_icon = (""
  # Pixbuf magic (0x47646b50)
  "GdkP"
  # length: header (24) + pixel_data (1024)
  "\0\0\4\30"
  # pixdata_type (0x1010002)
  "\1\1\0\2"
  # rowstride (64)
  "\0\0\0@"
  # width (16)
  "\0\0\0\20"
  # height (16)
  "\0\0\0\20"
  # pixel_data:
  "\377\377\377\0\201\330\201\377\24\234\24\377\23\231\23\377\25\236\25"
  "\377\27\244\27\377\30\251\30\377\32\256\32\377\34\263\34\377\14H\14\377"
  "\0\0\0\377\237x\27\377\377\300%\377\377\300%\377\377\334\205\377\377"
  "\377\377\0\201\330\201\377\4p\4\377\11~\11\377\15\211\15\377\21\223\21"
  "\377\24\236\24\377\30\250\30\377\34\263\34\377\13E\13\377\0\0\0\377\237"
  "x\27\377\377\300%\377\377\300%\377\377\300%\377\335\245\36\377\243\225"
  "r\377\24\234\24\377\11~\11\377\15\211\15\377\21\223\21\377\24\236\24"
  "\377\30\250\30\377\34\263\34\377\13E\13\377\0\0\0\377\237x\27\377\377"
  "\300%\377\377\300%\377\377\300%\377\335\245\36\377S;\2\377G2\0\377\23"
  "\231\23\377\15\211\15\377\21\223\21\377\24\236\24\377\30\250\30\377\34"
  "\263\34\377\23u\23\377\0\0\0\377\237x\27\377\377\300%\377\377\300%\377"
  "\377\300%\377\335\245\36\377S;\2\377G2\0\377G2\0\377\25\236\25\377\21"
  "\223\21\377\24\236\24\377\30\250\30\377\34\263\34\377\40\275\40\377\4"
  "\30\4\377]YK\377\377\300%\377\377\300%\377\377\300%\377\335\245\36\377"
  "S;\2\377G2\0\377G2\0\377:)\0\377\27\244\27\377\24\236\24\377\30\250\30"
  "\377\34\263\34\377\40\275\40\377\32\224\32\377\0\0\0\377\330\320\257"
  "\377\377\300%\377\377\300%\377\335\245\36\377S;\2\377G2\0\377G2\0\377"
  ":)\0\377\4\3\0\377\30\251\30\377\30\250\30\377\34\263\34\377\40\275\40"
  "\377#\310#\377\11""3\11\377]YK\377\367\356\310\377\370\350\264\377\330"
  "\277z\377S;\2\377G2\0\377G2\0\377:)\0\377\4\3\0\377\14H\14\377\32\256"
  "\32\377\34\263\34\377\40\275\40\377#\310#\377\"\267\"\377\0\0\0\377\330"
  "\320\257\377\367\356\310\377\367\356\310\377\326\313\243\377G2\0\377"
  "G2\0\377:)\0\377\4\3\0\377\14H\14\377\35\266\35\377\34\263\34\377\40"
  "\275\40\377#\310#\377'\322'\377\20Q\20\377MJ\77\377\367\356\310\377\367"
  "\356\310\377\367\356\310\377\367\356\310\377\307\274\226\377PH2\377\4"
  "\3\0\377\14H\14\377\36\271\36\377\33\261\33\377\36\271\36\377#\310#\377"
  "'\322'\377(\316(\377\0\0\0\377\0\0\0\377.-&\377\367\356\310\377\350\337"
  "\274\377|wd\377\17\17\15\377\4\31\4\377\25y\25\377\36\271\36\377\32\256"
  "\32\377\31\254\31\377\40\276\40\377'\322'\377+\335+\377\27r\27\377\0"
  "\0\0\377\0\0\0\377\0\0\0\377lhX\377\17\17\15\377\12""5\12\377\34\231"
  "\34\377\"\303\"\377\36\271\36\377\32\256\32\377\27\244\27\377\30\247"
  "\30\377\"\303\"\377+\335+\377/\347/\377\3\17\3\377\0\0\0\377\0\0\0\377"
  "\0\0\0\377\20T\20\377$\274$\377%\315%\377\"\303\"\377\36\271\36\377\32"
  "\256\32\377\27\244\27\377\23\231\23\377\26\241\26\377$\310$\377/\347"
  "/\3772\3622\377\24^\24\377\3\17\3\377\30u\30\377*\324*\377)\330)\377"
  "%\315%\377\"\303\"\377\36\271\36\377\32\256\32\377\27\244\27\377\23\231"
  "\23\377\17\217\17\377\24\234\24\377#\306#\3772\3622\3776\3736\3774\370"
  "4\3771\3551\377-\343-\377)\330)\377%\315%\377\"\303\"\377\36\271\36\377"
  "\32\256\32\377\27\244\27\377\23\231\23\377\17\217\17\377\13\204\13\377"
  "\25\236\25\377\201\330\201\377%\314%\3774\3704\3771\3551\377-\343-\377"
  ")\330)\377%\315%\377\"\303\"\377\36\271\36\377\32\256\32\377\27\244\27"
  "\377\23\231\23\377\17\217\17\377\13\204\13\377\6t\6\377\201\330\201\377"
  "\377\377\377\0\201\330\201\377$\311$\377%\313%\377#\306#\377!\301!\377"
  "\37\274\37\377\35\266\35\377\33\261\33\377\31\254\31\377\30\247\30\377"
  "\26\241\26\377\24\234\24\377\25\236\25\377\201\330\201\377\377\377\377"
  "\0"
)
icon_size = gtk.icon_size_lookup(gtk.ICON_SIZE_MENU)
pixbuf = gtk.gdk.pixbuf_new_from_inline(len(_icon), _icon, False)
pixbuf = pixbuf.scale_simple(icon_size[0], icon_size[1], gtk.gdk.INTERP_BILINEAR)

ueb_re = re.compile(r'<link(\s+(rel="(?P<alt>alternate)"|type="(?P<type>application/(x-)?wiki)"|title="(?P<title>[^"]*)"|href="(?P<href>[^"]*)"))+\s*/?>', re.IGNORECASE | re.DOTALL | re.MULTILINE)
head_re = re.compile(r'<head>(.*?)</head>', re.IGNORECASE | re.DOTALL | re.MULTILINE)


def substitute_entity(match):
    ent = match.group(2)
    if match.group(1) == "#":
        return unichr(int(ent))
    else:
        cp = n2cp.get(ent)

        if cp:
            return unichr(cp)
        else:
            return match.group()

entity_re = re.compile("&(#?)(\d{1,5}|\w{1,8});")
def decode_htmlentities(string):
    return entity_re.subn(substitute_entity, string)[0]

def _switch_page_cb(notebook, page, page_num, window):
    ui_show(window, window.get_active_child())
    
def _load_status_cb(embed, data, window):
    if not embed.get_property('load-status'):
        # page is loaded
        detect_ueb(window, embed)
    else:
        embed._has_ueb = False
    ui_show(window, embed)
        
def _ueb_button_pressed(button, event, window):
    embed = window.get_active_child()
    ueb_url = embed._ueb_url
    embed.load_url(ueb_url)

def ui_init(window):
    ueb_image = gtk.Image()
    ueb_image.set_from_pixbuf(pixbuf)
    ueb_image.show()

    eventbox = gtk.EventBox()    
    eventbox.set_visible_window(False)
    eventbox.connect("button-press-event", _ueb_button_pressed, window)
    # Pack the widgets
    eventbox.add(ueb_image)

    eventbox.show()    

    statusbar = window.get_statusbar()
    statusbar.add_widget(eventbox)
    statusbar._ueb_eventbox = eventbox
    eventbox.set_tooltip_text('')

def ui_show(window, embed):
    if embed != window.get_active_child():
        return

    statusbar = window.get_statusbar()
    eventbox = statusbar._ueb_eventbox
    
    try:
        if embed._has_ueb:
            statusbar._ueb_eventbox.set_tooltip_text(embed._ueb_title)
            eventbox.show()
        else:
            eventbox.hide()
    except: 
        eventbox.hide()

def ui_destroy(window):
    statusbar = window.get_statusbar()
    eventbox = statusbar._ueb_eventbox
    del statusbar._ueb_eventbox
    statusbar.remove_widget(eventbox)

def detect_ueb(window, embed):
    # Get the HTML code
    persist = epiphany.ephy_embed_factory_new_object(epiphany.EmbedPersist)
    persist.set_flags(epiphany.EMBED_PERSIST_NO_VIEW |epiphany.EMBED_PERSIST_COPY_PAGE)
    persist.set_embed(embed)
    embed._has_ueb = False
    page_string = persist.to_string()
    ueb_url, ueb_title = get_ueb_data(page_string)
    if ueb_url is not None:
        cur = embed.get_property('address')
        if not '://' in ueb_url:
            if ueb_url[0] == '/':
                ueb_url = '/'.join(cur.split('/')[:3]) + ueb_url
            else:
                ueb_url = '/'.join(cur.split('/')[:-1]) + ueb_url
        embed._ueb_url = decode_htmlentities(ueb_url)
        embed._ueb_title = decode_htmlentities(ueb_title)
        embed._has_ueb = True

def get_ueb_data(page_string):
    match = head_re.search(page_string)
    if not match:
        return None, None
    head_string = match.group(1)
    for match in ueb_re.finditer(head_string):
        if not match.group('alt'):
            continue
        if not match.group('type'):
            continue
        href = match.group('href')
        if not href:
            continue
        title = match.group('title')
        return href, title
    return None, None

def attach_window(window):
    notebook = window.get_notebook()
    ui_init(window)
    signal_tab_switch = notebook.connect_after("switch_page", _switch_page_cb, window);
    notebook._ueb_signal_tab_switch = signal_tab_switch

def detach_window(window):
    notebook = window.get_notebook()
    notebook.disconnect(notebook._ueb_signal_tab_switch)
    del notebook._ueb_signal_tab_switch
    ui_destroy(window)

def attach_tab(window, embed):
    signal_load_status = embed.connect_after("notify::load-status",_load_status_cb, window)
    embed._ueb_signal_load_status = signal_load_status

def detach_tab(window, embed):
    embed.disconnect(embed._ueb_signal_load_status)
    del embed._ueb_signal_load_status
