#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library: Actions.
 *   
 *   This module defines the set of built-in library actions.  
 */

#include "adv3.h"
#include "tok.h"

/* ------------------------------------------------------------------------ */
/*
 *   Special "debug" action - this simply breaks into the debugger, if the
 *   debugger is present. 
 */
class DebugAction: IAction
    execAction()
    {
        /* if the debugger is present, break into it */
        if (t3DebugTrace(T3DebugCheck))
            t3DebugTrace(T3DebugBreak);
        else
            "Debugger not present. ";
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Special internal action to note a change to the darkness level.  This
 *   command is invoked internally when a change to the darkness level
 *   occurs.  
 */
class NoteDarknessAction: IAction
    execAction()
    {
        /* 
         *   if we're in the dark, note that darkness has fallen;
         *   otherwise, show the player character's room description as
         *   though the player had typed "look" 
         */
        if (gActor.isLocationLit())
        {
            /* look around */
            gActor.lookAround(true, nil);
        }
        else
        {
            /* it is now dark */
            mainReport(&newlyDark);
        }
    }

    /* this is an internal command that takes no time */
    actionTime = 0

    /* this isn't a real action, so it's not repeatable */
    isRepeatable = nil

    /* this action doesn't do anything; don't include it in undo */
    includeInUndo = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Special "again" action.  This command repeats the previous command.
 */
class AgainAction: IAction
    /* for obvious reasons, 'again' is not itself repeatable with 'again' */
    isRepeatable = nil

    /* 
     *   the undo command itself is not undoable (but the underlying
     *   command that we repeat might be) 
     */
    includeInUndo = nil

    /* information on the most recent command */
    lastIssuingActor = nil
    lastTargetActor = nil
    lastTargetActorPhrase = nil
    lastAction = nil

    /* save the most recent command so that it can be repeated if desired */
    saveForAgain(issuingActor, targetActor, targetActorPhrase, action)
    {
        /* save the information */
        lastIssuingActor = issuingActor;
        lastTargetActor = targetActor;
        lastTargetActorPhrase = targetActorPhrase;
        lastAction = action;
    }

    /* 
     *   Execute the 'again' command.  This action is special enough that
     *   we override its entire action processing sequence - this is
     *   necessary in case we're repeating another special command, such
     *   as 'again', and in any case is desirable because we don't want
     *   'again' to count as a command in its own right; it's essentially
     *   just a macro that we replace with the original command. 
     */
    doAction(issuingActor, targetActor, targetActorPhrase,
             countsAsIssuerTurn)
    {
        /* if there's nothing to repeat, show an error and give up */
        if (lastAction == nil)
        {
            libMessages.noCommandForAgain();
            return;
        }

        /* 
         *   'again' cannot be executed with a target actor - the target
         *   actor must be the player character 
         */
        if (!targetActor.isPlayerChar)
        {
            libMessages.againCannotChangeActor();
            return;
        }

        /* 
         *   if the issuing actor isn't the same as the target actor, make
         *   sure the issuer can still talk to the target 
         */
        if (lastIssuingActor != lastTargetActor
            && !lastIssuingActor.canTalkTo(lastTargetActor))
        {
            /* complain that we can no longer talk to the target */
            libMessages.againCannotTalkToTarget(lastIssuingActor,
                                                lastTargetActor);
            return;
        }

        /* reset any cached information for the new command context */
        lastAction.resetAction();

        /* execute the command */
        executeAction(lastTargetActor, lastTargetActorPhrase,
                      lastIssuingActor, nil, lastAction);
    }

    /* 
     *   this command itself consumes no time on the game clock (although
     *   the action we perform might) 
     */
    actionTime = 0
;

/* ------------------------------------------------------------------------ */
/*
 *   PreSaveObject - every instance of this class is notified, via its
 *   execute() method, just before we save the game.  This uses the
 *   ModuleExecObject framework, so the sequencing lists (execBeforeMe,
 *   execAfterMe) can be used to control relative ordering of execution
 *   among instances.  
 */
class PreSaveObject: ModuleExecObject
    /*
     *   Each instance must override execute() with its specific pre-save
     *   code. 
     */
;

/*
 *   PostRestoreObject - every instance of this class is notified, via its
 *   execute() method, immediately after we restore the game. 
 */
class PostRestoreObject: ModuleExecObject
    /* 
     *   note: each instance must override execute() with its post-restore
     *   code 
     */

    /*
     *   The "restore code," which is the (normally integer) value passed
     *   as the second argument to restoreGame().  The restore code gives
     *   us some idea of what triggered the restoration.  By default, we
     *   define the following restore codes:
     *   
     *   1 - the system is restoring a game as part of interpreter
     *   startup, usually because the user explicitly specified a game to
     *   restore on the interpreter command line or via a GUI shell
     *   mechanism, such as double-clicking on a saved game file from the
     *   desktop.
     *   
     *   2 - the user is explicitly restoring a game via a RESTORE command.
     *   
     *   Games and library extensions can use their own additional restore
     *   codes in their calls to restoreGame().  
     */
    restoreCode = nil
;

/*
 *   PreRestartObject - every instance of this class is notified, via its
 *   execute() method, just before we restart the game (with a RESTART
 *   command, for example). 
 */
class PreRestartObject: ModuleExecObject
    /* 
     *   Each instance must override execute() with its specific
     *   pre-restart code.  
     */
;

/*
 *   PostUndoObject - every instance of this class is notified, via its
 *   execute() method, immediately after we perform an 'undo' command. 
 */
class PostUndoObject: ModuleExecObject
    /* 
     *   Each instance must override execute() with its specific post-undo
     *   code.  
     */
;

/* ------------------------------------------------------------------------ */
/*
 *   Special "save" action.  This command saves the current game state to
 *   an external file for later restoration. 
 */
class SaveAction: SystemAction
    execSystemAction()
    {
        local result;

        /* ask for a file */
        result = getInputFile(libMessages.getSavePrompt(), InFileSave,
                              FileTypeT3Save, 0);

        /* check the inputFile response */
        switch(result[1])
        {
        case InFileSuccess:
            /* perform the save on the given file */
            performSave(result[2]);

            /* 
             *   in case we're on a slow system, restore the real-time
             *   counter, so that the time we were writing the saved state
             *   file doesn't count against the elapsed real-time 
             */
            realTimeManager.setElapsedTime(origElapsedTime);

            /* done */
            break;

        case InFileFailure:
            /* advise of the failure of the prompt */
            libMessages.filePromptFailed();
            break;

        case InFileCancel:
            /* acknowledge the cancellation */
            libMessages.saveCanceled();
            break;
        }
    }

    /* perform a save */
    performSave(fname)
    {
        /* before saving the game, notify all PreSaveObject instances */
        PreSaveObject.classExec();
        
        /* 
         *   Save the game to the given file.  If an error occurs, the
         *   save routine will throw a runtime error.  
         */
        try
        {
            /* try saving the game */
            saveGame(fname);
        }
        catch (RuntimeError err)
        {
            /* the save failed - mention the problem */
            libMessages.saveFailed(err);
            
            /* done */
            return;
        }
        
        /* note the successful save */
        libMessages.saveOkay();
    }

    /* 
     *   Saving has no effect on game state, so it's irrelevant whether or
     *   not it's undoable; but it might be confusing to say we undid a
     *   "save" command, because the player might think we deleted the
     *   saved file.  To avoid such confusion, do not include "save"
     *   commands in the undo log.  
     */
    includeInUndo = nil
;

/*
 *   Subclass of Save action that takes a literal string as part of the
 *   command.  The filename must be a literal enclosed in quotes, and the
 *   string (with the quotes) must be stored in our fname_ property by
 *   assignment of a quotedStringPhrase production in the grammar rule.  
 */
class SaveStringAction: SaveAction
    execSystemAction()
    {
        /* 
         *   Perform the save, using the filename given in our fname_
         *   parameter, trimmed of quotes.  
         */
        performSave(fname_.getStringText());
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Special "restore" action.  This action restores game state previously
 *   saved with the "save" action.
 */
class RestoreAction: SystemAction
    execSystemAction()
    {
        local result;

        /* ask for a file */
        result = getInputFile(libMessages.getRestorePrompt(), InFileOpen,
                              FileTypeT3Save, 0);

        /* check the inputFile response */
        switch(result[1])
        {
        case InFileSuccess:
            /* 
             *   try restoring the file; use code 2 to indicate that the
             *   restoration was performed by an explicit RESTORE command 
             */
            if (performRestore(result[2], 2))
            {
                /* 
                 *   success - look around, to refresh their memory of the
                 *   state the game was in when saved 
                 */
                "\b";
                libGlobal.playerChar.lookAround(true, true);
            }
            else
            {
                /* 
                 *   failed - in case the failed restore took some time,
                 *   restore the real-time clock, so that the file-reading
                 *   time doesn't count against the game time 
                 */
                realTimeManager.setElapsedTime(origElapsedTime);
            }

            /* done */
            break;

        case InFileFailure:
            /* advise of the failure of the prompt */
            libMessages.filePromptFailed();
            break;

        case InFileCancel:
            /* acknowledge the cancellation */
            libMessages.restoreCanceled();
            break;
        }

        /* abandon any additional commands on the same command line */
        throw new TerminateCommandException();
    }

    /*
     *   Restore a file.  'code' is the restoreCode value for the
     *   PostRestoreObject notifications.  Returns true on success, nil on
     *   failure.  
     */
    performRestore(fname, code)
    {
        try
        {
            /* restore the file */
            restoreGame(fname);
        }
        catch (RuntimeError err)
        {
            /* failed - check the error to see what went wrong */
            switch(err.errno_)
            {
            case 1201:
                /* not a saved state file */
                libMessages.restoreInvalidFile();
                break;
                
            case 1202:
                /* saved by different game or different version */
                libMessages.restoreInvalidMatch();
                break;
                
            case 1207:
                /* corrupted saved state file */
                libMessages.restoreCorruptedFile();
                break;
                
            default:
                /* some other failure */
                libMessages.restoreFailed(err);
                break;
            }

            /* indicate failure */
            return nil;
        }

        /* note that we've successfully restored the game */
        libMessages.restoreOkay();
        
        /* set the appropriate restore-action code */
        PostRestoreObject.restoreCode = code;

        /* notify all PostRestoreObject instances */
        PostRestoreObject.classExec();

        /* indicate success */
        return true;
    }
;

/*
 *   Subclass of Restore action that takes a literal string as part of the
 *   command.  The filename must be a literal enclosed in quotes, and the
 *   string (with the quotes) must be stored in our fname_ property by
 *   assignment of a quotedStringPhrase production in the grammar rule.  
 */
class RestoreStringAction: RestoreAction
    execSystemAction()
    {
        /* 
         *   Perform the restore, using the filename given in our fname_
         *   parameter, trimmed of quotes.  Use code 2, the same as any
         *   other explicit RESTORE command.  
         */
        if (performRestore(fname_.getStringText(), 2))
        {
            /* success - look around */
            "\b";
            libGlobal.playerChar.lookAround(true, true);
        }

        /* abandon any additional commands on the same command line */
        throw new TerminateCommandException();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Restart the game from the beginning.
 */
class RestartAction: SystemAction
    execSystemAction()
    {
        /* confirm that they really want to restart */
        libMessages.confirmRestart();
        if (yesOrNo())
        {
            /* 
             *   The confirmation input will have put us into
             *   start-of-command mode for sequencing purposes; force the
             *   sequencer back to mid-command mode, so we can show
             *   inter-command separation before the restart. 
             */

            /* restart the game */
            doRestartGame();
        }
        else
        {
            /* confirm that we're not really restarting */
            libMessages.notRestarting();
        }
    }

    /* carry out the restart action */
    doRestartGame()
    {
        local sig;

        /* 
         *   Show a command separator, to provide separation from any
         *   introductory text that we'll show on restarting.  Note that
         *   we probably just asked for confirmation, which means that the
         *   command sequencer will be in start-of-command mode; force it
         *   back to mid-command mode so we show inter-command separation.
         */
        commandSequencer.setCommandMode();
        "<.commandsep>";

        /* before restarting, notify anyone interested of our intentions */
        PreRestartObject.classExec();

        /* set up a 'restarting' signal */
        sig = new RestartSignal();

        /* 
         *   call the intrinsic restartGame function to reset all of the
         *   static objects to their initial state 
         */
        restartGame();

        /* switch to the initial no-command mode */
        "<.commandnone>";

        /* throw our 'restarting' signal */
        throw sig;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Undo one turn. 
 */
class UndoAction: SystemAction
    /*
     *   "Undo" is so special that we must override the entire action
     *   processing sequence.  We do this because undoing will restore the
     *   game state as of the previous savepoint, which would leave all
     *   sorts of things unsynchronized in the normal action sequence.  To
     *   avoid problems, we simply leave out any other action processing
     *   and perform the 'undo' directly.  
     */
    doAction(issuingActor, targetActor, targetActorPhrase,
             countsAsIssuerTurn)
    {
        /* 
         *   don't allow this unless the player character is performing
         *   the command directly
         */
        if (!targetActor.isPlayerChar)
        {
            /* 
             *   tell them this command cannot be directed to another
             *   actor, and give up 
             */
            libMessages.systemActionToNPC();
            return;
        }

        /* perform the undo */
        performUndo(true);
    }

    /* 
     *   Perform undo.  Returns true if we were successful, nil if not.
     *   
     *   'asCommand' indicates whether or not the undo is being performed
     *   as an explicit command: if so, we'll save the UNDO command for use
     *   in AGAIN.  
     */
    performUndo(asCommand)
    {
        /* try undoing to the previous savepoint */
        if (undo())
        {
            local oldActor;
            local oldIssuer;
            local oldAction;

            /* notify all PostUndoObject instances */
            PostUndoObject.classExec();

            /* set up the globals for the command */
            oldActor = gActor;
            oldIssuer = gIssuingActor;
            oldAction = gAction;

            /* set the new globals */
            gActor = gPlayerChar;
            gIssuingActor = gPlayerChar;
            gAction = self;

            /* make sure we reset globals on the way out */
            try
            {
                /* success - mention what we did */
                libMessages.undoOkay(libGlobal.lastActorForUndo,
                                     libGlobal.lastCommandForUndo);
                
                /* look around, to refresh the player's memory */
                libGlobal.playerChar.lookAround(true, true);
            }
            finally
            {
                /* restore the parser globals to how we found them */
                gActor = oldActor;
                gIssuingActor = oldIssuer;
                gAction = oldAction;
            }
                
            /* 
             *   if this was an explicit 'undo' command, save the command
             *   to allow repeating it with 'again' 
             */
            if (asCommand)
                AgainAction.saveForAgain(gPlayerChar, gPlayerChar, nil, self);

            /* indicate success */
            return true;
        }
        else
        {
            /* no more undo information available */
            libMessages.undoFailed();

            /* indicate failure */
            return nil;
        }
    }

    /* 
     *   "undo" is not undoable - if we undo again after an undo, we undo
     *   the next most recent command
     */
    includeInUndo = nil
;

/* ------------------------------------------------------------------------ */
/*
 *   Quit the game. 
 */
class QuitAction: SystemAction
    execSystemAction()
    {
        /* confirm that they really want to quit */
        libMessages.confirmQuit();
        if (yesOrNo())
        {
            /* carry out the termination */
            terminateGame();
        }
        else
        {
            /* show the confirmation that we're not quitting */
            libMessages.notTerminating();
        }
    }

    /*
     *   Carry out game termination.  This can be called when we wish to
     *   end the game without asking for any additional player
     *   confirmation.  
     */
    terminateGame()
    {
        /* show the goodbye message */
        libMessages.terminating();
            
        /* throw a 'quitting' signal to end the game */
        throw new QuittingException;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   Pause the game.  This stops the real-time clock until the user
 *   presses a key.  Games that don't use the real-time clock will have no
 *   use for this. 
 */
class PauseAction: SystemAction
    execSystemAction()
    {
        local elapsed;
        
        /* 
         *   remember the current elapsed game real time - when we are
         *   released from the pause, we'll restore this time 
         */
        elapsed = realTimeManager.getElapsedTime();

        /* show our prompt */
        libMessages.pausePrompt();

        /* keep going until we're released */
    waitLoop:
        for (;;)
        {
            /* 
             *   Wait for a key, and see what we have.  Note that we
             *   explicitly do not want to allow any real-time events to
             *   occur, so we simply wait forever without timeout. 
             */
            switch(inputKey())
            {
            case ' ':
                /* space key - end the wait */
                break waitLoop;

            case 's':
            case 'S':
                /* mention that we're saving */
                libMessages.pauseSaving();
                
                /* 
                 *   set the elapsed time to the time when we started, so
                 *   that the saved position reflects the time at the
                 *   start of the pause 
                 */
                realTimeManager.setElapsedTime(elapsed);

                /* save the game - go run the normal SAVE command */
                SaveAction.execSystemAction();

                /* show our prompt again */
                "<.p>";
                libMessages.pausePrompt();

                /* go back to wait for another key */
                break;
                
            case '[eof]':
                /* end-of-file on keyboard input - throw an error */
                "\b";
                throw new EndOfFileException();

            default:
                /* ignore other keys; just go back to wait again */
                break;
            }
        }

        /* show the released-from-pause message */
        libMessages.pauseEnded();

        /* 
         *   set the real-time clock to the same elapsed game time
         *   that we had when we started the pause, so that the
         *   elapsed real time of the pause itself doesn't count
         *   against the game elapsed time 
         */
        realTimeManager.setElapsedTime(elapsed);
    }
;

/*
 *   Change to VERBOSE mode. 
 */
class VerboseAction: SystemAction
    execSystemAction()
    {
        /* set the global 'verbose' mode */
        libGlobal.verboseMode = true;

        /* acknowledge it */
        libMessages.acknowledgeVerboseMode(true);
    }
;

/*
 *   Change to TERSE mode. 
 */
class TerseAction: SystemAction
    execSystemAction()
    {
        /* set the global 'verbose' mode */
        libGlobal.verboseMode = nil;

        /* acknowledge it */
        libMessages.acknowledgeVerboseMode(nil);
    }
;

/* in case the score module isn't present */
property showScore;
property showFullScore;
property scoreNotify;

/*
 *   Show the current score.
 */
class ScoreAction: SystemAction
    execSystemAction()
    {
        /* show the simple score */
        if (libGlobal.scoreObj != nil)
            libGlobal.scoreObj.showScore();
        else
            libMessages.scoreNotPresent;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   Show the full score. 
 */
class FullScoreAction: SystemAction
    execSystemAction()
    {
        /* show the full score */
        if (libGlobal.scoreObj != nil)
            libGlobal.scoreObj.showFullScore();
        else
            libMessages.scoreNotPresent;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   Show the NOTIFY status. 
 */
class NotifyAction: SystemAction
    execSystemAction()
    {
        /* show the current notification status */
        if (libGlobal.scoreObj != nil)
            libMessages.showNotifyStatus(libGlobal.scoreObj.scoreNotify);
        else
            libMessages.commandNotPresent;
    }
;

/*
 *   Turn score change notifications on. 
 */
class NotifyOnAction: SystemAction
    execSystemAction()
    {
        /* turn notifications on, and acknowledge the status */
        if (libGlobal.scoreObj != nil)
        {
            libGlobal.scoreObj.scoreNotify = true;
            libMessages.acknowledgeNotifyStatus(true);
        }
        else
            libMessages.commandNotPresent;
    }
;

/*
 *   Turn score change notifications off. 
 */
class NotifyOffAction: SystemAction
    execSystemAction()
    {
        /* turn notifications off, and acknowledge the status */
        if (libGlobal.scoreObj != nil)
        {
            libGlobal.scoreObj.scoreNotify = nil;
            libMessages.acknowledgeNotifyStatus(nil);
        }
        else
            libMessages.commandNotPresent;
    }
;

/*
 *   Show version information for the game and the library modules the
 *   game is using.  
 */
class VersionAction: SystemAction
    execSystemAction()
    {
        /* show the version information for each library */
        foreach (local cur in ModuleID.getModuleList())
            cur.showVersion();
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   Show the credits for the game and the library modules the game
 *   includes. 
 */
class CreditsAction: SystemAction
    execSystemAction()
    {
        /* show the credits for each library */
        foreach (local cur in ModuleID.getModuleList())
            cur.showCredit();
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   Show the "about" information for the game and library modules.
 */
class AboutAction: SystemAction
    execSystemAction()
    {
        local anyOutput;
        
        /* watch for any output while showing module information */
        anyOutput = mainOutputStream.watchForOutput(new function()
        {
            /* show information for each module */
            foreach (local cur in ModuleID.getModuleList())
                cur.showAbout();
        });

        /* 
         *   if we didn't have any ABOUT information to show, display a
         *   message to this effect 
         */
        if (!anyOutput)
            libMessages.noAboutInfo;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/*
 *   A state object that keeps track of our logging (scripting) status.
 *   This is transient, because logging is controlled through the output
 *   layer in the interpreter, which does not participate in any of the
 *   persistence mechanisms.  
 */
transient scriptStatus: object
    /*
     *   Script file name.  This is nil when logging is not in effect, and
     *   is set to the name of the scripting file when a log file is
     *   active. 
     */
    scriptFile = nil
;

/*
 *   Turn scripting on.  This creates a text file that contains a
 *   transcript of all commands and responses from this point forward.
 */
class ScriptAction: SystemAction
    execSystemAction()
    {
        /* go set up scripting */
        setUpScripting(true);
    }

    /* set up scripting - ask for a file and start logging to it */
    setUpScripting(ack)
    {
        local result;

        /* ask for a file */
        result = getInputFile(libMessages.getScriptingPrompt(), InFileSave,
                              FileTypeLog, 0);

        /* check the inputFile result */
        switch(result[1])
        {
        case InFileSuccess:
            /* turn on scripting */
            performScripting(result[2], ack);
            break;

        case InFileFailure:
            /* advise of the failure of the prompt */
            libMessages.filePromptFailed();
            break;

        case InFileCancel:
            /* acknowledge the cancellation */
            libMessages.scriptingCanceled();
            break;
        }
    }

    /* turn on scripting to the given file */
    performScripting(fname, ack)
    {
        /* turn on logging */
        setLogFile(fname);

        /* remember that scripting is in effect */
        scriptStatus.scriptFile = fname;

        /* note that logging is active, if acknowledgment is desired */
        if (ack)
            libMessages.scriptingOkay();
    }

    /* we can't include this in undo, as it affects external files */
    includeInUndo = nil
;

/*
 *   Subclass of Script action taking a quoted string as part of the
 *   command syntax.  The grammar rule must set our fname_ property to a
 *   quotedStringPhrase subproduction. 
 */
class ScriptStringAction: ScriptAction
    execSystemAction()
    {
        /* set up scripting to the filename specified in the command */
        performScripting(fname_.getStringText(), true);
    }
;

/*
 *   Turn scripting off.  This stops recording the game transcript started
 *   with the most recent SCRIPT command. 
 */
class UnscriptAction: SystemAction
    execSystemAction()
    {
        /* turn off scripting */
        turnOffScripting(true);
    }

    /* turn off scripting */
    turnOffScripting(ack)
    {
        /* cancel scripting in the interpreter's output layer */
        setLogFile(nil);

        /* remember that scripting is no longer in effect */
        scriptStatus.scriptFile = nil;

        /* acknowledge the change, if desired */
        if (ack)
            libMessages.unscriptingOkay();
    }

    /* we can't include this in undo, as it affects external files */
    includeInUndo = nil
;

/* in case the footnote module is not present */
property showFootnote;

/*
 *   Footnote - this requires a numeric argument parsed via the
 *   numberPhrase production and assigned to the numMatch property.  
 */
class FootnoteAction: SystemAction
    execSystemAction()
    {
        /* ask the Footnote class to do the work */
        if (libGlobal.footnoteClass != nil)
            libGlobal.footnoteClass.showFootnote(numMatch.getval());
        else
            libMessages.commandNotPresent;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

/* base class for FOOTNOTES xxx commands */
class FootnotesAction: SystemAction
    execSystemAction()
    {
        if (libGlobal.footnoteClass != nil)
        {
            /* set my footnote status in the global setting */
            libGlobal.footnoteClass.showFootnotes = showFootnotes;

            /* acknowledge it */
            libMessages.acknowledgeFootnoteStatus(showFootnotes);
        }
        else
            libMessages.commandNotPresent;
    }

    /* 
     *   the footnote status I set when this command is activated - this
     *   must be overridden by each subclass 
     */
    showFootnotes = nil
;

class FootnotesFullAction: FootnotesAction
    showFootnotes = FootnotesFull
;

class FootnotesMediumAction: FootnotesAction
    showFootnotes = FootnotesMedium
;

class FootnotesOffAction: FootnotesAction
    showFootnotes = FootnotesOff
;

class FootnotesStatusAction: SystemAction
    execSystemAction()
    {
        /* show the current status */
        if (libGlobal.footnoteClass != nil)
            libMessages.showFootnoteStatus(libGlobal.footnoteClass
                                           .showFootnotes);
        else
            libMessages.commandNotPresent;
    }

    /* there's no point in including this in undo */
    includeInUndo = nil
;

class InventoryAction: IAction
    execAction()
    {
        /* show the actor's inventory in the current mode */
        gActor.showInventory(inventoryMode == InventoryTall);
    }

    /* current inventory mode - start in 'wide' mode by default */
    inventoryMode = InventoryWide;
;

class InventoryTallAction: IAction
    execAction()
    {
        /* set inventory mode to 'tall' */
        InventoryAction.inventoryMode = InventoryTall;

        /* run the inventory action */
        InventoryAction.execAction();
    }
;

class InventoryWideAction: IAction
    execAction()
    {
        /* set inventory mode to 'wide' */
        InventoryAction.inventoryMode = InventoryWide;

        /* run the inventory action */
        InventoryAction.execAction();
    }
;

class WaitAction: IAction
    execAction()
    {
        /* just show the "time passes" message */
        defaultReport(&timePasses);
    }
;

class LookAction: IAction
    execAction()
    {
        /* show the actor's current location in verbose mode */
        gActor.lookAround(true, true);
    }
;

class SleepAction: IAction
    execAction()
    {
        /* let the actor handle it */
        gActor.goToSleep();
    }
;

DefineTAction(Take)
    getAllDobj(actor, scopeList)
    {
        local loc;
        
        /* 
         *   Include only objects that are directly in the actor's
         *   location, or within fixed items in the actor's location.
         */
        loc = actor.location;
        return scopeList.subset(
            {x: x.isDirectlyIn(loc) || x.isInFixedIn(loc) });
    }
;

DefineTIAction(TakeFrom)
    getAllDobj(actor, scopeList)
    {
        /* include only objects contained within the indirect object */
        return scopeList.subset({x: x != getIobj()
                                    && x.isDirectlyIn(getIobj())});
    }
;

DefineTAction(Drop)
    getAllDobj(actor, scopeList)
    {
        /* include only objects directly held by the actor */
        return scopeList.subset({x: x.isDirectlyIn(actor)});
    }
;

DefineTAction(Examine)
;

DefineTAction(Read)
;

DefineTAction(LookIn)
;

DefineTAction(Search)
;

DefineTAction(LookUnder)
;

DefineTAction(LookBehind)
;

DefineTAction(LookThrough)
;

DefineTAction(Feel)
;

DefineTAction(Taste)
;

DefineTAction(Smell)
;

DefineTAction(ListenTo)
;

/*
 *   Base class for undirected sensing, such as "listen" or "smell" with
 *   no object.  We'll scan 
 */
class SenseImplicitAction: IAction
    /* the sense in which I operate */
    mySense = nil

    /* the object property to display this sense's description */
    descProp = nil
    
    /* the default message to display if we find nothing specific to sense */
    defaultMsgProp = nil

    /* the Lister we use to show the items */
    resultLister = nil

    /* execute the action */
    execAction()
    {
        local senseTab;
        local presenceList;
            
        /* get a list of everything in range of this sense for the actor */
        senseTab = gActor.senseInfoTable(mySense);

        /* get a list of everything with a presence in this sense */
        presenceList = senseInfoTableSubset(senseTab,
            {obj, info: obj.(mySense.presenceProp)});

        /* 
         *   if there's anything in the list, show it; otherwise, show a
         *   default report 
         */
        if (presenceList.length() != 0)
        {
            /* show the list using our lister */
            resultLister.showList(gActor, nil, presenceList, 0, 0,
                                  senseTab, nil);
        }
        else
        {
            /* there's nothing to show - say so */
            defaultReport(defaultMsgProp);
        }
    }
;

class SmellImplicitAction: SenseImplicitAction
    mySense = smell
    descProp = &smellDesc
    defaultMsgProp = &nothingToSmell
    resultLister = smellActionLister
;

class ListenImplicitAction: SenseImplicitAction
    mySense = sound
    descProp = &soundDesc
    defaultMsgProp = &nothingToHear
    resultLister = listenActionLister
;

DefineTIAction(PutIn)
    getAllDobj(actor, scopeList)
    {
        local loc;

        /* get the actor's location */
        loc = actor.location;
        
        /*
         *   Include objects that are directly in the actor's location, or
         *   within fixed items in the actor's location, or directly in
         *   the actor's inventory.
         *   
         *   Exclude the indirect object (since we obviously can't put the
         *   indirect object in itself), and exclude everything already
         *   directly in the indirect object.  
         */
        return scopeList.subset({x:
                                (x.isDirectlyIn(loc)
                                 || x.isInFixedIn(loc)
                                 || x.isDirectlyIn(actor))
                                && x != getIobj()
                                && !x.isDirectlyIn(getIobj())});
    }
;

DefineTIAction(PutOn)
    getAllDobj(actor, scopeList)
    {
        /* use the same strategy that we do in PutIn */
        local loc = actor.location;
        return scopeList.subset({x:
                                (x.isDirectlyIn(loc)
                                 || x.isInFixedIn(loc)
                                 || x.isDirectlyIn(actor))
                                && x != getIobj()
                                && !x.isDirectlyIn(getIobj())});
    }
;

DefineTIAction(PutUnder)
;

DefineTAction(Wear)
;

DefineTAction(Doff)
;

DefineTIAction(AskFor)
    /* 
     *   Resolve the direct object first for this command.  This will
     *   allow us to scope the indirect (the object we're asking for) to
     *   the possessions of the direct object (the person we're asking),
     *   since we'll know the direct object by the time we get around to
     *   resolving the indirect object.  
     */
    resolveFirst = DirectObject
;

DefineTopicAction(AskAbout, IndirectObject)
    getDefaultDobj(resolver)
    {
        local obj;
        
        /* 
         *   check to see if the actor has a default interlocutor; if so,
         *   use it as the default actor to be addressed here, otherwise
         *   use the default handling 
         */
        obj = resolver.getTargetActor().getDefaultInterlocutor();
        if (obj != nil)
            return [new ResolveInfo(obj, 0)];
        else
            return inherited(resolver);
    }
;

DefineTopicAction(TellAbout, IndirectObject)
    getDefaultDobj(resolver)
    {
        local obj;

        /* check the actor for a default interlocutor */
        obj = resolver.getTargetActor().getDefaultInterlocutor();
        if (obj != nil)
            return [new ResolveInfo(obj, 0)];
        else
            return inherited(resolver);
    }
;

class HelloAction: IAction
    execAction()
    {
        /* 
         *   if the current actor is the player character, issue a generic
         *   response, since the player just typed "hello" as though to
         *   the game itself; if there's an NPC target actor, the player
         *   is trying to greet the actor 
         */
        if (gActor.isPlayerChar)
            mainReport(&sayHello);
        else
            gActor.greetingsFrom(libGlobal.playerChar);
    }
;

DefineTAction(Kiss)
;

class YellAction: IAction
    execAction()
    {
        /* yelling generally has no effect; issue a default response */
        mainReport(&okayYell);
    }
;

DefineTAction(TalkTo)
;

DefineTIAction(GiveTo)
    getDefaultIobj(resolver)
    {
        local obj;

        /* check the actor for a default interlocutor */
        obj = resolver.getTargetActor().getDefaultInterlocutor();
        if (obj != nil)
            return [new ResolveInfo(obj, 0)];
        else
            return inherited(resolver);
    }
;

DefineTIAction(ShowTo)
    getDefaultIobj(resolver)
    {
        local obj;

        /* check the actor for a default interlocutor */
        obj = resolver.getTargetActor().getDefaultInterlocutor();
        if (obj != nil)
            return [new ResolveInfo(obj, 0)];
        else
            return inherited(resolver);
    }
;

DefineTAction(Follow)
    /*
     *   For resolving our direct object, we want to include in the scope
     *   any item that isn't present but which the actor saw departing the
     *   present location.  
     */
    initResolver(issuingActor, targetActor)
    {
        /* inherit the base resolver initialization */
        inherited(issuingActor, targetActor);

        /* 
         *   add to the scope all of the actor's followable objects -
         *   these are the objects which the actor has witnessed leaving
         *   the actor's present location 
         */
        scope_ += targetActor.getFollowables();
    }
;

DefineTAction(Attack)
;

DefineTIAction(AttackWith)
    /* 
     *   for the indirect object, limit 'all' and defaults to the items in
     *   inventory 
     */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTAction(Throw)
;

DefineTIAction(ThrowAt)
;

DefineTIAction(ThrowTo)
;

DefineTAction(Dig)
;

DefineTIAction(DigWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

class JumpAction: IAction
    preCond = [actorStanding]
    execAction()
    {
        /* show the default report for jumping in place */
        mainReport(&okayJump);
    }
;

DefineTAction(JumpOver)
;

DefineTAction(JumpOff)
;

class JumpOffIAction: IAction
    execAction()
    {
        mainReport(&cannotJumpOffHere);
    }
;

DefineTAction(Push)
;

DefineTAction(Pull)
;

DefineTAction(Move)
;

DefineTIAction(MoveWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTIAction(MoveTo)
;

DefineTAction(Turn)
;

DefineTIAction(TurnWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineLiteralTAction(TurnTo, IndirectObject)
;

DefineLiteralTAction(SetTo, IndirectObject)
;

DefineLiteralTAction(TypeOn, DirectObject)
;

DefineLiteralTAction(EnterOn, DirectObject)
;

DefineTAction(Consult)
;

DefineTopicAction(ConsultAbout, IndirectObject)
;

DefineTAction(Switch)
;

DefineTAction(Flip)
;

DefineTAction(TurnOn)
;

DefineTAction(TurnOff)
;

DefineTAction(Light)
;

DefineTAction(Burn)
;

DefineTIAction(BurnWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTAction(Extinguish)
;

DefineTIAction(AttachTo)
;

DefineTIAction(DetachFrom)
;

DefineTAction(Detach)
;

DefineTAction(Break)
;

DefineTAction(Cut)
;

DefineTIAction(CutWith)
;

DefineTAction(Climb)
;

DefineTAction(ClimbUp)
;

DefineTAction(ClimbDown)
;

DefineTAction(Open)
;

DefineTAction(Close)
;

DefineTAction(Lock)
;

DefineTAction(Unlock)
;

DefineTIAction(LockWith)
    /* 
     *   Resolve the direct object (the lock) first, so that we know what
     *   we're trying to unlock when we're verifying the key.  This allows
     *   us to (optionally) boost the likelihood of a known good key for
     *   disambiguation. 
     */
    resolveFirst = DirectObject

    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTIAction(UnlockWith)
    /* resolve the direct object first, for the same reason as in LockWith */
    resolveFirst = DirectObject

    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTAction(Eat)
;

DefineTAction(Drink)
;

DefineTAction(Pour)
;

DefineTIAction(PourInto)
;

DefineTIAction(PourOnto)
;

DefineTAction(Clean)
;

DefineTIAction(CleanWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTAction(SitOn)
;

DefineTAction(LieOn)
;

DefineTAction(StandOn)
;

class StandAction: IAction
    execAction()
    {
        /* let the actor handle it */
        gActor.standUp();
    }
;

DefineTAction(Board)
;

DefineTAction(GetOutOf)
    getAllDobj(actor, scopeList)
    {
        /* 'all' for 'get out of' is the actor's immediate container */
        return scopeList.subset({x: actor.isDirectlyIn(x)});
    }
;

DefineTAction(GetOffOf)
    getAllDobj(actor, scopeList)
    {
        /* 'all' for 'get off of' is the actor's immediate container */
        return scopeList.subset({x: actor.isDirectlyIn(x)});
    }
;

class GetOutAction: IAction
    execAction()
    {
        /* let the actor handle it */
        gActor.disembark();
    }
;

DefineTAction(Fasten)
;

DefineTIAction(FastenTo)
;

DefineTAction(Unfasten)
;

DefineTIAction(UnfastenFrom)
;

DefineTAction(PlugIn)
;

DefineTIAction(PlugInto)
;

DefineTAction(Unplug)
;

DefineTIAction(UnplugFrom)
;

DefineTAction(Screw)
;

DefineTIAction(ScrewWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

DefineTAction(Unscrew)
;

DefineTIAction(UnscrewWith)
    /* limit 'all' for the indirect object to items in inventory */
    getAllIobj(actor, scopeList)
    {
        return scopeList.subset({x: x.isIn(actor)});
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Travel Action - this is the base class for verbs that attempt to move
 *   an actor to a new location via one of the directional connections
 *   from the current location.
 *   
 *   Each grammar rule for this action must set the 'dirMatch' property to
 *   a DirectionProd match object that gives the direction.  
 */
class TravelAction: IAction
    execAction()
    {
        local conn;
        
        /* 
         *   Perform the travel via the connector, if we have one.  If
         *   there's no connector defined for this direction, show a
         *   default "you can't go that way" message. 
         */
        if ((conn = getConnector()) != nil)
        {
            /* 
             *   we have a connector - use the pseudo-action TravelVia with
             *   the connector to carry out the travel 
             */
            replaceAction(TravelVia, conn);
        }
        else
        {
            /* no connector - show a default "can't go that way" error */
            mainReport(&cannotGoThatWay);
        }
    }

    /* get the direction object for the travel */
    getDirection() { return dirMatch.dir; }

    /*
     *   Get my travel connector.  My connector is given by the travel
     *   link property for this action as defined in the actor's current
     *   location. 
     */
    getConnector()
    {
        /* ask the location for the connector in my direction */
        return gActor.location.getTravelConnector(getDirection(), gActor);
    }
;

/*
 *   This class makes it convenient to synthesize a TravelAction given a
 *   Direction object.  
 */
class TravelDirAction: TravelAction
    construct(dir)
    {
        /* remember my direction */
        dir_ = dir;
    }
    
    /* get my direction */
    getDirection() { return dir_; }
    
    /* my direction, normally specified during construction */
    dir_ = nil
;

/*
 *   To make it more convenient to use directional travel actions as
 *   synthesized commands, define a set of action classes for the specific
 *   directions.  
 */
class NorthAction: TravelAction
    getDirection = northDirection
;
class SouthAction: TravelAction
    getDirection = southDirection
;
class EastAction: TravelAction
    getDirection = eastDirection
;
class WestAction: TravelAction
    getDirection = westDirection
;
class NortheastAction: TravelAction
    getDirection = northeastDirection
;
class NorthwestAction: TravelAction
    getDirection = northwestDirection
;
class SoutheastAction: TravelAction
    getDirection = southeastDirection
;
class SouthwestAction: TravelAction
    getDirection = southwestDirection
;
class InAction: TravelAction
    getDirection = inDirection
;
class OutAction: TravelAction
    getDirection = outDirection
;
class UpAction: TravelAction
    getDirection = upDirection
;
class DownAction: TravelAction
    getDirection = downDirection
;
class ForeAction: TravelAction
    getDirection = foreDirection
;
class AftAction: TravelAction
    getDirection = aftDirection
;
class PortAction: TravelAction
    getDirection = portDirection
;
class StarboardAction: TravelAction
    getDirection = starboardDirection
;

/*
 *   Non-directional travel actions 
 */

DefineTAction(GoThrough)
;

DefineTAction(Enter)
;

/*
 *   An internal action for traveling via a connector.  This isn't a real
 *   action, and shouldn't have a grammar defined for it.  The purpose of
 *   this action is to allow real actions that cause travel via a
 *   connector to be implemented by mapping to this internal action, which
 *   we implement on the base travel connector class.  
 */
DefineTAction(TravelVia)
    /* 
     *   The direct object of this synthetic action isn't necessarily an
     *   ordinary simulation object: it could be a TravelConnector
     *   instead.  Don't return it in the list of objects to the command.
     *   This synthetic action doesn't necessarily involve any simulation
     *   objects.  
     */
    getCurrentObjects = []
;

/* "go back" */
class GoBackAction: IAction
    execAction()
    {
        /* ask the actor to handle it */
        gActor.reverseLastTravel();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Combined pushing-and-traveling action ("push crate north", "drag sled
 *   into cave").  All of these are based on a base action class, which
 *   defines the methods invoked on the object being pushed; the
 *   subclasses provide a definition of the connector that determines
 *   where the travel takes us.  
 */
DefineTAction(PushTravel)
;

/*
 *   For directional push-and-travel commands, we define a common base
 *   class that does the work to find the connector based on the room's
 *   directional connector.
 *   
 *   Subclasses for grammar rules must define the 'dirMatch' property to
 *   be a DirectionProd object for the associated direction.  
 */
class PushTravelDirAction: PushTravelAction
    /*
     *   Get the direction we're going.  By default, we return the
     *   direction associated with the dirMatch match object from our
     *   grammar match.  
     */
    getDirection() { return dirMatch.dir; }
    
    /*
     *   Get the connector for travel.  We'll ask the location for the
     *   connector in our direction.  
     */
    getConnector()
    {
        /* get the direction object from the DirectionProd */
        return gActor.location.getTravelConnector(getDirection(), gActor);
    }
;

/*
 *   To make it easy to synthesize actions for pushing objects, define
 *   individual subclasses for the various directions.
 */
class PushNorthAction: PushTravelDirAction
    getDirection = northDirection
;

class PushSouthAction: PushTravelDirAction
    getDirection = southDirection
;

class PushEastAction: PushTravelDirAction
    getDirection = eastDirection
;

class PushWestAction: PushTravelDirAction
    getDirection = westDirection
;

class PushNorthwestAction: PushTravelDirAction
    getDirection = northwestDirection
;

class PushNortheastAction: PushTravelDirAction
    getDirection = northeastDirection
;

class PushSouthwestAction: PushTravelDirAction
    getDirection = southwestDirection
;

class PushSoutheastAction: PushTravelDirAction
    getDirection = southeastDirection
;

class PushUpAction: PushTravelDirAction
    getDirection = upDirection
;

class PushDownAction: PushTravelDirAction
    getDirection = downDirection
;

class PushInAction: PushTravelDirAction
    getDirection = inDirection
;

class PushOutAction: PushTravelDirAction
    getDirection = outDirection
;

class PushForeAction: PushTravelDirAction
    getDirection = foreDirection
;

class PushAftAction: PushTravelDirAction
    getDirection = aftDirection
;

class PushPortAction: PushTravelDirAction
    getDirection = portDirection
;

class PushStarboardAction: PushTravelDirAction
    getDirection = starboardDirection
;

/*
 *   Base class for two-object push-travel commands, such as "push boulder
 *   out of cave" or "drag sled up hill".  For all of these, the connector
 *   is given by the indirect object.
 */
class PushTravelViaIobjAction: TIAction, PushTravelAction
    /* my travel connector is the indirect object */
    getConnector() { return getIobj(); }
;

DefineTIActionSub(PushTravelThrough, PushTravelViaIobjAction)
;

DefineTIActionSub(PushTravelEnter, PushTravelViaIobjAction)
;

DefineTIActionSub(PushTravelGetOutOf, PushTravelViaIobjAction)
;

DefineTIActionSub(PushTravelClimbUp, PushTravelViaIobjAction)
;

DefineTIActionSub(PushTravelClimbDown, PushTravelViaIobjAction)
;

/*
 *   The "exits" verb.  This verb explicitly shows all of the exits from
 *   the current location. 
 */
class ExitsAction: IAction
    execAction()
    {
        /* 
         *   if we have an exit lister object, invoke it; otherwise,
         *   explain that this command isn't supported in this game 
         */
        if (gExitLister != nil)
            gExitLister.showExitsCommand();
        else
            libMessages.commandNotPresent;
    }
;

/* in case the exits module isn't included */
property showExitsCommand, exitsOnOffCommand;

/*
 *   Turn exit display on.  The grammar rule should set the property 'on_'
 *   or 'off_' to a non-nil token to indicate the new setting.  
 */
class ExitsOnOffAction: SystemAction
    execSystemAction()
    {
        /* turn exits display on */
        if (gExitLister != nil)
            gExitLister.exitsOnOffCommand(on_ != nil);
        else
            libMessages.commandNotPresent;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Parser debugging verbs 
 */

#ifdef PARSER_DEBUG

grammar predicate(ParseDebug):
    'parse-debug' 'on'->onOrOff_
    | 'parse-debug' 'off'->onOrOff_
    | 'parse-debug'
    : IAction
    execAction()
    {
        local newMode;

        /* 
         *   get the mode - if the mode is explicitly stated in the
         *   command, use the stated new mode, otherwise invert the current
         *   mode 
         */
        newMode = (onOrOff_ == 'on'
                   ? true
                   : onOrOff_ == 'off'
                   ? nil
                   : !libGlobal.parserDebugMode);

        /* set the new mode */
        libGlobal.parserDebugMode = newMode;

        /* mention the change */
        "Parser debugging is now
        <<libGlobal.parserDebugMode ? 'on' : 'off'>>.\n";
    }
;

#endif

