/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <iostream>

#include <sigc++/method_slot.h>
#include <sigc++/retype_return.h>
#include <sigcx/ref_slot.h>
#include <sigcx/convert.h>
#include <sigcx/tunnel.h>

#include "yehia/script.h"
#include "yehia/plugin.h"

namespace
{

using namespace std;
using namespace Yehia;

class scriptPlugin : public Plugin
{
  public:
    scriptPlugin(PluginManager& mgr) : Plugin(mgr) { }
    virtual std::string description() const { return "scripting plugin"; }
};

Script::Any call_signal_emit(const std::list<Script::Any>& args)
{
  using namespace Script;

  if (args.size() < 1)
    throw BadParam();
  
  Signal *signal = MarshalTraits<Signal *>::unmarshal(args.front());
  
  if (!signal)
    throw Script::BadParam();
  
  std::list<Any> rest_args(++args.begin(), args.end());
  return signal->emit(rest_args);
}

class myPluginNodeIterator : public PluginNode::iterator
{
  public:
    myPluginNodeIterator(const PluginNode::iterator& it) 
        : PluginNode::iterator(it) { }
    virtual ~myPluginNodeIterator() { }
};

class myPluginNodeRecursiveIterator : public PluginNode::recursive_iterator
{
  public:
    myPluginNodeRecursiveIterator(const PluginNode::iterator& it) 
        : PluginNode::recursive_iterator(it) { }
    virtual ~myPluginNodeRecursiveIterator() { }
};

bool pn_iterator_next(myPluginNodeIterator& it) { 
  return (++it).node != NULL;
}

PluginNode *pn_iterator_object(const myPluginNodeIterator& it) { 
  return new PluginNode(it);
}

PluginNode *pn_rec_iterator_object(const myPluginNodeRecursiveIterator& it) { 
  return new PluginNode(it);
}

bool pn_rec_iterator_next(myPluginNodeRecursiveIterator& it) { 
  return (++it).node != NULL;
}

std::list<std::string > ucxxinl1__(PluginManager& self)
{

      return self.arch_indep_paths();
    }
std::list<std::string > ucxxinl2__(PluginManager& self)
{

      return self.arch_dep_paths();
    }
PluginNode * ucxxinl3__(PluginManager& self)
{

      return new PluginNode(self.plugins());
    }
myPluginNodeIterator * ucxxinl4__(PluginNode& self, const string& name)
{

      PluginNode::const_iterator it = self.find(name);
      return new myPluginNodeIterator(it);
    }
myPluginNodeIterator * ucxxinl5__(PluginNode& self, const string& name, Plugin *plugin)
{

      return new myPluginNodeIterator(self.insert(name, plugin).node);
    }
void ucxxinl6__(PluginNode& self, PluginNode::iterator& it)
{

      self.erase(it);
    }
void ucxxinl7__(PluginNode& self, PluginNode::iterator& start, PluginNode::iterator& end)
{

      self.erase(start, end);
    }
myPluginNodeIterator * ucxxinl8__(PluginNode& self)
{

      return new myPluginNodeIterator(self.begin());
    }
//
// Yehia Script interface for namespace yehia
//
void yehia_ns_yehia_register(Yehia::Script::Language& lang)
{
  using namespace Yehia::Script;
  using SigC::retype_return;
  using SigC::slot;
  
  ObjectFactory& factory = lang.factory();
  Signature bases;
  
  NamespaceBuilder nsyehia(factory, lang.root_namespace(), "yehia");
  
  ClassBuilder<SigC::Object> clObject(factory, nsyehia, "Object");
  clObject.constructor(Constructor0<SigC::Object>());
  clObject.method("reference", slot(&SigC::Object::reference));
  clObject.method("unreference", slot(&SigC::Object::unreference));
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(SigC::Object));
  ClassBuilder<Signal> clSignal(factory, nsyehia, "Signal", bases);
  clSignal.constructor(Constructor0<ScriptSignal>());
  clSignal.method("connect", slot(&Signal::connect));
  {

      Signature sig;
      sig.push_back(&clSignal.class_typeid());
      sig.push_back(&typeid(Any));
      clSignal.add_method("emit", slot(&call_signal_emit), sig);
      }
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(SigC::Object));
  ClassBuilder<SigCX::Dispatcher> clDispatcher(factory, nsyehia, "Dispatcher", bases);
  clDispatcher.method("exit", slot(&SigCX::Dispatcher::exit));
  clDispatcher.method("run", slot(&SigCX::Dispatcher::run));
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(SigCX::Dispatcher));
  ClassBuilder<SigCX::StandardDispatcher> clStandardDispatcher(factory, nsyehia, "StandardDispatcher", bases);
  clStandardDispatcher.constructor(Constructor0<SigCX::StandardDispatcher>());
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(SigC::Object));
  ClassBuilder<ErrorHandler> clErrorHandler(factory, nsyehia, "ErrorHandler", bases);
  clErrorHandler.constructor(Constructor0<ErrorHandler>());
  clErrorHandler.constructor(Constructor1<ErrorHandler, ErrorHandler *>());
  clErrorHandler.getter("parent", slot(&ErrorHandler::parent));
  clErrorHandler.setter("parent", slot(&ErrorHandler::set_parent));
  clErrorHandler.method("emit_error", slot(&ErrorHandler::emit_error));
  clErrorHandler.method("emit_log", slot(&ErrorHandler::emit_log));
  clErrorHandler.getter("log", SignalWrapper<ErrorHandler>(&ErrorHandler::log).getter());
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(ErrorHandler));
  ClassBuilder<PluginManager> clPluginManager(factory, nsyehia, "PluginManager", bases);
  clPluginManager.class_method("instance", slot(&PluginManager::instance));
  clPluginManager.getter("arch_indep_paths", slot(&ucxxinl1__));
  clPluginManager.getter("arch_dep_paths", slot(&ucxxinl2__));
  clPluginManager.getter("plugins", slot(&ucxxinl3__));
  clPluginManager.method("load_plugin", slot(&PluginManager::load_plugin));
  clPluginManager.method("release_plugin", slot(&PluginManager::release_plugin));
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(ErrorHandler));
  ClassBuilder<Plugin> clPlugin(factory, nsyehia, "Plugin", bases);
  clPlugin.constructor(Constructor1<scriptPlugin, PluginManager&>());
  clPlugin.getter("description", slot(&Plugin::description));
  clPlugin.getter("manager", slot(&Plugin::manager));
  ClassBuilder<PluginNode> clPluginNode(factory, nsyehia, "PluginNode");
  clPluginNode.getter("full_name", slot(&PluginNode::full_name));
  clPluginNode.getter("name", slot(&PluginNode::name));
  clPluginNode.getter("plugin", slot(&PluginNode::plugin));
  clPluginNode.setter("plugin", slot(&PluginNode::set_plugin));
  clPluginNode.method("is_ancestor_of", slot(&PluginNode::is_ancestor_of));
  clPluginNode.method("find", slot(&ucxxinl4__));
  clPluginNode.method("insert", slot(&ucxxinl5__));
  clPluginNode.method("erase", slot((bool (PluginNode::*)(const std::string&))&PluginNode::erase));
  clPluginNode.method("erase", slot(&ucxxinl6__));
  clPluginNode.method("erase", slot(&ucxxinl7__));
  clPluginNode.getter("begin", slot(&ucxxinl8__));
  ClassBuilder<myPluginNodeIterator> clPluginNodeIterator(factory, nsyehia, "PluginNodeIterator");
  clPluginNodeIterator.constructor(WrapConstructor1<myPluginNodeIterator, const myPluginNodeIterator&>());
  clPluginNodeIterator.method("advance", slot(&::pn_iterator_next));
  clPluginNodeIterator.getter("object", slot(&::pn_iterator_object));
  bases.erase(bases.begin(), bases.end());
  bases.push_back(&typeid(myPluginNodeIterator));
  ClassBuilder<myPluginNodeRecursiveIterator> clPluginNodeRecursiveIterator(factory, nsyehia, "PluginNodeRecursiveIterator", bases);
  clPluginNodeRecursiveIterator.constructor(WrapConstructor1<myPluginNodeRecursiveIterator, const myPluginNodeIterator&>());
  clPluginNodeRecursiveIterator.constructor(WrapConstructor1<myPluginNodeRecursiveIterator, const myPluginNodeRecursiveIterator&>());
  clPluginNodeRecursiveIterator.method("advance", slot(&::pn_rec_iterator_next));
  clPluginNodeRecursiveIterator.getter("object", slot(&::pn_rec_iterator_object));
}

//
// Plugin 'yehia'
//
class yehiaPlugin : public Yehia::Plugin
{
  public:
    yehiaPlugin(Yehia::PluginManager& mgr) : Plugin(mgr) {
      using namespace Yehia::Script;
      std::list<Language *> langs = LanguageManager::instance().languages();
      
      for (std::list<Language *>::iterator it = langs.begin();
           it != langs.end(); ++it)
      {
        SigCX::tunnel<void, Yehia::Script::Language&>(slot(&yehia_ns_yehia_register), **it, (**it).tunnel());
      }
      LanguageManager::instance().language_registered.connect(SigC::slot(*this, &yehiaPlugin::got_new_language));
    }
    virtual std::string description() const { return "Yehia scripting support"; }
    void got_new_language(const std::string& name) {
      using namespace Yehia::Script;
      Language *lang = LanguageManager::instance().language(name);
      if (lang)
      {
        SigCX::tunnel<void, Yehia::Script::Language&>(slot(&yehia_ns_yehia_register), *lang, (*lang).tunnel());
      }
    }
};

extern "C" Yehia::Plugin *yehia_yehia_plugin_init(Yehia::PluginManager *mgr)
{
  try
  {
    Yehia::Plugin *plugin = SigC::manage(new yehiaPlugin(*mgr));
    plugin->reference();
    return plugin;
  }
  catch (...)
  {
    mgr->set_error("yehiaPlugin initialisation failed");
    return 0;
  }
}


}

