/*
 * app.c
 */

/*
 * yank  -  yet another NoteKeeper
 * Copyright (C) 1999, 2000, 2001 Michael Humann <m.hussmann@home.ins.de>
 *
 * This program is free software; you  can redistribute it and/or modify it
 * under the terms of  the GNU General  Public License as published  by the
 * Free Software Foundation;  either version 2 of  the License, or (at your
 * option) any later version.
 *
 * This program  is  distributed in the  hope  that it will  be useful, but
 * WITHOUT  ANY   WARRANTY;  without   even   the  implied    warranty   of
 * MERCHANTABILITY or    FITNESS FOR A PARTICULAR   PURPOSE.   See  the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the  GNU General Public License along
 * with this program; if not, write to the  Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
  
#include "app.h"

#ifdef USE_GAL
#include <gal/e-paned/e-hpaned.h>
#endif /* USE_GAL */
#include <signal.h>
#include <stdlib.h>
#include <stdio.h>

#include "callbacks.h"
#include "e-dateedit.h"
#include "fileio.h"
#include "menus.h"
#include "preferences.h"
#include "run_pipe.h"
#include "time_label.h"
#ifdef USE_GLADE_NOTES
#include "gdd_notes.h"
#endif /* USE_GLADE_NOTES */
#include "text.xpm"
#include "box.xpm"
#include "box2.xpm"
#include "circle.xpm"
#include "circle2.xpm"

/*
 * widgets
 */

GtkWidget        *title_entry = NULL;
GtkWidget        *user_entry = NULL;
static GtkWidget *todo_check = NULL;
static GtkWidget *prio_entry = NULL;
static GtkWidget *dused_entry = NULL;
static GtkWidget *destm_entry = NULL;
static GtkObject *ad_complete = NULL;    /* adjustment for the ... */
static GtkWidget *complete_entry = NULL; /* ... completion entry */
static GtkWidget *de_deadline = NULL;
static GtkWidget *tlabel_created = NULL;
static GtkWidget *tlabel_changed = NULL;
static GtkWidget *label_changes = NULL;
static GtkWidget *de_expire = NULL;
static GtkWidget *note_id_entry = NULL;
GtkWidget        *text_entry = NULL;
GtkWidget        *note_tree = NULL;      /* tree with notes on the left */
GtkWidget        *b_ok = NULL;
GtkWidget        *b_apply = NULL;
GtkWidget        *b_cancel = NULL;
GtkWidget        *todolist = NULL;       /* todolist on the right */
static GtkWidget *todolist_scrw = NULL;  /* window which hosts the list */
static GtkWidget *status = NULL;         /* statusbar */
static GtkWidget *prio_date_box = NULL;  /* hbox used for showing/ hiding */
static GtkWidget *hpaned1 = NULL;        /* needed for default_gui_size() */
static GtkWidget *display_vbox = NULL;   /* container for oac_vbox& views */
static GtkWidget *oac_vbox = NULL;       /* box with buttons, notes */
static GtkWidget *frame_vbox = NULL;     /* container for notes */
static GtkWidget *std_vbox = NULL;       /* std-notes */


/*
 * pixmaps
 */

GtkWidget *pix_text_xpm;
GtkWidget *pix_box_xpm;
GtkWidget *pix_box2_xpm;
GtkWidget *pix_circle_xpm;
GtkWidget *pix_circle2_xpm;

/*
 * prototypes
 */

static GtkWidget  *gui_ext_flags(void);
static gint       determine_use_fontset(void);
static void       set_font_str(GtkWidget *, gchar *);
static void       set_font(GtkWidget *, GdkFont *);
static void       box_flipflop(GtkBox *box, GtkWidget *widget);

/* ------------------------------------------------------ */
/*
 * draw the main gui
 */

GtkWidget*
yank_app_new(const gchar *geometry, const gchar *file)
{
    yank_preferences      *prefs;
    static GtkTargetEntry drop_types[] =
    {
        { "text/uri-list", 0, 1 },
    };
    static gint n_drop_types = sizeof (drop_types)/sizeof (drop_types [0]);
    gchar     *treetitle[] =
    {
        _("NoteTree"),
        NULL
    };
    gchar     *todotitles[] =
    {
        _("Deadline"),
        _("Priority"),
        _("Complete"),
        _("Title"),
        _("User"),
        NULL
    };
    GtkWidget   *app;
    GtkWidget   *scrolledwindow1;
/*     GtkWidget   *scrolledwindow2; */
    GtkWidget   *scrolledwindow3;
    GtkWidget   *hbox;
    GtkWidget   *hbox3;
    GtkWidget   *vbox;
    GtkWidget   *label;
    GdkFont     *font;
    GtkObject   *prio_adjust;
    GtkObject   *dused_adjust;
    GtkObject   *destm_adjust;
    typedef struct
    {
        GtkAdjustment  *used_days_adjust;
        GtkAdjustment  *estm_days_adjust;
        GtkAdjustment  *complete_adjust;
    } adjustments;
    adjustments* my_adjustments;
    gint        got_crash;
    GtkWidget   *b_toggle_ext_flags;
    GtkWidget   *hbox_ext_flags;
    
    app = gnome_app_new(PACKAGE, _(PACKAGE));
    yank_main_app(app);
    
    gtk_window_set_policy(GTK_WINDOW(app), FALSE, TRUE, FALSE);
    gtk_window_set_wmclass(GTK_WINDOW(app), PACKAGE, PACKAGE);
    yank_root_win(GTK_WINDOW(app));

#ifdef USE_GAL
    hpaned1 = e_hpaned_new();
#else  /* USE_GAL */
    hpaned1 = gtk_hpaned_new();
#endif /* USE_GAL */
    gtk_widget_show(hpaned1);

    /*
     * main tree
     */
    
    scrolledwindow1 = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow1),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    /* gtk_widget_show will be called later */

#ifdef USE_GAL
    e_paned_add1(E_PANED(hpaned1), scrolledwindow1); 
#else  /* USE_GAL */
    gtk_container_add(GTK_CONTAINER(hpaned1), scrolledwindow1);
#endif /* USE_GAL */

    gtk_widget_push_visual(gtk_widget_get_default_visual ());
    gtk_widget_push_colormap(gtk_widget_get_default_colormap ());
    note_tree = gtk_ctree_new_with_titles(1, 0, treetitle);
    gtk_widget_show(note_tree);
    gtk_container_add(GTK_CONTAINER(scrolledwindow1), note_tree);
    gtk_clist_set_column_width(GTK_CLIST(note_tree), 0, 80);
    gtk_clist_set_column_auto_resize(GTK_CLIST(note_tree), 0, TRUE);
    gtk_clist_set_row_height(GTK_CLIST(note_tree), 19);
    gtk_clist_set_selection_mode(GTK_CLIST(note_tree), GTK_SELECTION_EXTENDED);
    gtk_clist_set_reorderable(GTK_CLIST(note_tree), TRUE);
    gtk_clist_set_use_drag_icons(GTK_CLIST(note_tree), TRUE);
    gtk_clist_column_titles_show(GTK_CLIST(note_tree));
    gtk_signal_connect(GTK_OBJECT(note_tree), "tree_select_row",
                       GTK_SIGNAL_FUNC (note_tree_row_selected), NULL);
    gtk_signal_connect(GTK_OBJECT(note_tree), "tree_unselect_row",
                       GTK_SIGNAL_FUNC (note_tree_row_unselected), NULL);
    gtk_signal_connect(GTK_OBJECT(note_tree), "click_column",
                       GTK_SIGNAL_FUNC (cb_note_tree_col_selected), NULL);
    gtk_signal_connect(GTK_OBJECT(note_tree), "drag_data_received",
		       GTK_SIGNAL_FUNC(cb_note_tree_reordered), NULL);
    /*
     * FIXME:
     * dnd on the note-tree disables reordering of notes by dnd
     */
/*     gtk_signal_connect(GTK_OBJECT(note_tree), "drag_data_received", */
/*                        GTK_SIGNAL_FUNC(cb_note_tree_drop), NULL); */
/*     gtk_drag_dest_set(note_tree, GTK_DEST_DEFAULT_MOTION | */
/*                       GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP, */
/*                       drop_types, n_drop_types, GDK_ACTION_COPY); */

    display_vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(display_vbox);
    
#ifdef USE_GAL
    e_paned_add2(E_PANED(hpaned1), display_vbox); 
#else  /* USE_GAL */
    gtk_container_add(GTK_CONTAINER(hpaned1), display_vbox);
#endif
    /*
     * todo-list
     */
    
    todolist_scrw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(todolist_scrw),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    gtk_widget_show(todolist_scrw);
    gtk_box_pack_start(GTK_BOX(display_vbox), todolist_scrw, TRUE, TRUE, 0);
    
    todolist = gtk_clist_new_with_titles(5, todotitles);
    gtk_signal_connect(GTK_OBJECT(todolist), "select_row",
                       GTK_SIGNAL_FUNC (cb_todo_row_selected), NULL);
    gtk_signal_connect(GTK_OBJECT(todolist), "click_column",
                       GTK_SIGNAL_FUNC (cb_todo_col_selected), NULL);
    gtk_signal_connect(GTK_OBJECT(todolist), "drag_data_received",
                       GTK_SIGNAL_FUNC(cb_todo_list_drop), NULL);
    gtk_drag_dest_set(todolist, GTK_DEST_DEFAULT_MOTION |
                      GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
                      drop_types, n_drop_types, GDK_ACTION_COPY);
    gtk_widget_show(todolist);
    gtk_container_add(GTK_CONTAINER(todolist_scrw), todolist);
    gtk_clist_column_titles_show(GTK_CLIST(todolist));
    gtk_clist_set_column_auto_resize(GTK_CLIST(todolist), 0 , TRUE);
    gtk_clist_set_column_auto_resize(GTK_CLIST(todolist), 3 , TRUE);
    gtk_clist_set_column_justification(GTK_CLIST(todolist), 1,
                                       GTK_JUSTIFY_RIGHT);
    gtk_clist_set_column_justification(GTK_CLIST(todolist), 2,
                                       GTK_JUSTIFY_RIGHT);
    

    /*
     * text/ data
     */
    
    oac_vbox = gtk_vbox_new(FALSE, 3);
    gtk_widget_hide(oac_vbox);
    gtk_box_pack_start(GTK_BOX(display_vbox), oac_vbox, TRUE, TRUE, 0);

    frame_vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(frame_vbox);
    gtk_box_pack_start(GTK_BOX(oac_vbox), frame_vbox, TRUE, TRUE, 0);
    
    vbox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(vbox);
    yank_register_note_widget(vbox);
    std_vbox = vbox;
    
    hbox = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox);
    gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 3);

    prio_date_box = gtk_vbox_new(FALSE, 3);
    gtk_widget_show(prio_date_box);
    gtk_box_pack_start(GTK_BOX(vbox), prio_date_box, FALSE, FALSE, 0);
    
    hbox_ext_flags = gui_ext_flags();
    gtk_box_pack_start(GTK_BOX(vbox), hbox_ext_flags, FALSE, FALSE, 3);

    label = gtk_label_new(_("Title:"));
    font = gtk_widget_get_style(label)->font;
    gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

    title_entry = gtk_entry_new();
    gtk_widget_set_usize(title_entry, -1, gdk_string_height(font, "W") + 10);
    gtk_widget_show(title_entry);
    gtk_box_pack_start(GTK_BOX(hbox), title_entry, TRUE, TRUE, 5);

    label = gtk_label_new(_("User:"));
    font = gtk_widget_get_style(label)->font;
    gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
    gtk_widget_show(label);
    gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

    user_entry = gtk_entry_new();
    gtk_widget_set_usize(user_entry, gdk_string_width(font, "WWWWWWWW"),
                         gdk_string_height(font, "W") + 10);
    gtk_widget_show(user_entry);
    gtk_box_pack_start(GTK_BOX(hbox), user_entry, FALSE, FALSE, 5);

    todo_check = gtk_check_button_new_with_label(_("Done"));
    gtk_widget_set_usize(todo_check, -1, gdk_string_height(font, "W") + 10);
    gtk_widget_show(todo_check);
    gtk_box_pack_start(GTK_BOX(hbox), todo_check, FALSE, FALSE, 5);

    /*
     * button for extra flags& data
     */
    
    b_toggle_ext_flags = gtk_button_new();
    {
        GtkWidget *p_up;
        GtkWidget *p_down;
        GtkWidget *box;
        
        box = gtk_hbox_new(FALSE, 0);
        gtk_container_add(GTK_CONTAINER(b_toggle_ext_flags), box);
        gtk_widget_show(box);
        p_up = gnome_stock_pixmap_widget_at_size(
            GTK_WIDGET(yank_root_win(NULL)), GNOME_STOCK_BUTTON_UP, 12, 14);
        p_down = gnome_stock_pixmap_widget_at_size(
            GTK_WIDGET(yank_root_win(NULL)), GNOME_STOCK_BUTTON_DOWN, 12, 14);
        gtk_widget_hide(p_up);
        gtk_widget_show(p_down);
        gtk_container_add(GTK_CONTAINER(box), p_up);
        gtk_container_add(GTK_CONTAINER(box), p_down);
        gtk_object_set_data(GTK_OBJECT(b_toggle_ext_flags), "up", p_up);
        gtk_object_set_data(GTK_OBJECT(b_toggle_ext_flags), "down", p_down);
    }
    gtk_widget_show(b_toggle_ext_flags);
    gtk_box_pack_start(GTK_BOX(hbox), b_toggle_ext_flags, FALSE, FALSE, 5);
    gtk_signal_connect(GTK_OBJECT(b_toggle_ext_flags), "clicked",
                       GTK_SIGNAL_FUNC(cb_toggle_ext_flags), hbox_ext_flags);
    
    /*
     * deadline/ prio/ complete
     */
    
    {
        GtkWidget *prio_hbox1;
        GtkWidget *prio_hbox2;
        
        prio_hbox1 = gtk_hbox_new(FALSE, 0);
        gtk_widget_show(prio_hbox1);
        gtk_box_pack_start(GTK_BOX(prio_date_box), prio_hbox1, FALSE, FALSE, 3);
        prio_hbox2 = gtk_hbox_new(FALSE, 0);
        gtk_widget_show(prio_hbox2);
        gtk_box_pack_start(GTK_BOX(prio_date_box), prio_hbox2, FALSE, FALSE, 3);
        
        label = gtk_label_new(_("Deadline:"));
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox1), label, FALSE, FALSE, 5);
        de_deadline = e_date_edit_new();
        e_date_edit_set_show_date(E_DATE_EDIT(de_deadline), TRUE);
        e_date_edit_set_show_time(E_DATE_EDIT(de_deadline), TRUE);
        e_date_edit_set_allow_no_date_set(E_DATE_EDIT(de_deadline), TRUE);
        e_date_edit_set_time_popup_range(E_DATE_EDIT(de_deadline), 0, 23);
        gtk_widget_show(de_deadline);
        gtk_box_pack_start(GTK_BOX(prio_hbox1), de_deadline, FALSE, FALSE, 0);
        
        label = gtk_label_new(_("Priority:"));
        gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox1), label, FALSE, FALSE, 5);
        
        prio_adjust = gtk_adjustment_new(0, 0, 9999, 1, 10, 10);
        prio_entry = gtk_spin_button_new(GTK_ADJUSTMENT(prio_adjust), 1, 0);
        /* NOTE: this is note 100% ok */
        gtk_widget_set_usize(GTK_WIDGET(prio_entry),
                             gdk_string_width(font, "999999") + 10,
                             gdk_string_height(font, "W") + 10);
        gtk_widget_show(prio_entry);
        gtk_box_pack_start(GTK_BOX(prio_hbox1), prio_entry, FALSE, FALSE, 0);

/* days used */
        label = gtk_label_new(_("Days used:"));
        gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), label, FALSE, FALSE, 5);
        
        dused_adjust = gtk_adjustment_new(0, 0, 9999, 1, 10, 10);
        dused_entry = gtk_spin_button_new(GTK_ADJUSTMENT(dused_adjust), 1, 1);
        /* NOTE: this is note 100% ok */
        gtk_widget_set_usize(GTK_WIDGET(dused_entry),
                             gdk_string_width(font, "999999") + 10,
                             gdk_string_height(font, "W") + 10);
        gtk_widget_show(dused_entry);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), dused_entry, FALSE, FALSE, 0);

/* days estimated */
        label = gtk_label_new(_("Days Estimated:"));
        gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), label, FALSE, FALSE, 5);
        
        destm_adjust = gtk_adjustment_new(0, 0, 9999, 1, 10, 10);
        destm_entry = gtk_spin_button_new(GTK_ADJUSTMENT(destm_adjust), 1, 1);
        /* NOTE: this is note 100% ok */
        gtk_widget_set_usize(GTK_WIDGET(destm_entry),
                             gdk_string_width(font, "999999") + 10,
                             gdk_string_height(font, "W") + 10);
        gtk_widget_show(destm_entry);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), destm_entry, FALSE, FALSE, 0);
        
        label = gtk_label_new(_("Complete:"));
        gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), label, FALSE, FALSE, 5);
        
        ad_complete = gtk_adjustment_new(0, 0, 100, 1, 10, 1);
        complete_entry = gtk_spin_button_new(GTK_ADJUSTMENT(ad_complete), 1, 0);
        gtk_widget_set_usize(GTK_WIDGET(complete_entry),
                             gdk_string_width(font, "99999") + 10,
                             gdk_string_height(font, "W") + 10);
        gtk_widget_show(complete_entry);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), complete_entry, FALSE, FALSE, 0);

        label = gtk_label_new(_("%"));
        gtk_widget_set_usize(label, -1, gdk_string_height(font, "W") + 10);
        gtk_widget_show(label);
        gtk_box_pack_start(GTK_BOX(prio_hbox2), label, FALSE, FALSE, 5);
    }
    
    /*
     * additional note data
     */

    note_id_entry = gtk_entry_new();
/*     gtk_widget_show(note_id_entry); */
    gtk_widget_hide(note_id_entry);
    gtk_box_pack_start(GTK_BOX(hbox), note_id_entry, FALSE, FALSE, 5);

    /*
     * note text
     */
    
    scrolledwindow3 = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolledwindow3),
                                   GTK_POLICY_AUTOMATIC,
                                   GTK_POLICY_AUTOMATIC);
    gtk_widget_show(scrolledwindow3);
    gtk_box_pack_start(GTK_BOX(vbox), scrolledwindow3, TRUE, TRUE, 0);
                       
    text_entry = gtk_text_new(NULL, NULL);
    gtk_text_set_editable(GTK_TEXT(text_entry), TRUE);
    gtk_widget_show(text_entry);
    gtk_container_add(GTK_CONTAINER(scrolledwindow3), text_entry);

    /* text dnd */
    gtk_signal_connect(GTK_OBJECT(text_entry), "drag_data_received",
                       GTK_SIGNAL_FUNC(cb_text_entry_drop), NULL);
    gtk_drag_dest_set(text_entry, GTK_DEST_DEFAULT_MOTION |
                      GTK_DEST_DEFAULT_HIGHLIGHT | GTK_DEST_DEFAULT_DROP,
                      drop_types, n_drop_types, GDK_ACTION_COPY);

    hbox3 = gtk_hbox_new(FALSE, 0);
    gtk_widget_show(hbox3);
    gtk_box_pack_start(GTK_BOX(oac_vbox), hbox3, FALSE, FALSE, 3);

    b_ok = gnome_stock_button(GNOME_STOCK_BUTTON_OK);
    gtk_widget_show(b_ok);
    gtk_box_pack_start(GTK_BOX(hbox3), b_ok, TRUE, FALSE, 5);
    gtk_signal_connect(GTK_OBJECT(b_ok), "clicked",
                       GTK_SIGNAL_FUNC(cb_b_ok), NULL);

    b_apply = gnome_stock_button(GNOME_STOCK_BUTTON_APPLY);
    gtk_widget_show(b_apply);
    gtk_box_pack_start(GTK_BOX(hbox3), b_apply, TRUE, FALSE, 5);
    gtk_signal_connect(GTK_OBJECT(b_apply), "clicked",
                       GTK_SIGNAL_FUNC(cb_b_apply), NULL);
        
    b_cancel = gnome_stock_button(GNOME_STOCK_BUTTON_CANCEL);
    gtk_widget_show(b_cancel);
    gtk_box_pack_start(GTK_BOX(hbox3), b_cancel, TRUE, FALSE, 5);
    gtk_signal_connect(GTK_OBJECT(b_cancel), "clicked",
                       GTK_SIGNAL_FUNC(cb_b_cancel), NULL);

    /*
     * a bit suboptimal
     */
    
    my_adjustments=(adjustments*) malloc(sizeof(adjustments));
    my_adjustments->complete_adjust=GTK_ADJUSTMENT(ad_complete);
    my_adjustments->used_days_adjust=GTK_ADJUSTMENT(dused_adjust);
    my_adjustments->estm_days_adjust=GTK_ADJUSTMENT(destm_adjust);
    
    gtk_signal_connect(GTK_OBJECT(title_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(user_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(de_deadline), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(de_expire), "changed",
		       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(prio_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(dused_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(dused_adjust), "value_changed",
                       GTK_SIGNAL_FUNC(cb_updated_used_days), my_adjustments);
    gtk_signal_connect(GTK_OBJECT(destm_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(destm_adjust), "value_changed",
                       GTK_SIGNAL_FUNC(cb_updated_used_days), my_adjustments);
    gtk_signal_connect(GTK_OBJECT(ad_complete), "value_changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(ad_complete), "value_changed",
                       GTK_SIGNAL_FUNC(cb_complete_updates_done), todo_check);
    gtk_signal_connect(GTK_OBJECT(ad_complete), "value_changed",
                       GTK_SIGNAL_FUNC(cb_updated_complete), my_adjustments);
    gtk_signal_connect(GTK_OBJECT(todo_check), "toggled",
                       GTK_SIGNAL_FUNC(note_changed), NULL);
    gtk_signal_connect(GTK_OBJECT(text_entry), "changed",
                       GTK_SIGNAL_FUNC(note_changed), NULL);

    /*
     * keyboard-control
     */
    
    gtk_signal_connect(GTK_OBJECT(title_entry), "activate",
                       GTK_SIGNAL_FUNC(cb_finished_note), NULL);
    gtk_signal_connect(GTK_OBJECT(user_entry), "activate",
                       GTK_SIGNAL_FUNC(cb_finished_note), NULL);
    gtk_signal_connect(GTK_OBJECT(text_entry), "activate",
                       GTK_SIGNAL_FUNC(cb_finished_note), NULL);

    /*
     * create pixmaps
     */
    

    pix_text_xpm = gnome_pixmap_new_from_xpm_d(text_xpm);
    gtk_widget_show(pix_text_xpm);
    pix_box_xpm = gnome_pixmap_new_from_xpm_d(box_xpm);
    gtk_widget_show(pix_box_xpm);
    pix_box2_xpm = gnome_pixmap_new_from_xpm_d(box2_xpm);
    gtk_widget_show(pix_box2_xpm);
    pix_circle_xpm = gnome_pixmap_new_from_xpm_d(circle_xpm);
    gtk_widget_show(pix_circle_xpm);
    pix_circle2_xpm = gnome_pixmap_new_from_xpm_d(circle2_xpm);
    gtk_widget_show(pix_circle2_xpm);    
    
    /*
     * menus
     */
    
    gnome_app_set_contents(GNOME_APP(app), hpaned1);
    status = gnome_appbar_new(FALSE, TRUE, GNOME_PREFERENCES_USER);
    gnome_app_set_statusbar(GNOME_APP(app), status);
    yank_install_menus_and_toolbar(app);
    yank_tree_item_context_menu(note_tree);
    yank_todo_item_context_menu(todolist);
    yank_text_item_context_menu(title_entry);
    yank_text_item_context_menu(user_entry);
    yank_text_item_context_menu(text_entry);
    
    gtk_signal_connect(GTK_OBJECT(app), "delete_event",
                       GTK_SIGNAL_FUNC(cb_delete_event), NULL);

    /*
     * initialize preferences
     * also preloads plugins
     */
    
    load_preferences();
    prefs = get_preferences();

#ifdef USE_GLADE_NOTES
    if (gdd_notes_build_list() == 0)
    {
        gdd_notes_build_menu(yank_testing_menu_path());
    }
#endif /* USE_GLADE_NOTES */
    
    gtk_text_set_word_wrap(GTK_TEXT(text_entry), prefs->wordwrap);
    gtk_text_set_line_wrap(GTK_TEXT(text_entry), prefs->linewrap);
    
    
    if (prefs->yank_width * prefs->yank_height)
    {
        gtk_window_set_default_size(GTK_WINDOW(app), prefs->yank_width,
                                    prefs->yank_height);
    }
    else
    {
        gtk_window_set_default_size(GTK_WINDOW(app), 600, 400);
    }
    
    if (prefs->use_custom_font)
    {
	if (prefs->note_font)
        {
	    set_note_font_str(prefs->note_font);
	}
	if (prefs->note_tree_font)
	{
	    set_note_tree_font_str(prefs->note_tree_font);
        }
        if (prefs->todolist_font)
        {
            set_todolist_font_str(prefs->todolist_font);
        }
    }
    else
    {
        set_default_font();
    }
    
    
    /*
     * parse geometry if given
     */
    
    if (geometry != NULL)
    {
        gint x, y, w, h;
        if (gnome_parse_geometry(geometry, &x, &y, &w, &h ))
        {
            if (x != -1)
            {
                gtk_widget_set_uposition(app, x, y);
            }
            
            if (w != -1)
            {
                gtk_window_set_default_size(GTK_WINDOW(app), w, h);
            }
        }
        else
        {
            g_error(_("Could not parse geometry string `%s'"), geometry);
        }
    }
    
    if (prefs->note_tree_width)
    {
        gtk_widget_set_usize(scrolledwindow1, prefs->note_tree_width, -1);
    }
    else
    {
        gtk_widget_set_usize(scrolledwindow1, 200, -1);
    }
    gtk_widget_show(scrolledwindow1);

    /*
     * try to read files from killed sessions
     */

    got_crash = get_crash_file();
    
    
    /*
     * load file from the command-line
     */
    
    if (file != NULL && ! got_crash)
    {
        load_notes_from_cli((gchar *)file, GTK_CTREE(note_tree), NULL);
    }

    /*
     * read default file
     */
    
    if (file == NULL && ! got_crash)
    {
        get_def_file();
    }
    
    /*
     * auto load last file
     */
    
    if (file == NULL && ! got_crash && prefs->auto_load_last_file)
    {
        auto_load_last_file(); 
    }
    
    /*
     * register some signal-handlers
     */

    if (signal(SIGTERM, handle_sigterm) == SIG_ERR)
    {
        g_warning("Can't handle SIGTERM");
    }
    if (signal(SIGCHLD, handle_sigchild) == SIG_ERR)
    {
        g_warning("Can't handle SIGCHILD");
    }
    if (signal(SIGPIPE, handle_sigpipe) == SIG_ERR)
    {
        g_warning("Can't handle SIGPIPE");
    }

    /*
     * init autosave
     */
    
    if (prefs->auto_save_minutes)
    {
        autosave_notes(NULL);
    }
    
    return (app);
}

/* ------------------------------------------------------ */
/*
 * adjust the notebook
 */

inline void
show_todo_tab(void)
{
    box_flipflop(GTK_BOX(display_vbox), todolist_scrw);
}

/* ------------------------------------------------------ */
/*
 * adjust the notebook
 */

inline void
show_text_tab(void)
{
    box_flipflop(GTK_BOX(display_vbox), oac_vbox);
}

/* ------------------------------------------------------ */
/*
 * wipe the text fields
 */

void
clear_text_page(gboolean add, gint *uncommitted)
{
    guint len;
    gint  ucm;

    ucm = *uncommitted;
    gtk_entry_set_text(GTK_ENTRY(title_entry), "");
    gtk_entry_set_text(GTK_ENTRY(user_entry), g_get_user_name());
    if (add == TRUE)
    {
        time_label_set_time(TIME_LABEL(tlabel_created), time(NULL));
    }
    else
    {
        time_label_set_time(TIME_LABEL(tlabel_created), -1);
    }
    time_label_set_time(TIME_LABEL(tlabel_changed), -1);
    if (add == TRUE)
    {
        gtk_label_set_text(GTK_LABEL(label_changes), _("0"));
    }
    else
    {
        gtk_label_set_text(GTK_LABEL(label_changes), "");
    }
    e_date_edit_set_time(E_DATE_EDIT(de_expire), -1);
    gtk_entry_set_text(GTK_ENTRY(note_id_entry), "");
    e_date_edit_set_time(E_DATE_EDIT(de_deadline), -1);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(prio_entry), 3);
    gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(todo_check), FALSE);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(destm_entry), 0);
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(dused_entry), 0);
    gtk_adjustment_set_value(GTK_ADJUSTMENT(ad_complete), 0);
    len = gtk_text_get_length(GTK_TEXT(text_entry));
    if (len)
    {
        gtk_text_freeze(GTK_TEXT(text_entry));
        gtk_text_set_point(GTK_TEXT(text_entry), 0);
        gtk_text_forward_delete(GTK_TEXT(text_entry), len);
        gtk_text_thaw(GTK_TEXT(text_entry));
    }
    *uncommitted = ucm;
}

/* ------------------------------------------------------ */
/*
 * show the correct portion of the note
 */

void
view_note_as(_notetype type,
             GtkWidget *widget)
{
    switch (type)
    {
    case TextNote:
        box_flipflop(GTK_BOX(frame_vbox), std_vbox);
        gtk_widget_hide(prio_date_box);
        gtk_widget_hide(todo_check);
        break;
    case CheckNote:
        box_flipflop(GTK_BOX(frame_vbox), std_vbox);
        gtk_widget_hide(prio_date_box);
        gtk_widget_show(todo_check);
        break;
    case TodoNote:
        box_flipflop(GTK_BOX(frame_vbox), std_vbox);
        gtk_widget_show(prio_date_box);
        gtk_widget_show(todo_check);
        break;
#ifdef USE_GLADE_NOTES
    case Testing:
        box_flipflop(GTK_BOX(frame_vbox), widget);
        break;
#endif /* USE_GLADE_NOTES */
    case None:
    default:
        g_warning(_("You shouldn't see this warning - please report."));
        break;
    };
}

/* ------------------------------------------------------ */
/*
 * insert data into a struct _note_data
 */

note_data * 
fill_note_data_from_page(_notetype notetype)
{
    note_data *note;
    gchar     *l_ptr;
    
    note = (note_data *) malloc(sizeof (note_data));
    if (note == NULL)
    {
        g_warning(_("Error while allocating memory for note (0)"));
        return (NULL);
        /* notreached */
    }

    note->notetype = notetype;
#ifdef USE_GLADE_NOTES
    if (notetype == Testing)
    {
        gdd_notes_fill_note_data_from_page(note);
        return (note);
        /* notreached */
    }
#endif /* USE_GLADE_NOTES */
    
    note->title = g_strdup(gtk_entry_get_text(GTK_ENTRY(title_entry)));
    note->user = g_strdup(gtk_entry_get_text(GTK_ENTRY(user_entry)));
    if (note->title == NULL)
    {
        g_warning(_("Error while allocating memory for note (1)"));
        return (NULL);
    }
    note->text = gtk_editable_get_chars(GTK_EDITABLE(text_entry), 0,
                                        gtk_text_get_length(GTK_TEXT(
                                            text_entry)));
    if (note->text == NULL)
    {
        g_warning(_("Error while allocating memory for note (2)"));
        return (NULL);
    }
    if (notetype == CheckNote || notetype == TodoNote)
    {
        note->todo = gtk_toggle_button_get_active(
            GTK_TOGGLE_BUTTON(todo_check));
    }
    else
    {
        note->todo = 0;
    }
    
    note->expire = e_date_edit_get_time(E_DATE_EDIT(de_expire));
    note->created = time_label_get_time(TIME_LABEL(tlabel_created));
    note->changed = time(NULL);
    time_label_set_time(TIME_LABEL(tlabel_changed), note->changed);

    gtk_label_get(GTK_LABEL(label_changes), &l_ptr);
    note->changes = atoi(l_ptr);
    note->changes++;
    l_ptr = g_strdup_printf("%ld", note->changes);
    gtk_label_set_text(GTK_LABEL(label_changes), l_ptr);
    g_free(l_ptr);
    
    note->id = g_strdup(gtk_entry_get_text(GTK_ENTRY(note_id_entry)));
    if (note->id == NULL)
    {
        g_warning(_("Error while allocating memory for note (3)"));
        return (NULL);
    }
    
    if (notetype == TodoNote)
    {
        note->deadline = e_date_edit_get_time(E_DATE_EDIT(de_deadline));
        note->prio = gtk_spin_button_get_value_as_int(
            GTK_SPIN_BUTTON(prio_entry));
        note->estm_days = gtk_spin_button_get_value_as_float(
            GTK_SPIN_BUTTON(destm_entry));
        note->used_days = gtk_spin_button_get_value_as_float(
            GTK_SPIN_BUTTON(dused_entry));

        note->complete = GTK_ADJUSTMENT(ad_complete)->value;

    }
    else
    {
        note->deadline = -1;
        note->prio = 0;
        note->estm_days = 0.0;
        note->used_days = 0.0;
        note->complete = 0;
    }
    
    return (note);
}

/* ------------------------------------------------------ */
/*
 * display the note text
 */

void
fill_page_from_note_data(note_data *note)
{
    gint  len;
    gchar *buff;

#ifdef USE_GLADE_NOTES
    if (note->notetype == Testing)
    {
        gdd_notes_fill_page_from_note_data(note);
        return;
        /* notreached */
    }
#endif /* USE_GLADE_NOTES */
    
    if (note->title != NULL)
    {
        gtk_entry_set_text(GTK_ENTRY(title_entry), note->title);
    }
    if (note->user != NULL)
    {
        gtk_entry_set_text(GTK_ENTRY(user_entry), note->user);
    }
    if (note->text != NULL)
    {
        len = strlen(note->text);
        gtk_text_insert(GTK_TEXT(text_entry), NULL, NULL, NULL, note->text,
                        len);
        gtk_editable_set_position(GTK_EDITABLE(text_entry), 0);
    }

    e_date_edit_set_time(E_DATE_EDIT(de_expire), note->expire);
    time_label_set_time(TIME_LABEL(tlabel_created), note->created);
    time_label_set_time(TIME_LABEL(tlabel_changed), note->changed);

    buff = g_strdup_printf("%ld", note->changes);
    gtk_label_set_text(GTK_LABEL(label_changes), buff);
    g_free(buff);
    
    if (note->id != NULL)
    {
        gtk_entry_set_text(GTK_ENTRY(note_id_entry), note->id);
    }
    
    if (note->notetype == CheckNote || note->notetype == TodoNote)
    {
        gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(todo_check),
                                     note->todo);
    }
    if (note->notetype == TodoNote)
    {
        e_date_edit_set_time(E_DATE_EDIT(de_deadline), note->deadline);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(prio_entry), note->prio);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(destm_entry), note->estm_days);
        gtk_spin_button_set_value(GTK_SPIN_BUTTON(dused_entry), note->used_days);

        gtk_signal_handler_block_by_func(
            GTK_OBJECT(ad_complete), GTK_SIGNAL_FUNC(cb_complete_updates_done),
            todo_check);
        gtk_adjustment_set_value(GTK_ADJUSTMENT(ad_complete), note->complete);
        gtk_signal_handler_unblock_by_func(
            GTK_OBJECT(ad_complete), GTK_SIGNAL_FUNC(cb_complete_updates_done),
            todo_check);
    }
}

/* ------------------------------------------------------ */
/*
 * get/ set pointer to the root-window
 */

GtkWindow *
yank_root_win(GtkWindow *win)
{
    static GtkWindow *rwin = NULL;

    if (win != NULL)
    {
        rwin = win;
    }
    
    return (rwin);
}

/* ------------------------------------------------------ */
/*
 * (un-)display toolbar, statusbar
 */

void
show_tool_status(GnomeApp *app, yank_preferences *pref)
{
    GnomeDockItem *gditem;
    
    gditem = gnome_app_get_dock_item_by_name(app, GNOME_APP_TOOLBAR_NAME);
    if (pref->hide_toolbar)
    {
        gtk_widget_hide(GTK_WIDGET(gditem));
    }
    else
    {
        gtk_widget_show(GTK_WIDGET(gditem));
    }
    if (pref->hide_statusbar)
    {
        gtk_widget_hide(app->statusbar);
    }
    else
    {
        gtk_widget_show(app->statusbar);
    }
    gtk_widget_queue_resize(GTK_WIDGET(app));
}

/* ------------------------------------------------------ */
/*
 * reset gui to default sizes
 */

void
default_gui_size(GtkWidget *w, gpointer p)
{
    gdk_window_resize(GTK_WIDGET(yank_root_win(NULL))->window, 600, 400);
    
    /*
     * FIXME: this works only once
     */

#ifdef USE_GAL
    gtk_widget_set_usize(E_PANED(hpaned1)->child1, 200, -1);
#else  /* USE_GAL */
    gtk_widget_set_usize(GTK_PANED(hpaned1)->child1, 200, -1);
#endif /* USE_GAL */
}

/* ------------------------------------------------------ */
/*
 * sets the text in statusbar ...
 */

inline
void
set_status_text(const gchar *txt)
{
    gnome_appbar_set_status(GNOME_APPBAR(status), txt);
}

/* ------------------------------------------------------ */
/*
 * set/ get main widget
 */

GtkWidget*
yank_main_app(GtkWidget *app)
{
    static GtkWidget *main_app = NULL;

    if (app != NULL)
    {
        main_app = app;
    }

    return (main_app);
}

/* ------------------------------------------------------ */
/*
 * change font style
 */

static void
set_font(GtkWidget *w, GdkFont *font)
{
    GtkStyle *style;
    
    style = gtk_style_copy(gtk_widget_get_style(w));
    gdk_font_unref(style->font);
    gdk_font_ref(font);
    style->font = font;
    gtk_widget_set_style(w, style);
    gtk_style_unref(style);
}

/* ------------------------------------------------------ */
/*
 * change font style 
 */

static void
set_font_str(GtkWidget *w, gchar *fontstr)
{
    GdkFont *font;
    gchar   *msg;

    if (determine_use_fontset())
    {
        font = gdk_fontset_load(fontstr);
    }
    else
    {
        font = gdk_font_load(fontstr);
    }
    
    if (font != NULL)
    {
        set_font(w, font);
    }
    else
    {
        msg = g_strdup_printf(_("Can't load font %s"), fontstr);
        set_status_text(msg);
        g_free(msg);
    }   
}

/* ------------------------------------------------------ */
/*
 * change fonts for notes
 */

void
set_note_font_str(gchar *fontstr)
{
    GdkFont *font;
    gint    size;
    gint    width;
    
    gtk_widget_ensure_style(text_entry);
    set_font_str(text_entry, fontstr);
    gtk_widget_ensure_style(title_entry);
    set_font_str(title_entry, fontstr);
    gtk_widget_ensure_style(user_entry);
    set_font_str(user_entry, fontstr);
    gtk_widget_ensure_style(de_deadline);
    set_font_str(de_deadline, fontstr);
    gtk_widget_ensure_style(prio_entry);
    set_font_str(prio_entry, fontstr);
    gtk_widget_ensure_style(dused_entry);
    set_font_str(dused_entry, fontstr);
    gtk_widget_ensure_style(destm_entry);
    set_font_str(destm_entry, fontstr);
    gtk_widget_ensure_style(complete_entry);
    set_font_str(complete_entry, fontstr);
    
    font = gtk_widget_get_style(de_deadline)->font;
    size = gdk_string_height(font, "W") + 10;
    width = gdk_string_width(font, _("99/99/99999"));
    gtk_widget_set_usize(de_deadline, -1, size);
    width = gdk_string_width(font, _("99:999"));
    gtk_widget_set_usize(title_entry, -1, size);
    width = gdk_string_width(font, _("WWWWWWWW"));
    gtk_widget_set_usize(user_entry, width, size);
    gtk_widget_set_usize(prio_entry, -1, size);
    gtk_widget_set_usize(dused_entry, -1, size);
    gtk_widget_set_usize(destm_entry, -1, size);
    gtk_widget_set_usize(complete_entry, -1, size);
}

/* ------------------------------------------------------ */
/*
 * set tree font
 */

void
set_note_tree_font_str(gchar *fontstr)
{
    gtk_widget_ensure_style(note_tree);
    set_font_str(note_tree, fontstr);
}

/* ------------------------------------------------------ */
/*
 * set todolist font
 */

void
set_todolist_font_str(gchar *fontstr)
{
    gtk_widget_ensure_style(todolist);
    set_font_str(todolist, fontstr);
}

/* ------------------------------------------------------ */
/*
 * set default font
 */

void
set_default_font()
{
    GtkWidget *dummy_widget;
    GdkFont   *font;
    gint      size;
    gint      width;
    
    dummy_widget = gtk_label_new("");
    gtk_widget_ensure_style(dummy_widget);
    font = gtk_widget_get_style(dummy_widget)->font;
    
    gtk_widget_ensure_style(text_entry);
    set_font(text_entry, font);
    gtk_widget_ensure_style(title_entry);
    set_font(title_entry, font);
    gtk_widget_ensure_style(user_entry);
    set_font(user_entry, font);
    gtk_widget_ensure_style(de_deadline);
    set_font(de_deadline, font);
    gtk_widget_ensure_style(prio_entry);
    set_font(prio_entry, font);
    gtk_widget_ensure_style(dused_entry);
    set_font(dused_entry, font);
    gtk_widget_ensure_style(destm_entry);
    set_font(destm_entry, font);
    gtk_widget_ensure_style(complete_entry);
    set_font(complete_entry, font);
    
    size = gdk_string_height(font, "W") + 10;
    
    gtk_widget_set_usize(title_entry, -1, size);
    width = gdk_string_width(font, _("WWWWWWWW"));
    gtk_widget_set_usize(user_entry, width, size);
    width = gdk_string_width(font, _("99/99/99999"));
    gtk_widget_set_usize(de_deadline, -1, size);
    width = gdk_string_width(font, _("99:999"));
    gtk_widget_set_usize(prio_entry, -1, size);
    gtk_widget_set_usize(dused_entry, -1, size);
    gtk_widget_set_usize(destm_entry, -1, size);
    gtk_widget_set_usize(complete_entry, -1, size);
    
    gtk_widget_ensure_style(note_tree);
    set_font(note_tree, font);
    gtk_widget_ensure_style(todolist);
    set_font(todolist, font);
    
    gtk_widget_destroy(dummy_widget);
}


/* ------------------------------------------------------ */
/*
 * stolen from gedit
 */

static gint
determine_use_fontset(void)
{
    GtkWidget    *dummy_widget;
    static  gint ret = -1;

    if (ret == -1)
    {
        dummy_widget = gtk_text_new(NULL, NULL);
        gtk_widget_ensure_style(dummy_widget);
        ret = (dummy_widget->style->font->type == GDK_FONT_FONTSET) ? 1 : 0;
        gtk_widget_destroy(dummy_widget);
    }
    
    return (ret);
}

/* ------------------------------------------------------ */
/*
 * Save recent file list and quit
 */

void
yank_app_exit(GtkWidget * app)
{
    gchar const *filename;

    if ((filename = get_filename()))
    {
        if (app == NULL)
        {
            app = yank_main_app(NULL);
        }
	/*
	 * Add filename to list of recent files 
	 */
	yank_recent_files_update(app, filename);
    }
    recent_files_write_config();
    gtk_main_quit();
}

/* ------------------------------------------------------ */
/*
 * extended flags& data
 */

static GtkWidget*
gui_ext_flags(void)
{
    GdkFont   *font;
    GtkWidget *hbox1;
    GtkWidget *table1;
    GtkWidget *label1;
    GtkWidget *label3;
    GtkWidget *label6;
    
    hbox1 = gtk_hbox_new(FALSE, 0);
    gtk_widget_hide(hbox1);
    
    table1 = gtk_table_new(2, 4, FALSE);
    gtk_widget_show(table1);
    gtk_box_pack_start(GTK_BOX(hbox1), table1, FALSE, FALSE, 3);
    gtk_table_set_col_spacings(GTK_TABLE(table1), 4);
    gtk_table_set_row_spacings(GTK_TABLE(table1), 4);
    
    label1 = gtk_label_new(_("Created:"));
    font = gtk_widget_get_style(label1)->font;
    gtk_widget_show(label1);
    gtk_table_attach(GTK_TABLE(table1), label1, 0, 1, 0, 1,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(label1), 1, 0.5);
    
    tlabel_created = time_label_new();
    time_label_set_format(TIME_LABEL(tlabel_created), "%b %d %Y %X");
    gtk_widget_show(tlabel_created);
    gtk_table_attach(GTK_TABLE(table1), tlabel_created, 1, 2, 0, 1,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(tlabel_created), 0, 0.5);
    
    label3 = gtk_label_new(_("Changed:"));
    gtk_widget_show(label3);
    gtk_table_attach(GTK_TABLE(table1), label3, 2, 3, 0, 1,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(label3), 1, 0.5);
    
    tlabel_changed = time_label_new();
    time_label_set_format(TIME_LABEL(tlabel_changed), "%b %d %Y %X");
    gtk_widget_show(tlabel_changed);
    gtk_table_attach(GTK_TABLE(table1), tlabel_changed, 3, 4, 0, 1,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(tlabel_changed), 0, 0.5);
    
    label6 = gtk_label_new(_("Changes:"));
    gtk_widget_show(label6);
    gtk_table_attach(GTK_TABLE(table1), label6, 2, 3, 1, 2,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(label6), 1, 0.5);
    
    label_changes = gtk_label_new("");
    gtk_widget_show(label_changes);
    gtk_table_attach(GTK_TABLE(table1), label_changes, 3, 4, 1, 2,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(label_changes), 0, 0.5);
    
    label6 = gtk_label_new(_("Expires:"));
    gtk_widget_show(label6);
    gtk_table_attach(GTK_TABLE(table1), label6, 0, 1, 1, 2,
                     (GtkAttachOptions)(GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);
    gtk_misc_set_alignment(GTK_MISC(label6), 1, 0.5);

    de_expire = e_date_edit_new();
    e_date_edit_set_show_date(E_DATE_EDIT(de_expire), TRUE);
    e_date_edit_set_show_time(E_DATE_EDIT(de_expire), FALSE);
    e_date_edit_set_allow_no_date_set(E_DATE_EDIT(de_expire), TRUE);
    gtk_widget_show(de_expire);
    gtk_table_attach(GTK_TABLE(table1), de_expire, 1, 2, 1, 2,
                     (GtkAttachOptions)(GTK_EXPAND | GTK_FILL),
                     (GtkAttachOptions)(0), 0, 0);

    return (hbox1);
}

/* ------------------------------------------------------ */
/*
 * display only widget inside the box and hide all others
 */

static void
box_flipflop(GtkBox *box, GtkWidget *widget)
{
    GtkBoxChild *child;
    GList       *list;

    g_return_if_fail(box != NULL);
    g_return_if_fail(GTK_IS_BOX(box));

    for (list = g_list_first(GTK_BOX(box)->children);
         list != NULL;
         list = g_list_next(list))
    {
        child = list->data;
        if (child->widget == widget)
        {
            gtk_widget_show(widget);
        }
        else
        {
            gtk_widget_hide(child->widget);
        }
    }
    
    gtk_widget_queue_resize(GTK_WIDGET(box));
}

/* ------------------------------------------------------ */
/*
 * add a new note-widget
 */

void
yank_register_note_widget(GtkWidget *note_widget)
{
    g_return_if_fail(note_widget != NULL);
    g_return_if_fail(GTK_IS_WIDGET(note_widget));
    
    gtk_widget_hide(note_widget);
    gtk_box_pack_start(GTK_BOX(frame_vbox), note_widget, TRUE, TRUE, 0);
}

/* ------------------------------------------------------ */
