// 	$Id: HarmonicsFile.cc,v 1.8 2003/10/14 18:04:23 flaterco Exp $	
/*  HarmonicsFile  Operations for reading harmonics files.

    Copyright (C) 1998  David Flater.

    Vastly altered by Jan Depner 2002-09 to remove old .txt and .xml
    based code and replace with libtcd.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "common.hh"

// libtcd's open_tide_db is called each time a HarmonicsFile is
// created.  Intentionally, there is no matching call to close_tide_db
// (consquently, no need for a destructor).  See
// http://www.flaterco.com/xtide/tcd_notes.html and XTide changelog
// for version 2.6.3.

HarmonicsFile::HarmonicsFile (Dstr &filename, Settings *in_settings) {
  settings = in_settings;

  myfilename = &filename;

  // Do some sanity checks before invoking libtcd.  (open_tide_db just
  // returns false regardless of what the problem was.)
  {
    FILE *fp = fopen (filename.aschar(), "r");
    if (fp) {
      char c = fgetc (fp);
      if (c != '[') {
        Dstr details (filename.aschar());
        details += " is apparently not a TCD file.\n\
We do not use harmonics.txt or offsets.xml anymore.  Please see\n\
http://www.flaterco.com/xtide/files.html for a link to the current data.";
        barf (CORRUPT_HARMONICS_FILE, details);
      }
      fclose (fp);
    } else {
      // This should never happen.  We statted this file in
      // [xx]TideContext.cc.
      Dstr details (filename.aschar());
      details += ": ";
      details += strerror (errno);
      details += ".";
      barf (CANT_OPEN_FILE, details);
    }
  }

  if (!open_tide_db ((*myfilename).aschar())) {
    Dstr details (*myfilename);
    details += ": libtcd returned generic failure.";
    barf (CORRUPT_HARMONICS_FILE, details);
  }
  db = get_tide_db_header ();
  mynumconst = db.constituents;

  version_string = (char *) db.last_modified;
  version_string += " / ";
  version_string += (char *) db.version;

  ConstituentSet *c = new ConstituentSet (mynumconst);
  unsigned i;
  for (i=0; i<mynumconst; i++) {
    (*c)[i].speed (DegreesPerHour (get_speed (i)));
  }
  Year startyear (db.start_year);
  Year endyear (startyear + (db.number_of_years - 1));
  for (i=0; i<mynumconst; i++) {
    (*c)[i].readboth (i, startyear, endyear);
  }
  constituents = c;

  assert (constituents);
  assert (constituents->constituents);
  assert (!(constituents->constituents[0].firstvalidyear().isNull()));
}

StationRef *
HarmonicsFile::getNextStationRef () {
  int i;
  if ((i = get_next_partial_tide_record (&rec.header)) == -1) return NULL;
  if (rec.header.record_type == 1) {
    ReferenceStationRef *sr = new ReferenceStationRef();
    sr->harmonicsFileName = myfilename;
    sr->filePosition = i;
    sr->name = (char *) rec.header.name;
    sr->coordinates.lat(rec.header.latitude);
    sr->coordinates.lng(rec.header.longitude);
    sr->timezone = (char *) get_tzfile (rec.header.tzfile);
    return sr;
  } else {
    read_tide_record (rec.header.record_number, &rec);
    SubordinateStationRef *sr = new SubordinateStationRef();
    sr->rsr = NULL;
    sr->harmonicsFileName = myfilename;
    sr->name = (char *) rec.header.name;
    sr->coordinates.lat(rec.header.latitude);
    sr->coordinates.lng(rec.header.longitude);
    sr->timezone = (char *) get_tzfile (rec.header.tzfile);
    sr->source = (char *) get_station (rec.header.reference_station);
    if (strlen ((char *) rec.comments) > 5) sr->note = (char *) rec.comments;

    // XTide will simplify this later if need be.
    HairyOffsets *s = new HairyOffsets ();
    PredictionValue::Unit tu;

    Dstr value = (char *) ret_time (rec.min_time_add);
    s->min.timeAdd = Interval (value);
    value = (char *) get_level_units (rec.level_units);
    if (value == "Unknown") 
	value = (char *) get_level_units (rec.avg_level_units);
    tu = PredictionValue::Unit (value);
    s->min.levelAdd = PredictionValue (tu, rec.min_level_add);
    if (rec.min_level_multiply != 0.0) 
      s->min.levelMultiply (rec.min_level_multiply);

    if (rec.min_direction != 361) {
      Angle::qualifier tq = Angle::NONE;
      double tv = (double) rec.min_direction;
      value = (char *) get_dir_units (rec.direction_units);
      if (value == "degrees true") tq = Angle::TRU;
      s->min.currentAngle = NullableAngle (Angle::DEGREES, tq, tv);
    }

    value = (char *) ret_time (rec.max_time_add);
    s->max.timeAdd = Interval (value);
    value = (char *) get_level_units (rec.level_units);
    if (value == "Unknown") 
	value = (char *) get_level_units (rec.avg_level_units);
    tu = PredictionValue::Unit (value);
    s->max.levelAdd = PredictionValue (tu, rec.max_level_add);
    if (rec.max_level_multiply != 0.0) 
      s->max.levelMultiply (rec.max_level_multiply);

    if (rec.max_direction != 361) {
      Angle::qualifier tq = Angle::NONE;
      double tv = (double) rec.max_direction;
      value = (char *) get_dir_units (rec.direction_units);
      if (value == "degrees true") tq = Angle::TRU;
      s->max.currentAngle = NullableAngle (Angle::DEGREES, tq, tv);
    }

    // For a discussion of why zero is not the same as null here,
    // see http://www.flaterco.com/xtide/tcd_notes.html

    if (rec.flood_begins != NULLSLACKOFFSET) {
      value = (char *) ret_time (rec.flood_begins);
      s->floodbegins.timeAdd = Interval (value);
    } else
      s->floodbegins.timeAdd = (s->max.timeAdd + s->min.timeAdd) / 2;

    if (rec.ebb_begins != NULLSLACKOFFSET) {
      value = (char *) ret_time (rec.ebb_begins);
      s->ebbbegins.timeAdd = Interval (value);
    } else
      s->ebbbegins.timeAdd = (s->max.timeAdd + s->min.timeAdd) / 2;

    sr->offsets = s;

    return sr;
  }
}

Station *HarmonicsFile::getStation (StationRef &in_ref,
				    TideContext *in_context) {
  const ReferenceStationRef *rsr = (const ReferenceStationRef *) &in_ref;
  Dstr meridian, units;
  double d_datum;
  unsigned i;

  read_tide_record (rsr->filePosition, &rec);
  if (in_context->settings->in == "y") infer_constituents (&rec);

  meridian = (char *) ret_time (rec.zone_offset);
  units = (char *) get_level_units (rec.units);
  PredictionValue::Unit d_units (units);
  d_datum = rec.datum_offset;
  PredictionValue::Unit a_units;
  if (d_units.mytype == PredictionValue::Unit::KnotsSquared) {
    a_units = PredictionValue::Unit (PredictionValue::Unit::KnotsSquared);
    d_units = PredictionValue::Unit (PredictionValue::Unit::Knots);
  } else
    a_units = d_units;

  ConstantSet *constants = new ConstantSet (mynumconst);
  constants->datum = PredictionValue (d_units, d_datum);

  for (i=0; i<mynumconst; i++) {
    constants->amplitudes[i] = Amplitude (a_units, (double) rec.amplitude[i]);
    // Note negation of phase
    constants->phases[i] = Degrees (-rec.epoch[i]);
  }

  // Normalize the meridian to UTC
  // To compensate for a negative meridian requires a positive offset.
  SimpleOffsets fixMeridianOffsets;
  fixMeridianOffsets.timeAdd = -Interval (meridian);
  constants->adjust (fixMeridianOffsets, *constituents);

  assert (constituents);
  assert (constituents->constituents);
  assert (!(constituents->constituents[0].firstvalidyear().isNull()));
  ConstantSetWrapper *csw = new ConstantSetWrapper (constituents, constants);
  ReferenceStation *s = new ReferenceStation (in_context, csw);
  s->stationRef = &in_ref;
  s->name = rsr->name;
  s->timeZone = rsr->timezone;
  s->coordinates = rsr->coordinates;
  s->myUnits = d_units;
  // FIXME if there are ever units of velocity other than knots
  if (d_units.mytype == PredictionValue::Unit::Knots) {
    s->isCurrent = 1;
    if (a_units.mytype == PredictionValue::Unit::KnotsSquared)
      s->isHydraulicCurrent = 1;
    else
      s->isHydraulicCurrent = 0;
  } else
    s->isCurrent = s->isHydraulicCurrent = 0;
    
  // FIXME if there are ever units of velocity other than knots
  if (in_context->settings->u != "x" && (!(s->isCurrent)))
    s->setUnits (PredictionValue::Unit (in_context->settings->u));
  return s;
}
