/*

    xpuyopuyo - pai.c         Copyright(c) 1999-2003 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Computer AI Code (revision 1, the stupid rule-based AI's)
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#include <xpuyopuyo.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <config.h>
#include <pai.h>
#include <pmanip.h>
#include <ppiece.h>


/* Structure for storing rules for each AI */
/* Positive scores indicate beneficial actions,
   negative scores indicate nonbenificial acts.
   Zero scores indicate AI is indifferent about an action */
typedef struct _airule_int {
   int      rules[P_AIS_NUM_RULES];          /* AI's ruleset (must be 1st in structure) */
   const char *name;                         /* AI's name */
   const char *desc;                         /* Description for AI */
   int      score;                           /* Final score for AI's selected play */
   int      index;                           /* Index of this AI in aiinfo struct */
   int      generation;                      /* Generation of AI (0 == factory) */
   int      wins;                            /* Number of tournament wins for AI */
   int      losses;                          /* Number of tournament losses for AI */
} airule_int;


/* Fallout is a structure for simulating block falls for the AI. */
typedef struct _fallout {
   int x[2];            /* X coordinate of each blob in piece */
   int y[2];            /* Y coordinate of each blob in piece */
   int c[2];            /* Color of each blob in piece */
} fallout;
         
         
/*    DIST     score_distance_factor;
      BLOK1    score_blocked_color_below;
      BLOK2    score_blocked_color_side;
      SAME     score_same_color;
      MATCH    score_match;
      ROCK     score_rocks;
      HIGH     score_high_penalty;
      INTF     score_interference;
      NEXT     score_blocking_next;       
      BMAT     score_both_match;          
      SMAT     score_simul_match;            
      DIAG     score_diagonal_match;   
      SIDE     score_side_attraction;
      PEAK     score_peaks;               */
static const airule_int default_rules[P_AI_COUNT + 1] = {
/* DIST  BLOK1 BLOK2 SAME  MATCH ROCK  HIGH  INTF  NEXT  BMAT  SMAT  DIAG  SIDE  PEAK  */
 {{+6,   -22,  -4,   +18,  +12,  +7,   -100, -30,  -17,  +15,  +30,  +5,   0,    0 },  "Default",                  "" },
 {{0,    0,    0,    +20,  +15,  0,    -100, 0,    0,    +25,  +50,  0,    0,    0 },  "Trainer",                  "" },
 {{+2,   -10,  -3,   +35,  +10,  -10,  -100, -50,  -50,  +100, +10,  +10,  0,    0 },  "Bovine",                   "" },
 {{0,    0,    0,    +20,  +30,  0,    -100, 0,    0,    +70,  +50,  0,    0,    0 },  "Ziggy",                    "" },
 {{-4,   -28,  -13,  +26,  +20,  +4,   -100, -28,  -20,  +34,  +1,   +5,   0,    0 },  "Alpha",                    "" },
 {{-2,   -10,  -6,   +16,  +21,  +2,   -100, +8,   -24,  +33,  +37,  0,    0,    0 },  "Bob",                      "" },
 {{+9,   -11,  -45,  +17,  +20,  +16,  -100, -39,  -24,  +58,  +43,  -4,   0,    0 },  "Ytinasni",                 "" },
 {{-5,   -5,   -5,   +5,   +10,  +1,   -100, -5,   -3,   +15,  +25,  +20,  0,    0 },  "Blue Sky",                 "" },
 {{0,    0,    0,    +1,   0,    0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "Chains",                   "" },
 {{0,    0,    0,    0,    0,    0,    -100, 0,    0,    0,    +1,   0,    0,    0 },  "Matches",                  "" },
 {{0,    -1,   -5,   0,    0,    0,    -100, -10,  -10,  0,    0,    0,    0,    0 },  "Interference",             "" },
 {{0,    0,    0,    0,    0,    0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "Oddity",                   "" },
 {{+2,   -3,   -3,   +10,  +20,  +1,   -100, -10,  -4,   +40,  +60,  +2,   0,    0 },  "Turpin",                   "" },
 {{-1,   +1,   +1,   +5,   +10,  -10,  -100, +10,  +1,   -1,   +10,  +2,   0,    +10 },"The Architect",            "" },
 {{+20,  0,    0,    +2,   +2,   +5,   -100, 0,    0,    +10,  +20,  +1,   -5,   -10 },"Singularity",              "" },
 {{-10,  +10,  -5,   +2,   +20,  0,    -100, +3,   0,    +15,  +30,  -15,  +10,  0 },  "Wallcrawler",              "" },
#if USE_AIBREED
 {{0,    0,    0,    0,    +1,   0,    -100, 0,    0,    0,    0,    -10,  0,    0 },  "AI17",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, 0,    0,    0,    0,    +10,  0,    0 },  "AI18",                     "" },
 {{0,    -2,   -2,   0,    -10,  0,    -100, 0,    0,    +2,   +30,  0,    0,    0 },  "AI19",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, 0,    0,    -1,   -1,   0,    0,    0 },  "AI20",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, 0,    -5,   0,    0,    0,    0,    0 },  "AI21",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, 0,    +5,   0,    0,    0,    0,    0 },  "AI22",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, -5,   0,    0,    0,    0,    0,    0 },  "AI23",                     "" },
 {{0,    0,    0,    0,    +1,   0,    -100, +5,   0,    0,    0,    0,    0,    0 },  "AI24",                     "" },
 {{0,    0,    0,    0,    +1,   -10,  -100, 0,    0,    0,    0,    0,    0,    0 },  "AI25",                     "" },
 {{0,    0,    0,    0,    +1,   +10,  -100, 0,    0,    0,    0,    0,    0,    0 },  "AI26",                     "" },
 {{0,    0,    0,    -1,   +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI27",                     "" },
 {{0,    0,    0,    +2,   +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI28",                     "" },
 {{0,    -5,   -5,   0,    +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI29",                     "" },
 {{0,    +5,   +5,   0,    +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI30",                     "" },
 {{+5,   0,    0,    0,    +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI31",                     "" },
 {{-5,   0,    0,    0,    +1,   0,    -100, 0,    0,    0,    0,    0,    0,    0 },  "AI32",                     "" },
#endif /* Using AI breed? */
 {{0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0,    0 },  "Null",                     "" }
};
 
 
static const char *ai_rule_names[P_AIS_NUM_RULES] = {
   "distance_factor",
   "blocked_color_below",
   "blocked_color_side",
   "same_color",
   "match",
   "rocks",
   "high_penalty",
   "interference",
   "blocking_next",
   "both_match",
   "simul_match",
   "diagonal_match",
   "side_attraction",
   "peaks"
};


static inline int p_ai_get_value(pfield *f, fallout *b, int x, int y) {
/* p_ai_get_value

   Returns the color of the blob at playing field coordinate (x, y).  Since
   the coordinate may be one of the blobs from the simulated fallout, we
   need to check each blob in b to see if it is at the same location.  */

   int i;
   
   for(i = 0; i < 2; i++) if(b->x[i] == x && b->y[i] == y) return(b->c[i]);
   return(p_field_get(f, x, y));

}


static inline int p_ai_fallout(pfield *f, fallout *b, int x, int y) {
/* p_ai_fallout

   This function accepts the playing field and current position
   of the fallout blobs.  It scans down the column until a non-passable
   blob is hit, then returns the Y coordinate of the last passable
   block in the field.  Useful for measuring heights of columns or
   for determining how far each block in the fallout should fall. 
   Note, any blob at (x, y) is ignored in scanning for the first
   non-passable blob; it is usaully assumed that this blob, if it
   exists, is the one we are trying to drop.  */

   y++;
   while(P_IS_CLEAR(p_ai_get_value(f, b, x, y))) y++;
   return(y - 1);

}


static inline void p_ai_drop_blocks(pfield *f, fallout *b) {
/* p_ai_drop_blocks

   Drops both blocks in fallout.  Note, the first block in <b>
   should always be ABOVE or at the same height as the second
   block; never below.  Otherwise, one block will "hang" in the
   air.  We do this check by swapping the order of iteration,
   if necessary.  */

   int i;

   if(b->y[1] > b->y[0]) {     
      /* Block 1 is below block 0 */
      for(i = 1; i >= 0; i--) b->y[i] = p_ai_fallout(f, b, b->x[i], b->y[i]);
   } else {
      for(i = 0; i < 2; i++) b->y[i] = p_ai_fallout(f, b, b->x[i], b->y[i]);
   }

}


static inline void p_ai_set_blocks(fallout *b, int x1, int y1, int c1, int x2, int y2, int c2) {
/* p_ai_set_blocks

   Sets up the falling blocks.  Nothing more than a shortcut to manual
   initialisation                                                       */

   b->x[0] = x1;
   b->y[0] = y1;
   b->c[0] = c1;
   b->x[1] = x2;
   b->y[1] = y2;
   b->c[1] = c2;

}


static inline int p_ai_neighbors(pfield *f, fallout *b, int x, int y) {
/* p_ai_neighbors

   Returns the number of blocks neighboring (x, y) with the same color.
   If (x, y) is not a colored blob, zero is returned.  Otherwise, a
   number from 1 to 5 is returned (the blob at (x, y) is counted) */

   int color;
   int sum;
   
   sum = 0;
   
   color = p_ai_get_value(f, b, x, y);
   if(P_IS_COLOR(color)) {
      /* Blob has a color */
      color = P_COLOR_VALUE(color);
      sum++;
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y)) == color) sum++;
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y)) == color) sum++;
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y - 1)) == color) sum++;
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y + 1)) == color) sum++;
   }
   
   return(sum);

}


static inline int p_ai_extended_neighbors(pfield *f, fallout *b, int x, int y) {
/* p_ai_extended_neighbors 

   This function does a little more than the above.  It checks the blobs
   to the left, right, top and bottom.  If any have the same color, then
   it continues the calculation for one more iteration.  If there is a
   chain of three blobs including the one at (x, y), this function will
   count all of the blobs on that chain.  We can stop after this though
   because there can be no four-blob chains (else they would have already
   matched!) Self is counted, and zero is returned if (x, y) is not a
   colored blob */

   int color;     /* Color at (x, y) */
   int sum;       /* Total number of matching blobs */
   int tl;        /* Cell topleft of (x, y) has already been counted, if nonzero */
   int tr;        /* Cell topright of (x, y) has already been counted, if nonzero */
   int bl;        /* Cell botleft of (x, y) has already been counted, if nonzero */
   int br;        /* Cell botright of (x, y) has already been counted, if nonzero */
   
   /* Default values */
   sum = 0;
   tl = 0;
   tr = 0;
   bl = 0;
   br = 0;
   
   color = p_ai_get_value(f, b, x, y);
   if(P_IS_COLOR(color)) {
      /* Blob has a color */
      color = P_COLOR_VALUE(color);
      sum++;
      /* Check left */
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y)) == color) {
         sum++;
         if(!tl && P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y - 1)) == color) sum++, tl++;
         if(!bl && P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y + 1)) == color) sum++, bl++;
         if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 2, y)) == color) {
            sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 2, y - 1)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 2, y + 1)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 3, y)) == color) sum++;
         }
      }
      /* Check right */
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y)) == color) {
         sum++;
         if(!tr && P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y - 1)) == color) sum++, tr++;
         if(!br && P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y + 1)) == color) sum++, br++;
         if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 2, y)) == color) {
            sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 2, y - 1)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 2, y + 1)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 3, y)) == color) sum++;
         }
      }
      /* Check up */
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y - 1)) == color) {
         sum++;
         if(!tl && P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y - 1)) == color) sum++, tl++;
         if(!tr && P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y - 1)) == color) sum++, tr++;
         if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y - 2)) == color) {
            sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y - 2)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y - 2)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y - 3)) == color) sum++;
         }
      }
      /* Check down */
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y + 1)) == color) {
         sum++;
         if(!bl && P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y + 1)) == color) sum++, bl++;
         if(!br && P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y + 1)) == color) sum++, br++;
         if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y + 2)) == color) {
            sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y + 2)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y + 2)) == color) sum++;
            if(P_COLOR_VALUE(p_ai_get_value(f, b, x, y + 3)) == color) sum++;
         }
      }
   }
   
   return(sum);

}


static inline int p_ai_neighbor_rocks(pfield *f, fallout *b, int x, int y) {
/* p_ai_neighbor_rocks 

   Returns the number of neighboring rocks, or zero if (x, y) has no
   neighboring rocks.  Hopefully (x, y) is not itself a rock, because
   it is not counted. */

   int sum;
   
   sum = 0;
   if(P_IS_ROCK(p_ai_get_value(f, b, x - 1, y))) sum++;
   if(P_IS_ROCK(p_ai_get_value(f, b, x + 1, y))) sum++;
   if(P_IS_ROCK(p_ai_get_value(f, b, x, y - 1))) sum++;
   if(P_IS_ROCK(p_ai_get_value(f, b, x, y + 1))) sum++;
   
   return(sum);

}


static inline int p_color_in(int color, int *next) {
/* p_color_in

   Returns truth if the specified color is in the next piece */

   return(P_COLOR_VALUE(color) == P_COLOR_VALUE(next[0]) || P_COLOR_VALUE(color) == P_COLOR_VALUE(next[1]));

}


static int p_ai_simulate(pfield *f, fallout *b, pfield *f2, int mintomatch) {
/* p_ai_simulate

   This runs a simulation with the specified fallout blocks.  This
   is achieved by copying the playfield (a slow process) and dropping
   the block in that playfield.  The original field f and fallout blocks
   are required, and the fallout blocks should already have been
   "dropped".  f2 may be a temporary field to work in, and must be
   non-null.  The number of matches found is returned. */

   int j;
   int match;
   
   match = 0;

   /* Setup temporary field */   
   p_field_copy(f2, f);
   p_field_set(f2, b->x[0], b->y[0], b->c[0]);
   p_field_set(f2, b->x[1], b->y[1], b->c[1]);
   
   /* While j nonzero, run more matches until none are left */
   j = 1;
   while(j) {
      /* The following returns the number of simultaneous matches */
      j = p_manip_clear_matches(f2, mintomatch);
      if(j) {
         match += j;
         /* If matches found, we need to collapse the playfield so 
            more matches might be made. */
         p_manip_collapse(f2);
      }
   }
           
   if(match < 0) match = 0;
   return(match);

}


static int p_ai_calc_score(pfield *f, fallout *b, airule_int *a, int *next, int mintomatch) {
/* p_ai_calc_score

   Calculate the AI score for field f, with initial position of the
   block specified in fallout structure b, using airuleset a.  The
   next block is also specified as an array of two integers.  
   Large positive scores are best */

   int i;
   int j;
   int x;      /* X coordinate for current blob (shorthand) */
   int y;      /* Y coordinate for current blob (shorthand) */
   int c;      /* Color of current blob (not P_COLOR_VALUE-stripped) */
   int score;  /* Cumulative score */
   int match;  /* If 1, one of the blobs made at least 1 match.  If 2, both */
   int sum;    /* Temporary variables ... */
   int color;
   int color2;
   pfield *f2;
   
   /* Initialise */
   score = 0;
   match = 0;
   for(i = 0; i < 2; i++) {
      /* Get current parameters */
      x = b->x[i];
      y = b->y[i];
      c = b->c[i];

      /* Base score, use distance down */
      score += y * a->rules[P_AIS_DISTANCE_FACTOR];
      if(y < P_AIS_HIGH_THRESHOLD) score += a->rules[P_AIS_HIGH_PENALTY] * (P_AIS_HIGH_THRESHOLD - y);

      /* Check tile beneath - blocking another color? */
      color = p_ai_get_value(f, b, x, y + 1);
      if(P_IS_COLOR(color) && P_COLOR_VALUE(color) != P_COLOR_VALUE(c)) {
         score += a->rules[P_AIS_BLOCKED_COLOR_BELOW] * p_ai_neighbors(f, b, x, y + 1);
         if(p_color_in(color, next)) score += a->rules[P_AIS_BLOCKING_NEXT];
      }
      color = p_ai_get_value(f, b, x - 1, y);
      if(P_IS_COLOR(color) && P_COLOR_VALUE(color) != P_COLOR_VALUE(c)) {
         score += a->rules[P_AIS_BLOCKED_COLOR_SIDE] * p_ai_neighbors(f, b, x - 1, y);
         if(p_color_in(color, next)) score += a->rules[P_AIS_BLOCKING_NEXT];
      }
      color = p_ai_get_value(f, b, x + 1, y);
      if(P_IS_COLOR(color) && P_COLOR_VALUE(color) != P_COLOR_VALUE(c)) {
         score += a->rules[P_AIS_BLOCKED_COLOR_SIDE] * p_ai_neighbors(f, b, x + 1, y);
         if(p_color_in(color, next)) score += a->rules[P_AIS_BLOCKING_NEXT];
      }
      
      /* Check for interference */
      color = p_ai_get_value(f, b, x - 1, y);
      if(P_IS_COLOR(color) && P_COLOR_VALUE(color) != P_COLOR_VALUE(c)) {
         color2 = p_ai_get_value(f, b, x + 1, y);
         if(P_IS_COLOR(color2) && P_COLOR_VALUE(color) == P_COLOR_VALUE(color2)) score += a->rules[P_AIS_INTERFERENCE];
         color2 = p_ai_get_value(f, b, x, y + 1);
         if(P_IS_COLOR(color2) && P_COLOR_VALUE(color) == P_COLOR_VALUE(color2)) score += a->rules[P_AIS_INTERFERENCE];
      } else {
         color = p_ai_get_value(f, b, x + 1, y);
         if(P_IS_COLOR(color) && P_COLOR_VALUE(color) != P_COLOR_VALUE(c)) {
            color2 = p_ai_get_value(f, b, x, y + 1);
            if(P_IS_COLOR(color2) && P_COLOR_VALUE(color) == P_COLOR_VALUE(color2)) score += a->rules[P_AIS_INTERFERENCE];
         }
      }
         
      /* Check neighbors */
      sum = p_ai_extended_neighbors(f, b, x, y);
      score += sum * a->rules[P_AIS_SAME_COLOR];
      if(sum >= 4) {
         score += sum * a->rules[P_AIS_MATCH];
         match++;
      }

      /* Check diagonal neighbors (up) */
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x - 1, y - 1)) == P_COLOR_VALUE(c)) {
         sum = p_ai_extended_neighbors(f, b, x - 1, y - 1);
         score += sum * a->rules[P_AIS_DIAGONAL_MATCH];
      }
      if(P_COLOR_VALUE(p_ai_get_value(f, b, x + 1, y - 1)) == P_COLOR_VALUE(c)) {
         sum = p_ai_extended_neighbors(f, b, x + 1, y - 1);
         score += sum * a->rules[P_AIS_DIAGONAL_MATCH];
      }

      score += p_ai_neighbor_rocks(f, b, x, y) * a->rules[P_AIS_ROCKS];
      
      /* If near edge, do side calc */
      if(x == 0 || x == f->width - 1) score += a->rules[P_AIS_SIDE];
      
      /* Peak vs neighbor calc */
      if(x > 0) {
         for(j = 0; j < f->height && P_IS_CLEAR(p_ai_get_value(f, b, x - 1, j)); j++);
         score += abs(j - y) * a->rules[P_AIS_PEAK];
      }
      if(x + 1 < f->width) {
         for(j = 0; j < f->height && P_IS_CLEAR(p_ai_get_value(f, b, x + 1, j)); j++);
         score += abs(j - y) * a->rules[P_AIS_PEAK];
      }
      
   }

   /* Simulation */
   f2 = p_field_new(f->width, f->height, f->number);
   score += p_ai_simulate(f, b, f2, mintomatch) * a->rules[P_AIS_SIMUL_MATCH];
   p_field_free(&f2);

   /* Did both blobs match? */
   if(match >= 2) {
      score += a->rules[P_AIS_BOTH_MATCH];
   }
   
   return(score);

}


static void p_ai_align(pfield *f, airule_int *a, int *blocks, int *next, int mintomatch, int numcolors) {
/* p_ai_align

   Runs all scores for all possible moves, and decides which move is the
   best for this AI.  Blocks and next are two-integer arrays which store 
   the colors of the current block and the next block.  There are 
   P_FIELD_WIDTH * 2 possible moves where the piece is oriented vertically,
   and P_FIELD_WITDH * 2 - 2 possible moves where the piece is oriented
   horizontally.  */

   fallout b;                             /* Fallout (see above) */
   int scores[f->width * 4 - 2];          /* Listing of scores */
   int *scorep;                           /* Pointer into above */
   int scoremax;                          /* Maximum score seen */
   int match;                             /* Set when a move is finally selected */
   int i;                                 
   
   /* Calculate scores for each possible move.  The scores are
      stored in the array interlaced; the two vertical moves
      starting at x=0 are stored, then the two horizontal moves
      starting at x=0 are stored, and so on.  Note, there are
      no horizontal moves at x=P_FIELD_WIDTH-1, only vertical
      moves */
   scorep = scores;
   scoremax = -9999;
   for(i = 0; i < f->width; i++) {
      p_ai_set_blocks(&b, i, 0, blocks[1], i, 1, blocks[0]);
      p_ai_drop_blocks(f, &b);
      *scorep = p_ai_calc_score(f, &b, a, next, mintomatch);
      if(*scorep > scoremax) scoremax = *scorep;
      scorep++;
      p_ai_set_blocks(&b, i, 0, blocks[0], i, 1, blocks[1]);
      p_ai_drop_blocks(f, &b);
      *scorep = p_ai_calc_score(f, &b, a, next, mintomatch);
      if(*scorep > scoremax) scoremax = *scorep;
      scorep++;

      /* Are there any horizontal blocks here? */
      if(i < f->width - 1) {
         p_ai_set_blocks(&b, i, 0, blocks[0], i + 1, 0, blocks[1]);
         p_ai_drop_blocks(f, &b);
         *scorep = p_ai_calc_score(f, &b, a, next, mintomatch);
         if(*scorep > scoremax) scoremax = *scorep;
         scorep++;
         p_ai_set_blocks(&b, i, 0, blocks[1], i + 1, 0, blocks[0]);
         p_ai_drop_blocks(f, &b);
         *scorep = p_ai_calc_score(f, &b, a, next, mintomatch);
         if(*scorep > scoremax) scoremax = *scorep;
         scorep++;
      }
   }
   
   /* Select a random choice with the maximum score (must be at least one!) */
   match = 0;
   while(!match) {
      i = rand() % (f->width * 4 - 2);
      if(scores[i] == scoremax) match = 1;
   }
   a->score = scores[i];

   /* Determine how the final piece should be set in the field, and rotate it */   
   switch(i % 4) {
      case 1:
         p_piece_rotate_old(f);
      case 3:
         p_piece_rotate_old(f);
      case 0:
         p_piece_rotate_old(f);
      case 2:
         break;
   }
   f->piece->x = i / 4;
   /* piece->y is always going to be the same */
   return;

}


void p_ai_decide(pfield *f, airule *a, int *cur, int *next, int mintomatch, int numcolors) {
/* p_ai_decide

   Interface to above function */

   p_ai_align(f, (airule_int *)a, cur, next, mintomatch, numcolors);
   return;

}


static inline char *p_ai_load_break(char *s, char k) {
/* p_ai_load_break */

   char *p;
   
   p = s;
   while(*p) {
      if(*p == k) {
         *p = 0;
         return(p + 1);
      }
      p++;
   }
   return(NULL);

}


aiinfo *p_ai_load_rules() {
/* p_ai_load_rules */

   airule_int *ai;
   char filename[P_AI_IO_BUFFER];
   char buf[P_AI_IO_BUFFER];
   char *p;
   char *peq;
   char *pcol;
   FILE *f;
   int whichai;
   int whichrule;
   int value;
   int i;
   int j;
   
   /* Allocate memory for data structure */
   if(!(ai = (airule_int *)malloc(sizeof(airule_int) * P_AI_COUNT))) return(NULL);

   /* Set default rules */
   for(i = 0; i < P_AI_COUNT; i++) {
      ai[i].name = default_rules[i].name;
      ai[i].desc = default_rules[i].desc;
      ai[i].index= i;
      ai[i].generation = 0;
      for(j = 0; j < P_AIS_NUM_RULES; j++) {
         ai[i].rules[j] = default_rules[i].rules[j];
      }
   }
   
   /* Attempt to read AI file to get current values */
   snprintf(filename, P_AI_IO_BUFFER, "%s/%s/%s.%d", getenv("HOME"), P_XPUYOPUYO_DIR, P_AI_FILE, P_AI_COUNT);
   if(!(f = fopen(filename, "r"))) {
      #if USE_AIBREED
      printf("ai_load:  Cannot find \"%s\", generating new default ai's.\n", filename);
      p_ai_save_rules((aiinfo *)ai);
      #endif /* AI breeding? */
   } else {
      while(fgets(buf, P_AI_IO_BUFFER, f) > 0) {
         p = buf;
         while(*p) {
            if(*p == '\n') *p = '\0';
            else p++;
         }
         if((peq = p_ai_load_break(buf, '='))) {
            if((pcol = p_ai_load_break(buf, ':'))) {
               value = -1;
               whichai = 0;
               while(whichai < P_AI_COUNT && value < 0) {
                  if(!strcmp(buf, default_rules[whichai].name)) {
                     value = 1;
                  } else whichai++;
               }
               if(value < 0) whichai = atoi(buf) % P_AI_COUNT;
               value = atoi(peq);
               if(!strcmp(pcol, "generation")) {
                  ai[whichai].generation = value;
               } else if(!strcmp(pcol, "wins")) {
                  ai[whichai].wins = value;
               } else if(!strcmp(pcol, "losses")) {
                  ai[whichai].losses = value;
               } else {
                  whichrule = 0;
                  while(whichrule < P_AIS_NUM_RULES && whichrule >= 0) {
                     if(!strcmp(pcol, ai_rule_names[whichrule])) {
                        ai[whichai].rules[whichrule] = value;
                        whichrule = -1;
                     } else whichrule++;
                  }
                  if(whichrule >= 0) {
                     whichrule   =    atoi(pcol) % P_AIS_NUM_RULES;
                     ai[whichai].rules[whichrule] = value;
                  }
               }
            } /* Found : */
         } /* Found = */
      } /* While reading file */
      fclose(f);
   }
               
   return((aiinfo *)ai);

}


void p_ai_save_rules(const aiinfo *ai_) {
/* p_ai_save_rules */

   const airule_int *ai = (airule_int *)ai_;
   char filename[P_AI_IO_BUFFER];
   FILE *f;
   int i;
   int j;
   
   snprintf(filename, P_AI_IO_BUFFER, "%s/%s/%s", getenv("HOME"), P_XPUYOPUYO_DIR, P_AI_FILE);

   if(!(f = fopen(filename, "w"))) {
      printf("ai_save:  Cannot write \"%s\" to save AI rules.\n", filename);
      return;
   }

   fprintf(f, "# xpuyopuyo ai rules file - automatically generated\n#\n");
   for(i = 0; i < P_AI_COUNT; i++) {
      for(j = 0; j < P_AIS_NUM_RULES; j++) {
         fprintf(f, "%s:%s=%d\n", ai[i].name, ai_rule_names[j], ai[i].rules[j]);
      }
      fprintf(f, "%s:generation=%d\n", ai[i].name, ai[i].generation);
      fprintf(f, "%s:wins=%d\n", ai[i].name, ai[i].wins);
      fprintf(f, "%s:losses=%d\n", ai[i].name, ai[i].losses);
   }

   fclose(f);
   return;

}


void p_ai_release_rules(aiinfo **ai_) {
/* p_ai_release_rules */

   if(!ai_ || !*ai_) return;
   free(*ai_);
   *ai_ = NULL;
   return;

}


airule *p_ai_get_rule(aiinfo *ai_, int index) {
/* p_ai_get_rule */

   airule_int *ai = (airule_int *)ai_;

   if(index < 0) {
      /* Select a random AI */
      index = rand() % P_AI_COUNT;
   } else if(index >= P_AI_COUNT) {
      /* Select default AI */
      index = P_AI_DEFAULT;
   }
                                 
   return((airule *)&(ai[index]));

}


#if USE_AIBREED
void p_ai_mutate(const aiinfo *ai, airule *d_, const airule *s_, /* TEMP int n, */ int quiet) {
/* p_ai_mutate */

   airule_int *s = (airule_int *)s_;
   airule_int *d = (airule_int *)d_;
   int mutaterule;
   int mutatedelta;
   #ifdef __TEMP_broken
   int div;
   #endif /* TEMP */

   if(rand() % 2 == 0) return;
   if(d->wins > s->wins && rand() % 2 == 0) return;

   mutaterule  = rand() % P_AIS_NUM_RULES;
   mutatedelta = rand() % (P_AIS_MUTATE_MAX - 1) + 1;
   if(rand() % 4 == 0) mutatedelta = -mutatedelta;
   if(d->rules[mutaterule] > s->rules[mutaterule]) mutatedelta = -mutatedelta;
   #ifdef __TEMP_broken
   div = 1;
   while(n < P_AI_COUNT) {
      div *= 2;
      n *= 2;
   }
   if(div >= P_AIS_MUTATE_MAX / 2) div = P_AIS_MUTATE_MAX / 2;
   mutatedelta /= div;
   #endif /* TEMP */
   if(d->wins < s->wins) mutatedelta = mutatedelta * 2;
   if(mutatedelta == 0) return;
   
   if(!quiet) printf("\nAI is mutating!  %s[%d].%s[%d] (%d) is %d (winner)\n", s->name, s->index, ai_rule_names[mutaterule], mutaterule, s->generation, s->rules[mutaterule]);
   if(!quiet) printf("                 %s[%d].%s[%d] (%d) is %d %c %d\n\n", d->name, d->index, ai_rule_names[mutaterule], mutaterule, d->generation, d->rules[mutaterule], mutatedelta < 0 ? '-' : '+', abs(mutatedelta));

   d->rules[mutaterule] += mutatedelta;
   d->generation++;
   p_ai_save_rules(ai);
   
   return;
   
}
#endif /* Breeding? */
   

const char *p_ai_get_name(const airule *a_) {
/* p_ai_get_name */

   return(((airule_int *)a_)->name);

}


const char *p_ai_get_desc(const airule *a_) {
/* p_ai_get_desc */

   return(((airule_int *)a_)->desc);
   
}


int p_ai_get_score(const airule *a_) {
/* p_ai_get_score */

   return(((airule_int *)a_)->score);
   
}


int p_ai_get_index(const airule *a_) {
/* p_ai_get_index */

   return(((airule_int *)a_)->index);
   
}


const char *p_ai_name_of(const aiinfo *ai_, int rule) {
/* p_ai_name_of */

   if(rule < 0) return("Random AI"); 
   return(((airule_int *)ai_)[rule].name);
   
}


const char *p_ai_desc_of(const aiinfo *ai_, int rule) {
/* p_ai_desc_of */

   if(rule < 0) return("Randomly selected AI rule"); 
   return(((airule_int *)ai_)[rule].desc);
   
}


int p_ai_rule_of(const aiinfo *ai, int aidx, int ridx) {

   return(((airule_int *)ai)[aidx].rules[ridx]);

}


int p_ai_generation_of(const aiinfo *ai, int aidx) {

   return(((airule_int *)ai)[aidx].generation);

}


int p_ai_win_count_of(const aiinfo *ai, int aidx) {

   return(((airule_int *)ai)[aidx].wins);

}


void p_ai_inc_win_count(const aiinfo *ai, int aidx) {

   ((airule_int *)ai)[aidx].wins++;
   return;

}


int p_ai_loss_count_of(const aiinfo *ai, int aidx) {

   return(((airule_int *)ai)[aidx].losses);

}


void p_ai_inc_loss_count(const aiinfo *ai, int aidx) {

   ((airule_int *)ai)[aidx].losses++;
   return;

}


int p_ai_default_rule_of(const aiinfo *ai, int aidx, int ridx) {

   return(default_rules[aidx].rules[ridx]);

}


const char *p_ai_rule_name(int ridx) {

   return(ai_rule_names[ridx]);

}


const char *p_ai_rule_text_of(const aiinfo *ai, int aidx) {
    
   return(p_ai_desc_of(ai, aidx));

}
