/*\
|*|  Functions for changing/checking the playlist.
\*/

#include "cdread.h"
#include "playlist.h"

extern GList *playlist;
extern pthread_mutex_t playlist_mutex;

#ifndef HAVE_GLOBAL_PLAYLIST_DELETE_NODE

#ifdef HAVE_GLOBAL_PLAYLIST_POSITION
extern PlaylistEntry *playlist_position;
#else
#warning Strange behaviour when deleting the currently playing song.
#endif

/*\ Copied directly from xmms source \*/
static void
playlist_delete_node(GList *node, gboolean *set_info_text, gboolean *restart_playing)
{
	/* Caller should hold playlist mutex */
	PlaylistEntry *entry;
	GList *playing_song;
	/*
	 * We call g_list_find manually here because
	 * we don't want an item in the shuffle_list
	 */
	entry = (PlaylistEntry *) node->data;
#ifdef HAVE_GLOBAL_PLAYLIST_POSITION
	playing_song = g_list_find(playlist, playlist_position);
	if (playing_song == node)
	{
		*set_info_text = TRUE;
		if (get_input_playing())
		{
			pthread_mutex_unlock(&playlist_mutex);
			input_stop();
			pthread_mutex_lock(&playlist_mutex);
			*restart_playing = TRUE;
		}
		if (g_list_next(playing_song))
			playlist_position = g_list_next(playing_song)->data;
		else if (g_list_previous(playing_song))
			playlist_position = g_list_previous(playing_song)->data;
		else if (node != playlist)
			playlist_position = playlist->data;
		else
			playlist_position = NULL;
	}
#endif /* HAVE_GLOBAL_PLAYLIST_POSITION */
	if (entry->filename)
		g_free(entry->filename);
	if (entry->title)
		g_free(entry->title);
	playlist = g_list_remove_link(playlist, node);
	g_free(entry);
	g_list_free_1(node);
}
#else /* HAVE_GLOBAL_PLAYLIST_DELETE_NODE */
void playlist_delete_node(GList *node, gboolean *set_info_text, gboolean *restart_playing);
#endif /* HAVE_GLOBAL_PLAYLIST_DELETE_NODE */

static GList *
list_try_replace(GList *node, GList *ln)
{
	PlaylistEntry *entry = (PlaylistEntry *) node->data;
	while (ln) {
		if (!strcmp((gchar *)ln->data, entry->filename)) {
			entry->length = -1;
			if (entry->title) g_free(entry->title);
			entry->title = 0;
			return ln;
		}
		ln = g_list_next(ln);
	}
	return 0;
}

/*\ Replace entries beginning with 'prefix' with entries from 'list'
|*|  Overlapping entries are only 'dirtied'
|*|  Extra entries in the playlist are removed.
|*|  Extra 'list' entries are added directly behind the last existing entry
|*| If no entries are found, nothing is changed, returns FALSE
|*|  'list' is freed.
|*|
|*| P.S. this is rather inefficient, and could be much better using
|*|      'internal' glib knowledge, but as you'd normally use pretty
|*|      small playlists with CDs, it's OK.  And a lot less iffy.
\*/
gboolean
playlist_replace(gchar *prefix, GList *list)
{
	gboolean restart_playing = FALSE,
		 set_info_text = FALSE,
		 regen_shuffle = FALSE;
	GList *node;
	GList *plist_pos_list;
	gint pos = -1;

	pthread_mutex_lock(&playlist_mutex);
	/*\ First, dirty all overlapping entries, remove them from 'list',
	|*| and remove all playlist entries not in 'list'
	\*/
	node = playlist;
	while (node) {
		PlaylistEntry *entry;
		GList *next;

		entry = (PlaylistEntry *) node->data;
		if (g_list_index(playlist, entry) == -1) {
			node = playlist;
			continue;
		}
		next = g_list_next(node);
		if (!memcmp(entry->filename, prefix, strlen(prefix))) {
			GList *tmp;
			pos = g_list_index(playlist, entry);
			tmp = list_try_replace(node, list);
			if (tmp) {
				pos++;
				list = g_list_remove_link(list, tmp);
				g_free(tmp->data);
				g_list_free_1(tmp);
			} else {
				regen_shuffle = TRUE;
				playlist_delete_node(node, &set_info_text,
						&restart_playing);
			}
		}
		node = next;
	}
	if (pos < 0) {
		pthread_mutex_unlock(&playlist_mutex);
		node = list;
		while (node) {
			g_free(node->data);
			node = g_list_next(node);
		}
		g_list_free(list);
		return FALSE;
	}
	/*\ Now add any remaining entries in 'list'
	|*|  after the position of the last matching entry.
	\*/
	while (list) {
		GList *next = g_list_next(list);
		PlaylistEntry *entry;
		
		CNEW(entry, 1);
		regen_shuffle = TRUE;

		entry->filename = (gchar *)list->data;
		entry->length = -1;
		playlist = g_list_insert(playlist, entry, pos);
		pos++;
		g_list_free_1(list);
		list = next;
	}
	pthread_mutex_unlock(&playlist_mutex);
	
	playlistwin_update_list();
	if (regen_shuffle) playlist_generate_shuffle_list();
	if (set_info_text) mainwin_set_info_text();
	if (restart_playing)
	{
		/* mainwin_set_song_info(0, 0, 0); */
		playlist_play();
	}
	return TRUE;
}

/*\
|*| Force a reread of all files with this prefix
\*/
void
playlist_dirty(gchar *prefix)
{
	GList *node;

	pthread_mutex_lock(&playlist_mutex);
	node = playlist;
	while (node) {
		PlaylistEntry *entry = (PlaylistEntry *)node->data;
		if (!memcmp(entry->filename, prefix, strlen(prefix))) {
			entry->length = -1;
			if (entry->title) g_free(entry->title);
			entry->title = 0;
		}
		node = g_list_next(node);
	}
	pthread_mutex_unlock(&playlist_mutex);
	playlist_start_get_info_scan();
}

/*\
|*| Return a GList containing playlist entries matching prefix, in order
|*|  (For PLAYORDER)
|*| returned GList must be g_list_free()d.
\*/
GList *
playlist_find(gchar *prefix)
{
	GList *node, *ret = 0;

	pthread_mutex_lock(&playlist_mutex);
	node = playlist;
	while (node) {
		PlaylistEntry *entry = (PlaylistEntry *)node->data;
		if (!memcmp(entry->filename, prefix, strlen(prefix))) {
			gchar *tmp;
			tmp = g_strdup(entry->filename + strlen(prefix));
			ret = g_list_insert(ret, tmp, -1);
		}
		node = g_list_next(node);
	}
	pthread_mutex_unlock(&playlist_mutex);
	return ret;
}

/*\
|*| Return the number of matching entries
\*/
gint
playlist_check(gchar *prefix)
{
	GList *node;
	gint fnd = 0;

	pthread_mutex_lock(&playlist_mutex);
	node = playlist;
	while (node) {
		PlaylistEntry *entry = (PlaylistEntry *)node->data;
		if (!memcmp(entry->filename, prefix, strlen(prefix)))
			++fnd;
		node = g_list_next(node);
	}
	pthread_mutex_unlock(&playlist_mutex);
	return fnd;
}
