/* $Id: xkbsel.c,v 1.5 1999/07/25 10:57:23 stano Exp $

   xkbsel simple text mode selector

   (C) 1999 Stanislav Meduna <stano@eunet.sk>
*/

#include <config.h>

#include <xkbselx.h>

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <libintl.h>
#include <limits.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>
#include <locale.h>

#define _(s) gettext(s)

static void usage(void);
static void help(void);

static int select_map(const char *arg);
static int select_next(void);
static int select_prev(void);

static int test(const char *arg);

Display *disp;

int main(int argc, char **argv)
{
	int c;

	int oper = 0;
	int res;

	char test_arg[PATH_MAX+1];
	char map_arg[MAX_DATA_PATH_LEN+1];

	setlocale(LC_ALL, "");
	textdomain(PACKAGE);

	*map_arg = 0;

	while(1)
	{
		static struct option long_options[] =
		{
        	     { "select", 1, 0, 's' },
        	     { "next", 0, 0, 'n' },
        	     { "previous", 0, 0, 'p' },
        	     { "verbose", 0, 0, 'v' },
        	     { "debug", 0, 0, 0 },
        	     { "help", 0, 0, 0 },
        	     { "version", 0, 0, 0 },
        	     { "test", 2, 0, 't' },
		     { 0, 0, 0, 0 }
		};

		int option_index = 0;

		c = getopt_long(argc, argv, "s:npvt::", long_options, &option_index);
		if (c == -1)
			break;

		switch(c)
		{
		case 0:
			/* Long option without corresponding short one */
			if (!strcmp("help", long_options[option_index].name))
			{
				help();
				exit(0);
			}
			else if (!strcmp("version", long_options[option_index].name))
			{
				printf("xkbsel (xkbsel) %s\n", VERSION);
				exit(0);
			}
			if (!strcmp("debug", long_options[option_index].name))
			{
				flag_debug = 1;
			}
			else
			{
				usage();
				exit(0);
			}
			break;

		case 't':
		case 's':
		case 'n':
		case 'p':
			if (oper)
			{
				fprintf(stderr, _("Options -s, -n and -p are mutually exclusive\n"));
				usage();
				exit(1);
			}
			oper = c;

			if (oper == 't')
			{
				if (optarg != NULL)
					strcpy(test_arg, optarg);
				else
					*test_arg = 0;
			}
			else if (oper == 's')
			{
				if (strlen(optarg) >= MAX_DATA_PATH_LEN)
				{
					fprintf(stderr, _("Map name too long\n"));
					exit(1);
				}
				strcpy(map_arg, optarg);
			}
			break;

		case 'v':
			flag_verbose = 1;
			break;

		case '?':
			{
				usage();
				exit(1);
			}
			break;

		default:
			break;
		}
	}

	if (! oper)
		oper = 's';

	if (optind < argc && oper != 's')
	{
		fprintf(stderr, _("Unexpected arguments for this mode\n"));
		exit(1);
	}

	while (optind < argc)
	{
		if (*map_arg)
		{
			fprintf(stderr, _("Only one map allowed\n"));
			exit(1);
		}
		
		if (strlen(argv[optind]) >= MAX_DATA_PATH_LEN)
		{
			fprintf(stderr, _("Map name too long\n"));
			exit(1);
		}
		strcpy(map_arg, argv[optind]);

		optind++;
	}

	if ((! oper || oper == 's') && ! *map_arg)
	{
		fprintf(stderr, _("Map name not set\n"));
		exit(1);
	}

	read_config();

	switch(oper)
	{
	case 't':
		res = test(test_arg);
		break;

	case 's':
		res = select_map(map_arg);
		break;

	case 'n':
		res = select_next();
		break;

	case 'p':
		res = select_prev();
		break;
	}

	if (res < 0)
		exit(2);

	exit(0);
}


static void usage(void)
{
	fprintf(stderr, _("Usage: xkbsel OPTION...\n"));
}


static void help(void)
{
	usage();

	printf(_("Selects a keyboard\n"));

	help_line("s", _("select=MAP"),  _("select a map (default)"));
	help_line("n", "next",           _("select next map"));
	help_line("p", "previous",       _("select previous map"));
	help_line("v", "verbose",        _("be verbose"));
	help_line("",  "debug",          _("enter debug mode"));

	help_line("",  "help",           _("display this help and exit"));
	help_line("",  "version",        _("output version information and exit"));
	printf(_("Report bugs to <stano@trillian.eunet.sk>\n"));
}


static int select_map(const char *arg)
{
	int r;
	sel_info_t info;
	char *p1, *p2;

	r = open_db();
	if (r < 0)
		return r;

	r = install_map(arg);
	if (r < 0)
	{
		close_db();
		return r;
	}

	close_db();

	disp = XOpenDisplay(display_name);

	if (disp == NULL)
	{
		fprintf(stderr, _("Cannot connect to X server\n"));
		return -1;
	}

	strcpy(info.map_name, arg);

	p1 = strchr(info.map_name, '(');
	p2 = strchr(info.map_name, ')');

	if (p1 != NULL && p2 != NULL && p2 > p1 && p2-p1 >= 2)
	{
		*p2 = 0;
		strncpy(info.shortcut, p1+1, sizeof(info.shortcut)-1);
		info.shortcut[sizeof(info.shortcut)-1] = 0;

	}
	else
	{
		int idx, found;

		/* Shortcut only - find full name */
		r = get_locale_map(arg, &info, &idx, &found);
		if (r < 0)
			return r;
		
		if (! found)
		{
			strcpy(info.shortcut, arg);
			strcpy(info.map_name, arg);
		}
	}

	r = publish_current_sel(disp, &info);

	XCloseDisplay(disp);

	if (r < 0)
		return r;

	return 0;
}

static int select_next(void)
{
	int r;
	int found;
	sel_info_t info;

	if (! n_locale_maps)
	{
		fprintf(stderr, _("No configured maps\n"));
		return -1;
	}

	disp = XOpenDisplay(display_name);

	if (disp == NULL)
	{
		fprintf(stderr, _("Cannot connect to X server\n"));
		return -1;
	}

	/* Get current selection */
	r = get_current_sel(disp, &info, &found);

	if (r < 0)
	{
		XCloseDisplay(disp);
		return r;
	}

	if (found)
	{
		int idx;
		
		r = get_locale_map(info.shortcut, NULL, &idx, &found);
		if (r < 0)
		{
			XCloseDisplay(disp);
			return r;
		}

		if (! found)
		{
			fprintf(stderr, _("Selected map %s is not from configuration - using first one\n"), info.shortcut);
			idx = -1;
		}

		idx++;
		if (idx == n_locale_maps)
			idx = 0;

		info = locale_maps[idx];
	}
	else
		info = locale_maps[0];

	r = open_db();
	if (r < 0)
	{
		XCloseDisplay(disp);
		return r;
	}

	r = install_map(info.map_name);
	if (r < 0)
	{
		XCloseDisplay(disp);
		close_db();
		return r;
	}

	close_db();

	r = publish_current_sel(disp, &info);

	XCloseDisplay(disp);

	if (r < 0)
		return r;

	return 0;
}

static int select_prev(void)
{
	int r;
	int found;
	sel_info_t info;

	if (! n_locale_maps)
	{
		fprintf(stderr, _("No configured maps\n"));
		return -1;
	}

	disp = XOpenDisplay(display_name);

	if (disp == NULL)
	{
		fprintf(stderr, _("Cannot connect to X server\n"));
		return -1;
	}

	/* Get current selection */
	r = get_current_sel(disp, &info, &found);

	if (r < 0)
	{
		XCloseDisplay(disp);
		return r;
	}

	if (found)
	{
		int idx;
		
		r = get_locale_map(info.shortcut, NULL, &idx, &found);
		if (r < 0)
		{
			XCloseDisplay(disp);
			return r;
		}

		if (! found)
		{
			fprintf(stderr, _("Selected map %s is not from configuration - using last one\n"), info.shortcut);
			idx = n_locale_maps;
		}

		if (idx == 0)
			idx = n_locale_maps;
		idx--;

		info = locale_maps[idx];
	}
	else
		info = locale_maps[n_locale_maps - 1];

	r = open_db();
	if (r < 0)
	{
		XCloseDisplay(disp);
		return r;
	}

	r = install_map(info.map_name);
	if (r < 0)
	{
		XCloseDisplay(disp);
		close_db();
		return r;
	}

	close_db();

	r = publish_current_sel(disp, &info);

	XCloseDisplay(disp);

	if (r < 0)
		return r;

	return 0;
}


static int test(const char *arg)
{
	return 0;
}

