/*  xhangglider
 *  Copyright (C) 1999 Yasuhiro Take
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <X11/Xlib.h>

#include "def.h"
#include "3ddraw.h"
#include "pilot.h"
#include "init.h"

#define MAX2(_x,_y) ((_x) > (_y) ? (_x) : (_y))
#define MIN2(_x,_y) ((_x) < (_y) ? (_x) : (_y))

PolygonData polygons_load, polygons_orig[8], polygons_shown;
static Display *dpy;
static Window win, vario;
static Pixmap pix, back, back2, clip[8][2];
static GC gc, variogc, bmapgc;
static Font font;
static XColor blue, red;

char datadir[256] = DEFAULT_DATA_PATH;
char backcolor[80] = "DeepSkyBlue";

double px[8], py[8], pz[8], direction_r[8];
int bank_r[8];
double relative_speed[8];
char nobankflag[8];

int pilot_num = 1, thermal_num = 5;

char DrawingMode = 0, Transient = 0, Vario = 0;
char Pylon = 0, Verbose = 0;
float Frequency = 0.1;

int scr_width, scr_height, scr_x, scr_y;
double thermal_x[5], thermal_y[5], pylon_x[5], pylon_y[5];
double viewpoint_z = 0.0;
double max_y = 450.0;
int pylon[8], pilot_mode[8];

Polygon *realloc_polygons(PolygonData *pd)
{
  Polygon *p;
  
  p = calloc(sizeof(Polygon), pd->n_pol + 10);
  
  memcpy(p, pd->pol, sizeof(Polygon) * pd->n_pol);
  
  pd->n_pol += 10;
  free(pd->pol);
  pd->pol = p;
  
  return p;
}

Polygon *new_polygon(PolygonData *pd)
{
  int i;

  for (;;) {
    for (i = 0; i < pd->n_pol; i++) {
      if (pd->pol[i].type == T_UNUSED) {
	memset(&(pd->pol[i]), 0, sizeof(Polygon));
	pd->pol[i].type = T_TRI;
	
	return &(pd->pol[i]);
      }
    }

    realloc_polygons(pd);
  }
}

int free_polygon(Polygon *pol)
{
  memset(pol, 0, sizeof(Polygon));

  return 0;
}

int get_num_of_valid_polygons(PolygonData *pd)
{
  int i;
  int cnt = 0;

  for (i = 0; i < pd->n_pol; i++) {
    if (pd->pol[i].type != T_UNUSED)
      cnt++;
  }

  return cnt;
}

#define COPY_AT_ONCE 1
#define COPY_EACH 2

int copy_polygons(PolygonData *src, PolygonData *dst, int mode)
{
  int n, i, j;

  if (mode == COPY_EACH)
    n = get_num_of_valid_polygons(src);
  else
    n = src->n_pol;
  
  if (n > dst->n_pol) {
    dst->pol = realloc(dst->pol, sizeof(Polygon) * n);
    dst->n_pol = n;
  }
  
  memset(dst->pol, 0, sizeof(Polygon) * dst->n_pol);

  if (mode == COPY_EACH) {
    j = 0;
    for (i = 0; i < src->n_pol; i++) {
      if (src->pol[i].type != T_UNUSED) {
	memcpy(&(dst->pol[j]), &(src->pol[i]), sizeof(Polygon));
	j++;
      }
    }
  } else {
    memcpy(dst->pol, src->pol, sizeof(Polygon) * src->n_pol);
  }

  return 0;
}

int xinit()
{
  int i;
  unsigned int j, x, y, w, h;
  XColor ecolor;
  XSetWindowAttributes at;
  
  dpy = XOpenDisplay(NULL);

  gc = XCreateGC(dpy, DefaultRootWindow(dpy), 0, 0);
  XSetGraphicsExposures(dpy, gc, False);
  
  win = RootWindow(dpy, 0);
  XGetGeometry(dpy, win, (Window *)(&j), &x, &y, &w, &h, &j, &j);

  if (scr_x == -1)
    scr_x = x;
  if (scr_y == -1)
    scr_y = y;
  if (scr_width == -1)
    scr_width = w;
  if (scr_height == -1)
    scr_height = h;

  if (Transient == 0) {
    XAllocNamedColor(dpy, DefaultColormap(dpy, 0), backcolor, &blue,
		     &ecolor);
    XSetWindowBackground(dpy, win, blue.pixel);
    XSetForeground(dpy, gc, blue.pixel);
    XClearWindow(dpy, win);
  } else {
    at.backing_store = Always;
    XChangeWindowAttributes(dpy, win, CWBackingStore, &at);
    XFlush(dpy);
    back = XCreatePixmap(dpy, RootWindow(dpy, 0), scr_width, scr_height,
			 DisplayPlanes(dpy, 0));
    XClearWindow(dpy, win);
    XCopyArea(dpy, win, back, gc, scr_x, scr_y, scr_width, scr_height, 0, 0);
    at.backing_store = NotUseful;
    XChangeWindowAttributes(dpy, win, CWBackingStore, &at);
    XFlush(dpy);
  }

  if (DrawingMode == DRAW_NOFLICKER)
    back2 = XCreatePixmap(dpy, RootWindow(dpy, 0), scr_width, scr_height,
			  DisplayPlanes(dpy, 0));

  if (DrawingMode != DRAW_NOFLICKER)
    pix = XCreatePixmap(dpy, DefaultRootWindow(dpy), scr_height, scr_height,
			DisplayPlanes(dpy, 0));

  if (DrawingMode == DRAW_USECLIP) {
    for (i = 0; i < pilot_num; i++) {
      clip[i][0] = XCreatePixmap(dpy, RootWindow(dpy, 0), scr_height,
				 scr_height, 1);
      clip[i][1] = XCreatePixmap(dpy, RootWindow(dpy, 0), scr_height,
				 scr_height, 1);
    }
    
    bmapgc = XCreateGC(dpy, clip[0][0], 0, 0);
  } else {
    for (i = 0; i < pilot_num; i++)
      clip[i][0] = clip[i][1] = 0;
  }

  if (Vario) {
    vario = XCreateSimpleWindow(dpy, DefaultRootWindow(dpy), 0, 0, 120,
				8 * pilot_num, 0, 0, WhitePixel(dpy, 0));

    variogc = XCreateGC(dpy, DefaultRootWindow(dpy), 0, 0);
    font = XLoadFont(dpy, "-adobe-courier-medium-r-normal-*-10-*");
    XAllocNamedColor(dpy, DefaultColormap(dpy, 0), "red", &red, &ecolor);
    XSetFont(dpy, variogc, font);
    
    XMapWindow(dpy, vario);
  }

  td_initialize(dpy, clip[0][0]);

  return 0;
}

int convert_colorname_to_pixel()
{
  int i, p;
  XColor col, tmp;

  for (p = 0; p < pilot_num; p++) {
    for (i = 0; i < polygons_orig[p].n_pol; i++) {
      if (polygons_orig[p].pol[i].type != T_UNUSED) {
	XAllocNamedColor(dpy, DefaultColormap(dpy, 0),
			 polygons_orig[p].pol[i].color.name, &col, &tmp);

	polygons_orig[p].pol[i].color.pixel = col.pixel;
      }
    }
  }

  return 0;
}
 
int draw_polygons_3d(int p, Drawable d, Pixmap clip, int xz_r, int xy_r,
		     Vector *v,int zoom, int x, int y)
{
  double cos_yx, sin_yx, cos_yz, sin_yz, b_v_yx, b_v_yz;
  
  copy_polygons(&(polygons_orig[p]), &polygons_shown, COPY_AT_ONCE);

  b_v_yx = sqrt(v->x * v->x + v->y * v->y);
  b_v_yz = sqrt(v->y * v->y + v->z * v->z);
  
  cos_yx = v->y / b_v_yx;
  sin_yx = v->x / b_v_yx;

  cos_yz = v->y / b_v_yz;
  sin_yz = -(v->z / b_v_yz);

  if (nobankflag[p] == 0)
    td_rotate_polygons(xz_r, 0, 0, &polygons_shown, DIM_X, DIM_Z);
  td_rotate_polygons(xy_r, 0, 0, &polygons_shown, DIM_X, DIM_Y);
  
  td_rotate_polygons(999, cos_yx, sin_yx, &polygons_shown, DIM_X, DIM_Y);
  td_rotate_polygons(999, cos_yz, sin_yz, &polygons_shown, DIM_Y, DIM_Z);

  td_draw_polygons(d, clip, &polygons_shown, zoom, x, y);
  
  return 0;
}

double corect_r(double *r)
{
  if (*r <= -180.0)
    *r += 360.0;
  if (*r > 180.0)
    *r -= 360.0;

  return *r;
}

double lidgelift(double x, double y, double z)
{
  double ret = 0.0;
  
  if (z < 0.0 && -70.0 < x && x < 70.0) {
    if (z < -70.0)
      ret = 0.22;
    else if (z < -10.0)
      ret = 0.18;
    else if (z < 10.0)
      ret = 0.13;
    else if (z < 20.0)
      ret = 0.08;

    if (y > -z + 45.0)
      ret = 0;
    if (y > -z + 35.0)
      ret *= 0.75;
  }

  return ret;
}

double lift(double x, double y, double z)
{
  double ret;
  double rnd, d, r;
  int i;

  ret = 0.0;
  rnd = (double)(rand() % 5) / 100.0;
  
  if ((d = lidgelift(x, y, z)) > 0.0) {
    ret = d;
  } else {
    for (i = 0; i < thermal_num; i++) {
      d = (thermal_x[i] - x) * (thermal_x[i] - x);
      d += (thermal_y[i] - y) * (thermal_y[i] - y);

      r = 35.0;
      if (z < 70.0)
	r = 30.0;
      if (z < 0.0)
	r = 25.0;
      
      if (d <= r * r) {
	ret = (double)(i + 1) * 0.08 + 0.15;
	if (z > 0.0)
	  ret += 0.1;
	if (z > 70.0)
	  ret += 0.1;
	if (d <= r * r / 4.0)
	  ret += 0.1;
	
	if ((z - viewpoint_z) > thermal_y[i])
	  ret = 0;
      }
    }
  }

  if (ret > 0.0)
    return ret + rnd;
  else
    return 0.0;
}

int get_core(double x, double y, double z, double *core_x, double *core_y)
{
  double d;
  int i;

  for (i = 0; i < thermal_num; i++) {
    d = (thermal_x[i] - x) * (thermal_x[i] - x);
    d += (thermal_y[i] - y) * (thermal_y[i] - y);
    
    if (d <= 40.0 * 40.0) {
      *core_x = thermal_x[i];
      *core_y = thermal_y[i];

      return 1;
    }
  }

  return 0;
}


int world2display(double x, double y, double z, int *dx, int *dy)
{
  if (y > 1.0) {
    *dx = (int)(x * scr_height / y / 2.0) + scr_width / 2 + scr_x;
    *dy = (int)((-(z - viewpoint_z)) * scr_height / y / 2.0) + scr_height / 2 +
      scr_y;
  } else {
    *dx = *dy = 0;
  }

  return 0;
}

int draw_vario(int p, double updown, double z)
{
  int i;
  int x;
  char buf[20];
  char c[] = {0, 'G', 'A', 'P', 'S', 'L', 'C'};
  
  XClearArea(dpy, vario, 0, p * 8, 120, 8, True);

  if (updown > 0.0)
    XSetForeground(dpy, variogc, red.pixel);
  else
    XSetForeground(dpy, variogc, BlackPixel(dpy, 0));

  for (i = 0; i <= 8; i++)
    XDrawLine(dpy, vario, variogc, i * 10 + 40, p * 8, i * 10 + 40, p * 8 + 7);

  i = (int)(updown * 100.0);

  if (i < 0)
    x = 80 + i;
  else
    x = 80;

  XFillRectangle(dpy, vario, variogc, x, p * 8 + 2, abs(i) + 1, 4);

  sprintf(buf, "%c %d", c[pilot_mode[p]], (int)(z));
  XDrawString(dpy, vario, variogc, 0, p * 8 + 7, buf, strlen(buf));
  
  return 0;
}

int cmp_nearfar(void *v1, void *v2)
{
  int i1, i2;

  i1 = *((int *)(v1));
  i2 = *((int *)(v2));

  if (py[i1] > py[i2])
    return -1;
  if (py[i1] < py[i2])
    return 1;
  return 0;
}

int loop()
{
  int scr_basis;
  int zoom[8];
  int dx[8], dy[8], i, j, k, jj, jj2j[8];
  int dx_old[8], dy_old[8], zoom_old[8], draw[8], draw_old[8];
  int cx1 = 0, cy1 = 0, cx2 = 0, cy2 = 0;
  double updown, d;
  Vector v;

  double bank2curve[] = {0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0};
  double bank2speed[] = {1.00, 1.02, 1.04, 1.08, 1.15, 1.25, 1.4};
  double bank2sink[] = {0.1, 0.1, 0.11, 0.12, 0.13, 0.15, 0.2};

  for (i = 0; i < pilot_num; i++) {
    px[i] = (double)(rand() % 200 - 100);
    py[i] = (double)(rand() % 100 + 100);
    pz[i] = (double)(rand() % 50);

    /*
    direction_r[i] = (double)(rand() % 360 - 180);
    */
    direction_r[i] = 0;
    bank_r[i] = 0;

    pilot_mode[i] = PILOT_GLIDING;
    pylon[i] = 0;

    draw[i] = draw_old[i] = 0;

    jj2j[i] = i;
  }
  
  scr_basis = scr_height * 20;
  k = 0;
  
  for (;;) {
    if (DrawingMode == DRAW_NOFLICKER) {
      if (Transient) {
	XCopyArea(dpy, back, back2, gc, 0, 0, scr_width, scr_height, 0, 0);
      } else {
	XSetForeground(dpy, gc, blue.pixel);
	XFillRectangle(dpy, back2, gc, 0, 0, scr_width, scr_height);
      }
    }

    if (pilot_num > 1)
      qsort(jj2j, pilot_num, sizeof(int), cmp_nearfar);

    for (jj = 0; jj < pilot_num; jj++) {
      j = jj2j[jj];
      
      updown = 0.0;

      if (DrawingMode == DRAW_USECLIP) {
	XSetFunction(dpy, bmapgc, GXclear);
	XFillRectangle(dpy, clip[j][k], bmapgc, 0, 0, scr_height, scr_height);
      }
      
      if (bank_r[j] > 0) {
	direction_r[j] -= bank2curve[bank_r[j]];
	i = bank_r[j];
      } else if (bank_r[j] < 0) {
	direction_r[j] += bank2curve[-bank_r[j]];
	i = -bank_r[j];
      } else {
	i = 0;
      }
      
      updown -= bank2sink[i];
      updown += lift(px[j], py[j], pz[j]);
      
      corect_r(&(direction_r[j]));
      
      bank_r[j] = pilot(j, px[j], py[j], pz[j], bank_r[j], direction_r[j]);

      pz[j] += updown;
      px[j] += sin(direction_r[j] / 180.0 * M_PI) * relative_speed[j] *
	bank2speed[i];
      py[j] -= cos(direction_r[j] / 180.0 * M_PI) * relative_speed[j] *
	bank2speed[i];
      
      d = sqrt(px[j] * px[j] + py[j] * py[j] +
	       (pz[j] - viewpoint_z) * (pz[j] - viewpoint_z));
      
      v.x = px[j] / d;
      v.y = py[j] / d;
      v.z = (pz[j] - viewpoint_z) / d;
      
      if (py[j] > 1.0) {
	zoom[j] = scr_basis / (int)py[j];
	
	world2display(px[j], py[j], pz[j], &(dx[j]), &(dy[j]));
	dx[j] -= zoom[j] / 2;
	dy[j] -= zoom[j] / 2;

	if (dx[j] + zoom[j] <= scr_x || dy[j] + zoom[j] <= scr_y ||
	    dx[j] >= scr_x + scr_width || dy[j] >= scr_y + scr_height)
	  draw[j] = 0;
	else
	  draw[j] = 1;
      } else {
	draw[j] = 0;
      }

      if (draw[j]) {
	switch (DrawingMode) {
	  case DRAW_NOFLICKER:
	    draw_polygons_3d(j, back2, 0, bank_r[j] * 10, (int)direction_r[j],
			     &v, zoom[j], dx[j] - scr_x, dy[j] - scr_y);
	    break;
	    
	  case DRAW_USECLIP:
	    draw_polygons_3d(j, pix, clip[j][k], bank_r[j] * 10,
			     (int)direction_r[j], &v, zoom[j], 0, 0);
	    
	    if (draw_old[j]) {
	      XSetClipMask(dpy, gc, clip[j][1 - k]);
	      XSetClipOrigin(dpy, gc, dx_old[j], dy_old[j]);
	      XCopyArea(dpy, back, win, gc, dx_old[j], dy_old[j],
			zoom_old[j], zoom_old[j], dx_old[j], dy_old[j]);
	    }
	    
	    XSetClipMask(dpy, gc, clip[j][k]);
	    XSetClipOrigin(dpy, gc, dx[j], dy[j]);
	    XCopyArea(dpy, pix, win, gc, 0, 0, zoom[j], zoom[j], dx[j], dy[j]);
	    break;
	    
	  case DRAW_NORMAL:
	    if (Transient == 0)
	      XFillRectangle(dpy, pix, gc, 0, 0, zoom[j], zoom[j]);
	    else
	      XCopyArea(dpy, back, pix, gc, dx[j], dy[j], zoom[j], zoom[j], 0,
			0);
	    
	    draw_polygons_3d(j, pix, 0, bank_r[j] * 10, (int)direction_r[j],
			     &v, zoom[j], 0, 0);
	    
	    if (draw_old[j])
	      XClearArea(dpy, win, dx_old[j], dy_old[j], zoom_old[j],
			 zoom_old[j], False);
	    
	    XCopyArea(dpy, pix, win, gc, 0, 0, zoom[j], zoom[j], dx[j], dy[j]);
	}
      } else {
	if (draw_old[j] && DrawingMode != DRAW_NOFLICKER) {
	  if (DrawingMode == DRAW_USECLIP) {
	    XSetClipMask(dpy, gc, clip[j][1 - k]);
	    XSetClipOrigin(dpy, gc, dx_old[j], dy_old[j]);
	    XCopyArea(dpy, back, win, gc, dx_old[j], dy_old[j], zoom_old[j],
			zoom_old[j], dx_old[j], dy_old[j]);
	  } else {
	    XClearArea(dpy, win, dx_old[j], dy_old[j], zoom_old[j],
		       zoom_old[j], False);
	  }
	}
      }

      if (Vario)
	draw_vario(j, updown, pz[j]);
    }

    if (DrawingMode == DRAW_NOFLICKER) {
      for (j = 0; j < pilot_num; j++) {
	if (draw_old[j] && draw[j]) {
	  cx1 = MIN2(dx_old[j], dx[j]);
	  cy1 = MIN2(dy_old[j], dy[j]);
	  cx2 = MAX2(dx_old[j] + zoom_old[j], dx[j] + zoom[j]);
	  cy2 = MAX2(dy_old[j] + zoom_old[j], dy[j] + zoom[j]);
	} else if (draw_old[j]) {
	  cx1 = dx_old[j];
	  cy1 = dy_old[j];
	  cx2 = cx1 + zoom_old[j];
	  cy2 = cy1 + zoom_old[j];
	} else if (draw[j]) {
	  cx1 = dx[j];
	  cy1 = dy[j];
	  cx2 = cx1 + zoom[j];
	  cy2 = cy1 + zoom[j];
	}
	
	cx1 = MAX2(cx1, scr_x);
	cy1 = MAX2(cy1, scr_y);
	cx2 = MIN2(cx2, scr_x + scr_width);
	cy2 = MIN2(cy2, scr_y + scr_height);
	
	if (draw[j] || draw_old[j])
	  XCopyArea(dpy, back2, win, gc, cx1 - scr_x, cy1 - scr_y, cx2 - cx1,
		    cy2 - cy1, cx1, cy1);
      }
    }

    for (j = 0; j < pilot_num; j++) {
      dx_old[j] = dx[j];
      dy_old[j] = dy[j];
      zoom_old[j] = zoom[j];
      draw_old[j] = draw[j];
    }
    
    usleep((int)(1000000.0 * Frequency));

    k = 1 - k;
  }

  return 0;
}

char *get_option_exact(char *optionname, int step, int argc, char **argv)
{
  int i;

  for (i = 1; i < argc; i++) {
    if (strcmp(argv[i], optionname) == 0 && i + step < argc)
      return argv[i + step];
  }

  return NULL;
}

char *get_option_substr(char *optionname, int step, int argc, char **argv)
{
  int i;

  for (i = 1; i < argc; i++) {
    if (strstr(argv[i], optionname) != NULL && i + step < argc)
      return argv[i + step];
  }
  
  return NULL;
}

int main(int argc, char **argv)
{
  int i;
  
  scr_x = scr_y = scr_width = scr_height = -1;

  datadir[255] = '\0';
  backcolor[79] = '\0';

  if (get_option_substr("-help", 0, argc, argv)) {
    print_options();
    exit(0);
  }

  if (get_option_exact("-verbose", 0, argc, argv)) {
    Verbose = 1;
    fprintf(stderr, "Verbose mode.\n");
  }

  for (i = 0; i < 8; i++) {
    nobankflag[i] = 0;
    relative_speed[i] = 1.0;
  }
  
  read_deffile(get_option_exact("-deffile", 1, argc, argv));

  if (Transient == 0 && DrawingMode == DRAW_USECLIP)
    DrawingMode = DRAW_NORMAL;
  
  parse_option(argc, argv);

  if (Transient == 0 && DrawingMode == DRAW_USECLIP)
    DrawingMode = DRAW_NORMAL;
  
  if (DrawingMode != DRAW_NOFLICKER)
    scr_x = scr_y = scr_width = scr_height = -1;

  if (max_y < 100.0)
    max_y = 100.0;
  
  init();
  xinit();
  convert_colorname_to_pixel();
  
  loop();

  return 0;
}
  
    

    

    

    
      
      
	
	
	
	
	
  
  





