#include "config.h"
#include "i18n.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <fcntl.h>

#include <fox-1.2/fx.h>
#include <fox-1.2/fxkeys.h>
#include <fox-1.2/FXPNGIcon.h>

#include "icons.h"
#include "FileList.h"
#include "Properties.h"
#include "FilePanel.h"
#include "XFileExplorer.h"
#include "InputDialog.h"
#include "HistInputDialog.h"
#include "File.h"
#include "FileDict.h"
#include "MessageBox.h"
#include "OverwriteBox.h"


// Maximum length of a file name
#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif

// History size
#ifndef OPEN_WITH_HIST_SIZE
#define OPEN_WITH_HIST_SIZE 30
#endif

// Archives history size
#ifndef ADD_TO_ARCH_HIST_SIZE
#define ADD_TO_ARCH_HIST_SIZE 30
#endif

// Root directory string
#ifndef ROOTDIR
#define ROOTDIR "/"
#endif

// Global Variables
extern FXMainWindow *mainWindow;
extern FXString editor;
extern FXString viewer;
extern FXString homelocation;
extern FXString clipboard;
extern FXbool confirm_del;
extern char OpenHistory[OPEN_WITH_HIST_SIZE][100];
extern int OpenNum;
#if defined(linux)
extern FXStringDict* devices;
#endif

FXuint clptype=0;

// Map
FXDEFMAP(FilePanel) FilePanelMap[]={
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_DIRECTORY_UP,FilePanel::onCmdDirectoryUp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILTER,FilePanel::onCmdItemFilter),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_GO_HOME,FilePanel::onCmdGoHome),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_VIEW,FilePanel::onCmdEdit),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_EDIT,FilePanel::onCmdEdit),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PROPERTIES,FilePanel::onCmdProperties),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_DIR,FilePanel::onCmdNewDir),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_NEW_FILE,FilePanel::onCmdNewFile),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_COPY,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_COPY_CLP,FilePanel::onCmdClp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_MOVE,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_RENAME,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_CUT_CLP,FilePanel::onCmdClp),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_PASTE_CLP,FilePanel::onCmdPaste),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_DEL,FilePanel::onCmdFileDel),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_FILE_SLINK,FilePanel::onCmdFileMan),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN_WITH,FilePanel::onCmdOpenWith),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_OPEN,FilePanel::onCmdOpen),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_REFRESH,FilePanel::onRefresh),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_ADD_TO_ARCH,FilePanel::onCmdAddToArch),
                               FXMAPFUNC(SEL_RIGHTBUTTONRELEASE,FilePanel::ID_FILE_PANEL,FilePanel::onCmdPopupMenu),
                               FXMAPFUNC(SEL_DOUBLECLICKED,FilePanel::ID_FILE_PANEL,FilePanel::onCmdItemDblClcked),
                               FXMAPFUNC(SEL_FOCUSIN,FilePanel::ID_FILE_PANEL,FilePanel::onCmdItemClcked),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_STATUS,FilePanel::onUpdStatus),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_LABEL,FilePanel::onUpdLabel),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_DIRECTORY_UP,FilePanel::onUpdUp),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_COPY_CLP,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_CUT_CLP,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PASTE_CLP,FilePanel::onUpdPaste),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_PROPERTIES,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_DEL,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_MOVE,FilePanel::onUpdMenu),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_RENAME,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_FILE_SLINK,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_EDIT,FilePanel::onUpdSelMult),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_VIEW,FilePanel::onUpdSelMult),
#if defined(linux)
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_MOUNT,FilePanel::onCmdMount),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_UMOUNT,FilePanel::onCmdMount),
                               FXMAPFUNC(SEL_COMMAND,FilePanel::ID_RPM_QUERY,FilePanel::onCmdRpmQuery),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_MOUNT,FilePanel::onUpdMount),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_UMOUNT,FilePanel::onUpdMount),
                               FXMAPFUNC(SEL_UPDATE,FilePanel::ID_RPM_QUERY,FilePanel::onUpdSelMult),
#endif
                           };

// Object implementation 
FXIMPLEMENT(FilePanel,FXVerticalFrame,FilePanelMap,ARRAYNUMBER(FilePanelMap))

// Construct File Panel
FilePanel::FilePanel(FXchar* nm,FXComposite *p, FXuint name_size, FXuint size_size, FXuint type_size, 
             FXuint modd_size, FXuint user_size, FXuint grou_size, FXuint attr_size,
			 FXuint opts,FXint x,FXint y,FXint w,FXint h):
             FXVerticalFrame(p,opts,x,y,w,h,0,0,0,0)
{
	// Create some icons
	FXIcon *showhiddenicon=new FXPNGIcon(getApp(),showhidden);
	FXIcon *hidehiddenicon=new FXPNGIcon(getApp(),hidehidden);

    FXVerticalFrame* cont=new FXVerticalFrame(this,LAYOUT_FILL_Y|LAYOUT_FILL_X|FRAME_SUNKEN,0,0,0,0, 0,0,0,0, 1,1);

    name=nm;
    label=new FXLabel(cont," ",NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_RAISED);
    label->setTarget(this);
    label->setSelector(FXSEL(0,FilePanel::ID_LABEL));

    list=new FileList(cont,this,ID_FILE_PANEL,LAYOUT_FILL_X|LAYOUT_FILL_Y|ICONLIST_BIG_ICONS);
    list->setHeaderSize(0,name_size);
    list->setHeaderSize(1,size_size);
    list->setHeaderSize(2,type_size);
    list->setHeaderSize(3,modd_size);
    list->setHeaderSize(4,user_size);
    list->setHeaderSize(5,grou_size);
    list->setHeaderSize(6,attr_size);  

	statusbar=new FXHorizontalFrame(cont,JUSTIFY_LEFT|LAYOUT_FILL_X,0,0,0,0, 3,3,3,3);
    new FXToggleButton(statusbar,_("\tShow hidden files (Ctrl-F5)"),_("\tHide hidden files (Ctrl-F5)"),showhiddenicon,hidehiddenicon,this->list,
	                   FileList::ID_TOGGLE_HIDDEN,BUTTON_TOOLBAR|LAYOUT_LEFT|ICON_BEFORE_TEXT|FRAME_RAISED);
    statusbar->setTarget(this);
    statusbar->setSelector(FXSEL(SEL_UPDATE,FilePanel::ID_STATUS));
    status=new FXLabel(statusbar,_("Status"),NULL,JUSTIFY_LEFT|LAYOUT_LEFT|LAYOUT_FILL_X|FRAME_SUNKEN);

    homelocation=getenv("HOME");
    if(homelocation=="")
        homelocation=ROOTDIR;
	selmult=FALSE;
	current=NULL;
}

// Create X window
void FilePanel::create()
{
	FXVerticalFrame::create();
}

// Destructor
FilePanel::~FilePanel()
{
    delete list;
    delete label;
    delete current;
    delete next;
    delete status;
	delete statusbar;
}


// FilePanel Gets Focus
void FilePanel::focus()
{
    if(this == current)
        return;
    label->setBackColor(getApp()->getBaseColor());
    label->setTextColor(FXRGB(0,0,0));
    current=this;
    next->unfocus();
    list->setFocus();
}


// FilePanel Loses Focus
void FilePanel::unfocus()
{
    label->setBackColor(FXRGB(128,128,128));
    label->setTextColor(FXRGB(255,255,255));
    current=next;
    list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),NULL);
}


// Set Pointer to Another FilePanel
void FilePanel::Next(FilePanel* nxt)
{
    next=nxt;
}


// Update location history when changing directory (home, up or double click)
void FilePanel::updateLocation()
{
    FXString item;
	FXint i=0;
	FXComboBox* address=((XFileExplorer*) mainWindow)->address;
  	address->setNumVisible(5);
	FXint count=address->getNumItems();
    FXString p=list->getDirectory();

	// Remember latest directory in the location address
	if(!count)
	{
		count++;
		address->insertItem(0,address->getText());
	}
    while(i < count)
	{
    	item = address->getItem(i++);
    	if(!strcmp((const char*)&p[0],(const char*)&item[0]))
		{
			i--;
			break;
		}
	}
    if(i==count)
		address->insertItem(0,list->getDirectory());
}


// Double Click on File Item
long FilePanel::onCmdItemDblClcked(FXObject* sender,FXSelector sel, void* ptr)
{
    // Commented out to fix a problem with umount
	//chdir(current->list->getDirectory().text());

    FXchar path[MAXPATHLEN+1],name[2048],dir[MAXPATHLEN+1],*p;
    FXString buf;

    int item= (int) ptr;
    if(item > -1)
    {
        strcpy(name,list->getItemText(item).text());      //// FIX
        if((p=strchr(name,'\t')))
            *p=0;
        strcpy(path,list->getDirectory().text());         //// FIX
        strcpy(dir,FXFile::absolute(path,name).text());

        // If directory, open the directory
        if(list->isItemDirectory(item))
        {
            // Does not have access
			if(access(dir,R_OK|X_OK))
			{
            	MessageBox::error(this,MBOX_OK,_("Error"),_(" Permission to: %s denied."), dir);
                return 0;
			}
            if(dir[0]=='.' && dir[1]=='.' && dir[2]==0)// '..'  ?
                list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
            else
                list->setDirectory(dir);
            
			// Change directory in tree list  
			((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(dir);
			
			// Update location history			
			updateLocation();
        }
        else if(list->isItemFile(item))
        {
            FileAssoc *association=list->getItemAssoc(item);
            if(association)
                if(association->command.section(',',0).text() != NULL)
                {
                    FXString command=association->command.section(',',0)+" \""+list->getItemPathname(item);
                    command+="\" &";
                    system(command.text());
                }
                else if(list->isItemExecutable(item)) //executable
                {
                    buf="'";
                    buf+=dir;
                    buf+="' &";
                    system(buf.text());
                }    // If regular file return as the selected file
                else
                    current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
            else if(list->isItemExecutable(item))//executable
            {
                buf="'";
                buf+=dir;
                buf+="' &";
                system(buf.text());
            }    // If regular file return as the selected file
            else
                current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
        }
    }
    return 1;
}


// Focus when clicked
long FilePanel::onCmdItemClcked(FXObject* sender,FXSelector sel, void* ptr)
{
    focus();
    return 1;
}


// Go to parent directory
long FilePanel::onCmdDirectoryUp(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->handle(this,FXSEL(SEL_COMMAND,FileList::ID_DIRECTORY_UP),NULL);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(current->list->getDirectory());
	updateLocation();
    return 1;
}


// Go to home directory
long FilePanel::onCmdGoHome(FXObject* sender,FXSelector sel, void* ptr)
{
    current->list->setDirectory(homelocation);
    current->list->setFocus();
    ((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(homelocation);
	updateLocation();
    return 1;
}


// Copy/Move/Rename/SymLink file(s)
long FilePanel::onCmdFileMan(FXObject* sender,FXSelector sel,void*)
{
    FXString target,sources,name,curdir,param;
    curdir = current->list->getDirectory();

    FXint num;	
	num=current->list->getNumSelectedItems();

	// If no selected files in the file list, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);	
    	target = FXFile::directory(filepath);
		param = target+"\n"+FXStringVal(1)+"\n"+filepath;
	}
	
	// If exist selected files, use it
	else if (num>=1)
	{
    	int dotdot = 0;
    	for(int u=0;u<current->list->getNumItems (); u++)
        	if (current->list->isItemSelected (u))
        	{
            	name = current->list->getItemText(u).text();
            	name = name.section('\t',0);
            	if(name == "..")
            	{
               		dotdot = 1;
                	continue;
            	}
            	sources += curdir+PATHSEPSTRING+name+"\n";
        	}
    	if(dotdot)
        	num--;
		
		target = current->next->list->getDirectory();
		if(!current->next->shown() || FXSELID(sel)==ID_FILE_RENAME)
        	target = current->list->getDirectory();
		param = target+"\n"+FXStringVal(num)+"\n"+sources;
	}

    switch(FXSELID(sel))
    {
    case ID_FILE_COPY:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case ID_FILE_MOVE:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_MOVE);
        break;
    case ID_FILE_RENAME:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_RENAME);
        break;
    case ID_FILE_SLINK:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_SLINK);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));
    return 1;
}


// Delete files from the file list or the tree list
long FilePanel::onCmdFileDel(FXObject*,FXSelector,void*)
{
	File *f=NULL;
    current->list->setFocus();
	FXString dir=current->list->getDirectory();
    chdir(dir.text());

    // If we don't have permission to write to the parent directory
    if(access(dir.text(), W_OK))
    {
        MessageBox::error(this,MBOX_OK,_("Error"),_("Can't write to %s: Permission denied"),dir.text());
        return 0;
    }

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();

    // If no selected files, use the selected folder from the tree list (if any)
	if(num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);

    	if(confirm_del)
    	{
        	FXPNGIcon icon(getApp(),delete_big);
			FXString message=_("Delete folder ") + filepath + " ?";
         	MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
       		if(box.execute() != MBOX_CLICKED_OK)
            	return 0;
   		}

		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(getApp(),_("File Delete"),DELETE);
   		f->create();
		f->remove(filepath); 	

		// If action is cancelled in progress dialog
		if (f->isCancelled)
		{
			f->hide();
			MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
		}

    	// Return to parent dir in DirList		
		FXString parentdir=FXFile::directory(dir);
		((XFileExplorer*) mainWindow)->dirpanel->list->setDirectory(parentdir);
    	chdir(parentdir.text());

		getApp()->endWaitCursor();
  		delete f;
	}
	
	// If exist selected files, use it
	else if (num>=1)
	{
		if(confirm_del)
    	{
        	FXPNGIcon icon(getApp(),delete_big);
        	FXString message;
			if (num==1)
				message=_("  Delete selected item ?  ");
			else
				message=_("  Delete all selected items ?  ");
			MessageBox box(this,_("Confirm Delete"),message,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);
        	if(box.execute() != MBOX_CLICKED_OK)
            	return 0;
   		}

		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
   		f=new File(getApp(),_("File Delete"),DELETE);
   		f->create();

		// Overwrite initialisations
		FXbool overwrite=FALSE;
		FXbool overwrite_all=FALSE;
		FXbool skip_all=FALSE;

		// Delete selected files
    	FXchar name[MAXPATHLEN+1],*p;
		FXString filepath;    			
		for(int u=0;u< current->list->getNumItems (); u++)
    	{
        	if (current->list->isItemSelected (u))
        	{
            	// Get file path
				strcpy(name,current->list->getItemText(u).text());
            	if((p=strchr(name,'\t')))
                	*p=0;

				filepath=::filePath(name,dir);

    			// If we don't have permission to write to the file
    			if(access(filepath.text(), W_OK) && !access(filepath.text(), F_OK))
    			{
					// Get process uid and file uid
					FXuint uid=getuid();
					struct stat buf;
					lstatout(filepath.text(),&buf);
					FXuint fileuid=buf.st_uid;
					
					// If it is a file and we are owner of it
					if (isFile(filepath) && fileuid==uid)
					{
						// Dialog to confirm file deletion
						f->allowPdialog=FALSE;
						f->hide();
					
						// Overwrite dialog if necessary
						if (!(overwrite_all | skip_all))
    					{
							FXString msg=_("File ") + filepath + _(" is write-protected, delete it anyway?");
       						ConfirmOverWriteBox* dlg = new ConfirmOverWriteBox(this,_("Confirm Delete"),msg);
       						FXuint answer=dlg->execute();
							delete dlg;
       						switch(answer)
       						{
       							case 1:
           							overwrite = TRUE;
           							break;
       							case 2:
           							overwrite_all = TRUE;
           							break;
       							case 3:
									overwrite = FALSE;
           							break;
       							case 4:
           							skip_all = TRUE;
									break;
       						}
						}
						if((overwrite | overwrite_all) & !skip_all)
						{
							// Caution!! Don't delete parent directory!!
							if (strcmp(name,"..")!=0)
								f->remove(filepath); 
						}
						f->allowPdialog=TRUE;
					}
					
					// If it is a directory or we are not owner of the file
					else
					{
						f->allowPdialog=FALSE;
						f->hide();
						FXPNGIcon icon(getApp(),errorIcon);
						FXString str;
						if (isDirectory(filepath))
							str=_("Can't delete folder ") + filepath + _(" : Permission denied");
						else
							str=_("Can't delete file ") + filepath + _(" : Permission denied");
        				MessageBox box(this,_("Error"),str,&icon,MBOX_OK_CANCEL|DECOR_TITLE|DECOR_BORDER);        	
						FXuint answer=box.execute();
						if(answer == MBOX_CLICKED_CANCEL)
						{
							MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
							break;
						}					
						f->allowPdialog=TRUE;
					}
    			}

				// If we have permission to write
				else
				{
					// Caution!! Don't delete parent directory!!
					if (strcmp(name,"..")!=0)
						f->remove(filepath); 	

					// If action is cancelled in progress dialog
					if (f->isCancelled)
					{
						f->hide();
						MessageBox::error(this,MBOX_OK,_("Error"),_("Delete file operation cancelled!"));
						break;
					}
				}
        	}
    	}

		getApp()->endWaitCursor();
		delete f;
	}

	// Force FilePanel and DirPanel refresh
   	onRefresh(0,0,0);
	((XFileExplorer*) mainWindow)->dirpanel->list->onRefresh(0,0,0);
	
    return 1;
}


// Add files or directory to an archive
long FilePanel::onCmdAddToArch(FXObject* o,FXSelector,void*)
{
    FXString ext1, ext2, cmd, archive="";
    static char history[ADD_TO_ARCH_HIST_SIZE][100];
    static int n=0;
    int i;
	File *f;
	
	chdir(current->list->getDirectory().text());

    if(current->list->getNumSelectedItems()==0)
        return 0;

    FXchar name[MAXPATHLEN+1],*p;
    HistInputDialog* dialog;
    FXString archpath=current->list->getDirectory();

    dialog = new HistInputDialog(this,archive,_("Create new archive: (e.g. archive.tar.gz)"),_("Add To Archive"));

    for(int i=0;i<n;i++)
        dialog->appendItem(history[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        // Get string and preserve escape characters
		archive=archpath+PATHSEPSTRING+"\""+dialog->getText()+"\"";

        // Get extensions of the archive name
        ext1=dialog->getText().rafter('.',1);
        ext2=dialog->getText().rafter('.',2);
		
		// Handle different archive formats
		if (ext2=="tar.gz" || ext2=="TAR.GZ")
            cmd="/bin/tar -zcvf "+archive+" ";
		else if (ext2=="tar.bz2" || ext2=="TAR.BZ2")
            cmd="/bin/tar -jcvf "+archive+" ";
		else if (ext2=="tar.Z" || ext2=="TAR.Z")
           	cmd="/bin/tar -Zcvf "+archive+" ";
		else if (ext1=="tar" || ext1=="TAR")
           	cmd="/bin/tar -cvf "+archive+" ";
		else if (ext1=="gz" || ext1=="GZ")
			cmd="/usr/bin/gzip -v ";				
		else if (ext1=="tgz" || ext1=="TGZ")
            cmd="/bin/tar -zcvf "+archive+" ";
		else if (ext1=="bz2" || ext1=="BZ2")
			cmd="/usr/bin/bzip2 -v ";				
		else if (ext1=="tbz2" || ext1=="TBZ2")
            cmd="/bin/tar -jcvf "+archive+" ";
		else if (ext1=="Z")
            cmd="/usr/bin/compress -v ";
		else if (ext1=="zip" || ext1=="ZIP")
            cmd="/usr/bin/zip -r "+archive+" ";
		else
			cmd="/bin/tar -zcvf "+archive+" ";

        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected (u))
            {
                strcpy(name,current->list->getItemText(u).text());      //// FIX
                if((p=strchr(name,'\t')))
                    *p=0;
				
				// Don't include '..' in the list
				if (strcmp(name,"..")!=0)
				{
                	cmd+=" ";
                	cmd=cmd+"\""+name+"\"";
                	cmd+=" ";
				}
            }
					
		// Wait cursor
		getApp()->beginWaitCursor();

		// File object
		f=new File(getApp(),_("Create archive"),ARCHIVE);
		f->create();

		// Create archive
		f->archive(archive,cmd);
 
		getApp()->endWaitCursor();
		delete f; 

		// Force panel refresh
    	onRefresh(0,0,0);

        // Manage dialog history size
		n=dialog->getHistorySize();
        n++;
        cmd=dialog->getText();
        if(n>ADD_TO_ARCH_HIST_SIZE)
            n=ADD_TO_ARCH_HIST_SIZE;
        for(i=0;i<n-1;i++)
            if(!strcmp(history[i],cmd.text()))
                break;
        if(i==n-1)
        {
            strcpy(history[0],cmd.text());
            for(i=1;i<n;i++)
                strcpy(history[i],dialog->getHistoryItem(i-1).text());
        }
        else
            n--;
    }
    delete dialog;
    return 1;
}


// Edit file
long FilePanel::onCmdEdit(FXObject*,FXSelector s,void*)
{
    current->list->setFocus();
    chdir(current->list->getDirectory().text());
    FXchar path[MAXPATHLEN+1],name[2*MAXPATHLEN+1],*p;

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

	// Only process the case where one file is selected
	if (num==0 || num>1)
		return 0;

    FXString command;
    if(FXSELID(s)==ID_EDIT)
        command = editor;
    else
        command = viewer;

	FileAssoc *association=current->list->getItemAssoc(itm);
	if(association)
	{
		if(FXSELID(s)==ID_EDIT)
		{
			command=association->command.section(',',2);
			if(command.length()==0)
				command = editor;
		}
		else
		{
			command=association->command.section(',',1);
			if(command.length()==0)
				command = viewer;
		}
	}

	// Construct command string
	strcpy(name,current->list->getItemText(itm).text());
	if((p=strchr(name,'\t')))
		*p=0;
	strcpy(path,current->list->getDirectory().text());
	command+=" \"";
	command+=::filePath(name,path);
	command+="\"";
    command+=" &";

    // Viewer or editor command
    system(command.text());
    return 1;
}


// File or directory properties 
long FilePanel::onCmdProperties(FXObject* sender,FXSelector,void*)
{
    current->list->setFocus();
    chdir(current->list->getDirectory().text());

    FXint num, itm;	
	num=current->list->getNumSelectedItems(&itm);

	// If no selected files in the file list, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString filepath=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);
    	FXString path=FXFile::directory(filepath);
    	FXString filename=FXFile::name(filepath);
        PropertiesBox* attrdlg = new PropertiesBox(this,filename,path);
        if(attrdlg->execute())
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        return 1;
	}

	// There is one selected file in the file list
	else if (num==1)
    {
    	FXString path=current->list->getDirectory();
        FXString filename = current->list->getItemText(itm);
        filename=filename.section('\t',0);
        PropertiesBox* attrdlg = new PropertiesBox(this,filename,path);
        if(attrdlg->execute())
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        return 1;
    }
    
	// There are multiple selected files in the file list
	else if (num>1)
    {
    	FXString path=current->list->getDirectory();
        FXString *files = new FXString[num];

        int i= 0;
        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected (u))
            {
                files[i] = current->list->getItemText(u).section('\t',0);
                i++;
            }

        PropertiesBox* attrdlg = new PropertiesBox(this,files,num);
        if(attrdlg->execute())
        {
            current->list->setDirectory(ROOTDIR);
            current->list->setDirectory(path);
        }
        delete attrdlg;
        delete[]files;
    }
    return 0;
}


// Create new directory
long FilePanel::onCmdNewDir(FXObject*,FXSelector,void*)
{
	// Current item
	current->list->setFocus();

    FXString dirname="";	
    FXString dirpath = current->list->getDirectory();
    if(dirpath!=ROOTDIR)
        dirpath+=PATHSEPSTRING;

    InputDialog* dialog = new InputDialog(this,dirname,_("Create new folder..."),_("New Folder"));
    dialog->CursorEnd();
    if(dialog->execute())
    {
		dirname=dirpath+dialog->getText();
        if(dirname!=dirpath)
        {
			// Create the new dir
			FXint ret=mkdir(dirname.text(),0755);
			if (ret==-1)
			{
         		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
    			return 0;
			}
		}							
    }
    delete dialog;
	
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}


// Create new file
long FilePanel::onCmdNewFile(FXObject*,FXSelector,void*)
{
	// Current item
    current->list->setFocus();

    FXString filename="";	
    FXString filepath = current->list->getDirectory();
    if(filepath!=ROOTDIR)
        filepath+=PATHSEPSTRING;

    InputDialog* dialog = new InputDialog(this,filename,_("Create new file..."),_("New File"));
    dialog->CursorEnd();
    if(dialog->execute())
    {
		filename=filepath+dialog->getText();
        if(filename!=filepath)
        {
			// Test some error conditions
			if (exists(dequote(filename)))
        	{
				MessageBox::error(this,MBOX_OK,_("Error"),_("File or folder %s already exists"), filename.text());
        		return 0;
			}
			// Create the new file
			FILE *file=fopen(filename.text(),"w+");
			if (file==NULL)
			{
         		MessageBox::error(this,MBOX_OK,_("Error"),"%s",strerror(errno));
    			return 0;
			}
			fclose(file);								
		}							
    }
    delete dialog;
	
	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}

// Paste file(s)
long FilePanel::onCmdPaste(FXObject*,FXSelector sel,void*)
{
    if(clipboard.empty())
        return 1;

    FXString param;
    param = current->list->getDirectory()+"\n"+clipboard;
    switch(clptype)
    {
    case COPY_CLP:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_COPY);
        break;
    case CUT_CLP:
        sel = FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_CUT);
        break;
    }
    mainWindow->handle(this,sel,strdup(param.text()));
    clipboard = "";

	// Force panel refresh
	onRefresh(0,0,0);
    return 1;
}


// Clipboard
long FilePanel::onCmdClp(FXObject*,FXSelector sel,void*)
{
    FXString name,curdir;

    // Clipboard type
	clipboard = "";
	if (FXSELID(sel)==ID_CUT_CLP)
		clptype=CUT_CLP;
	else
		clptype=COPY_CLP;

	// Items number in the file list
    FXint num=current->list->getNumSelectedItems();
			
    // If no selected files, use the selected folder from the tree list (if any)
	if (num==0)
	{
    	DirItem* item=(DirItem*)((XFileExplorer*) mainWindow)->dirpanel->list->getCurrentItem();
		if (!item)
			return 0;
			
		FXString pathname=((XFileExplorer*) mainWindow)->dirpanel->list->getItemPathname((TreeItem*)item);
		clipboard = FXStringVal(1)+"\n"+pathname;
	}

	// If exist selected files, use it
	else if (num>=1)
	{
    	curdir = current->list->getDirectory();

    	int dotdot = 0;
    	for (int u= 0; u< current->list->getNumItems (); u++)
        	if (current->list->isItemSelected (u))
        	{
            	name = current->list->getItemText(u).text();
            	name = name.section('\t',0);
            	if(name == "..")
            	{
                	dotdot = 1;
                	continue;
            	}
            	clipboard += curdir+PATHSEPSTRING+name+"\n";
        	}
    	clipboard = FXStringVal(num)+"\n"+clipboard;
	}
    return 1;
}


// Open
long FilePanel::onCmdOpen(FXObject*,FXSelector,void*)
{
    FXString pathname, command;
	FileAssoc* association;
	   
	current->list->setFocus();
    chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
		return 0;
    
	for (int u= 0; u< current->list->getNumItems (); u++)
	{
		if (current->list->isItemSelected (u))
		{
			pathname=current->list->getItemPathname(u);
			association = current->list->getItemAssoc(u);
			
			// If there is an association
			if(association)
			{
                // Use it to open the file
				if(association->command.section(',',0).text() != NULL)
                {
                    command=association->command.section(',',0)+" \""+pathname;
                    command+="\" &";
                    system(command.text());
                }
                
				// or execute it
				else if(current->list->isItemExecutable(u))
                {
                    command="'";
                    command+=pathname;
                    command+="' &";
                    system(command.text());
                }
                
				// or call the "Open with..." dialog
				else
                    current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
			}
            
			// If no association but executable
			else if(current->list->isItemExecutable(u))
            {
                command="'";
                command+=pathname;
                command+="' &";
                system(command.text());
            }
			
			// Other cases
            else
                current->handle(this,FXSEL(SEL_COMMAND,ID_OPEN_WITH),NULL);
		}
	}
    return 1;
}

// Open with
long FilePanel::onCmdOpenWith(FXObject*,FXSelector,void*)
{
	char **str=NULL;
    current->list->setFocus();
    int i;
    chdir(current->list->getDirectory().text());

	if (current->list->getNumSelectedItems()==0)
		return 0;

    FXchar path[MAXPATHLEN+1],name[2048],dir[MAXPATHLEN+1],*p;
    HistInputDialog* dialog;
    FXString command="";
    
	// Dialog with history list and associate checkbox
	dialog = new HistInputDialog(this,command,_("Open selected file(s) with :"),_("Open With"), "", NULL, 1, _("A&ssociate"));
    for(int i=0;i<OpenNum;i++)
        dialog->appendItem(OpenHistory[i]);
    dialog->CursorEnd();
    if(dialog->execute())
    {
        command=dialog->getText();

        for (int u= 0; u< current->list->getNumItems (); u++)
            if (current->list->isItemSelected (u))
            {
                // Handles "associate" checkbox for "open with..." dialog
                if (dialog->getOption())
                {
                    strcpy(name,current->list->getItemFilename(u).text());
                    FXString ext = FXFile::extension(name).text();
                    if (ext == "") { ext = FXFile::name(name); }

                    FileAssoc* association = current->list->getItemAssoc(u);

                    if (association)
                    {
                        // Update existing association
                        FXString oldfileassoc = getApp()->reg().readStringEntry("FILETYPES", ext.text(),"");
						oldfileassoc.remove(0, oldfileassoc.section(';',0).section(',',0).length()-1);
                        oldfileassoc.prepend(dialog->getText());
                        getApp()->reg().writeStringEntry("FILETYPES", ext.text(), oldfileassoc.text());

						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(oldfileassoc.text())+1];
						strcpy(str[0],ext.text());
						strcpy(str[1],oldfileassoc.text());
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC), str);
                    }
                    else
                    {
                        // New association
						FXString command=dialog->getText().append(";Document;;;;");
						getApp()->reg().writeStringEntry("FILETYPES", ext.text(), command.text());
                  
						// Handle file association
						str=new char*[2];
						str[0]=new char[strlen(ext.text())+1];
						str[1]=new char[strlen(command.text())+1];
						strcpy(str[0],ext.text());
						strcpy(str[1],command.text());
						mainWindow->handle(this, FXSEL(SEL_COMMAND,XFileExplorer::ID_FILE_ASSOC),str);
                    }
                }
                // End
                strcpy(name,current->list->getItemText(u).text());      //// FIX
                if((p=strchr(name,'\t')))
                    *p=0;
                strcpy(path,current->list->getDirectory().text());         //// FIX
                strcpy(dir,FXFile::absolute(path,name).text());
                command+=" ";
                command=command+"\""+dir+"\"";
                command+=" ";
            }

        command+=" &";
        system(command.text());

        OpenNum=dialog->getHistorySize();
        OpenNum++;
        command=dialog->getText();
        if(OpenNum>OPEN_WITH_HIST_SIZE)
            OpenNum=OPEN_WITH_HIST_SIZE;
        for(i=0;i<OpenNum-1;i++)
            if(!strcmp(OpenHistory[i],command.text()))
                break;
        if(i==OpenNum-1)
        {
            strcpy(OpenHistory[0],command.text());
            for(i=1;i<OpenNum;i++)
                strcpy(OpenHistory[i],dialog->getHistoryItem(i-1).text());
        }
        else
            OpenNum--;
    }
    delete dialog;
    onRefresh(0, 0, 0);
    return 1;
}


long FilePanel::onCmdItemFilter(FXObject*,FXSelector,void*)
{
    current->list->setFocus();
#define FILTER_HIST_SIZE 15

    static char history[FILTER_HIST_SIZE][100];
    static int no=7;
    if (no==7)
	{
		strcpy(history[0],"*");
    	strcpy(history[1],"*.txt");
    	strcpy(history[2],"*.c");
    	strcpy(history[3],"*.h");
    	strcpy(history[4],"*.html");
    	strcpy(history[5],"*.pdf");
    	strcpy(history[6],"*.jpg");
	}
    int i;
    HistInputDialog* dialog;
    FXString pat=list->getPattern();
    dialog = new HistInputDialog(this,pat,_("Show files :"),_("Filter"));
    dialog->CursorEnd();
    dialog->SelectAll();
    for(int i=0;i<no;i++)
        dialog->appendItem(history[i]);

    if(dialog->execute() && (pat = dialog->getText()) != "")
    {
        list->setPattern(pat);
        no=dialog->getHistorySize();
        no++;
        if(no>FILTER_HIST_SIZE)
            no=FILTER_HIST_SIZE;

        for(i=0;i<no-1;i++)
            if(!strcmp(history[i],pat.text()))
                break;
        if(i==no-1)
        {
            strcpy(history[0],pat.text());
            for(i=1;i<no;i++)
                strcpy(history[i],dialog->getHistoryItem(i-1).text());
        }
        else
            no--;
    }
    delete dialog;

    return 1;
}


// File list context menu
long FilePanel::onCmdPopupMenu(FXObject* o,FXSelector s,void* p)
{
    // FilePanel focus
	focus();

	// Check if control key is pressed
	FXEvent* event=(FXEvent*)p;
    ctrlkey=FALSE;
	if (event->state&CONTROLMASK)
		ctrlkey=TRUE;

    if(list->getNumSelectedItems()<=1)
    {
        FXint x,y;
        FXuint state;
        list->getCursorPosition(x,y,state);

        int item=list->getItemAt(x,y);

        if (list->getCurrentItem ()>= 0)
            list->deselectItem(list->getCurrentItem());

        if (item>= 0)
        {
            list->setCurrentItem(item);
            if(!list->selectItem(item));
        }
    }

	// If control key was pressed, deselect all items
	if(ctrlkey)
		list->handle(o,FXSEL(SEL_COMMAND,FileList::ID_DESELECT_ALL),p);

	mainWindow->handle(this,FXSEL(SEL_COMMAND,XFileExplorer::ID_MENU),(void*)list->getCurrentItem());
	
    return 1;
}


// FilePanel and DirPanel refresh
long FilePanel::onRefresh(FXObject*,FXSelector,void*)
{
	getApp()->beginWaitCursor();
    current->list->setFocus();
    FXString dir=current->list->getDirectory();
    current->list->setDirectory(ROOTDIR);
    current->list->setDirectory(dir);
	((XFileExplorer*) mainWindow)->dirpanel->list->onRefresh(0,0,0);
	getApp()->endWaitCursor();
    return 1;
}


#if defined(linux)
// Mount/Unmount 
long FilePanel::onCmdMount(FXObject*,FXSelector sel,void*)
{
    FXString cmd, msg, text;
	unsigned int op;
	File *f;
	
    // Select the command and set the appropriate message
	current->list->setFocus();
    FXString dir=current->list->getDirectory();

	// If symbolic link, read the linked directory
	if (isLink(dir))
		dir=readlink(dir);

    if(FXSELID(sel)==ID_MOUNT)
    {
        op=MOUNT;
		msg=_("Mount");
        cmd="/bin/mount ";
    }
    else
    {
        op=UNMOUNT;
		msg=_("Unmount");
        cmd="/bin/umount ";
    }
    cmd+=dir;
    cmd+=" 2>&1";
    chdir(ROOTDIR);

	// Wait cursor
	getApp()->beginWaitCursor();

	// File object
	text=msg + _(" file system...");
	f=new File(getApp(),text.text(),op);
	f->create();
				   
	// Mount/unmount file system
	text=msg + _(" the directory :");
	f->mount(dir,text,cmd,op);
 
	// If action is cancelled in progress dialog
	if (f->isCancelled)
	{
		f->hide();
		text=msg + _(" operation cancelled!");
		MessageBox::error(this,MBOX_OK,_("Error"),text.text());
		delete f;
		return 0;
	}
					
	getApp()->endWaitCursor();
	delete f; 

	// Force panel refresh
    onRefresh(0,0,0);

    return 1;
}


// Update the Mount/Unmount buttons
long FilePanel::onUpdMount(FXObject* o,FXSelector sel,void*)
{
    // Current directory
	FXString dir=current->list->getDirectory();
	if (isLink(dir))
		dir=readlink(dir);

    if(devices->find(dir.text()))
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    return 1;
}


// Query RPM data base
long FilePanel::onCmdRpmQuery(FXObject* o,FXSelector sel,void*)
{
	// Name of the current selected file
	FXString file=current->list->getCurrentFile();

	// Query command
	FXString cmd = "rpm -qf " + file;
   	cmd+=" 2>&1";

	// Wait cursor
	getApp()->beginWaitCursor();

	// Perform the command
	FILE *pcmd=popen(cmd.text(),"r");
	if(!pcmd)
	{
		MessageBox::error(this,MBOX_OK,_("Error"),_("Failed command : %s"),cmd.text());
		return 0;
	}

	// Get command output
	char text[10000]={0};
	FXString buf;
	while(fgets(text,sizeof(text),pcmd))
		buf+=text;
	sprintf(text,"%s",buf.text());

	// Close the stream and display error message if any
	if(pclose(pcmd))
	{
		getApp()->endWaitCursor();
		MessageBox::error(this,MBOX_OK,_("Error"),"%s",text);
		return 0;
	}
	getApp()->endWaitCursor();
	
	// Output message
	FXString message=_("File ") + file + _(" belongs to the RPM package : ") + text;
	MessageBox::information(this,MBOX_OK,_("Information"),"%s",message.text());

   	return 1;
}
#endif


// Update the status bar
long FilePanel::onUpdStatus(FXObject* sender,FXSelector,void*)
{
    int item= -1;

    FXString str,path,name,ext,size,perm,usr,grp;
    path=list->getDirectory();
    FXchar hsize[64], buf[MAXPATHLEN+1],*p,*file;

   	FXint num=list->getNumSelectedItems();

	// To handle the update rename (ugly, I know)
	if (current==this)
	{
		if (num<=1) selmult=FALSE;
		else if (num>1) selmult=TRUE;
	}

    item = list->getCurrentItem();
    if(num>1)
    {
        unsigned long long sz=0;
        size=_("0 bytes");

        for (int u= 0; u< list->getNumItems (); u++)
            if (list->isItemSelected (u))
            {
				sz+=list->getItemFileSize(u);
				sprintf(hsize,"%llu",sz);
				::hSize(hsize,hsize);
				size=hsize;
            }
        status->setText(size+_(" in ")+FXStringVal(num)+_(" selected files"));
        return 0;
    }

    if(item<0)
    {
        num=list->getNumItems();
        status->setText(FXStringVal(num-1)+_(" Items"));
    }
    else
    {
        strcpy(buf,(list->getItemText(item)).text());
        file=buf;
        if((p=strchr(file,'\t')))
        {
            *p=0;
            name=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
			sprintf(hsize,"%llu",list->getItemFileSize(item));			
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            ext=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            usr=file;
            file=p+1;
        }
        if((p=strchr(file,'\t')))
        {
            *p=0;
            grp=file;
            perm=p+1;
        }
        if(strstr(ext.text(),_("Link")) || strstr(ext.text(),_("Broken link")))
        {
            FXint nread;

            nread=readlink((path+PATHSEPSTRING+name).text(),buf,sizeof(buf)-1);
            if(nread>0)
            {
                buf[nread]='\0';
                status->setText(name+" -> "+buf);
            }
        }
        else
        {
            ::hSize(hsize,hsize);
            size=hsize;
            status->setText(usr+","+grp+" "+perm+" ("+size+") ");
        }

    }
    return 1;
}


// Update label (title) in file list
long FilePanel::onUpdLabel(FXObject*,FXSelector,void*)
{
    char directory[MAXPATHLEN+1];
    register char *dir = directory;
    strcpy(dir,list->getDirectory().text());
    FXFont *normalFont = getApp()->getNormalFont();
    register int len = strlen(dir);
    register int text_width = normalFont->getTextWidth(dir,len);
    int lbl_width = label->getWidth();

    while((text_width > lbl_width) && len)
    {
        dir++;
        len--;
        dir[0] = '.';
        dir[1] = '.';
        dir[2] = '.';
        text_width = normalFont->getTextWidth(dir,len);
    }
	FXString path=dir;
	if (list->getPattern() != "*")
		path=path+PATHSEPSTRING+list->getPattern();
    label->setText(path);
    return 1;
}


long FilePanel::onUpdUp(FXObject* o,FXSelector,void*)
{
    FXButton *button=(FXButton*)o;
    int style=button->getButtonStyle();
    if(style & BUTTON_TOOLBAR)
    {
        if(current->list->getDirectory()!=ROOTDIR)
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
        else
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    }
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
    return 1;
}


long FilePanel::onUpdPaste(FXObject* o,FXSelector,void*)
{
    FXButton *button=(FXButton*)o;
    int style=button->getButtonStyle();
    if(style & BUTTON_TOOLBAR)
    {
        if(!clipboard.empty())
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
        else
            o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
    }
    else
        o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

    return 1;
}


// Update menu items and toolbar buttons that are related to file operations
long FilePanel::onUpdMenu(FXObject* o,FXSelector,void*)
{
   	// Menu item is disabled when nothing is selected
	FXint num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->dirpanel;
	DirItem* item=(DirItem*)dirpanel->list->getCurrentItem();
	
	if ((dirpanel->shown() && item) || (num>=1))
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);
	else
		o->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);

    return 1;
}


// Update the status of the menu items that should be disabled when selecting multiple files
long FilePanel::onUpdSelMult(FXObject* sender,FXSelector,void*)
{
   	// Menu item is disabled when nothing is selected or multiple selection
	FXint num=current->list->getNumSelectedItems();
	DirPanel* dirpanel=((XFileExplorer*) mainWindow)->dirpanel;
	DirItem* item=(DirItem*)dirpanel->list->getCurrentItem();
	
	if (current->selmult || (num==0 && !item) || (!dirpanel->shown() && num==0))
		sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_DISABLE),NULL);
  	else
    	sender->handle(this,FXSEL(SEL_COMMAND,FXWindow::ID_ENABLE),NULL);

	return 1;
}

