/**************************************************
 xdkcal.c -- Copyright(c) 1999 Jiro Sekiba <jir@hello.to>
 **************************************************/
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <regex.h>
#include <unistd.h>
#include <signal.h>
#include <getopt.h>
#include "window.h"
#include "config.h"

#ifdef X_LOCALE
#include <X11/Xlocale.h>
#else
#include <locale.h>
#endif

#define DEFAULT_FONT "-misc-*"
#define DEFAULT_DEFAULT_COLOR "#FFFFFF"
#define DEFAULT_HOLIDAY_COLOR "#FC0000"
#define DEFAULT_SATURDAY_COLOR "#0808FC"
#define DEFAULT_TODAY_COLOR "#08FC08"
#define DEFAULT_MONTH_COLOR "#CCCCCC"
#define DEFAULT_PAD "2"
#define DEFAULT_STYLE "0" /* NORMAL_STYLE */
#define DEFAULT_DRAW "1"  /* draw week/month */
#define DEFAULT_NUM "2"   /* number day of week 2 chars */
                            
#define HASH_MAX 103
#define VALUE_BUF 256

static int target_year = 0;
static int target_month = 0;
static int save_configuration = -1;

enum prop_list
{
  WINDOW_X,
  WINDOW_Y,
  FONTSET,
  DEFAULT_COLOR,
  HOLIDAY_COLOR,
  SATURDAY_COLOR,
  TODAY_COLOR,
  MONTH_COLOR,
#ifdef HAVE_LIBXEXT
  SHAPE,
#endif
  PADDING,
  STYLE,
  DRAW_MONTH,
  DRAW_WEEK,
  NUM_WEEK,
  NUM_PROPERTIES
};

static char filename[] = ".xdkcalrc";
static int loop = -1;

struct prop
{
  char name[32];
  char value[VALUE_BUF];
  unsigned int hash;
  regex_t pregex;
};

unsigned int prop_index[HASH_MAX];

struct prop properties[NUM_PROPERTIES] ={
    {"window_x","0",0},
    {"window_y","0",0},
    {"fontset",DEFAULT_FONT,0},
    {"default_color",DEFAULT_DEFAULT_COLOR,0},
    {"holiday_color",DEFAULT_HOLIDAY_COLOR,0},
    {"saturday_color",DEFAULT_SATURDAY_COLOR,0},
    {"today_color",DEFAULT_TODAY_COLOR,0},
    {"month_color",DEFAULT_MONTH_COLOR,0},
#ifdef HAVE_LIBXEXT
    {"shape","0",0},
#endif
    {"padding",DEFAULT_PAD,0},
    {"style",DEFAULT_STYLE,0},
    {"draw_month",DEFAULT_DRAW,0},
    {"draw_week",DEFAULT_DRAW,0},
    {"num_week",DEFAULT_NUM,0}
};

static int on_off(char *onoff,int def)
{
  if(strncmp(onoff,"on",2) == 0)
      return 1;
  else if(strncmp(onoff,"off",3) == 0)
      return 0;
  else
      return def;
}

unsigned int hash(char *str)
{
    int len,i;
    unsigned int tmp;
    
    len = strlen(str);
    tmp=1;
    for(i=0;i<len;i++)
    {
        tmp=tmp * str[i];
    }

    return tmp%HASH_MAX;
}

FILE *init_file(char *mode)
{
    char file[512];
    FILE *fp;
    
    strcpy(file,getenv("HOME"));
    strcat(file,"/");
    strcat(file,filename);

    if(!(fp = fopen(file,mode)))
    {
        fprintf(stderr,"can't open file: %s\n",file); 
        return NULL;
    }
    return fp;

}

int init_properties()
{
   int i;
   unsigned int hash_value;
   char tmp[512];

   for(i=0;i<NUM_PROPERTIES;i++)
   {
       hash_value = hash(properties[i].name);
       properties[i].hash = hash_value;
       prop_index[hash_value] = i;
       sprintf(tmp,"\\(%s\\) *= *\\(.*\\)",properties[i].name);
       regcomp(&(properties[i].pregex),tmp,0);
   }
   return 0;
}

int write_property()
{
    FILE *fp;
    int i;

    if(!(fp = init_file("w")))
        return 1;
    
    for(i=0;i<NUM_PROPERTIES;i++)
    {
        fprintf(fp,"%s = %s\n",properties[i].name,properties[i].value);
    }

    fclose(fp);
    return 0;
}

int substr(char *dest,char *src,int start, int end)
{
    int i;
    int index = 0;

    for(i=start;i<end;i++)
    {
        if(src[i]=='\n')
            break;
        dest[index] = src[i];
        index++;
    }
    dest[index]='\0';
    return 0;
}

int read_property()
{
    FILE *fp;
    char buf[1024];
    char key[32],value[1024];
    int i;

    regmatch_t match[5];

    if(!(fp = init_file("r")))
        return 1;

    while( fgets(buf,1024,fp))
    {
        for(i=0;i<NUM_PROPERTIES;i++)
        {
            if(!regexec(&(properties[i].pregex),buf,5,match,REG_NOTEOL))
            {
                substr(key,buf,match[1].rm_so,match[1].rm_eo);
                substr(value,buf,match[2].rm_so,match[2].rm_eo);
#ifdef DEBUG
                printf("key = %s, value = %s, hash(key) = %u\n",
                       key,value,hash(key));
#endif                
                strcpy(properties[prop_index[hash(key)]].value,value);
            }
        }
    }

    fclose(fp);
    return 0;
}

void short_usage()
{
  fprintf(stderr,
"Usage: %s [-h][-x window_x][-y window_y][-f fontset][-C default_color]
              [-H holiday_color][-T today_color][-M month_color][-t on/off]
              [-p padding][-s style_num][-m on/off][-y on/off]
              [-n num_week]  [month [year]]\n",PACKAGE);

}

void usage()
{
  fprintf(stderr,"%s %s - Desktop Calendar for X - by Jiro Sekiba(jir@hello.to)\n\n",PACKAGE,VERSION);
  short_usage();
  fprintf(stderr,"\nshort\tlong\t\t argument   description\n");
  fprintf(stderr,"  -h\t--help\t\t\t   display this help message\n");
  fprintf(stderr,"  -x\t--%s\t  x\t   position x\n",properties[WINDOW_X].name);
  fprintf(stderr,"  -y\t--%s\t  y\t   position y\n",properties[WINDOW_Y].name);
  fprintf(stderr,"  -f\t--%s\t fontset   fontset to write calendar\n",properties[FONTSET].name);
  fprintf(stderr,"  -C\t--%s\t color\t   default color\n",properties[DEFAULT_COLOR].name);
  fprintf(stderr,"  -H\t--%s\t color\t   holiday color\n",properties[HOLIDAY_COLOR].name);
  fprintf(stderr,"  -S\t--%s color\t   saturday color\n",properties[SATURDAY_COLOR].name);
  fprintf(stderr,"  -T\t--%s\t color\t   today color\n",properties[TODAY_COLOR].name);
  fprintf(stderr,"  -M\t--%s\t color\t   month color\n",properties[MONTH_COLOR].name);
#ifdef HAVE_LIBXEXT
  fprintf(stderr,"  -t\t--%s\t\t on/off    transparent(default off)\n",properties[SHAPE].name);
#endif
  fprintf(stderr,"  -p\t--%s\t padding   padding\n",properties[PADDING].name);
  fprintf(stderr,"  -s\t--%s\t\t style_num drawing style(0:normal 1:line)\n",properties[STYLE].name);
  fprintf(stderr,"  -m\t--%s\t on/off    draw month(default on)\n",properties[DRAW_MONTH].name);
  fprintf(stderr,"  -w\t--%s\t on/off    draw weeks(default on)\n",properties[DRAW_WEEK].name);
  fprintf(stderr,"  -n\t--%s\t num\t   number of chars for weeks(default 2)\n",properties[NUM_WEEK].name);
  fprintf(stderr,"  -N\t--no_save\t\t   don't save configuration\n");
}


void init_args(int argc, char *argv[])
{
  int opt_index = 0;
  int help_flag = 0;
  int err_flag = 0;
  int version_flag = 0;
  int c;
  struct option long_opts[]=
  {{"help",no_argument,NULL,'h'},
   {properties[WINDOW_X].name,required_argument, NULL ,'x'},
   {properties[WINDOW_Y].name,required_argument, NULL ,'y'},
   {properties[FONTSET].name,required_argument, NULL ,'f'},
   {properties[DEFAULT_COLOR].name,required_argument, NULL ,'C'},
   {properties[HOLIDAY_COLOR].name,required_argument, NULL ,'H'},
   {properties[SATURDAY_COLOR].name,required_argument, NULL ,'S'},
   {properties[TODAY_COLOR].name,required_argument, NULL ,'T'},
   {properties[MONTH_COLOR].name,required_argument, NULL ,'M'},
#ifdef HAVE_LIBXEXT
   {properties[SHAPE].name,required_argument, NULL ,'t'},
#endif
   {properties[PADDING].name,required_argument, NULL ,'p'},
   {properties[STYLE].name,required_argument, NULL ,'s'},
   {properties[DRAW_MONTH].name,required_argument, NULL ,'m'},
   {properties[DRAW_WEEK].name,required_argument, NULL ,'w'},
   {properties[NUM_WEEK].name,required_argument, NULL ,'n'},
   {"no_save",no_argument, NULL ,'N'},
   {"version",no_argument, NULL ,'v'},
   {0,0,0,0}  /* <- Tnx for kota */
  };
  
  while(1)
  {
    c = getopt_long_only(argc,argv,
                         "hx:y:f:C:H:S:T:M:t:p:s:m:w:n:Nv",
                         long_opts,&opt_index);
    if(c == -1)
        break;
    switch(c)
    {
      case 'h':
        help_flag = -1;
        break;
      case 'x':
        strncpy(properties[WINDOW_X].value,optarg,VALUE_BUF);
        break;
      case 'y':
        strncpy(properties[WINDOW_Y].value,optarg,VALUE_BUF);
        break;
      case 'f':
        strncpy(properties[FONTSET].value,optarg,VALUE_BUF);
        break;
      case 'C':
        strncpy(properties[DEFAULT_COLOR].value,optarg,VALUE_BUF);
        break;
      case 'H':
        strncpy(properties[HOLIDAY_COLOR].value,optarg,VALUE_BUF);
        break;
      case 'S':
        strncpy(properties[SATURDAY_COLOR].value,optarg,VALUE_BUF);
        break;
      case 'T':
        strncpy(properties[TODAY_COLOR].value,optarg,VALUE_BUF);
        break;
      case 'M':
        strncpy(properties[MONTH_COLOR].value,optarg,VALUE_BUF);
        break;
      case 't':
#ifdef HAVE_LIBXEXT
        sprintf(properties[SHAPE].value,"%i",on_off(optarg,0));
#else
        fprintf(stderr,"Sorry, you haven't got shape extension\n");
#endif
        break;
      case 'p':
        strncpy(properties[PADDING].value,optarg,VALUE_BUF);
        break;
      case 's':
        strncpy(properties[STYLE].value,optarg,VALUE_BUF);
        break;
      case 'm':
        sprintf(properties[DRAW_MONTH].value,"%i",on_off(optarg,1));
        break;
      case 'w':
        sprintf(properties[DRAW_WEEK].value,"%i",on_off(optarg,1));
        break;
      case 'n':
        strncpy(properties[NUM_WEEK].value,optarg,VALUE_BUF);
        break; 
      case 'N':
        save_configuration = 0;
        break;
      case 'v':
        version_flag = -1;
        break;
      default:
        err_flag = -1;
        break;
    }
  }

  if(err_flag)
  {
    short_usage();
    exit(1);
  }
  if(help_flag)
  {
    usage();
    exit(0);
  }
  if(version_flag)
  {
    fprintf(stderr,"%s -- version %s\n",PACKAGE,VERSION);
    exit(0);
  }

  if(optind < argc)
  {
    target_month = atoi(argv[optind]);
    printf("target_month = %i\n",target_month);
  }

  if(optind + 1 < argc)
  {
    target_year = atoi(argv[optind + 1]);
    printf("target_year = %i\n",target_year);
  }

}

int calc_sleep_time()
{
  time_t now,tomorrow;
  struct tm *tm_now;

  time(&now);
  tm_now = localtime(&now);

  tm_now->tm_mday++;
  tm_now->tm_sec = 0;
  tm_now->tm_min = 0;
  tm_now->tm_hour = 0;
  
  tomorrow = mktime(tm_now);
  return tomorrow - now;

}

void int_handler(int signum)
{
  loop=0;
}

int main(int argc, char *argv[])
{
  struct sigaction act,oact;
  time_t now;
  struct tm *tm_today;
  int sleep_time;

#ifdef DEBUG
  printf("LC_ALL = %s\n",setlocale(LC_ALL,""));
#else
  setlocale(LC_ALL,"");
#endif  

  time(&now);
  tm_today = localtime(&now);

  target_year = (tm_today->tm_year) + 1900;
  target_month = (tm_today->tm_mon) + 1;

  init_properties();
  read_property();
  init_args(argc,argv);

  xdkcal(target_year,
         target_month,
         atoi(properties[WINDOW_X].value),
         atoi(properties[WINDOW_Y].value),
         properties[FONTSET].value,
         properties[DEFAULT_COLOR].value,
         properties[HOLIDAY_COLOR].value,
         properties[SATURDAY_COLOR].value,
         properties[TODAY_COLOR].value,
         properties[MONTH_COLOR].value,
#ifdef HAVE_LIBXEXT
         atoi(properties[SHAPE].value),
#endif
         atoi(properties[PADDING].value),
         atoi(properties[STYLE].value),
         atoi(properties[DRAW_MONTH].value),
         atoi(properties[DRAW_WEEK].value),
         atoi(properties[NUM_WEEK].value)
         );
  if(save_configuration)
      write_property();

  act.sa_handler = int_handler;
  act.sa_flags = SA_NODEFER;
  sigaction(SIGINT,&act,&oact);
  sigaction(SIGTERM,&act,&oact);
  repaint();

  do
  {
      repaint();
      sleep_time = calc_sleep_time();
#ifdef DEBUG      
      printf("sleep %i seconds\n",sleep_time);
#endif      
      sleep(sleep_time);
  }while(loop);

  destroy_data();
  return 0;
}

