/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/
// written by Karsten Laux, June 1999  

#ifndef _SCREENSURFACE_H_
#define _SCREENSURFACE_H_

#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif

#include <wftk/surface.h>
#include <wftk/mutex.h>

namespace wftk {

class Region;

/// interface to the underlying SDL_Surface
class ScreenSurface : public Surface
{
 public:
  /// Initialize the target screen surface
  ScreenSurface(int w, int h, int bitsPerPixel, bool fullscreen, bool resizeable, Uint32 extra_flags);
  ///
  ~ScreenSurface() {sdlSurface_ = 0;} // keep Surface from calling SDL_FreeSurface()

  /// Update the surface
  void update();
  /// Update a rectangle on the surface
  void update(const Rect& dest);
  /// Update a region on the surface
  void update(const Region& dest);
  /// If doublebuffered(), flip the surface and its buffer, otherwise update()
  void flip();
  /// returns true if the surface is double buffered
  bool doublebuffered() const {return (sdlSurface_->flags & SDL_DOUBLEBUF) != 0;}
  /// returns true if the surface is full screen
  bool fullscreen() const {return (sdlSurface_->flags & SDL_FULLSCREEN) != 0;}
  /// returns true if the surface is resizeable
  bool resizeable() const {return (sdlSurface_->flags & SDL_RESIZABLE) != 0;}
  /// ???
  static bool setGamma(float red, float green, float blue);
  /// ???
  static bool setGammaRamp(Uint16 *red, Uint16 *green, Uint16 *blue);
  /// The SDL flags that this screen surface was created with
  Uint32 flags_;

  /// lock this before accessing the screen, need for things like SMPEG
  /**
   * ScreenSurface doesn't do it's own locking, since these functions
   * are typically called in the draw loop, and it's safer to have
   * locking at the begin/end of the whole cycle and nowhere else.
   **/
  Mutex mutex;

  /// resize the SDL window
  void resize(unsigned w, unsigned h);

  /** 
   * This signal is emitted inside the mutex.grab() that RootWindow
   * wraps around its call to ScreenSurface::resize(), so responding
   * to it is a safe way to make child surfaces of the screen valid
   * again
   **/
  SigC::Signal2<void,int,int> Resized;

 private:
  ScreenSurface(const ScreenSurface&);
  ScreenSurface& operator=(const ScreenSurface&);
};

}

#endif



