/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _FONT_H
#define _FONT_H

#include <string>
#include <exception>

#include <wftk/surface.h>
#include <wftk/color.h>
#include <wftk/point.h>
#include <wftk/resources.h>
#include <wftk/region.h>

// pity we have to export this
#include <wftk/ref_map.h>

namespace wftk {

/**FreeType based surface font.
   This class uses the benefit of truetype fonts 
   (antialiasing, *many* free fonts) as well as the speed gain
   of surface fonts. When creating an object of Font the surfaces 
   for the characters are all rendered in best quality. 
   When the font-object is asked
   to create a surface from a string it just blits these character surfaces,
   which is quite fast.
*/
class Font 
{
 public:

  /// an exception thrown when a font fails to load properly
  class BadFont : public std::exception {};

   /**Constructor.
     fontfilename has to be the name of a truetype font file.
     The alpha channel of the color is ignored.
  */
  Font(const std::string& fontfilename, unsigned ptsize = 12,
	const Color& color = textColor(), unsigned face_index = 0) throw(BadFont);
  /// load a font from memory
  Font(const unsigned char* buffer, unsigned buf_size, unsigned ptsize = 12,
	const Color& color = textColor(), unsigned face_index = 0) throw(BadFont);
   
  /// Copy a Font (inexpensive, fonts are refcounted)
  Font(const Font& f) : glyphs_(f.glyphs_) {if(glyphs_) glyphs_->ref();}
  /// Create an invalid Font
  Font() : glyphs_(0) {}

  ///
  ~Font() {if(glyphs_) glyphs_->unref();}
  /// Copy another Font into this one
  Font& operator=(const Font& f);

  /// internal type
  friend class FontData;

  /// Glyph class: represents a single ASCII value as a surface
  class Glyph : public Surface
  {
   public:
    /// set the glyph image
    void set(FontData&, const Color&, unsigned char);

    /// holds the info from FT_Glyph_Slot_Rec
    struct Metrics
    {
      long width;
      long height;

      long horiBearingX;
      long horiBearingY;
      long horiAdvance;

      long vertBearingX;
      long vertBearingY;
      long vertAdvance;

      long linearHoriAdvance;
      long linearVertAdvance;
      Point advance;

      int bitmap_left;
      int bitmap_top;
    };

    /// get the glyph's metrics
    const Metrics& metrics() const {return metrics_;}

   private:
    friend class Font;
    // turns off SDL_SRCALPHA temporarily, since alpha->alpha blits don't do blending
    void copy(Surface& target, const Point& dest, const Region& destmask) const;

    Metrics metrics_;
  };

  /** return surface representation of a char.
      if no font is loaded, the surface returned is empty.
  */
  const Glyph &getChar(unsigned char c) const
	{return glyphs_ ? (*glyphs_)[c] : bad_glyph_;}

  /**return bitmap representation of a string.
     if no font is loaded, the bitmap returned is empty.
  */
  Surface *getString(const std::string & txt) const
	{Point p; return getString(txt, p);}
  /// Get the string blitted to a surface, and the offset from the starting pen position to the upper left hand corner
  Surface *getString(const std::string&, Point&) const;
  /// Draw a string on the surface, with the pen starting at pos
  int blitString(const std::string& txt, Surface& target, const Point& pos) const
	{return blitString(txt, target, pos, target.rect());}
  /// Draw a string on the surface, with the pen starting at pos, with a masking region taken relative to the target
  int blitString(const std::string& txt, Surface& target, const Point& pos, const Region& mask, bool copy = false) const;
  /// Get the area covered by a string, with (0, 0) being the starting pen position
  Rect getExtents(const std::string&) const;

  /// info from FT_Face_Rec
  struct Metrics
  {
    unsigned short units_per_EM;
    short ascender;
    short descender;
    short height;

    short max_advance_width;
    short max_advance_height;

    short underline_position;
    short underline_thickness;

    long xMin, xMax, yMin, yMax;
  };

  /// the font height in pixels
  int getHeight() const {return glyphs_ ? glyphs_->metrics().height / 64: -1;}

  /// get the font's metrics
  const Metrics& metrics() const {return glyphs_ ? glyphs_->metrics() : bad_metrics_;}
	
  /// Return true if the font is valid
  bool valid() const {return glyphs_ != 0;}

  /// Return the foreground color of the font.
  Color color() const {return glyphs_? glyphs_->color() : Color();}
  /// Set the color of the font.
  void setColor(const Color&);

  /// The font specified by the resource "text_font"
  static const Font& textFont();
  /// Get the default text color, specified by the resource "text_color"
  static const Color& textColor();

  struct ResLoad {
    std::pair<Font,bool> operator()(const std::string&);
  };
  struct ResInval {
    typedef const Font& OutType;
    OutType operator()(const std::string&) const {return textFont();}
  };
  /** Load font from truetype file and associate with a named resource
   *
   * Use the \b load() function to load a font from a file on disk and
   * register it with wftk's Resources engine.  The font specification is a
   * string, formatted thus: \n
   * &lt; \e fontname, \e size, \e foreground, \e background &gt;
   * @param fontname font name to be registered
   * @param size size of font in points
   * @param foreground foreground color of font, \c 0xAARRBBGG or \c 0xRRBBGG
   * 
   * \code
   *   Font::registry.load("button_font", "wf_opal.ttf, 16, 0xF0F0F0");
   * \endcode
   * Note that the wftk will attempt to use the "text_font" and "button_font"
   * resources by default for widgets / labels that don't have a font specified. 
   * 
   * Use the \b find() function to retrieve a font.  If the
   * named font has not been registered, find() will return the default text font.
   * \code
   *  MultiLineEdit* title = new MultiLineEdit(bg, 50, 10, 400, 70, Font::registry.find("button_font"));
   * \endcode
   */
  static ResourceRegistry<Font,ResLoad,ResInval> registry;
  /// refcounted resource
  typedef Resource<Font> Resource;

 private:
  /** The SurfaceTable class defines the set of surfaces that we 
   *  will use when drawing this font to screen.
   */
  class SurfaceTable
  {
   public:
    SurfaceTable(FontData& font, const Color& color) : font_(font), color_(color) {}

    const Color& color() const {return color_;}

    const Metrics& metrics() const;

    const Glyph& operator[](unsigned char);

    const FontData& font() const {return font_;}

    void ref();
    void unref();
    // reference another surface table for the same font, with a different color
    SurfaceTable* ref(const Color&);

   private:
    // unimplemented
    SurfaceTable(const SurfaceTable&);
    SurfaceTable& operator=(const SurfaceTable&);

    FontData& font_;
    Color color_;
    Glyph glyphs_[256];
  } *glyphs_;

  class MapType : public RefMap<Color, SurfaceTable> {
   public:
    MapType(FontData& font) : font_(font) {}

   private:
    // create the entries in the RefMap
    virtual Font::SurfaceTable* create(const Color& color)
	{return new Font::SurfaceTable(font_, color);}

    FontData& font_;

  };

  // for when there's no font loaded
  static Glyph bad_glyph_;
  static Metrics bad_metrics_;
};

} // namespace wftk

#endif // _FONT_H
