/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _DIALOG_H
#define _DIALOG_H

#include <wftk/singlecontainer.h>
#include <list>

namespace wftk {

/**
*  Dialog. A class for dialog-type windows
*
*  At creation Dialog is hidden. Call exec() to popup the dialog.
*/
class Dialog : public SingleContainer
{

 public:
  /// Create a dialog
  Dialog();
  virtual ~Dialog();

  /// signal sent when dialog is closed
  SigC::Signal0<void> done;

  /**
   * run the dialog window
   *\param root the ScreenArea that this window is subordinate to. Will take the root window per default
   *\param blocking whether the exec() will block until the dialog closes
   *\param whether this dialog will block input going to other widgets. If
   * more than one modal dialog exists, the last created one is the only one
   * that can receive input. 'modal' defaults to true if 'root' is null,
   * false otherwise.
   *
   * A blocking exec() will increase the dialog's refcount by 1.
   * This is so we can support two paradigms of dialog calls.
   *
   * For non-blocking calls:
   *
   *  (new MyDialog())->exec();
   *
   * In this case, the dialog is created with a floating reference.
   * The exec() call places the dialog in a parent window (the root
   * window by default), which bumps the refcount and drops the floating
   * reference. When the dialog is closed, it is removed from its parent
   * window, its refcount drops, and it is deleted.
   *
   * For blocking calls:
   *
   * MyDialog dialog;
   * dialog.exec(0, true);
   *
   * We never want to delete this dialog, since it's declared on the
   * stack. The exec() call bumps its refcount by one, so even after
   * the dialog closes and exec() exits it still has a nonzero refcount.
   * With no additional intervention, the dialog will live until it
   * goes out of scope.
   *
   * For more complicated situations, the user can make appropriate
   * ref() and unref() calls to get the desired dialog behavior.
   *
   * A dialog will also play the "open" and "close" named Sample resources
   * when it is opened and closed.
   */
  void exec(ScreenArea* root, bool blocking, bool modal);
  /// A version of 'exec' to set default args
  /**
   * This version is only here to make 'modal' default to 'root == 0'.
   **/
  void exec(ScreenArea* root = 0, bool blocking = false)
	{exec(root, blocking, root == 0);}

  /// Forcibly close this dialog
  void close();

  /// Returns true if the dialog is visible
  bool isOpen() {return open_;}

  /** 
   * make sure only the top modal dialog gets mouse clicks
   *\return the dialog that is on top ?
   */
  static Dialog* getTop() {return !highest_.empty() ? highest_.front() : 0;}

 protected:
  /// Toss packing call upwards to parent
  virtual void packingUpdateParent();
  /// we don't want this propagating out of the dialog
  virtual bool buttonEvent(Mouse::Button, bool pressed, const Point&) {return true;}
 private:
  ///
  bool open_;
  ///
  ScreenArea* root_;
  ///
  typedef std::list<Dialog*> HighStack;
  ///
  static HighStack highest_;
};


} // namespace wftk

#endif // !_DIALOG
