/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef _TABLE_H
#define _TABLE_H

#include <wftk/screenarea.h>
#include <vector>

namespace wftk {

/// an automatically expanding table
class Table : public ScreenArea
{
 public:
  ///
  Table() : top_(0), left_(0) {}
  ///
  ~Table() {freeGrid();} // ScreenArea destructor will clear the children's parents

  /// returns true if packing succeeded, false on overlap
  bool pack(ScreenArea*, unsigned x, unsigned y, unsigned w = 1, unsigned h = 1);

  /// insert a row before the indexed one
  /**
   * widgets spanning this row are stretched across it, all other cells start empty
   **/
  void insertRow(unsigned index) {insertEdge(index, false);}
  /// insert a column before the indexed one
  /**
   * widgets spanning this column are stretched across it, all other cells start empty
   **/
  void insertColumn(unsigned index) {insertEdge(index, true);}

  /// remove a widget
  void remove(ScreenArea*);
  /// remove all widgets
  void clear();

  /// returns the child which owns this cell, if any
  ScreenArea* child(unsigned x, unsigned y) const
	{GridElem* elem = gridPos(x, y); return elem ? elem->content : 0;}

  /// get prefered, minimum sizes for a row
  PackingInfo::Expander getRowPackingInfo(unsigned) const;
  /// get prefered, minimum sizes for a column
  PackingInfo::Expander getColumnPackingInfo(unsigned) const;

 protected:
  ///
  virtual void handleResize(Uint16 w, Uint16 h);
  ///
  virtual void setPackingInfo();
 private:

  struct GridElem
  {
    GridElem() : right(0), down(0), content(0) {}

    GridElem* traverse(unsigned, unsigned);

    GridElem *right, *down;
    ScreenArea *content;
  };

  struct GridEdge
  {
    GridEdge() : elems(0), next(0), pixels(0) {}

    GridEdge* traverse(unsigned);

    PackingInfo::Expander packing;
    GridElem *elems;
    GridEdge *next;

    // used to cache row/column size during handleResize()
    Uint16 pixels;
  };

  GridEdge *top_, *left_;

  PackingInfo::Weights x_weight_, y_weight_;

  struct GridSpan
  {
    Uint16 min;
    Uint16 pref;
    GridEdge* start;
    unsigned char length;

    Uint16 min_overage;
    Uint16 pref_overage;

    void calcOverage();

    // for sorting spans
    bool operator<(const GridSpan&) const;
  };

  typedef std::vector<GridSpan> SpanList;

  // free memory used by grid elements
  void freeGrid();

  // insert either a row or column, returns true if one was added to the grid
  bool insertEdge(unsigned index, bool down);

  void packingFromEdge(GridEdge*, PackingInfo::Expander&, PackingInfo::Weights&);

  void setPixels(GridEdge*, PackingInfo::Weights&);
  // this function trashes the SpanList
  void incorporateSpans(GridEdge*, SpanList&);

  // get the GridElem at the specified location, or 0 if it's off the grid
  GridElem* gridPos(unsigned, unsigned) const;
};

}

#endif // !_TABLE_H
