/*
 * Copyright (C) 2003  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <Foundation/NSBundle.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSException.h>
#include <Foundation/NSString.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSValue.h>
#include <AppKit/NSClipView.h>
#include <AppKit/NSNibLoading.h>
#include <AppKit/NSTextField.h>
#include <AppKit/NSTableColumn.h>
#include "DocumentWindowController.h"
#include "AppController.h"
#include "DocumentWindow.h"
#include "DocumentPosition.h"
#include "Match.h"
#include "ExtendedImageView.h"


/*
 * A data source for a table that displayes all
 * matches for a search.
 */
@interface MatchesTableDataSource : NSObject
{
   DocumentWindowController* controller;
}

- (id) init;
- (void) dealloc;

- (void) setController: (DocumentWindowController*)_controller;

@end


/*
 * Non-Public methods of DocumentWindowController.
 */
@interface DocumentWindowController(FindPrivate)
- (unsigned) _countMatches;
- (void) _showMatchAt: (int)index autoscroll: (BOOL)scroll;
- (Match*) _matchAt: (int)index;
- (NSTableView*) _matchesTable;
@end



/*
 * The part of DocumentWindowController that supports
 * text-searching in documents.
 */
@implementation DocumentWindowController(Find)

/*
 * Display a Panel to search for text in searchable Documents.
 */
- (void) showFindPanel: (id)sender
{
   //[(DocumentWindow*)[self window] enterFindMode];
   if (!findPanel)
   {
      [NSBundle loadNibNamed: @"find.gorm" owner: self];

      [pageCurrentlySearched setStringValue: @""];
      [totalPagesToSearch setStringValue: @""];
      [searchPageLabel setStringValue: @""];
      [ofLabel setStringValue: @""];
   }

   NSAssert(findPanel, @"could not load find panel");

   NSAssert(matchesTable, @"table with matches not found");
   [[matchesTable dataSource] setController: self];
   [matchesTable setTarget: self];
   [matchesTable setDoubleAction: @selector(gotoSelectedMatch:)];

   [findPanel setTitle: [NSString stringWithFormat: @"Find in %@",
                                  (NSWindow*)[[self window] title]]];

   [findPanel orderFrontRegardless];
   [findPanel makeKeyWindow];
   [findPanel display];
}


/*
 * Find the previous occurence of the last searched
 * text (from the current document position). This
 * is only possible if the user has entered a text
 * to search for in the find panel.
 */
- (void) findPrevious: (id)sender
{
   if (currentMatch != -1)
   {
      currentMatch--;
      if (currentMatch < 0)
      {
         currentMatch = [matches count] - 1;
      }
      [self _showMatchAt: currentMatch autoscroll: YES];
   }
}


/*
 * Find the next occurence of the last searched text.
 */
- (void) findNext: (id)sender
{
   if (currentMatch != -1)
   {
      currentMatch++;
      if (currentMatch >= [matches count])
      {
         currentMatch = 0;
      }
      [self _showMatchAt: currentMatch autoscroll: YES];
   }
}


/*
 * Go to the match that is currently selected in
 * the matches table.
 */
- (void) gotoSelectedMatch: (id)sender
{
   if ([[self _matchesTable] selectedRow] == -1)
   {
      return;
   }

   currentMatch = [[self _matchesTable] selectedRow];
   [self _showMatchAt: currentMatch autoscroll: YES];
}


/*
 * If no search is in progress, a serach will be initiated
 * by delegating this method to findAll. Otherwise, the
 * current serach is cancelled.
 */
- (void) findAllOrCancel: (id)sender
{
   if ([searchService isRunning])
   {
      [self cancelSearch: sender];
   }
   else
   {
      [self findAll: sender];
   }
}


/*
 * Initiate a search for a text that has been entered
 * by the user.
 */
- (void) findAll: (id)sender
{
   int                   startPage;
   NSRect                startPosition;

   NSAssert(searchText, @"searchText not connected");

   if (searchService)
   {
      [self killCurrentSearch];
   }
      
   if ([[searchText stringValue] length] == 0)
   {
      NSRunAlertPanel(@"Search for what?",
                      @"Please specify the text you are looking for.",
                      @"Continue",
                      nil,
                      nil);
      return;
   }

   if ((![self contentSelection]) || 
       ([self currentPage] != [[self contentSelection] page]))
      // (![[searchText stringValue] isEqualToString: lastSearchText]))
   {
      startPage  = [self currentPage];
      startPosition = NSMakeRect(0, 0, 0, -1);
      // -1 is max height
   }
   else
   {
      startPage = [[self contentSelection] page];

      startPosition = [[self contentSelection] boundingRect];
      startPosition.origin.x = startPosition.origin.x + startPosition.size.width;
      startPosition.origin.y = 
         startPosition.origin.y + (startPosition.size.height / 2);
      startPosition.size.height = 0;
   }

   searchService =
      [[SearchService alloc] initWithText: [searchText stringValue]
                              forDocument: [self pdfDocument]
                           startingAtPage: startPage
                               atPosition: startPosition
                                 delegate: self];
}


/*
 * Aborts the current search.
 */
- (void) cancelSearch: (id)sender
{
   if (searchService && [searchService isRunning])
   {
      [searchService stopAndReleaseWhenDone: NO];
   }
}


/*
 * Kills the current search. This really stops the
 * search immediatly.
 */
- (void) killCurrentSearch
{
   NSAssert(searchService, @"no search in progress");

   NSLog(@"kill current search");
   [searchService setDelegate: nil];
   [searchService stopAndReleaseWhenDone: YES];
   searchService = nil;
}


/*
 * SearchServiceDelegate implementation.
 */
- (void) serviceWillStartSearch: (SearchService*)aService
{
   // clear matches from previous search
   [matches release];
   matches = [[NSMutableArray alloc] initWithCapacity: 0];
   currentMatch = -1;
   [[self _matchesTable] reloadData];

   [searchPageLabel setStringValue: @"Searching Page"];
   [totalPagesToSearch setIntValue: [[self pdfDocument] countPages]];
   [ofLabel setStringValue: @"of"];
   [findButton setTitle: @"Abort"];
}


/*
 * SearchServiceDelegate implementation.
 */
- (BOOL) service: (SearchService*)aService willStartSearchingPage: (int)aPage
{
   [pageCurrentlySearched setIntValue: ([aService countSearchedPages] + 1)];
   return YES;
}


/*
 * SearchServiceDelegate implementation.
 */
- (void) service: (SearchService*)aService didFinishedSearchingPage: (int)aPage
{
   return;
}


/*
 * SearchServiceDelegate implementation.
 */
- (void) service: (SearchService*)aService didFoundMatches: (NSArray*)someMatches
{
   //NSLog(@"found %d matches", [someMatches count]);

   [matches addObjectsFromArray: someMatches];
   [[self _matchesTable] reloadData];

   if (currentMatch == -1)
   {
      currentMatch = 0;
      [self _showMatchAt: currentMatch autoscroll: NO];
   }
}


/*
 * SearchServiceDelegate implementation.
 */
- (void) serviceDidFinishSearch: (SearchService*)aService
{
   [searchPageLabel setStringValue: @""];
   [ofLabel setStringValue: @""];
   [totalPagesToSearch setStringValue: @""];
   [pageCurrentlySearched setStringValue: @""];
   [findButton setTitle: @"Find"];

   AUTORELEASE(searchService);
   searchService = nil;
}

@end


/* ----------------------------------------------------- */
/*  Category FindPrivate                                 */
/* ----------------------------------------------------- */

@implementation DocumentWindowController(FindPrivate)

/*
 * Count the number of matches that are available
 * for the current search.
 */
- (unsigned) _countMatches
{
   return [matches count];
}


/*
 * Display a match from a search. The position for
 * the match is obtained from the matches array. The
 * method does not protect access to the matches array.
 * If autoscrolling is enabled, this method will ensure
 * that the match is visible.
 */
- (void) _showMatchAt: (int)index autoscroll: (BOOL)scroll
{
   NSPoint  p;
   Match*   match;

   match = [matches objectAtIndex: index];

   [self setContentSelection: [match position]];
   [self setCurrentPage: [[match position] page]];

   // ensure that the selection is visible
   p = NSMakePoint([[match position] boundingRect].origin.x * [self scaleFactor],
                   [[match position] boundingRect].origin.y * [self scaleFactor]);

   if (scroll)
   {
      [[[[(DocumentWindow*)[self window] documentView] scrollView] documentView]
         scrollPoint: p];
   }

   [[self _matchesTable] selectRow: index byExtendingSelection: NO];
   [[self _matchesTable] scrollRowToVisible: index];
}


/*
 * Returns a match at a particular position
 * in the matches list. Returns nil if the
 * given index is invalid.
 */
- (Match*) _matchAt: (int)index
{
   if (index >= [matches count])
   {
      return nil;
   }

   return [matches objectAtIndex: index];
}


/*
 * Returns the table view that holds the matches.
 */
- (NSTableView*) _matchesTable
{
   return matchesTable;
}


/*
 * To be invoked by the NSNotificationCenter when
 * a new match was found in the document that is
 * assiociated with this controller.
 */
// - (void) _newMatchArrived: (id)notification
// {
//    Match*      match;
//    unsigned    index;
//    NSString*   context;

//    match   = [[notification userInfo] objectForKey: UserInfoKeyMatch];
//    index   = [[[notification userInfo] objectForKey: UserInfoKeyMatchIndex] intValue];

//    //NSLog(@"found match at %d (%@)", [[match position] page], [match context]);
//    [[self _matchesTable] reloadData];

//    [matchesLock lock];
//    if (currentMatch == -1)
//    {
//       currentMatch = 0;
//       [self _showMatchAt: currentMatch autoscroll: NO];
//    }
//    [matchesLock unlock];
// }

@end


/* ----------------------------------------------------- */
/*  MatchesTableDataSource                               */
/* ----------------------------------------------------- */

@implementation MatchesTableDataSource

- (id) init
{
   if ((self = [super init]))
   {
      controller = nil;
   }
   return self;
}


- (void) dealloc
{
   [super dealloc];
}


- (void) setController: (DocumentWindowController*)_controller
{
   controller = _controller;
}


- (int) numberOfRowsInTableView: (NSTableView*)aTableView
{
   return [controller _countMatches];
}


- (id) tableView: (NSTableView*)aTableView
objectValueForTableColumn: (NSTableColumn*)aTableColumn
             row: (int)rowIndex
{
   id      result;
   Match*  match;

   match = [controller _matchAt: rowIndex];
   if (!match)
   {
      return @"??";
   }

   if ([[aTableColumn identifier] isEqualToString: @"page"])
   {
      result = [NSNumber numberWithInt: [[match position] page]];
   }
   else if ([[aTableColumn identifier] isEqualToString: @"context"])
   {
      result = [NSString stringWithFormat: @"... %@ ...", [match context]];
   }
   else
   {
      result = @"invalid_column";
   }

   return result;
}

@end

