/*
 * Copyright (c) 1993-1994 Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the Computer Systems
 *	Engineering Group at Lawrence Berkeley Laboratory.
 * 4. Neither the name of the University nor of the Laboratory may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

static const char rcsid[] =
    "@(#) $Header: tkwidget.cc,v 1.10 96/03/16 13:12:53 van Exp $ (LBL)";

#include <stdlib.h>
#include "tkwidget.h"
#include "Tcl.h"

TkWidget::TkWidget(const char* path, const char* classname,
		   int reqw, int reqh, u_long extra_events) :
	 TclObject(path),
	 width_(0),
	 height_(0),
	 callback_pending_(0),
	 damage_(0)
{
	Tcl& tcl = Tcl::instance();
	tk_ = Tk_CreateWindowFromPath(tcl.interp(), tcl.tkmain(),
				      (char*)path, 0);
	if (tk_ == 0)
		abort();
	Tk_SetClass(tk_, (char*)classname);
	if (reqw != 0 || reqh != 0)
		Tk_GeometryRequest(tk_, reqw, reqh);
	u_long eventmask = extra_events | (ExposureMask|StructureNotifyMask);
	Tk_CreateEventHandler(tk_, eventmask, handle, (ClientData)this);
	/*
	 * Start with null geometry.  A ConfigureNotify event will
	 * set this right and initiate the resize action.
	 */
	Tk_MakeWindowExist(tk_);

	tcl.eval("winfo depth .");
	mono_ = strcmp(tcl.result(), "1") == 0;
}

Pixmap TkWidget::make_stipple(float level) const
{
	static Pixmap stipple[3];

	int index = int(3. * level);
	if (index > 2)
		index = 2;
	else if (index < 0)
		index = 0;
	Pixmap p = stipple[index];
	if (p == 0) {
		static char bits[3][2] = {
			{ 0x08, 0x02},
			{ 0x05, 0x0a},
			{ 0x0a, 0x0f}
		};
		p = XCreateBitmapFromData(Tk_Display(tk_), Tk_WindowId(tk_), 
					  bits[index], 4, 2);
		stipple[index] = p;
	}
	return (p);
}

XColor* TkWidget::getcolor(const char* name, const char* backup) const
{
	Tcl& tcl = Tcl::instance();
	XColor* c = Tk_GetColor(tcl.interp(), tk_, Tk_GetUid((char*)name));
	if (c == 0) {
		fprintf(stderr, "vat: cannot lookup color: %s (trying %s)\n",
			name, backup);
		c = Tk_GetColor(tcl.interp(), tk_, (char*)backup);
		if (c == 0) {
			fprintf(stderr, "vat: cannot lookup color: %s\n",
				backup);
			exit(1);
		}
	}
	return (c);
}

void TkWidget::set_gcv(XGCValues& v, u_long& mask,
		       Font fid, XColor* fc, XColor* bc) const
{
	mask = GCLineWidth|GCGraphicsExposures;
	v.graphics_exposures = 0;
	v.line_width = 1;
	if (fc != 0) {
		v.foreground = fc->pixel;
		mask |= GCForeground;
	}
	if (bc != 0) {
		v.background = bc->pixel;
		mask |= GCBackground;
	}
	if (fid != 0) {
		v.font = fid;
		mask |= GCFont;
	}
	if (mono_ && fc != 0) {
		float level = fc->red + fc->green + fc->blue;
		level /= 3. * 65536.;
		if (level > 0.2 && level < 0.8) {
			mask |= GCStipple|GCFillStyle;
			v.stipple = make_stipple(level);
			v.fill_style = FillStippled;
		}
	}
}

GC TkWidget::lookup_gc(Font fid, const char* fg, const char* bg) const
{
	XGCValues v;
	u_long mask;
	XColor* fc = (fg != 0) ? getcolor(fg, "black") : 0;
	XColor* bc = (bg != 0) ? getcolor(bg, "white") : 0;
	set_gcv(v, mask, fid, fc, bc);
	return (Tk_GetGC(tk_, mask, &v));
}

void TkWidget::resize()
{
}

void TkWidget::handle(const XEvent&)
{
}

void TkWidget::handle(ClientData cd, XEvent* ep)
{
	TkWidget* w = (TkWidget*)cd;
	
	switch (ep->type) {

	default:
		w->handle(*ep);

	case Expose:
		w->damage_ = 1;
		if (ep->xexpose.count == 0)
			w->redraw();
		break;

	case DestroyNotify:
		/* This will delete the SiteBox as well.
		   XXX can handle be called again after this
		   happens?  E.g., do we have to use defereed
		   freeing?  This will be tricky to incorporate
		   with delete. */
#ifdef notyet
		if (w->ka_ != 0)
			w->ka_->kill(w);
#endif
		break;

	case MapNotify:
		if (w->damage_)
			w->redraw();
		break;

	case ConfigureNotify:
		if (w->width_ != ep->xconfigure.width ||
		    w->height_ != ep->xconfigure.height) {
			w->width_ = ep->xconfigure.width;
			w->height_ = ep->xconfigure.height;
			w->resize();
			w->damage_ = 1;
			w->redraw();
		}
		break;
	}
}

void TkWidget::display(ClientData cd)
{
	TkWidget* w = (TkWidget*)cd;
	w->callback_pending_ = 0;
	if (!Tk_IsMapped(w->tk_)) {
		w->damage_ |= 2;
		return;
	}
	if (w->damage_ != 0) {
		w->damage_ = 0;
		w->draw();
	} else
		w->update();
}
