/* $Id: process.c,v 1.7 2002/10/30 00:23:37 bjk Exp $ */
/*
    Copyright (C) 2001-2002  Ben Kibbey <bjk@arbornet.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include "common.h"

#ifdef HAVE_KVM_H
#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/user.h>
#endif

#ifdef HAVE_LIBKVM
pid_t parentpid(uid_t uid)
{
    static int firstrun;
    int cnt, i;
    pid_t pid = 0;
    char errbuf[LINE_MAX];
    struct kinfo_proc *kp;

    if (!kd && firstrun)
	return -1;

    if (!kd) {
	firstrun = 1;

#ifdef BSD_KVM
	if ((kd = kvm_openfiles(_PATH_DEVNULL, _PATH_DEVNULL, _PATH_DEVNULL,
#else
	/* Darwin */
	if ((kd = kvm_openfiles(_PATH_DEVNULL, _PATH_MEM, _PATH_DEVNULL,
#endif
				O_RDONLY, errbuf)) == NULL) {
	    warnx("%s", errbuf);
	    return -1;
	}
    }

    if ((kp = kvm_getprocs(kd, KERN_PROC_RUID, uid, &cnt)) == NULL) {
        warnx("kvm_getprocs(): %s", kvm_geterr(kd));
	return -1;
    }

    for (i = 0; i < cnt; i++) {
	if (kp[i].kp_eproc.e_flag & EPROC_SLEADER && kp[i].kp_eproc.e_tdev !=
		-1) {
	    pid = kp[i].kp_proc.p_pid;
	    break;
	}
    }

    if (!pid)
	pid = kp[0].kp_eproc.e_ppid;

    return (pid > 1) ? pid : -1;
}
#elif defined(HAVE_PROCFS)
#include <sys/types.h>
#include <sys/stat.h>

#ifdef __svr4__
#include <unistd.h>
#include <procfs.h>
#endif

pid_t parentpid(uid_t uid)
{
    static int firstrun;
    pid_t ppid = 0;
    struct dirent *ent;
    struct stat st;
#ifdef __svr4__
    int fd;
    struct pstatus pstat;
#else
    FILE *fp;
#endif

    if (!procdir && firstrun)
	return -1;

    if (!procdir) {
	firstrun = 1;

	if ((procdir = opendir("/proc")) == NULL) {
	    warn("%s", "/proc");
	    return -1;
	}
    }

    rewinddir(procdir);

    while ((ent = readdir(procdir)) != NULL) {
	char filename[FILENAME_MAX];
#ifndef __svr4__
	int i;
	char buf[LINE_MAX], *t;
#endif

	if (!isdigit((unsigned char)*ent->d_name))
	    continue;

#ifdef __linux__
	snprintf(filename, sizeof(filename), "/proc/%s/stat", ent->d_name);
#else
	snprintf(filename, sizeof(filename), "/proc/%s/status", ent->d_name);
#endif

	if (stat(filename, &st) == -1)
	    continue;

	if (st.st_uid == uid) {
#ifdef __svr4__
	    if ((fd = open(filename, O_RDONLY)) == -1)
		continue;

	    if (pread(fd, &pstat, sizeof(struct pstatus), 0) != 
		    sizeof(struct pstatus)) {
		close(fd);
		continue;
	    }

	    ppid = pstat.pr_ppid;
	    close(fd);
#else
	    if ((fp = fopen(filename, "r")) == NULL)
		continue;

	    if ((t = fgets(buf, sizeof(buf), fp)) == NULL) {
		fclose(fp);
		continue;
	    }

#ifdef __linux__
	    if ((i = sscanf(buf, "%*i %*s %*c %*i %*i %i", &ppid)) < 1) {
#else
	    if ((i = sscanf(buf, "%*s %*i %li", &ppid)) < 1) {
#endif
		fclose(fp);
		continue;
	    }

	    fclose(fp);
#endif

	    break;
	}
    }

    return (ppid > 1) ? ppid : -1;
}
#endif
