/********************************************************************

    File:       upsd.c

    Purpose:    Monitor a UPS for power failure conditions.  The
                UPS can be a local device or we can monitor a master
                upsd over the network.  Can also be used to shut off
                UPS power.

                Copyright 1996, Bob Hauck
                Copyright 2000, Michael Robinton
                
                This program is free software; you can redistribute it
                and/or modify it under the terms of the GNU General Public
                License as published by the Free Software Foundation;
                either version 2 of the License, or (at your option) any
                later version.

                This program is distributed in the hope that it will be
                useful, but WITHOUT ANY WARRANTY; without even the implied
                warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
                PURPOSE.  See the GNU General Public License for more details.

                You should have received a copy of the GNU General Public
                License along with this program; if not, write to the Free
                Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
                USA.

    Language:   GCC 2.7.0

    Author:     Bob Hauck <bobh@wasatch.com>
    Modified:   Michael Robinton <michael@bzs.org>
    Modified:	Russ Magee <rmagee@mtroyal.ab.ca>

    Revision 1.9  2000/06/09 14:54:00  Michael Robinton/Russ Magee
    Integrate Russ Magee's pin remapping contribution.
    RM..
    Removed pin assignment #defines -- now read from upsd.conf
    Changed all code to use logical UPS lines, supported by mapping.c,h

    Revision 1.8  2000/06/07 10:10:00  Michael Robinton
    Remove alarmtimer function here and replace with non-blocking IO
    in net.c to accomodate connection failures for socksified client
    Pass timeout value to NET_Check

    Revision 1.7  1998/06/11 09:36:00  Michael Robinton
    Add console wait message

    Revision 1.6  1998/05/12 09:51:00  Michael Robinton
    Change name of Status to NetStatus to reflect it's usage.
    Change name of LastFail to LastValidStatus to reflect it's purpose.
    Add latching logic to detection of UPS status change.
    One-shot mode returns LastValidStatus if -recall option is set.
    "NetStatus" used by NET_Serve is uninitialized until EvalStatus has
    a chance to 'Mode_Count' down, set it to LastValidStatus during init.
    Added -d Mode_Dummy testing mode which pretends to be a master and
    responds to network querys with the status contents of /tmp/upsds
    or in 'local' mode gets it's input from /tmp/upsds rather than
    the UPS interface or the network.
    EvalStatus now handles case S_ERROR. Eliminate race startup 
    conditions in EvalStatus that cause spurious SIGPWR ok's.
    Add support for POWERFAILNOW (sysinit 2.52 and up), init 
    defaults to 'powerfail' on LOW BAT on older sysinits.
    Add RememberStatus, Mode_Recall, Mode_Wait to allow continuation 
    of alarm recovery action without multiple shutdowns in runlevel 1 
    allowing for clean shut down of raid md devices and clean restart
    of slave upsd's on un-switched or non-failing power sources

    Revision 1.5  1997/02/14 23:04:37  bobh
    *** empty log message ***

    Revision 1.4  1997/01/20 22:35:48  bobh
    Fix "shutdown already running" on startup.

    Revision 1.3  1996/12/15 02:07:38  bobh
    Add an alarm signal to prevent hangs in slave mode when
    the master has hung and is unable to fork a child to handle
    the request.

    Revision 1.2  1996/11/24 18:27:45  bobh
    Wasn't handling the case of poll time == 0 correctly.

    Revision 1.1  1996/11/23 16:45:12  bobh
    Initial revision

**********************************************************************/
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <sys/stat.h>
#include <errno.h>
#include "common.h"
#include "mapping.h"
#include "net.h"
#include "ups.h"

/*  define Dummy modes and transfer file  */
#define Dmaster  1
#define Dlocal   2
#define DUMSTAT  "/tmp/upsds"

/*  Flags set by command-line options
 */
static int   Mode_Wait;
static int   Mode_Recall;
static int   Mode_Kill;
static int   Mode_NoMaster;
static int   Mode_LowBat;
static int   Mode_Test;
static int   Mode_Port = 401;
static int   Mode_Delay = 10;
static int   Mode_Slave;
static int   Mode_Count = 2;
static int   Mode_Dummy;
static int   Mode_Enhanced;

/* Last status of UPS
 */
static int LastValidStatus = S_OK;
static int LastValidFail;
static int LastFail;
static int Remember = 1;

/*-------------------------------------------------------------------*
     ValidStatus
     
     Check for ValidStatus
     
     Parameters:	status
     
     Returns:		TRUE if valid, FALSE if not valid
 *-------------------------------------------------------------------*/
 static int ValidStatus (int status)
    {
    switch (status)
	{   
	case S_OK	:   return (TRUE);
	case S_ONBAT	:   return (TRUE);
	case S_LOBAT	:   return (TRUE);
	case S_ERROR	:   if (!Mode_Slave)
				return (TRUE);
	}
    return (FALSE);
    }
 
/*-------------------------------------------------------------------*
     RememberStatus

     Remember the last 'valid' status and true status of the ups
     by saving a status file UPSSTAT.
     The output string is formated in bytes as follows:
     
     Byte(s)
       0	Last valid status, used by -recall option
       1	:
       2	Current status
       3	<space>
     4 - 13 	Last valid status for humans to read
       14	:
    15 - 24	True current status for humans to read
       25	<newline>
           
     Local function, not exported.

     Parameters:  	fail {one of}
	S_NOCHANGE     -1 Don't Know
	S_OK            0 Power = OK
	S_ONBAT         1 On Battery
	S_LOBAT         2 Lo Battery
	S_ERROR         3 Cable Fail

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
static void RememberStatus (int fail)
    {
    int fdstat = -1;
    char *colon = ":";
    char *lastvc;			/*  last valid code pointer	*/
    char *lastvm;			/*  last valid message Pointer	*/
    char *curntc;			/*  current code pointer	*/
    char *curntm;			/*  current message pointer	*/
    
    typedef struct
	{
	char nmbr[2], msg[12];
	} STATSTRINGS;
	
    STATSTRINGS S_Codes[5] =
	{
	    {"0:", " Power = OK\n",},
	    {"1:", " On Battery\n",},
	    {"2:", " Lo Battery\n",},
	    {"3:", " No Connect\n",},
	    {"?:", " Don't Know\n",},
	};

    /* Create the memory file needed for runlevel (1) unless a dummy master.
     */
    unlink (UPSSTAT);
    if ((fdstat = open (UPSSTAT, O_CREAT|O_WRONLY, 0644)) >= 0)
        {
	if (fail < 0)
	    fail = S_ERROR +1;
	    
	lastvc = &S_Codes[LastValidStatus].nmbr[0];
	lastvm = &S_Codes[LastValidStatus].msg[0];
	curntc = &S_Codes[fail].nmbr[0];
	curntm = &S_Codes[fail].msg[1];

	write (fdstat, lastvc, 2);
	write (fdstat, curntc, 1);
	write (fdstat, lastvm, 11);
	write (fdstat, colon, 1);
	write (fdstat, curntm, 11);
	close(fdstat);
	Remember = 0;
	}
    }

/*-------------------------------------------------------------------*
    RecallStatus
    
    Recall UPS last status from file
        
    Parameters:  whichfile - TRUE use UPSSTAT, FALSE use DUMSTAT

    Returns:  S_* status
 *-------------------------------------------------------------------*/
static int RecallStatus (int whichfile)
    	{
    	int fdstat = -1;
    	int status = S_NOCHANGE;

        if (whichfile)
            fdstat = open (UPSSTAT, O_RDONLY);
        else
            fdstat = open (DUMSTAT, O_RDONLY);
        
        if (fdstat >= 0)
            {
            read(fdstat, &status, 1);
            close(fdstat);
            status = status & 0xF;      /* convert ascii to binary */
            if ((status > S_ERROR) || (status < 0))
                status = S_NOCHANGE;
            }
        return (status);
        }

/*-------------------------------------------------------------------*
     FetchUPStatus
     
     Fetch the UPS status from net or UPS, local function not exported.
     
     Parameters:	when applicable:
			server, server from which to fetch status
			device, serial port descriptor
     
     Returns:		S_* status
 *-------------------------------------------------------------------*/
static int FetchUPStatus (unsigned long server, int device)
    {
    int status;
    
    if (Mode_Dummy)
	{
	NetStatus = RecallStatus (FALSE);
	if ((Mode_Dummy == Dmaster) &&
	    (NetStatus == S_NOCHANGE))
	    NetStatus = S_DUMMY;	/* send bogus message if master */
	    
	return (NetStatus);
	}

    if (Mode_Slave)
	{                                                    
	/*	Set an alarm so we won't hang if the server goes
	 *	dead during the connect() or read() call.
	 */
 	status = NET_Check (server, Mode_Port, Mode_Delay + 1);
	return (status);
	}
    else
	{
	return (UPS_Check (device));
	}
    }    

/*-------------------------------------------------------------------*
     NotifyInit

     Notify init of power fails by sending SIGPWR.  Local function,
     not exported.

     Parameters:  ok - UPS status, only S_OK, S_ONBAT, S_LOBAT

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
static void NotifyInit (int ok)
    {
    int fd = -1;

    /* Create the info file needed by init.
     */
    unlink (PWRSTAT);
    
    if ((fd = open (PWRSTAT, O_CREAT|O_WRONLY, 0644)) >= 0)
        {
        switch (ok)
            {
            case S_OK:		/*  Power back  */
		write (fd, "OK\n", 3);
                break;

            case S_ONBAT:		/*  On battery  */
                if (!Mode_LowBat)
		    write (fd, "FAIL\n", 5);
                break;

            case S_LOBAT:		/*  Low battery */
		if (Mode_Enhanced)
		    write (fd, "LOW\n", 4);
		else
                    write (fd, "FAIL\n", 5);
               break;

	    default:
		write (fd, "OK\n", 3);
	        break;
            }
        close (fd);
        }

    /*  Send the signal unless we're in test or wait mode.
     */
    if (!Mode_Test)
        kill (1, SIGPWR);
    }
    
/*-------------------------------------------------------------------*
     EvalStatus

     Evaluate the UPS status and log changes.  If we are on battery,
     notify init to start a shutdown.  Will delay notification for
     Mode_Count update cycles so that users are not bothered by
     spurious notifications from short power glitches.  Local function,
     not exported.

     Parameters:  Fail - S_* UPS status code.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
static void EvalStatus (int Fail)
    {
    static int Count    = -1;

    if (LastFail != Fail)	/*  Update memory if there is any change  */
        {
        LastFail = Fail;
        ++Remember;
        }

    if (ValidStatus (Fail))	/*  Act only on valid status  */
        {
        if (LastValidFail != Fail)
            {
            LastValidFail = Fail;
            Count = 0;
            }
        else
            {
            if (Count == -1)	/* Take care of init and rollover */
                Count = Mode_Count;
                
            ++Count;
            }

        if (Count == Mode_Count)
            {
            LogError ("log", StatusString [LastValidFail]);
      
            NetStatus = LastValidFail;
            LastValidStatus = LastValidFail;	/* Only update LastValidStatus here */
            RememberStatus (LastValidFail);	/* to avoid race conditions         */

	    if (Mode_Dummy != Dmaster)
                NotifyInit (LastValidFail);
            }
        }
        
    if (Remember)
        RememberStatus (Fail);
                        
    }


/*-------------------------------------------------------------------*
     Daemon

     Go into the background, become a daemon process.

     Parameters:  None.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
static void Daemon (void)
    {
    switch (fork ())
        {
        case 0: /* I am the child. */
            setsid ();
            chdir ("/");
            umask (0);
            ++IsDaemon;
            break;
            
        case -1: /* Failed to become daemon. */
            LogError ("daemon", "can't fork");
            exit (1);
            
        default: /* I am the parent. */
            exit (0);
        }
    }


/*-------------------------------------------------------------------*

  M A I N   P R O G R A M
  
 *-------------------------------------------------------------------*/
int main (int argc, char *argv [])
    {
    int           option;
    int           status = -1;
    int           fd     = -1;
    unsigned long inaddr = (unsigned) -1;
    char *startmsg = {VERSION " startup complete"};
    char *waitmsg = {VERSION " waiting for power recovery"};

    
    ProgName = argv [0];

    /*  Evaluate the command line.
     */
    while ((option = getopt (argc, argv, "ed:rw:ktli:shp:mc:")) > 0)
        {
        switch (option)
            {
            case 'e':    /*  Enhanced mode, support 'init' powerfailnow */
		Mode_Enhanced = TRUE;
		break;
		
            case 'd':    /*  Dummy test mode, return /etc/upstatus */
            
		    switch (*optarg)
			{
			case 'm':	/*  Dummy is a master  */
			    Mode_Dummy = Dmaster;
			    break;

			case 'l':	/*  Dummy is local mode  */
			    Mode_Dummy = Dlocal;
			    break;
			
			default:
			    LogError ("main", "unknown Dummy mode argument");
			    Usage ();
			    exit (1);
			}
                    break;
                    
            case 'r':    /*  Recall Status  */
                Mode_Recall = TRUE;
                break;

            case 'w':    /*  Wait in foreground for power recovery */
                if ((Mode_Wait = atoi (optarg)) <= 0)
                    {
                    Mode_Wait = 0;
                    LogError ("main", "bad or missing Wait poll time");
                    Usage ();
                    exit (1);
                    }
                else
                    {
                    Mode_Test = TRUE;
                    Mode_NoMaster = TRUE;
                    LastValidStatus = S_NOCHANGE;
                    }
                break;
                                    
            case 'k':    /*  Kill Power  */
                Mode_Kill = TRUE;
                break;

            case 't':    /*  Test Mode  */
                Mode_Test = TRUE;
                break;

            case 'l':    /*  Don't Shutdown until low bat  */
                Mode_LowBat = TRUE;
                break;

            case 'i':    /*  Poll Interval, < 0 means "one shot"  */
                if (optarg)
                    {
                    Mode_Delay = atoi (optarg);
                    if (Mode_Delay < 0)
                        {
                        Mode_Test = TRUE;
                        Mode_NoMaster = TRUE;
                        }
                    }
                else
                    LogError ("main", "bad poll delay spec");
                break;

            case 's':    /*  Slave mode  */
                Mode_Slave = TRUE;
                break;

            case '?':    /*  Help  */
            case 'h':
                Usage ();
                exit (0);
                break;

            case 'p':    /*  IP Port for Master  */
                if (optarg)
                    Mode_Port = atoi (optarg);
                else
                    LogError ("main", "bad port number");
                break;

            case 'm':    /*  Disable Master Mode  */
                Mode_NoMaster = TRUE;
                break;

            case 'c':
                if (optarg)
                    {
                    if ((Mode_Count = atoi (optarg)) < 0)
                        {
                        Mode_Count = 0;
                        LogError ("main", "count must be >= 0");
                        }
                    }
                else
                    LogError ("main", "bad poll count");
                break;
            }
        }

    if (optind >= argc)
        {
        LogError ("main", "no device specified");
        Usage ();
        exit (1);
        }

/* If Dummy master mode is selected, ignore most everything else
 * since all we need to know is the PORT number, if local just use 
 * existing setup but get data from DUMSTAT
 */
    if (Mode_Dummy == Dmaster)
        {
        Mode_Wait = 0;
        Mode_Recall = FALSE;
        Mode_Kill = FALSE;
        Mode_NoMaster = FALSE;
        Mode_LowBat = FALSE;
        Mode_Test = FALSE;
        Mode_Delay = 1;
        Mode_Slave = FALSE;
        Mode_Count = 1;
        }

    /*  Set up the devices...
     */
    if (Mode_Slave)
        {
        if ((inaddr = NET_GetServerAddr (argv [optind])) == (unsigned) -1)
            exit (1);
        }
    else
        {
        
        if (Config_Read (CFGFILE, Mode_Test) == FALSE)
            exit (1);
            
        if ((fd = UPS_Open (argv [optind])) < 0)
            exit (1);

        if (Mode_Kill)
            UPS_Setup (fd, 1);

	if (Mode_Dummy == 0)
            UPS_Setup (fd, 0);
        }

/* Get memory of ups status if any
 */
    if (Mode_Recall)
        {
        if (ValidStatus (status = RecallStatus (TRUE)))
	    LastValidStatus = status;
	}
        
    LastFail = LastValidStatus;
    LastValidFail = LastValidStatus;
    NetStatus = LastValidStatus; /*  Eliminate net startup race condition  */

    unlink (PWRSTAT);
    unlink (UPSSTAT);
    if (Mode_Dummy == 0)
	unlink (DUMSTAT);
    
    /*  Go into the background...
     */
    if (!(Mode_Test))		/* or ! Mode_Wait  */
        Daemon ();
    
    /*  Now start monitoring...
     */
    if (Mode_Wait > 0)
        {
        /*  Foreground-wait mode, used with RO file system
         *  such as RAID or for SLAVE waiting for power 
         *  recovery status from MASTER that is shutdown.
         */
        --IsDaemon;          /* Logging and messages OFF */
        Mode_Delay = Mode_Wait;
        LogError ("main", waitmsg);
        }               
    else
        LogError ("main", startmsg);
    
    while (1)
        {
        status = FetchUPStatus ( inaddr, fd);
        if (Mode_Delay < 0)
            {
            /*  One-shot mode, useful in scripts...returns the
             *  ups status as an exit code > 100.
             */
            if (ValidStatus (status))
                LastValidStatus = status;

            RememberStatus (status);	/*  save status for -recall mode  */

	    if (Mode_Recall)
	 	exit (LastValidStatus + 100);
	    else
		exit (status + 100);
            }
        else
            {
            /*  Normal continuous mode...
             */
	    EvalStatus (status);
            if (Mode_Wait > 0)
                {
                if (LastValidStatus == S_OK)
                    exit (0);

                sleep (Mode_Wait);
                }
            else
                {
                if (Mode_NoMaster || Mode_Slave)
                    sleep (Mode_Delay);
                else
                    NET_Serve (Mode_Port, Mode_Delay);
                }
            }
        }

    /*  Error! (shouldn't happen)
     */
    return 1;
    }
