namespace eval chown {
    variable chown

    # Define the chown array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set chown {
	list.reset ""
	list.clear ""
	entry.dir ""
	entry.file ""
	ddlistbox.users ""
	ddlistbox.groups ""
	user ""
	group ""
	dialog ""
    }
}

# chown::create --
#
#   Method to create the dialog box for the chown command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::create { } {
    global tkWorld
    variable chown

    # Put the focus on the chown dialog if it is already open.
    if [winfo exists $chown(dialog)] {
	switch -- [wm state $chown(dialog)] {
	    normal {
		raise $chown(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $chown(dialog)
	    }
	}
	focus $chown(dialog)
	return
    } else {
	set chown(dialog) [dialog::create .chown Chown]
    }

    # There is only 1 tab with this command dialog.
    set tab1 [tabnotebook::page [dialog::interior \
	    $chown(dialog)] "Options"]

    # Use a frame to encapsulate the file and dir selections
    # so that the frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]

    button $f1.file_button \
	    -text "File" \
	    -width 5 \
	    -command chown::open
    set chown(entry.file) [entry $f1.file_entry \
	    -width 35 \
	    -textvariable chown::chown(file)]
    label $f1.dir_label \
	    -text "Directory" \
	    -width 8
    set chown(entry.dir) [entry $f1.dir_entry \
	    -width 35 \
	    -textvariable chown::chown(dir)]
    checkbutton $f1.recursive \
	    -text "Recursive" \
	    -variable chown::chown(recursive) \
	    -onvalue "R" \
	    -offvalue ""
    grid $f1.file_button $f1.file_entry \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.dir_label $f1.dir_entry $f1.recursive \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build the frame which allows the user to change
    # the user and group.
    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]

    label $f2.label_user \
	    -text "User Name" \
	    -width 10
    set chown(ddlistbox.users) [ddlistbox::create \
	    $f2 chown::chown(user) [system::users]]
    set pad [frame $f2.pad \
	    -width 10]
    label $f2.label_group \
	    -text "Group Name" \
	    -width 10
    set chown(ddlistbox.groups) [ddlistbox::create \
	    $f2 chown::chown(group) [system::groups]]
    grid $f2.label_user $chown(ddlistbox.users) $pad \
	    $f2.label_group $chown(ddlistbox.groups) \
	    -sticky ew \
	    -padx 2 \
	    -pady 2

    # Now build the individual checkbutton options available to
    # the user for the chown command.
    set f3 [frame $tab1.f3 \
	    -class TabnotebookFrame]

    checkbutton $f3.changes \
	    -text Changes \
	    -variable chown::chown(changes) \
	    -onvalue c \
	    -offvalue ""
    checkbutton $f3.silent \
	    -text Silent \
	    -variable chown::chown(silent) \
	    -onvalue f \
	    -offvalue ""
    checkbutton $f3.verbose \
	    -text Verbose \
	    -variable chown::chown(verbose) \
	    -onvalue v \
	    -offvalue ""
    grid $f3.changes $f3.silent $f3.verbose \
	    -sticky ew \
	    -padx 5 \
	    -pady 5

    # Build the tab.
    pack $f1 $f2 $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $chown(entry.file)

    # Define the lists for the reset and clear methods, include the
    # drop down listboxes.
    set chown(list.reset) "user group recursive changes silent verbose"
    set chown(list.clear) "file dir"
}

# chown::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::ok { } {
    global tkWorld
    variable chown

    # Build the command line for the recursive argument.
    set cmd_arg ""
    if [string length $chown(recursive)] {
	set cmd_arg "-$chown(recursive) "
    }

    set cmd_arg2 ""
    foreach x "changes silent verbose" {
	if [string length $chown::chown($x)] {
	    append cmd_arg2 $chown::chown($x)
	}
    }
    if [string length $cmd_arg2] {
	set cmd_arg2 "-$cmd_arg2"
    }

    if [string length $cmd_arg] {
	set cmd_arg "$cmd_arg $cmd_arg2"
    } else {
	set cmd_arg $cmd_arg2
    }
    unset cmd_arg2

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    if [string length $cmd_arg] {
	$tkWorld(cmd_center) insert insert "chown $cmd_arg \
		$chown(user)::$chown(group) "
    } else {
	$tkWorld(cmd_center) insert insert \
		"chwon $chown(user)::$chown(group) "
    }
    if [string length $chown(dir)] {
	$tkWorld(cmd_center) insert insert "$chown(dir) "
    }
    if [string length $chown(file)] {
	$tkWorld(cmd_center) insert insert "$chown(file) "
    }


    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# chown::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::reset { } {
    variable chown

    # Reset the drop down listboxes also.
    foreach x $chown(list.reset) {
	set chown($x) ""
    }
}

# chown::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::clear { } {
    variable chown

    # Reset the data structure elements and bg/fg.
    foreach x $chown(list.clear) {
	set chown($x) ""
	$chown(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $chown(entry.file)
}

# chown::help --
#
#   Method to invoke the Chown Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::help { } {
    global tkWorld

    help::create "help/chown.html" "Chown Command Help"
}

# chown::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::close { } {
    variable chown
    
    balloonhelp::cancel
    destroy $chown(dialog)
}

# chown::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc chown::open { } {
    variable chown

    # Insert the file list and move the cursor.
    $chown(entry.file) insert insert [file::select]
    $chown(entry.file) icursor end

    # Set the focus on the entry with the file list in it.
    focus $chown(entry.file)
}
