namespace eval wdir {
    global tkWorld
    variable wdir

    # Define the images used by the wdir::contents::dir and
    # wdir::contents::ls procedures.
    foreach f [glob [file join $tkWorld(image_dir) working *.gif]] {
	set iname [file rootname [file tail $f]]
	set wdir(image.$iname) [image create photo -file $f -format gif]
    }

    # Define the initial array elements that are not images.
    array set wdir {
	button.up ""
	dialog ""
	entry.wd ""
	listbox.cd ""
    }

    # define the button properties to use depending on the OS.
    switch $tkWorld(ostype) {
	unix {
	    set wdir(borderwidth) 1
	    set wdir(height) 1
	    set wdir(width) 5
	}
	default {
	    set wdir(borderwidth) 2
	    set wdir(height) 1
	    set wdir(width) 10
	}
    }
}

# resolve_link --
#
#   Method to resolve the final directory from a single link
#   or set of links to a file or directory.
#
# Args
#
#   p_link - Starting link or directory.  Directories do
#            not get resolved.
#
# Returns
#
#   Final destination directory that a link points to.  If a
#   directory is passed to the procedure, then the directory
#   name is returned since there is nothing to resolve.

proc wdir::resolve_link { p_link } {
    switch [file type $p_link] {
	link {
	    set p_link [file join [file dirname $p_link] \
		    [file readlink $p_link]]
	    wdir::resolve_link $p_link
	}
	file {
	    return [file dirname $p_link]
	}
	directory {
	    return $p_link
	}
    }
}

# wdir::create --
#
#   Method to create the working directory selector dialog box.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to select a directory, then reopen
#   it to change the working directory.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::create { } {
    global tkWorld
    variable wdir

    # If the dialog exists, bring it to the front and exit.
    # Otherwise create the dialog.
    if [winfo exists $wdir(dialog)] {
	switch -- [wm state $wdir(dialog)] {
	    normal {
		raise $wdir(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $wdir(dialog)
	    }
	}
	focus $wdir(dialog)
	return
    } else {
	set wdir(dialog) [toplevel .wdir \
		-borderwidth 1 \
		-relief flat]
    }

    # Create the entry to display the working directory that
    # also lets people type in the starting directory.
    set f1 [frame $wdir(dialog).f1]
    set wdir(entry.wd) [entry $f1.entry_wd \
	    -textvariable tkWorld(working_dir)]    
    pack $wdir(entry.wd) \
	    -fill x \
	    -padx 2 \
	    -pady 0

    # Now create the listbox that we use to display the directories.
    set wdir(listbox.cd) [scroll_listbox::interior \
	    [scroll_listbox::create $wdir(dialog).lb \
	    [expr [$wdir(entry.wd) cget -width] + 12] 20]]

    # Now build the buttons accross the bottom.
    set f2 [frame $wdir(dialog).f2]
    set wdir(button.home) [button $f2.button_home \
	    -text Home \
	    -width $wdir(borderwidth) \
	    -width $wdir(width) \
	    -height $wdir(height) \
	    -command {wdir::change_dir home}]
    foreach t "Refresh Up Help Close" {
	set b [string tolower $t]
	switch $b {
	    "refresh" {
		set c listbox_insert
	    }
	    default {
		set c $b
	    }
	}
	set wdir(button.$b) [button $f2.button_$b \
		-text $t \
		-width $wdir(borderwidth) \
		-width $wdir(width) \
		-height $wdir(height) \
		-command wdir::$c]
    }
    grid $wdir(button.home) $wdir(button.refresh) $wdir(button.up) \
	    $wdir(button.help) $wdir(button.close) \
	    -sticky n \
	    -padx 2 \
	    -pady 0

    # Build it, and they will come.
    grid $f1 \
	    -sticky ew \
	    -padx 0 \
	    -pady 2

    grid [seperator::create $wdir(dialog).s1] \
	    -sticky ew \
	    -padx 0 \
	    -pady 0

    grid $wdir(dialog).lb \
	    -sticky news \
	    -padx 2 \
	    -pady 2

    grid [seperator::create $wdir(dialog).s2] \
	    -sticky ew \
	    -padx 0 \
	    -pady 0

    grid $f2 \
	    -sticky ew \
	    -padx 5 \
	    -pady 5

    # Configure the rows so that when the user stretches the
    # dialog, everything looks good.
    grid rowconfigure $wdir(dialog) 0 \
	    -weight 0
    grid rowconfigure $wdir(dialog) 1 \
	    -weight 0
    grid rowconfigure $wdir(dialog) 2 \
	    -weight 1
    grid rowconfigure $wdir(dialog) 3 \
	    -weight 0

    # Define the balloon help for the dialog.
    balloonhelp::for $wdir(entry.wd) \
	    "The current working directory"
    balloonhelp::for $wdir(button.refresh) \
	    "Refresh the working directory listing"
    balloonhelp::for $wdir(button.home) \
	    "Navigate to your home directory"
    balloonhelp::for $wdir(button.up) \
	    "Chage to the parent directory"
    balloonhelp::for $wdir(listbox.cd) \
	    "Double click on a selection to change the\
	    working directory"
    balloonhelp::for $wdir(button.close) \
	    "Close the working directory dialog"

    # Create the binding for the listbox to create a new list
    # when the user clicks on a directory.
    bind $wdir(listbox.cd) <Double-1> wdir::change_dir

    # Setup the binding to let the user change the working
    # directory by typing it in.
    bind $wdir(entry.wd) <Return> wdir::listbox_insert

    # Create the initial directory listing
    wdir::listbox_insert

    # Define the Unix window manager options.
    switch $tkWorld(ostype) {
	windows -
	macintosh {
	    # Do nothing
	}
	default {
	    wm title $wdir(dialog) "Change Working Directory"
	    wm iconbitmap $wdir(dialog) \
		    @[file join $tkWorld(image_dir) wdir.xbm]
	    wm resizable $wdir(dialog) 0 1
	}
    }
}

# wdir::change_dir --
#
#   Method to change the working directory to the
#   current selection.
#
# Args
#
#   p_dir - Target directory to go to if not selected
#           from the listbox.
#
# Returns
#
#   None.

proc wdir::change_dir { {p_dir PERFORM_SELECTION} } {
    global tkWorld
    variable wdir

    switch -glob $p_dir {
	"home" {
	    set tkWorld(working_dir) $tkWorld(user_home)
	    wdir::listbox_insert
	}
	"PERFORM_SELECTION" {
	    # First get the currect selection index.
	    set idx [$wdir(listbox.cd) curselection]

	    # Get the directory from the selection index.
	    set tkWorld(working_dir) [file join \
		    $tkWorld(working_dir) [$wdir(listbox.cd) get $idx]]
	    wdir::listbox_insert
	}
	".." {
	    wdir::up
	}
	"../*" {
	    wdir::up
	    regsub "../" $p_dir {} p_dir
	    wdir::change_dir $p_dir
	}
	default {
	    regsub "$tkWorld(working_dir)/" $p_dir {} p_dir
	    set tkWorld(working_dir) [file join $tkWorld(working_dir) $p_dir]
	}
    }
}

# wdir::listbox_insert --
#
#   Method to the list the directory object in the current
#   working directory.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::listbox_insert { } {
    global tkWorld
    variable wdir

    # First delete the elements in the listbox
    if {[string compare {} $wdir(listbox.cd)]} {
	$wdir(listbox.cd) delete 0 end
    } else {
	return
    }

    # List all of the ojects in the current working directory.
    foreach x [lsort [glob -nocomplain \
	    [file join $tkWorld(working_dir) *] \
	    [file join $tkWorld(working_dir) .*]]] {

	# If the file is a directory, then process.
	if [file isdirectory $x] {

	    # Skip the typical unix . and .. listings.
	    switch [file tail $x] {
		.  -
		.. {
		    continue
		}
	    }

	    # Insert the directory into the listbox.
	    $wdir(listbox.cd) insert end [file tail $x]
	}
    }
}

# wdir::close --
#
#   Method to close the working directory selector dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::close { } {
    variable wdir

    balloonhelp::cancel
    destroy $wdir(dialog)
}

# wdir::help --
#
#   Method to invoke the Working Directory Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::help { } {
    global tkWorld

    help::create "help/wdir.html" "Working Directory Help"
}

# wdir::up --
#
#   Method to change to the parent directory of the
#   current working directory.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::up { } {
    global tkWorld
    variable wdir

    set tkWorld(working_dir) [file dirname $tkWorld(working_dir)]
    wdir::listbox_insert
}

#---------------------------------------------------------------------
namespace eval wdir::tag {
    # No private variables
}

# wdir::tag::bindings  --
#
#   Method to create the tag bindings for the files in the LW.
#
# Args
#
#   p_tag - Tag to define the bindings for.
#
# Returns
#
#   None.

proc wdir::tag::bindings { p_tag } {
    global tkWorld
    
    set cmd "wdir::tag::enter $p_tag"
    $tkWorld(log_window) tag bind $p_tag <Enter> $cmd
    set cmd "wdir::tag::leave $p_tag"
    $tkWorld(log_window) tag bind $p_tag <Leave> $cmd
    set cmd "wdir::popup::create $p_tag %X %Y"
    $tkWorld(log_window) tag bind $p_tag <Button-3> $cmd
}

# wdir::tag::enter --
#
#   Method to modify the properties of the supplied tag.
#
# Args
#
#   t - Tag to change properties of.
#
# Returns
#
#   None.

proc wdir::tag::enter { t } {
    global tkWorld

    # Just change the color scheme.
    $tkWorld(log_window) tag configure $t \
	    -foreground #000000 \
	    -background #d9d9d9
	    
    $tkWorld(log_window) configure \
	    -cursor hand2
}

# wdir::tag::leave --
#
#   Method to modify the properties of the supplied tag.
#
# Args
#
#   t - Tag to change properties of.
#
# Returns
#
#   None.

proc wdir::tag::leave { t } {
    global tkWorld

    # Revert back to the original color scheme.
    $tkWorld(log_window) tag configure $t \
	    -background #ffffff \
	    -foreground #000000

    $tkWorld(log_window) configure \
	    -cursor xterm
}

#---------------------------------------------------------------------
namespace eval wdir::popup {
    # No private variables
}

proc wdir::popup::create { t x y } {
    global tkWorld
    variable wdir

    # Kill the old command menu if it already exists.
    if [winfo exists .wdir_popup] {
	destroy .wdir_popup
    }

    # If the file is a directory, do a directory listing based
    # on the type of listing the user performed.
    regsub "tag_(dir|ls)_" $t "" t

    # Buile the menu which will appear when the user right clicks
    # on an active element.
    set m [menu .wdir_popup]

    # Get the file type to build the rest of the menu
    if [catch {file type [file join $tkWorld(working_dir) $t]} ftype] {
	# This will occur when the current working directory changes.
	return
    }

    # Build the default menu items which appear regardless of
    # the file type.

    $m add command \
	    -label "CC Insert" \
	    -command [list wdir::popup::cc_insert $t]
    $m add separator
    $m add command \
	    -label "Move" \
	    -command [list wdir::popup::rename $t]

    switch $ftype {
	socket {
	    set img $wdir::wdir(image.socket)
	}
	link {
	    set img $wdir::wdir(image.link)
	}
	directory {
	    set img $wdir::wdir(image.folder)
	    $m add command \
		    -label "Copy" \
		    -command [list wdir::popup::copy dir $t]
	    $m add command \
		    -label "Remove" \
		    -command [list wdir::popup::remove dir $t]
	    $m add separator
	    $m add command \
		    -label "dir" \
		    -command [list wdir::popup::filelist dir $t]
	    $m add command \
		    -label "ls" \
		    -command [list wdir::popup::filelist ls $t]
	}
	file {
	    set img $wdir::wdir(image.unknown)
	    $m add command \
		    -label "Copy" \
		    -command [list wdir::popup::copy file $t]
	    $m add command \
		    -label "Remove" \
		    -command [list wdir::popup::remove file $t]

	    # Add the seperator to the menu.
	    $m add separator

	    # Determine the file extension
	    set ext [file extension $t]

	    $m add cascade \
		    -label "Open With" \
		    -menu $m.openwith
	    set m2 [menu $m.openwith -tearoff 0]

	    # Now determine the executable to use if the
	    # system has it.
	    switch -glob [string tolower $ext] {
		.1 -
		.2 -
		.3 -
		.4 -
		.5 -
		.6 -
		.7 -
		.8 {
		    set img $wdir::wdir(image.man)
		    wdir::popup::filedefault $m $t
		}
		.f {
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.f_src)
		}
		.h {
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.h_src)
		}
		.cc -
		.c++ -
		.c {
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.c_src)
		}
		
		.ps {
		    foreach cmd [system::which "gv ghostview"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.postscript)
		}
		.pdf {
		    foreach cmd [system::which "gv ghostview xpdf"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    set img $wdir::wdir(image.pdf)
		}
		.fig {
		    foreach cmd [system::which "xfig"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    
		}
		.cbb {
		    foreach cmd [system::which "cbb"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		}
		.as -
		.ag -
		.aw {
		    foreach cmd [system::which "applix"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		}	    
		.gif -
		.png -
		.xbm -
		.ppm -
		.tiff -
		.xpm -
		.xwd -
		.bmp -
		.jpg {
		    foreach cmd [system::which "xv xpaint kpaint"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    set img $wdir::wdir(image.image)
		}
		.dvi {
		    foreach cmd [system::which "xdvi gv ghostview"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.dvi)
		}
		.tex {
		    foreach cmd [system::which "tex latex"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.tex)
		}
		.prn -
		.txt {
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.txt)
		}
		.htm* {
		    foreach cmd [system::which \
			    "netscape netscape-communicator"] {
			wdir::popup::systemadd $m2 $cmd $t
		    }
		    wdir::popup::filedefault $m $t
		    set img $wdir::wdir(image.html)
		}
		.java {
		    set img $wdir::wdir(image.java_src)
		}
		.tar -
		.tgz {
		    set img $wdir::wdir(image.tgz)
		}
		default {
		    wdir::popup::filedefault $m $t
		}
	    }
	}
    }

    # Add the properties box in the next release.
    #$m add separator
    #$m add command \
	    -label Properties \
	    -command [list wdir::popup::propbox::create $t $img]

    # Create the popup at the mouse point on the screen.
    tk_popup .wdir_popup $x $y
}

# wdir::popup::cc_insert  --
#
#   Method to insert the selected file into the CC.
#
# Args
#
#   filename - Name of the file to insert into the CC.
#
# Returns
#
#   None.

proc wdir::popup::cc_insert { filename } {
    global tkWorld

    # Put the slash to indicate a directory.
    if {[file isdirectory [file join $tkWorld(working_dir) $filename]]} {
	append filename "/"
    }

    $tkWorld(cmd_center) insert end "$filename "
}

# wdir::popup::rename --
#
#   Method to insert the command to rename a file into the CC.
#
# Args
#
#   filename - name fo the file to issue the rename
#              command on in the CC.
#
# Returns
#
#   None.

proc wdir::popup::rename { filename } {
    global tkWorld

    $tkWorld(cmd_center) insert end "mv $filename "
}

# wdir::popup::copy --
#
#   Method to insert the command to copy a file into the CC.
#
# Args
#
#   filename - name fo the file to issue the copy
#              command on in the CC.
#
# Returns
#
#   None.

proc wdir::popup::copy { filetype filename } {
    global tkWorld

    # Determine the command based on the file type.
    switch $filetype {
	dir -
	directory {
	    set cmd "cp -r"
	}
	default {
	    set cmd cp
	}
    }
    
    $tkWorld(cmd_center) insert end \
	    "$cmd $filename copy_of_$filename"
}

# wdir::popup::remove --
#
#   Method to insert the command to remove a file into the CC.
#
# Args
#
#   filetype - The file type (i.e. directory).
#   filename - Name fo the file to issue the remove
#              command on in the CC.
#
# Returns
#
#   None.

proc wdir::popup::remove { filetype filename } {
    global tkWorld
    
    # Determine the command based on the file type.
    switch $filetype {
	dir -
	directory {
	    set cmd rmdir
	}
	default {
	    set cmd rm
	}
    }

    $tkWorld(cmd_center) insert end "$cmd $filename "
}

# wdir::popup::filelist  --
#
#   Method to list the contents of the selected directory.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::popup::filelist { type dir } {
    global tkWorld

    # Change the working directory.
    set tkWorld(working_dir) [file join $tkWorld(working_dir) $dir]

    # Now use the parent method to list the contents.
    wdir::contents::list $type
}

# wdir::popup::filedefault  --
#
#   Method to add the default menu items for text files.
#
# Args
#
#   m - Menu to add the command to.
#   f - File to execute the command on (ie glbpreview.ps)
#
# Returns
#
#   None.

proc wdir::popup::filedefault { m f } {

    # Add the edit using menu item.
    $m add cascade \
	    -label "Edit Using" \
	    -menu $m.editors
    set m2 [menu $m.editors -tearoff 0]
    $m2 add command \
	    -label Emacs \
	    -command [list wdir::popup::emacs $f]
    $m2 add command \
	    -label Vi \
	    -command [list wdir::popup::vi $f]
    foreach cmd [system::which "kedit xedit axe jed xjed joe jove nedit"] {
	wdir::popup::systemadd $m2 $cmd $f
    }
    # Add the preview menu cascade item.
    $m add cascade \
	    -label "Preview" \
	    -menu $m.preview
    set m3 [menu $m.preview -tearoff 0]
    $m3 add command \
	    -label More \
	    -command [list wdir::popup::preview more $f]
    $m3 add command \
	    -label Head \
	    -command [list wdir::popup::preview head $f]
    $m3 add command \
	    -label Tail \
	    -command [list wdir::popup::preview tail $f]
}

# wdir::popup::systemadd  --
#
#   Method to add a system command to the popup menu
#
# Args
#
#   m   - Menu to add the commands to.
#   cmd - command to add to the list with path (ie /usr/local/bin/xv)
#   f   - file to execute the command on (ie glbspreview.ps)
#
# Returns
#
#   None.

proc wdir::popup::systemadd { m cmd f } {
    global tkWorld

    # Set the label for the command
    set l [file tail $cmd]

    # Set the file so that the command knows where to find it.
    set f [file join $tkWorld(working_dir) $f]

    # Add the command to the menu.
    $m add command \
	    -label $l \
	    -command [list exec $cmd $f &]
}

# wdir::popup::emacs --
#
#   Method to open the selected file in the Emacs editor.
#
# Args
#
#   filename
#
# Returns
#
#   None.

proc wdir::popup::emacs { filename } {
    global tkWorld
    
    # If we have the gui version of emacs, then use it.  Otherwise
    # fire off the correct xterm and run standard emacs-nox in it.
    if [catch {exec which emacs} cmd] {
	if [catch {exec which nxterm} xterm] {
	    set xterm xterm
	}
	set cmd "$xterm -e emacs"
    }
    
    # Run it!
    eval exec "$cmd [file join $tkWorld(working_dir) $filename]" &

    update
}

# wdir::popup::vi --
#
#   Method to open the selected file in the Vi editor.
#
# Args
#
#   filename
#
# Returns
#
#   None.

proc wdir::popup::vi { filename } {
    global tkWorld
    
    # If we have the gui version of vi, then use it.  Otherwise
    # fire off the correct xterm and run standard vi in it.
    if [catch {exec which gvim} cmd] {
	if [catch {exec which nxterm} xterm] {
	    set xterm xterm
	}
	set cmd "$xterm -e vi"
    }
    
    # Run the editor!
    eval exec "$cmd [file join $tkWorld(working_dir) $filename]" &
}

# wdir::popup::preview --
#
#   Method to more the contents of the selected file into the CC.
#
# Args
#
#   cmd      - The command the user wants to use (more, head, tail)
#   filename - Name of the file to issue the more command on.
#
# Returns
#
#   None.

proc wdir::popup::preview { cmd filename } {
    global tkWorld
    
    cmd_center::reset
    $tkWorld(cmd_center) insert end \
	    "$cmd [file join $tkWorld(working_dir) $filename] "
    cmd_center::run
}

#---------------------------------------------------------------------
namespace eval wdir::popup::propbox {
    variable propInfo

    array set propInfo {
	dialog ""
	filename.label1 ""
	filename.label2 ""
	button.close ""
    }
}

# wdir::popup::propbox::create  --
#
#   Method to create a properties dialog popup
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::popup::propbox::create { name img } {
    global tkWorld
    variable propInfo

    # If the dialog exists, bring it to the front and exit.
    # Otherwise create the dialog.
    if [winfo exists $propInfo(dialog)] {
	switch -- [wm state $propInfo(dialog)] {
	    normal {
		raise $propInfo(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $propInfo(dialog)
	    }
	}
	focus $propInfo(dialog)
	return
    } else {
	set propInfo(dialog) [toplevel .propInfo \
		-borderwidth 1 \
		-relief flat]
    }

    set f1 [frame $propInfo(dialog).f1 \
	    -relief groove \
	    -borderwidth 2]
    set propInfo(filename.label1) [label $f1.filename_label1 \
	    -image $img]
    set propInfo(filename.label2) [label $f1.filename_label2 \
	    -text $name \
	    -width 40 \
	    -anchor w]
    grid $propInfo(filename.label1) $propInfo(filename.label2) \
	    -padx 5\
	    -pady 5 \
	    -sticky ew

    # Now build the buttons accross the bottom.
    set f2 [frame $propInfo(dialog).f2]
    set propInfo(button.close) [button $f2.button_close \
	    -text Close \
	    -width 5 \
	    -command wdir::popup::propbox::close]
    grid $propInfo(button.close) \
	    -padx 2 \
	    -pady 0

    pack $f1 $f2 \
	    -side top \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5
}

proc wdir::popup::propbox::close { } {
    variable propInfo

    balloonhelp::cancel
    destroy $propInfo(dialog)
}    

#---------------------------------------------------------------------
namespace eval wdir::contents {
    variable contentInfo

    array set contentInfo {
	index 0
    }
}

# wdir::contents::list --
#
#   Display the file listing for the current working directory
#   in the LW.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::contents::list { {type ls} } {
    global tkWorld

    # Remove all of the current tags from the text in the LW.
    foreach t [$tkWorld(log_window) tag names] {
        $tkWorld(log_window) tag remove $t 1.0 end
    }

    wdir::contents::header

    # Now determine the command to run.
    switch $type {
	ls {
	    wdir::contents::ls
	}
	dir {
	    wdir::contents::dir
	}
    }

    # Configure the toolbar for the LW and move the cursor
    # to the end of the LW.
    toolbar::group_state log_window active
    $tkWorld(log_window) see end
}

# wdir::contents::header  --
#
#   Method to insert a label with the directory name in the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::contents::header { } {
    global tkWorld
    variable contentInfo

    # Create the label with the working directory as its text.
    set l [label $tkWorld(log_window).wdir_label[incr contentInfo(index)] \
	    -text $tkWorld(working_dir) \
	    -anchor w \
	    -foreground #ffffff \
	    -background #00009f \
	    -relief raised \
	    -width [$tkWorld(log_window) cget -width]]

    # Put the label in the LW.
    #$tkWorld(log_window) image create end \
	    -image $wdir::wdir(image.dir)
    $tkWorld(log_window) window create end \
	    -window $l \
	    -padx 0 \
	    -pady 10
}

# wdir::contents::dir --
#
#   Method to produce the directory listing with highlighting on
#   the filename so that the user can click on a file to put the
#   text in the LW for command processing.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::contents::dir { } {
    global tkWorld
    variable wdir

    $tkWorld(log_window) insert end "\n"

    # First get the list of all the file in the current
    # working directory.
    set x(list) [lsort [glob -nocomplain $tkWorld(working_dir)/.* \
	    $tkWorld(working_dir)/*]]

    if {![string length $x(list)]} {
	return
    }

    # Build the listing of all of the directories and find the
    # length of the longest directory name.
    set maxlen 0
    foreach f $x(list) {
	set len [expr [string length [file tail $f]] + 2]
	if {$len > $maxlen} {
	    set maxlen $len
	}
    }

    # Calculate how many columns to display based on a
    # 70 column display with room for images.
    set cols [expr 65/$maxlen]

    # Set the column counter.
    set colcnt 0
    foreach f $x(list) {
	# First get the file type and skip files that are
	# not directories.
	set img [wdir::parse::filetype $f]

	# We have a file so increment the column counter.
	incr colcnt
	
	set f "[file tail $f]"

	# Insert the listing in the LW.
	set t "tag_dir_$f"
	$tkWorld(log_window) image create end \
		-image $img \
		-padx 2 \
		-pady 1
	$tkWorld(log_window) insert end "$f" $t
	wdir::tag::bindings $t

	# Now insert the padding for the file.
	set l [expr [string length $f] + 1]
	set p [str::rpad "" [expr $maxlen - $l]]
	$tkWorld(log_window) insert end $p

	if {![expr $colcnt % $cols]} {
	    $tkWorld(log_window) insert end "\n"
	    set colcnt 0
	}
    }

    $tkWorld(log_window) insert end "\n"
}

# wdir::contents::ls --
#
#   Method to produce the ls -lF listing with highlighting on
#   the filename so that the user can click on a file to put the
#   text in the LW for command processing.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::contents::ls { } {
    global tkWorld
    variable wdir

    $tkWorld(log_window) insert end "\n"

    # First get the list of all the file in the current
    # working directory.
    set x(list) [lsort [glob -nocomplain $tkWorld(working_dir)/.* \
	    $tkWorld(working_dir)/*]]

    foreach f $x(list) {
	# First get the file type and define the image to display.
	# If it is a link, the resolve the link.
	#set x(ftype) [file type $f]
	if {[set x(ftype) [file type $f]] == "link"} {
	    if [catch {[wdir::resolve_link $f]} errmsg] {
		$tkWorld(log_window) image create end \
			-image $wdir::wdir(image.smallwarning) \
			-padx 2 \
			-pady 0
		$tkWorld(log_window) insert end \
			"Warning: $errmsg\n"
		continue
	    }
	}
	set img [wdir::parse::filetype $f]

	# Get the permissions for the file
	set x(perms) [wdir::parse::permissions \
		[file attributes $f -permissions]]

	# Set the owner and the user of the file.
	set x(owner) [file attributes $f -owner]
	set x(group) [file attributes $f -group]

	# Get the file size.
	set x(fsize) [file size $f]

	# Get the number of link and the file create time
	if {[catch {file stat $f y} errmsg]} {
	    $tkWorld(log_window) image create end \
		    -image $wdir::wdir(image.smallwarning) \
		    -padx 2 \
		    -pady 0
	    $tkWorld(log_window) insert end \
		    "Warning: $errmsg\n"
	    continue
	}
	set x(nlink) $y(nlink)
	if {$x(ftype) == "directory"} {
	    set x(ctime) [clock format $y(ctime) -format "%b %d  %Y"]
	} else {
	    set x(ctime) [clock format $y(ctime) -format "%b %d %R"]
	}

	# Strip the working directory name out of the filename.
	set f [file tail $f]

	# Insert the listing in the LW.
	$tkWorld(log_window) image create end \
		-image $img \
		-padx 2 \
		-pady 0
	$tkWorld(log_window) insert end "$x(perms)\
		[str::lpad $x(nlink) 3]\
		[str::rpad $x(owner) 8]\
		[str::rpad $x(group) 8]\
		[str::lpad $x(fsize) 8]\
		[str::lpad $x(ctime) 12] "

	set t "tag_ls_$f"
	$tkWorld(log_window) insert end "$f" $t
	wdir::tag::bindings $t

	# Start a new line for the next record.
	$tkWorld(log_window) insert end "\n"
    }

    $tkWorld(log_window) insert end "\n"
}

#---------------------------------------------------------------------
namespace eval wdir::parse {
    # No private variables
}

# wdir::parse::filetype  --
#
#   Method to decipher the prefix string or suffix string
#   based on the file type and return the image to use.
#
# Args
#
#   f - The file to determine the type to.
#
# Returns
#
#   The image identifier for the image to use.

proc wdir::parse::filetype { f } {
    variable wdir

    # Based on the file type, return the image
    switch [file type $f] {
	directory {
	    set img $wdir::wdir(image.smallfolder)
	}
	link {
	    set img $wdir::wdir(image.smalllink)
	}
	socket {
	    set img $wdir::wdir(image.smallsocket)
	}
	default {
	    set img $wdir::wdir(image.smallfile)
	}
    }
    if {[file type $f] == "file" && [file executable $f]} {
	set img $wdir::wdir(image.smallexec)
    }
    
    return $img
}

# wdir::parse::permissions --
#
#   Method to parse the output of the file attributes ... -perms
#   command.
#
# Args
#
#   p - The permission bits i.e. 04755
#
# Returns
#
#   The string interpretation of the last 3 permission bits i.e.
#   rwxr-xr-x
    
proc wdir::parse::permissions { p } {
    # Index of the end of the string
    set e [string length $p]

    # Index of the user bit which is the 3rd bit from the
    # right (00>7<55).
    set u [expr $e - 3]

    # The group bit is the second from the right (007>5<5)
    set g [expr $u + 1]

    # The other bit is the last character in the string
    set o [expr $g + 1]

    # Now build a string to parse each bit i.e from
    # "755" to "7 5 5" so that we can use the foreach command.
    set parse_str "[string range $p $u $u] [string range $p $g $g] \
	    [string range $p $o $o]"
    
    # Define the string to return to the calling function.
    set perm_str ""

    # Now define the string for each bit based on its
    # numeric value.
    foreach b $parse_str {
	switch $b {
	    0 {
		set s "---"
	    }
	    1 {
		set s "--x"
	    }
	    2 {
		set s "-w-"
	    }
	    3 {
		set s "-wx"
	    }
	    4 {
		set s "r--"
	    }
	    5 {
		set s "r-x"
	    }
	    6 {
		set s "rw-"
	    }
	    7 {
		set s "rwx"
	    }
	}

	# Now that we have the string for the bit, append it to
	# the string for all the permissions i.e. rwxr-xr-x.
	append perm_str $s
    }

    # Send back the full string i.e. rwxr-xr-x.
    return $perm_str
}

#---------------------------------------------------------------------
namespace eval wdir::expand {
    # No private variables
}

# wdir::expand::filename  --
#
#   Method to perform filename completion 
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc wdir::expand::filename { } {
    global tkWorld
    variable wdir

    set t $tkWorld(cmd_center)

    # Define the indices used to find the string to match.
    # Start from the cursor and work backward to find the text
    # to match.  Assume a string in the CC of the form:
    #
    # cmd xxx<tab>
    #
    set iend [$t index insert]
    set istart [$t search -backward -exact " " insert 0.0]

    # If the user has not entered a command, then disable filename
    # completion in the command center.
    if {![string compare {} $istart]} {
	stderr::log 0006
	return
    }

    # Get the string to match.
    set m [$t get "$istart +1c" $iend]
    
    # If a command is entered, but the user has not enter a partial
    # string, then disable filename completion.
    if {![string compare {} $m]} {
	bell
	return
    }
    
    # Define the list to search from.  If there is no match, then
    # let the user know and do not complete the filename.
    if {[catch {glob [file join $tkWorld(working_dir) $m*]} flist]} {
	bell
	$t delete $iend end
	return
    } else {
	# Retrieve the best match from the users partial string.
	set match [wdir::expand::bestmatch $flist]

	# Process if a match is found.
	if {[string compare {} $match]} {
	    # Remove the extra characters from the CC.
	    $t delete "$istart +1c" $iend

	    # Create the insert string with the working directory
	    # stripped off and then insert it.
	    regsub "$tkWorld(working_dir)/" $match {} istr
	    $t insert end $istr

	    # Append the unix directory indicator character or
	    # if necessary, popup a list of possible completions.
	    if {[file isdirectory $match] && \
		    [string range $match end end] != "/"} {
		$t insert end "/"
	    } elseif {[llength [set mlist [glob $match*]]] > 1} {
		# Strip out the working directory for each element
		# in the expansion list and in the match.
		regsub -all "$tkWorld(working_dir)/" $mlist {} mlist
		regsub "$tkWorld(working_dir)/" $match {} match

		# Now create the popup with the matches.
		wdir::expand::selection $mlist $match
	    }
	}
    }
}

# wdir::expand::bestmatch  --
#
#   This procedure taken from Jeff Hobbs tkcon.tcl package.
#
#   tkConExpandBestMatch - finds the best unique match in a list of
#   names.  The extra $e in this argument allows us to limit the
#   innermost loop a little further.  This improves speed as $l
#   becomes large or $e becomes long.
#
# Args
#
#   l - list to find best unique match in.
#   e - currently best known unique match.
#
# Returns
#
#   The longest unique match in the list.

proc wdir::expand::bestmatch {l {e {}}} {
    set ec [lindex $l 0]
    if {[llength $l]>1} {
	set e  [string length $e]; incr e -1
	set ei [string length $ec]; incr ei -1
	foreach l $l {
	    while {$ei>=$e && [string first $ec $l]} {
		set ec [string range $ec 0 [incr ei -1]]
	    }
	}
    }
    return $ec
}

# wdir::expand::selection  --
#
#   Method to create a popup dialog containing all of the possible
#   filename completions.
#
# Args
#
#   flist - The list of filename completions.
#   match - The partial match that the user has keyed in with
#           the tkWorld working directory embedded in it.
#
# Returns
#
#   None.

proc wdir::expand::selection { flist match } {
    global tkWorld

    # Build the filename completion dialog.
    if {[winfo exists .wdir_filename_selection]} {
	destroy .wdir_filename_selection
    }

    # Define the parent menu to place the completion list in.
    set m [menu .wdir_filename_selection\
	    -tearoff 0]

    # Add each file to the menu for a completion.
    foreach f $flist {
	# Set the string to insert into the CC based on the remaining
	# characters that make the completion: World completes tkWorld
	# if the user typed in tk.
	regsub $match $f {} istr

	# Insert the typical unix directory indicator to the insert
	# string.
	if {[file isdirectory [file join $tkWorld(working_dir) $f]]} {
	    append istr "/"
	}
	
	# Add the menu item which inserts the insert string into the CC.
	$m add command \
		-label $f \
		-command "$tkWorld(cmd_center) insert insert $istr"
    }

    # Now define where to put the popup based on the CC position.
    set x [winfo rootx $tkWorld(cmd_center)]
    set y [expr [winfo rooty $tkWorld(cmd_center)] + 2]

    # For the user, push the popup to the right based on the current
    # location of the insert cursor.
    incr x [lindex [$tkWorld(cmd_center) bbox insert] 0]
    incr y [lindex [$tkWorld(cmd_center) bbox insert] 1]

    # Create the menu to select completions from.
    tk_popup .wdir_filename_selection $x $y
}