# balloonhelp --
#
#   This namespace is used to implement balloon help on any widget in an
#   application.  As soon as the mouse enter the widget, the supplied help
#   message appears just below the mouse position.  When the mouse leaves
#   the widget, the help message is removed from the screen.

namespace eval balloonhelp {
    variable balloonhelp
    
    # Define the balloonhelp array structure.
    array set balloonhelp {
	active 1
	pending 0
	font {helvetica 12 bold}
    }

    # Define the cross-platform specific font.
    if {$tkWorld(ostype) == "windows"} {
	set balloonhelp(font) {helvetica 10 bold}
    }

    # Create the widget to display the message.
    toplevel .balloonhelp \
	    -class balloonhelp \
	    -background black \
	    -borderwidth 1 \
	    -relief flat

    label .balloonhelp.info \
	    -bg yellow \
	    -fg black \
	    -wraplength 3i \
	    -justify left \
	    -font $balloonhelp(font)

    pack .balloonhelp.info \
	    -side left \
	    -fill y

    wm overrideredirect .balloonhelp 1
    wm withdraw .balloonhelp

    # Create a binding for the balloon help message that does not
    # conflict with the name of the widget.
    bind balloon_help <Enter> {balloonhelp::pending %W}
    bind balloon_help <Leave> {balloonhelp::cancel}
}

# balloonhelp::for --
#
#   Method used by the application to create the help message for the
#   widget.
#
# Args
#
#   w   - Widget to put the balloon help on
#   msg - The message to display in the balloon help.
#
# Returns
#
#   None

proc balloonhelp::for { w msg } {
    variable balloonhelp

    set balloonhelp($w) $msg

    # Add the balloon help binding to the front of the bindtags list
    # so that it is the first tag to be evaluated (pending another
    # element is not added as the first tag later in the sourcing
    # application).
    bindtags $w "balloon_help [bindtags $w]"
}

# balloonhelp::control --
#
#   Method to verify the state of balloon help and cancel if turned off.
#
# Args
#
#   state - 1 for on
#           0 for off
#
# Returns
#
#   None

proc balloonhelp::control { state } {
    variable balloonhelp

    if {$state} {
        set balloonhelp(active) 1
    } else {
        balloonhelp::cancel
        set balloonhelp(active) 0
    }
}

# balloonhelp::pending --
#
#   Method for pending display of a help message
#
# Args
#
#   w - Widget for the help message. 
#
# Returns
#
#   None

proc balloonhelp::pending { w } {
    variable balloonhelp

    balloonhelp::cancel
    set balloonhelp(pending) [after 1500 [list balloonhelp::show $w]]
}

# balloonhelp::cancel --
#
#   Method for removing the help message from the screen after the
#   user has moved the mouse away from the specified widget.
#
# Args
#
#   None
#
# Returns
#
#   None

proc balloonhelp::cancel { } {
    variable balloonhelp

    # If in a pending state, first cancel that so that it is refreshed,
    # then remove the help message.
    if {[info exists balloonhelp(pending)]} {
        after cancel $balloonhelp(pending)
        unset balloonhelp(pending)
    }

    wm withdraw .balloonhelp
}

# balloonhelp::show --
#
#   Method for displaying the specified message in the balloon help
#   label widget.
#
# Args
#
#   w - Widget for the help message.
#
# Returns
#
#   None

proc balloonhelp::show { w } {
    variable balloonhelp

    # If balloon help is active, place the message just below and right
    # of the current mouse position so that it does not invade the real
    # estate of the widget the message is for.
    if {$balloonhelp(active)} {
        .balloonhelp.info configure -text $balloonhelp($w)

        set x [expr [winfo rootx $w] + 10]
        set y [expr [winfo rooty $w] + [winfo height $w] + 2]

        wm geometry .balloonhelp +$x+$y
        wm deiconify .balloonhelp

        raise .balloonhelp
    }

    # We are no longer pending since the message has been displayed.
    unset balloonhelp(pending)
}

