namespace eval comm {
    variable comm

    # Define the comm array structure so that all variables are
    # defined for the callbacks in the radiobuttons and checkbuttons.
    array set comm {
	list.reset ""
	list.clear ""
	entry.file ""
	dialog ""
    }
}

# comm::create --
#
#   Method to create the dialog box for the comm command.
#
# Note
#
#   This dialog will not grab focus so the user can keep it open
#   and run other tkWorld dialogs.  Imagine how tedious it would be
#   if you had to close the dialog to run your command, then reopen
#   it to modify it.  By not making this a modal dialog, we do not
#   have to implement any last command saving characteristics since
#   the user can just leave the dialog open.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::create { } {
    global tkWorld
    variable comm

    # Put the focus on the comm dialog if it is already open.
    if [winfo exists $comm(dialog)] {
	switch -- [wm state $comm(dialog)] {
	    normal {
		raise $comm(dialog)
	    }
	    withdrawn -
	    iconic {
		wm deiconify $comm(dialog)
	    }
	}
	focus $comm(dialog)
	return
    } else {
	set comm(dialog) [dialog::create .comm Comm]
    }

    # There is only 1 tab.
    set tab1 [tabnotebook::page [dialog::interior \
	    $comm(dialog)] "Options"]

    # Use a frame to encapsulate the file selection so that the
    # frame can be centered accross the grid columns.
    set f1 [frame $tab1.f1 \
	    -class TabnotebookFrame]

    button $f1.file_1_button \
	    -text "File 1" \
	    -width 5 \
	    -command {comm::open file_1}
    set comm(entry.file_1) [entry $f1.file_1_entry \
	    -width 35 \
	    -textvariable comm::comm(file_1)]
    button $f1.file_2_button \
	    -text "File 2" \
	    -width 5 \
	    -command {comm::open file_2}
    set comm(entry.file_2) [entry $f1.file_2_entry \
	    -width 35 \
	    -textvariable comm::comm(file_2)]
    grid $f1.file_1_button $f1.file_1_entry \
	    -sticky w \
	    -padx 2 \
	    -pady 2
    grid $f1.file_2_button $f1.file_2_entry \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    set f2 [frame $tab1.f2 \
	    -class TabnotebookFrame]
    label $f2.label_cols \
	    -text "Exclude Columns" \
	    -width 14 \
	    -anchor e
    set comm(entry.exclude_columns) [entry $f2.entry_cols \
	    -width 20 \
	    -textvariable comm::comm(exclude_columns)]
    grid $f2.label_cols $f2.entry_cols \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    set f3 [frame $tab1.f3 \
	    -class TabnotebookFrame]
    message $f3.msg_text \
	    -text "List the numbers of the columns that\
	    should be excluded from the comparison in the\
	    Exclude Columns entry above" \
	    -width 350
    grid $f3.msg_text \
	    -columnspan 2 \
	    -sticky w \
	    -padx 2 \
	    -pady 2

    # Build the only tab.
    pack $f1 $f2 $f3 \
	    -side top \
	    -fill x \
	    -padx 5 \
	    -pady 5 \
	    -ipadx 5 \
	    -ipady 5

    # Duhhhhhh......
    focus $comm(entry.file_1)

    # Define the lists for the reset and clear methods
    set comm(list.reset) ""
    set comm(list.clear) "file_1 file_2 exclude_columns"
}

# comm::ok --
#
#   Method to insert the command the user has created into the CC.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::ok { } {
    global tkWorld
    variable comm

    # Insert the Tcl command list in the Command Center with the
    # proper formatting of a space between each argument on the
    # command line.  If there are no options given by the user,
    # then don't display it in the CC.

    $tkWorld(cmd_center) insert insert "comm "
    if [string length $comm(file_1)] {
	$tkWorld(cmd_center) insert insert "$comm(file_1) "
    }
    if [string length $comm(file_2)] {
	$tkWorld(cmd_center) insert insert "$comm(file_2) "
    }
    if [string length $comm(exclude_columns)] {
	if [regsub -all "( |\t|\n)*" $comm(exclude_columns) \
		"" comm(exclude_columns)] {
	}
	$tkWorld(cmd_center) insert insert "-$comm(exclude_columns) "
    }

    # Activate the buttons in the toolbar for the command center.
    toolbar::group_state cmd_center active
    toolbar::button_state $toolbar::toolbar(stop) disabled
}

# comm::reset --
#
#   Method to reset the radio and checkbuttons in the dialog.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::reset { } {
    variable comm

    # Reset all of the list elements.
    foreach x $comm(list.reset) {
	set comm($x) ""
    }
}

# comm::clear --
#
#   Method to clear entry items of their text and reset the
#   background and foreground properties.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::clear { } {
    variable comm

    # Reset the data structure elements and bg/fg.
    foreach x $comm(list.clear) {
	set comm($x) ""
	$comm(entry.$x) configure -bg #ffffff -fg #000000
    }

    focus $comm(entry.file)
}

# comm::help --
#
#   Method to invoke the Comm Command Help.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::help { } {
    global tkWorld

    help::create "comm/index.html" "Comm Command Help"
}

# comm::close --
#
#   Close the dialog up.
#
# Args
#
#   None.
#
# Returns
#
#   None.

proc comm::close { } {
    variable comm
    
    balloonhelp::cancel
    destroy $comm(dialog)
}

# comm::open --
#
#   Method to add a file to the file entry and move the cursor
#   index of the entry to the end.
#
# Args
#
#   x - Descriptor for the entry to select the file for.
#
# Returns
#
#   None.

proc comm::open { x } {
    variable comm

    # Insert the file list and move the cursor.
    $comm(entry.$x) insert insert [file::select]
    $comm(entry.$x) icursor end

    # Set the focus on the entry with the file list in it.
    focus $comm(entry.$x)
}
