/*==================================================================
 * vbank.c - Virtual sound font bank file (*.bnk) routines
 *
 * Smurf Sound Font Editor
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * Borrowed code from awesfx utilities
 * in files loadbank.c, bool.c, dynload.c in awelib/
 * awesfx utilities are Copyright (C) 1996-1999 Takashi Iwai
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Smurf homepage: http://smurf.sourceforge.net
 *==================================================================*/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <glib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <unistd.h>

#include "vbank.h"
#include "i18n.h"
#include "sfont.h"
#include "smurfcfg.h"
#include "util.h"
#include "uif_sfont.h"		/* probably won't depend on this in future */

static void vbank_sort (VBnkData *vbnk);
static gint vbank_map_sort_func (gconstpointer a, gconstpointer b);
static gint vbank_compare_bank_prenum (gint bank1, gint pset1,
				       gint bank2, gint pset2);
static void vbank_GHFunc_find_sfont (gpointer key, gpointer value,
				     gpointer data);
static gint vbank_hash_sffiles (gchar *dirname, gboolean recurse);
static gint vbank_parse (FILE *fp, VBnkData *vbnk);
static gint vbank_parse_item(gchar *arg, VBnkItem *item, gchar **strp);
static void vbank_parse_preset(VBnkPtr *vptr, char *arg);
static int vbank_parse_arg(char *arg);
static char *strschr(char *str, char *dels);

/* hash tables to index sound font file names in virtual bank search path */

/* vbank_ipfile_hash is an index of sound font file names but with the '.' in
   the extension changed to a '\0' which makes 2 strings. So the hash key is
   effectively the sound font file name without the extension, but the extension
   can still be found by retrieving the original key */
static GHashTable *vbank_ipfile_hash = NULL;
static GHashTable *vbank_sfdir_hash = NULL; /* stores directory names */

VBnkData *
vbank_new (gint untitlecount)
{
  VBnkData *vbnk;

  vbnk = g_malloc0 (sizeof (VBnkData));

  vbnk->fname = g_strdup_printf (_("untitled%d.bnk"), untitlecount);
  vbnk->up2date = TRUE;
  vbnk->beensaved = FALSE;

  return (vbnk);
}

VBnkData *
vbank_load (gchar *fname)
{
  FILE *fp;
  VBnkData *vbnk;

  if (!(fp = fopen (fname, "r")))
    {
      logit (LogFubar, _("Failed to open virtual bank '%s'"), fname);
      return (NULL);
    }

  vbnk = g_malloc0 (sizeof (VBnkData));

  if (!vbank_parse (fp, vbnk))
    return (NULL);

  vbank_sort (vbnk);

  vbnk->fname = g_strdup (fname);
  vbnk->up2date = TRUE;
  vbnk->beensaved = FALSE;

  return (vbnk);
}

gint
vbank_save (VBnkData * vbnk, FILE *fp)
{
  VBnkItem *item;
  GSList *p;
  gchar *s, *s2, *name;
  gint i;

  if (fprintf (fp, _("## Virtual Sound Font Bank file written by:\n")) <= 0)
    goto agh;
  if (fprintf (fp, "## Smurf Sound Font Editor v" VERSION "\n\n") <= 0)
    goto agh;

  if (vbnk->defname && fprintf (fp, _("default %s\n\n"), vbnk->defname) <= 0)
    goto agh;

  p = vbnk->items;
  while (p)
    {
      item = (VBnkItem *)(p->data);

      /* write source preset/bank[/keynote] */
      if (fprintf (fp, "%d/%d", item->src.psetnum, item->src.bank) <= 0)
	goto agh;
      if (item->src.keynote >= 0 && fprintf (fp, "/%d", item->src.keynote) <= 0)
	goto agh;

      /* write destination :preset/bank[/keynote] */
      if (fprintf (fp, ":%d/%d", item->map.psetnum, item->map.bank) <= 0)
	goto agh;
      if (item->map.keynote >= 0 && fprintf (fp, "/%d", item->map.keynote) <= 0)
	goto agh;

      /* get name of source sound font (no path or extension) */
      s = item->sfname;

      if (s)			/* if sound font file name, then format it */
	{
	  i = strlen (s);

	  /* allocate space for largest possible '\' escaped sound font name */
	  name = g_malloc (i * 2 + 1);
	  s2 = name;

	  /* '\' escape all spaces and back slashes */
	  while (i--)
	    {
	      if (*s == ' ')
		{
		  *(s2++) = '\\';
		  *(s2++) = ' ';
		}
	      else if (*s == '\\')
		{
		  *(s2++) = '\\';
		  *(s2++) = '\\';
		}
	      else *(s2++) = *s;
	      s++;
	    }
	  *s2 = '\0';

	  i = fprintf (fp, ":%s", name);
	  g_free (name);
	  if (i <= 0) goto agh;
	}

      if (fputc ('\n', fp) == EOF)
	goto agh;

      p = g_slist_next (p);
    }

  return (OK);

 agh:
  return (logit (LogFubar | LogErrno, _("Failed to write to vbank")));
}

void
vbank_close (VBnkData *vbnk)
{
  GSList *p;

  if (vbnk->fname) g_free (vbnk->fname);
  if (vbnk->defname) g_free (vbnk->defname);

  p = vbnk->items;
  while (p)
    {
      vbank_item_free ((VBnkItem *)(p->data));
      p = g_slist_next (p);
    }
}

VBnkItem *
vbank_item_alloc (void)
{
  return ((VBnkItem *)g_malloc0 (sizeof (VBnkItem)));
}

void
vbank_item_free (VBnkItem *item)
{
  if (item->sfname) g_free (item->sfname);
  g_free (item);
}

VBnkItem *
vbank_add_map (gint mapbank, gint mappset, gint mapnote, gint srcbank,
	       gint srcpset, gint srcnote, gchar *srcsfname, VBnkData *vbnk)
{
  VBnkItem *item;

  g_return_val_if_fail (srcsfname != NULL, NULL);

  item = vbank_item_alloc ();

  item->map.bank = mapbank;
  item->map.psetnum = mappset;
  item->map.keynote = mapnote;

  item->src.bank = srcbank;
  item->src.psetnum = srcpset;
  item->src.keynote = srcnote;

  item->sfname = srcsfname;

  vbnk->items = g_slist_insert_sorted (vbnk->items, item, vbank_map_sort_func);

  return (item);
}

void
vbank_delete_map (VBnkItem *item, VBnkData *vbnk)
{
  vbnk->items = g_slist_remove (vbnk->items, item);
  vbank_item_free (item);
}

/* finds the first available preset map greater or equal to bank:preset and
   stores result in bank:preset */
void
vbank_find_free_map (gint *bank, gint *psetnum, VBnkData *vbnk)
{
  VBnkItem *item;
  GSList *p;
  gint bnk, pr;

  p = vbnk->items;
  bnk = *bank;
  pr = *psetnum;
  while (p)
    {
      item = (VBnkItem *)(p->data);
      p = g_slist_next (p);

      /* if item->bank:preset < start bank:preset, then skip */
      if (item->map.bank < *bank
	  || (item->map.bank == *bank && item->map.psetnum < *psetnum))
	continue;

      if (bnk < item->map.bank
	  || (bnk == item->map.bank && pr < item->map.psetnum))
	break;

      if (++pr > 127)
	{
	  pr = 0;
	  bnk++;
	}
    }
  *bank = bnk;
  *psetnum = pr;
}

static void
vbank_sort (VBnkData *vbnk)
{
  vbnk->items = g_slist_sort (vbnk->items, vbank_map_sort_func);
}

/* a GCompareFunc for the sorting of VBnkItems */
static gint
vbank_map_sort_func (gconstpointer a, gconstpointer b)
{
  VBnkItem *aitem, *bitem;

  aitem = (VBnkItem *)a;
  bitem = (VBnkItem *)b;

  return (((aitem->map.bank - bitem->map.bank) << 8)
    + (aitem->map.psetnum - bitem->map.psetnum));
}

/* check the extension */
gint
vbank_is_virtual_bank (gchar *path)
{
  char *p;
  if ((p = strrchr(path, '.')) == NULL)
    return FALSE;
  if (strcmp(p + 1, "bnk") == 0)
    return TRUE;
  return FALSE;
}

void
vbank_set_fname (VBnkData *vbnk, gchar *fname)
{
  if (vbnk->fname) g_free (vbnk->fname);
  vbnk->fname = g_strdup (fname);
}

void
vbank_set_defsf (VBnkData *vbnk, gchar *fname)
{
  g_return_if_fail (fname != NULL);

  if (vbnk->defname) g_free (vbnk->defname);

  vbnk->defname = fname;
}

/* get virtual bank items (returns: GList of VBnkPresetMap's)
   ** GList and list items should be freed ** */
GList *
vbank_get_preset_maps (VBnkData *vbnk)
{
  GList *psetlist = NULL;
  VBnkItem *item;
  VBnkPresetMap *psetmap;
  UISFont *uisf, *defuisf;
  IPPreset *pset;
  gchar *path;
  GSList *lvi, *lpr;
  gint rv;

  /* find default sound font if its opened already */
  path = NULL;
  if (vbnk->defname && (path = vbank_locate_instp_by_name (vbnk->defname))
      && (defuisf = uisf_find_instp_by_fname (path, NULL)))
    lpr = uisf->sf->preset;
  else lpr = NULL;

  if (path) g_free (path);

  lvi = vbnk->items;

  /* loop over virtual bank items and default sound font presets, virtual bank
     items override default presets */
  while (lvi || lpr)
    {
      if (lvi)			/* any virtual bank items left? */
	{
	  item = (VBnkItem *)(lvi->data);
	  rv = 1;
	}

      /* allocate preset map (an IPPreset and its bank:preset # to map to) */
      psetmap = g_malloc (sizeof (VBnkPresetMap));

      if (lpr)			/* any default presets left? */
	{
	  psetmap->preset = (IPPreset *)(lpr->data);
	  rv = -1;
	}

      /* check ordering of current vbank item and default preset */
      if (lvi && lpr)
	rv = vbank_compare_bank_prenum (psetmap->preset->bank,
					psetmap->preset->prenum,
					item->map.bank, item->map.psetnum);

      if (rv <= 0)		/* advance default preset, if needed */
	lpr = g_slist_next (lpr);

      if (rv >= 0)		/* vbank item comes next or overrides def? */
	{
	  lvi = g_slist_next (lvi);

	  /* ?: sound font loaded for this item and src preset is valid? */
	  if (!(pset = vbank_get_item_preset (item)))
	    {
	      g_free (psetmap);
	      continue;		/* ?: No, skip it */
	    }
	  psetmap->preset = pset;
	  psetmap->sf = uisf->sf;
	  psetmap->bank = item->map.bank;
	  psetmap->psetnum = item->map.psetnum;
	}
      else
	{
	  psetmap->sf = defuisf->sf;
	  psetmap->bank = psetmap->preset->bank;
	  psetmap->psetnum = psetmap->preset->prenum;
	}

      psetlist = g_list_append (psetlist, psetmap);
    }

  return (psetlist);
}

/* find a virtual bank item's preset in opened sfonts (NULL if not found) */
IPPreset *
vbank_get_item_preset (VBnkItem *item)
{
  gchar *path;
  UISFont *uisf;
  GSList *p;

  if (item->sfname
      && (path = vbank_locate_instp_by_name (item->sfname))
      && (uisf = uisf_find_instp_by_fname (path, NULL))
      && (p = instp_find_preset (uisf->sf, NULL, item->src.bank,
				 item->src.psetnum, NULL)))
    return ((IPPreset *)(p->data));

  return (NULL);
}

/* compares two bank:preset pairs */
static gint
vbank_compare_bank_prenum (gint bank1, gint pset1, gint bank2, gint pset2)
{
  if (bank1 < bank2) return (-1);
  if (bank1 > bank2) return (1);
  if (pset1 < pset2) return (-1);
  if (pset1 > pset2) return (1);
  return (0);
}

/* searches for unloaded sound fonts required by a virtual bank and returns
   a list of "found" file paths and "notfound" sound font names, both lists
   (and list items) should be freed */
void
vbank_find_unloaded_sfonts (VBnkData *vbnk, GList **found, GList **notfound)
{
  GHashTable *sfnames;
  GList *nlists[2] = { NULL, NULL};	/* 2 GLists for found and notfound */
  VBnkItem *item;
  GSList *p;

  sfnames = g_hash_table_new (g_str_hash, g_str_equal);

  if (vbnk->defname)
    g_hash_table_insert (sfnames, vbnk->defname, GINT_TO_POINTER (TRUE));

  /* loop over virtual bank items, and insert unloaded sound font names into
     hash (gets rid of duplicates) */
  p = vbnk->items;
  while (p)
    {
      item = (VBnkItem *)(p->data);

      if (item->sfname)
	g_hash_table_insert (sfnames, item->sfname, GINT_TO_POINTER (TRUE));

      p = g_slist_next (p);
    }

  g_hash_table_foreach (sfnames, vbank_GHFunc_find_sfont, nlists);

  if (!found)
    {
      g_list_foreach (nlists[0], (GFunc)g_free, NULL);
      g_list_free (nlists[0]);
    }
  else *found = nlists[0];

  if (!notfound)
    {
      g_list_foreach (nlists[1], (GFunc)g_free, NULL);
      g_list_free (nlists[1]);
    }
  else *notfound = nlists[1];
}

/* GHFunc used by vbank_find_unloaded_sfonts to fill found and not found lists
   with found sfont paths and not found sfont names */
static void
vbank_GHFunc_find_sfont (gpointer key, gpointer value, gpointer data)
{
  GList **nlists = (GList **)data;
  gchar *sfname = (gchar *)key;
  gchar *path;

  /* can we locate the sound font? (full file path) */
  if (!(path = vbank_locate_instp_by_name (sfname)))
    nlists[0] = g_list_append (nlists[0], path); /* ?: YES, add path to found */
  else nlists[1] = g_list_append (nlists[1], g_strdup (sfname)); /* ?: Nope */
}

/* looks up a sound font by name (without extension) in the sfont file hash
   and returns a full path to the found file, or NULL if not found.
   ** Return string should be freed ** */
gchar *
vbank_locate_instp_by_name (gchar *sfname)
{
  gchar *origkey, *dir;
  gchar *filepath;

  if (!vbank_ipfile_hash) vbank_update_ipfile_hash ();

  /* lookup the sfont name and get original key string */
  if (!g_hash_table_lookup_extended (vbank_ipfile_hash, sfname,
				     (gpointer *)(&origkey),(gpointer *)(&dir)))
    return (NULL);

  /* construct full path to file (original hash key contains 2 adjacent strings,
     first is the sfont name (without extension) second is the 3 letter
     extension) */
  filepath =
    g_strconcat (dir, origkey, ".", origkey + strlen (origkey) + 1, NULL);

  return (filepath);
}

/* turns a full path into just the base file name without ".sf2" extension
   ** returned string should be freed ** */
gchar *
vbank_base_fname (gchar *path)
{
  gchar *s;

  s = g_basename (path);
  if (strlen (s) > 4 && g_strcasecmp (s + strlen (s) - 4, ".sf2") == 0)
    return (g_strndup (s, strlen (s) - 4));
  else return (g_strdup (s));
}

/* indexes all sound font files in directories of vbank search path */
void
vbank_update_ipfile_hash (void)
{
  gchar *path;
  gchar *s, *s2;
  gboolean recurse;

  /* free old hashes if any */

  if (vbank_ipfile_hash)
    {
      /* free keys only (value strings are in vbank_sfdir_hash) */
      g_hash_table_foreach (vbank_ipfile_hash,
			    (GHFunc)g_free, NULL);
      g_hash_table_destroy (vbank_ipfile_hash);
    }

  if (vbank_sfdir_hash)
    {
      /* free keys only (value is just TRUE) */
      g_hash_table_foreach (vbank_sfdir_hash, (GHFunc)g_free, NULL);
      g_hash_table_destroy (vbank_sfdir_hash);
    }

  vbank_ipfile_hash = g_hash_table_new (g_str_hash, g_str_equal);
  vbank_sfdir_hash = g_hash_table_new (g_str_hash, g_str_equal);

  path = g_strdup (smurfcfg_get_val (SMURFCFG_VBNK_SEARCH_PATH)->v_string);

  /* no search path? */
  if (!path || !*path) return;

  s = path;
  while (s && *s)		/* loop over directory names in search path */
    {
      if ((s2 = strchr (s, ':')))
	{
	  *s2 = '\0';
	  s2++;
	}

      /* check for "slash*" (start of C comment :) at end of directory name,
	 signifying recursive search */
      if (strlen (s) >= 2
	  && strcmp (s + strlen (s) - 2, G_DIR_SEPARATOR_S "*") == 0)
	{
	  s[strlen (s) - 1] = '\0'; /* truncate string at the '*' */
	  recurse = TRUE;
	}
      else recurse = FALSE;	/* no recurse into sub directories */

      vbank_hash_sffiles (s, recurse);	/* hash sfont names */
      s = s2;
    }

  g_free (path);
}

/* adds all sound font file names in "dirname" to vbank_ipfile_hash,
 called by vbank_update_ipfile_hash and itself (recursively) */
static gint
vbank_hash_sffiles (gchar *dirname, gboolean recurse)
{
  gchar *dname;			/* dup of dirname but with / at end if needed */
  DIR *dh;
  struct dirent *ent;
  gchar *fname, *fullpath;
  struct stat statnfo;
  gboolean found = FALSE;	/* new sound font file found? */
  gchar *s;

  if (!dirname || !*dirname)
    return (logit (LogWarn, _("Empty directory name in virtual bank"
			     " search path")));

  /* if directory doesn't end with a slash, then add one */
  if (dirname [strlen (dirname) - 1] != G_DIR_SEPARATOR)
    dname = g_strconcat (dirname, G_DIR_SEPARATOR_S, NULL);
  else dname = g_strdup (dirname); /* duplicate to keep g_free consistent */

  /* directory already searched? */
  if (g_hash_table_lookup (vbank_sfdir_hash, dname))
    {
      g_free (dname);
      return (OK);
    }

  if (!(dh = opendir (dname)))	/* attempt to open the directory */
    return (logit (LogWarn | LogErrno, _("Failed to open directory '%s' in"
					 " virtual bank search path"), s));

  /* loop over contents of directory */
  while ((ent = readdir (dh)))
    {
      fname = ent->d_name;

      /* skip self and parent directory */
      if (strcmp (fname, ".") == 0 || strcmp (fname, "..") == 0)
	continue;

      if (recurse)		/* recurse into sub dirs? */
	{
	  fullpath = g_strconcat (dname, fname, NULL); /* full path to file */

	  if (stat (fullpath, &statnfo) == -1) /* stat the file */
	    {
	      logit (LogWarn | LogErrno, _("Failed to get stats on file '%s'"),
		    fullpath);
	      g_free (fullpath);
	      continue;
	    }

	  if (S_ISDIR (statnfo.st_mode)) /* if directory, then recurse it */
	    {
	      vbank_hash_sffiles (fullpath, TRUE); /* call us recursively */
	      g_free (fullpath);
	      continue;
	    }

	  g_free (fullpath);
	}

      /* is it a sound font file? (.sf2 extension) */
      if (strlen (fname) <= 4
	  || g_strcasecmp (fname + strlen (fname) - 4, ".sf2") != 0)
	continue;

      /* make file name string dynamically allocated so we can mess with it */
      fname = g_strdup (fname);

      /* make file name 2 strings, first is sound font name without extension,
	 second string (immediately following the first) is the extension,
         therefore file name (without extension) gets used as hash key */
      fname [strlen (fname) - 4] = '\0';

      /* check if file by this name already found */
      if ((s = g_hash_table_lookup (vbank_ipfile_hash, fname)))
	{
	  logit (LogWarn, _("Sound font file '%s' conflicts with '%s' in"
			   " virtual sound font search path"), fname, s);
	  g_free (fname);
	  continue;
	}

      /* insert into file hash table: sfont name as a key, file's directory
	 as the value */
      g_hash_table_insert (vbank_ipfile_hash, fname, dname);
      found = TRUE;
    }

  /* if any sfonts were found, insert directory name into directory hash
     as key and value (hash used to check for duplicates), keep dname */
  if (found)
    g_hash_table_insert (vbank_sfdir_hash, dname, GINT_TO_POINTER (TRUE));
  else g_free (dname);		/* no sfonts found, free dname */

  return (OK);
}

/* parse a virtual bank file into a virtual bank structure */
static gint
vbank_parse (FILE *fp, VBnkData *vbnk)
{
  char line[256], *p, *p2, *name;
  VBnkItem *item;
  int len;
  gint linenum = 0;

  while (fgets(line, sizeof(line) - 1, fp))
    {
      linenum++;

      /* discard the linefeed */
      len = strlen(line);
      if (len > 0 && line[len-1] == '\n') line[len-1] = 0; 

      /* skip spaces & comments */
      for (p = line; isspace(*p); p++)
	;
      if (!*p || *p == '#' || *p == '!' || *p == '%')
	continue;

      if (isalpha(*p))
	{
	  char *arg;

	  arg = strschr(p, " \t\r\n");
	  if (arg)
	    {
	      char *tmp = strschr(arg, " \t\r\n");
	      if (tmp) *tmp = 0;
	      arg++;
	    }

	  switch (*p)
	    {
	    case 'i':
	    case 'I':		/* include other bank file */
	      break;
	    case 'd':
	    case 'D':		/* set default font file */
	      if (vbnk->defname) g_free(vbnk->defname);
	      vbnk->defname = g_strdup(arg);
	      break;
	    default:
	      return (logit (LogFubar, _("Illegal command on line %d"),
					linenum));
	      break;
	    }
	  continue;
	}
      else if (*p == '*' || *p == '-' || isdigit(*p))
	{
	  item = vbank_item_alloc ();
	  if (! vbank_parse_item (p, item, &name))
	    {
	      vbank_item_free (item);
	      logit (LogWarn, _("Ignoring invalid item on line %d"), linenum);
	      continue;
	    }

	  if (item->src.bank == -1 || item->src.psetnum == -1)
	    {
	      vbank_item_free (item);
	      logit (LogWarn, "Virtual bank wildcard maps not supported yet");
	      continue;
	    }
	} else continue;

      if (name && *name)	/* if there is a sound font name string */
	{
	  /* convert "\ " to spaces and "\\" to back slashes, to support
	     file names with spaces (original AWE .bnk format doesn't) */
	  p = p2 = name;
	  while (*p2)
	    {
	      if (p2[0] == '\\' && p2[1] == ' ')
		{
		  *p = ' ';
		  p2++;
		}
	      else if (p2[0] == '\\' && p2[1] == '\\')
		{
		  *p = '\\';
		  p2++;
		}
	      else if (strchr (" \t\n", *p2))
		{
		  *p = '\0';
		  break;
		}
	      else *p = *p2;

	      p++;
	      p2++;
	    }
	  *p = '\0';		/* terminate sound font name */
	}

      if (name == NULL || !*name) /* without font name -- its a preset link */
	item->sfname = NULL;
      else item->sfname = g_strdup (name);

      vbnk->items = g_slist_append (vbnk->items, item);
    }

  return (OK);
}

/* parse source and mapping presets */
static gint
vbank_parse_item(gchar *arg, VBnkItem *item, gchar **strp)
{
  char *next;

  if ((next = strschr(arg, ":=")) != NULL)
    *next++ = 0;
  vbank_parse_preset(&item->src, arg);
  if (next == NULL) {
    item->map = item->src;
    if (strp) *strp = NULL;
    return TRUE;
  }
  arg = next;
  if ((next = strschr(arg, ":=")) != NULL)
    *next++ = 0;
  vbank_parse_preset(&item->map, arg);
  if (strp) *strp = next;
  return TRUE;
}

/* parse preset/bank/keynote */
static void
vbank_parse_preset(VBnkPtr *vptr, char *arg)
{
  char *next;

  if ((next = strschr(arg, "/: \t\n")) != NULL)
    *next++ = 0;
  vptr->psetnum = vbank_parse_arg(arg);
  vptr->bank = 0;
  vptr->keynote = -1;
  arg = next;
  if (arg) {
    if ((next = strschr(arg, "/: \t\n")) != NULL)
      *next++ = 0;
    vptr->bank = vbank_parse_arg(arg);
    if (next)
      vptr->keynote = vbank_parse_arg(next);
  }
}

/* ascii to digit; accept * and - characters */
static int
vbank_parse_arg(char *arg)
{
  if (isdigit(*arg))
    return atoi(arg);
  else
    return -1;
}

/* search delimiers */
static char *
strschr(char *str, char *dels)
{
  char *p;
  for (p = str; *p; p++) {
    if (strchr(dels, *p))
      return p;
  }
  return NULL;
}
