/***************************************************************************
    smb4kmounter.cpp  -  The core class, that mounts shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003 by Alexander Reinholdt
    email                : dustpuppy@mail.berlios.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// Qt includes
#include <qapplication.h>
#include <qdir.h>
#include <qfile.h>
#include <qtextstream.h>
#include <qregexp.h>
#include <qmap.h>
#include <qptrlist.h>

// KDE includes
#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>

// system includes
#include <sys/types.h>
#include <unistd.h>

#ifdef __FreeBSD__
#include <pwd.h>
#include <sys/param.h>
#include <sys/ucred.h>
#include <sys/mount.h>
#include <cerrno>
#include <qfileinfo.h>
#endif

// Application specific includes
#include "smb4kmounter.h"
#include "smb4kcore.h"


Smb4KMounter::Smb4KMounter( QObject *parent, const char *name ) : QObject( parent, name )
{
  m_config = kapp->config();

  m_proc = new KProcess( this, "MounterProcess" );
  m_proc->setUseShell( true );

  m_state = Idle;
  m_lastJob = false;

  // The timer for the queue.
  m_timer1 = new QTimer( this );
  m_timer1->start( 50, false );

  // The timer for the checks.
  m_timer2 = new QTimer( this );
  m_timer2->start( 1000, false );

  m_working = false;

  // Do some initial actions:
  // - Import all mounted SMB shares,
  // - Mount all shares that have to be mounted.
  m_queue.setAutoDelete( true );

  m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  m_queue.enqueue( new QString( QString( "%1:" ).arg( MountRecent ) ) );

  connect( m_proc,   SIGNAL( processExited( KProcess * ) ),               this, SLOT( slotProcessExited( KProcess * ) ) );
  connect( m_proc,   SIGNAL( receivedStdout( KProcess *, char *, int ) ), this, SLOT( slotReceivedStdout( KProcess *, char *, int ) ) );
  connect( m_proc,   SIGNAL( receivedStderr( KProcess *, char *, int ) ), this, SLOT( slotReceivedStderr( KProcess *, char *, int ) ) );
  connect( m_timer1, SIGNAL( timeout() ),                                 this, SLOT( init() ) );
  connect( m_timer2, SIGNAL( timeout() ),                                 this, SLOT( check() ) );
}


Smb4KMounter::~Smb4KMounter()
{
  m_config->setGroup( "Mount Options" );
  bool unmount_all = m_config->readBoolEntry( "Unmount All", false );
  
  if ( !unmount_all && m_state != UnmountAll )
  {
    abort();
  }
    
  m_mounted_shares.clear();
}


/***************************************************************************
   Aborts any action of the mounter.
***************************************************************************/


void Smb4KMounter::abort()
{
  m_config->setGroup( "Super User Privileges" );
  bool run_suid = m_config->readBoolEntry( "Run SUID", false );
  QString suid_program = m_config->readEntry( "SUID Program", QString::null );
      
  m_queue.clear();

  if ( run_suid && !suid_program.isEmpty() )
  {
    if ( m_proc->isRunning() )
    {
      KProcess p;
      p.setUseShell( true );
      p << QString( "%1 smb4k_kill -15 %2" ).arg( suid_program ).arg( m_proc->pid() );
      p.start( KProcess::DontCare, KProcess::NoCommunication );
    }
  }
  else
  {
    if ( m_proc->isRunning() )
    {
      m_proc->kill();
    }
  }
}


/***************************************************************************
   Mounts recently used shares.
***************************************************************************/


void Smb4KMounter::mountRecent()
{
  m_config->setGroup( "Mount Options" );
  
  if ( m_config->readBoolEntry( "Mount Recent", false ) )
  {
    if ( m_config->hasGroup( "Recently Mounted Shares" ) )
    {
      m_config->setGroup( "Recently Mounted Shares" );
      
      QString temp;
      
      for ( int index = 0; ; index++ )
      {
        temp = m_config->readEntry( QString( "%1" ).arg( index ), QString::null );
        
        if ( !temp.isEmpty() )
        {
          Smb4KShare *share = findShareByName( temp );
          
          if ( !share || share->getUID() != (int)getuid() && share->getGID() != (int)getgid() )
          {
#ifndef __FreeBSD__
            mountShare( QString::null, temp.section( "/", 2, 2 ), QString::null, temp.section( "/", 3, 3 ) );
#else
            mountShare( QString::null, temp.section( "/", 2, 2 ).section( "@", 1, 1 ), QString::null, temp.section( "/", 3, 3 ) );
#endif            
          }
          else
            continue;
        }
        else
          break;
      }
      
      emit running( MOUNTER_STOP, false );
    }
    else
    {
      emit running( MOUNTER_STOP, false );
    }
  }
  else
  {
    emit running( MOUNTER_STOP, false );
  }
  
  m_working = false;
}


/***************************************************************************
   Imports all shares, that are mounted externally.
***************************************************************************/

void Smb4KMounter::import()
{
  m_config->setGroup( "Mount Options" );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  
#ifndef __FreeBSD__

  QValueList<Smb4KShare *> shares;
    
  // /proc/mounts is read to import the externally mounted shares.
  QFile file( "mounts" );
  QDir *dir = new QDir();

  if ( QFile::exists( "/proc/mounts" ) )
  {
    dir->setPath( "/proc" );
    QDir::setCurrent( dir->path() );
  }
  else
  {
    if ( !m_proc_error )
    {
      m_proc_error = true;
      emit error( ERROR_FILE_NOT_FOUND, QString( "%1/%2" ).arg( dir->path() ).arg( file.name() ) );
    }
  }

  if ( !dir->path().isEmpty() )
  {
    if ( file.open( IO_ReadOnly ) )
    {
      QTextStream ts( &file );
      
      while( !ts.atEnd() )
      {
        QString line = ts.readLine().stripWhiteSpace();
        
        if ( line.contains( "smbfs", false ) != 0 )
        {
          QString share_and_path = line.section( "smbfs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();
          
          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }
          
          int uid = line.section( "uid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();
          int gid = line.section( "gid=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace().toInt();  
          
          // Test if share is broken
          bool broken = false;
            
          QDir *d = new QDir( path );
          
          if ( !d || !d->exists() || !d->isReadable() )
          {
            broken = true;
          }
          
          delete d;
                              
          shares.append( new Smb4KShare( name, path, "smbfs", uid, gid, broken ) );
        }
        else if ( line.contains( "cifs", false ) != 0 )
        {
          QString share_and_path = line.section( "cifs", 0, 0 ).stripWhiteSpace();
          QString name = share_and_path.section( " ", 0, 0 ).stripWhiteSpace().replace( "\\040", "\040" );
          QString path = share_and_path.section( " ", 1, 1 ).stripWhiteSpace();
          
          if ( path.contains( "\\040" ) != 0 || path.contains( "\040" ) != 0 )
          {
            name.replace( "_", "\040" );
            path.replace( "\\040", "\040" );
          }
          
          QString login = line.section( "username=", 1, 1 ).section( ",", 0, 0 ).stripWhiteSpace();
          
          bool foreign = true;
          
          if ( QDir( path ).canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || QDir( path ).canonicalPath().startsWith( QDir::home().canonicalPath() ) )
          {
            foreign = false;            
          }
          
          // Test if share is broken
          bool broken = false;
          
          QDir *d = new QDir( path );
          
          if ( !d || !d->exists() || !d->isReadable() )
          {
            broken = true;
          }
          
          delete d;
                    
          shares.append( new Smb4KShare( name, path, "cifs", login, foreign, broken ) );
        }
      }
      file.close();
    }
    else
    {
      emit error( ERROR_READING_FILE, QString( "%1/%2" ).arg( dir->path() ).arg( file.name() ) );
    }
  }
  
  delete dir;

  // Check, if the entries are already in the list.
  // If yes, replace the entry in the temporary by the one from the 
  // old list of shares. This way we won't get into trouble with 
  // shares that contain spaces.
  if ( m_mounted_shares.count() != 0 )
  {
    for ( QValueList<Smb4KShare *>::Iterator it = shares.begin(); it != shares.end(); ++it )
    {
      Smb4KShare *s = findShareByPath( (*it)->getPath() );
      
      if ( s )
      {
        s->setBroken( (*it)->isBroken() );
        
        delete *it;
        *it = s;

//         m_mounted_shares.remove( it );
//         m_mounted_shares.append( s );
      }
      else
      {
        continue;
      }
    }
  }
  
  m_mounted_shares = shares;
      
  emit importedShares();
  
  m_working = false;

#else

  struct statfs *buf;
  int count = getmntinfo( &buf, 0 );
   
  if ( count == 0 )
  {
    int err_code = errno;
    
    emit error( ERROR_IMPORTING_SHARES, strerror( err_code ) );
    
    m_working = false;
    return;
  }
  
  QValueList<Smb4KShare *> shares;

  for ( int i = 0; i < count; ++i )
  {
    if ( !strcmp( buf[i].f_fstypename, "smbfs" ) )
    {
      QString share( buf[i].f_mntfromname );
      QString path( buf[i].f_mntonname );
      QString fs( buf[i].f_fstypename );
      
      QFileInfo info( QString( buf[i].f_mntonname )+"/." );
      
      int uid = (int)info.ownerId();
      int gid = (int)info.groupId();
      
      // Test if share is broken
      bool broken = false;
          
      Smb4KShare *test = findShareByPath( path );
          
      if ( test && test->isBroken() )
      {
        broken = true;
      }
      else
      {          
        QDir *d = new QDir( path );
          
        if ( !d || !d->exists() || !d->isReadable() )
        {
          broken = true;
        }
          
        delete d;
      }
      
      shares.append( new Smb4KShare( share, path, fs, uid, gid, broken ) );
    }
  }
  
  // Apparently, under FreeBSD we do not need to delete 
  // the pointer (see manual page).
  
  m_mounted_shares = shares;
  
  emit importedShares();
    
  m_working = false;
  
#endif
}


/***************************************************************************
   This function is run on exit of the whole program. It cares about the
   last second actions, that are to be done.
***************************************************************************/

void Smb4KMounter::exit()
{
  m_config->setGroup( "Mount Options" );
  bool remount = m_config->readBoolEntry( "Mount Recent", false );
  bool unmount_on_exit = m_config->readBoolEntry( "Unmount All", false );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  
  m_lastJob = true;
  
  m_config->deleteGroup( "Recently Mounted Shares" );
  
  if ( remount )
  {
    m_config->setGroup( "Recently Mounted Shares" );
    
    int i = 0;
    
    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( (*it)->getUID() == (int)getuid() || (*it)->getCanonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || (*it)->getCanonicalPath().startsWith( QDir::home().canonicalPath() ) )
      {
        m_config->writeEntry( QString( "%1" ).arg( i++ ), (*it)->getName() );
      }
      else
      {
        continue;
      }
    }
  }
  
  QDir *dir = new QDir();
  
  dir->cd( mount_prefix );
  
  QStringList dirs = dir->entryList( QDir::Dirs, QDir::DefaultSort );
  
  for ( QStringList::ConstIterator it = dirs.begin(); it != dirs.end(); ++it )
  {
    if ( *it != "." && *it != ".." )
    {
      dir->cd( *it );
    
      QStringList subdirs = dir->entryList( QDir::Dirs, QDir::DefaultSort );
    
      for ( QStringList::ConstIterator i = subdirs.begin(); i != subdirs.end(); ++i )
      {
        if ( *i != "." && *i != ".." )
        {
          dir->rmdir( *i );
        }
      }
    
      dir->cdUp();
      dir->rmdir( *it );
    }
  }
  
  delete dir;
    
  if ( unmount_on_exit )
  {
    unmountAll();
  }
}


/***************************************************************************
   Mounts a share. (Public part)
***************************************************************************/

void Smb4KMounter::mountShare( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  QString *input = new QString( QString( "%1:%2:%3:%4:%5" ).arg( Mount ).arg( workgroup ).arg( host ).arg( ip ).arg( share ) );
  m_queue.enqueue( input );
}



/***************************************************************************
   Mounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::mount( const QString &workgroup, const QString &host, const QString &ip, const QString &share )
{
  m_config->setGroup( "Super User Privileges" );
  bool run_suid = m_config->readBoolEntry( "Run SUID", false );
  QString suid_program = m_config->readEntry( "SUID Program", QString::null );
  
  m_config->setGroup( "Samba" );
  m_filesystem = m_config->readEntry( "Mount Filesystem", "smbfs" );

  m_config->setGroup( "Mount Options" );
  QString mount_prefix = m_config->readPathEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  bool lowercase = m_config->readBoolEntry( "Force Lowercase", false );
  
  m_workgroup = workgroup.stripWhiteSpace();
  m_host = host.stripWhiteSpace();
  m_ip = ip.stripWhiteSpace();
  m_share = QString( "'%1'" ).arg( share.stripWhiteSpace() );
  QString share_dir = share.stripWhiteSpace();
  
  // Test whether the share is already mounted.
  QString test_name = QString( "//%1/%2" ).arg( m_host, m_share );
  
  Smb4KShare *test_share;
  
  if ( (test_share = findShareByPath( QString( "%1%2/%3" ).arg( mount_prefix, m_host, share_dir ) )) != NULL )
  {
    if ( !test_share->isForeign() )
    {
      emit error( ERROR_ALREADY_MOUNTED, QString::null );
      emit running( MOUNTER_STOP, false );
      m_working = false;
      return;
    }
  }
  else
  {
    if ( (test_share = findShareByName( QString( "//%1/%2" ).arg( m_host, m_share ) )) != NULL )
    {
      if ( !test_share->isForeign() )
      {
        emit error( ERROR_ALREADY_MOUNTED, QString::null );
        emit running( MOUNTER_STOP, false );
        m_working = false;
        return;
      }
    }
  }
  
  delete test_share;
  
  // Create the mount point.
  QDir *dir = new QDir( mount_prefix );
  
  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      emit error( ERROR_MKDIR_FAILED, dir->path() );
      emit running( MOUNTER_STOP, false );
      m_working = false;
      return;
    }
  }
  
  QString host_dir = lowercase ? host.lower() : host;
  
  dir->setPath( dir->path()+"/"+host_dir );
  
  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      emit error( ERROR_MKDIR_FAILED, dir->path() );
      emit running( MOUNTER_STOP, false );
      m_working = false;
      return;
    }
  }
  
  share_dir = lowercase ? share_dir.lower() : share_dir;
  
  dir->setPath( dir->path()+"/"+share_dir );
  
  if ( !dir->exists() )
  {
    if ( !dir->mkdir( dir->canonicalPath() ) )
    {
      emit error( ERROR_MKDIR_FAILED, dir->path() );
      emit running( MOUNTER_STOP, false );
      m_working = false;
      return;
    }
  }
  
  m_path = QDir::cleanDirPath( dir->path() );
  
  delete dir;
  
  // Mount the share.
  Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( workgroup, host, share );
  
  QString command;
    
#ifndef __FreeBSD__

  if ( !run_suid || suid_program.isEmpty() )
  {
    if ( m_filesystem != "cifs" )
    {
      command = "smb4k_mount --no-suid ";
    }
    else
    {
      command = QString( "smb4k_mount --suid -t %1 " ).arg( m_filesystem );
    }
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid -t %2 " ).arg( suid_program ).arg( m_filesystem );
  }

  command.append( QString( "//%1/%2 %3 " ).arg( KProcess::quote( host ) ).arg( KProcess::quote( share ) ).arg( KProcess::quote( m_path ) ) );

  command.append( "-o " );
    
  if ( !workgroup.isEmpty() )
  {
    command.append( QString( "workgroup=%1," ).arg( KProcess::quote( workgroup ) ) );
  }

  if ( !ip.isEmpty() )
  {
    command.append( QString( "ip=%1," ).arg( KProcess::quote( ip ) ) );
  }

  QString options = getMountOptions();
  
  if ( !options.stripWhiteSpace().isEmpty() )
  {
    command.append( options );
  }
    
  // We have to make sure, that if the user uses super or sudo,
  // the GID and the UID aren't the ones of root, except this
  // is explicitly wanted:
  if ( command.contains( "uid=", true ) == 0 && command.contains( "gid=", true ) == 0 && run_suid )
  {
    command.append( QString( "uid=%1,gid=%2," ).arg( QString( "%1" ).arg( (int)getuid() ) ).arg( QString( "%1" ).arg( (int)getgid() ) ) );
  }

  // (4) The authentication data:
  if ( !auth->user().isEmpty() )
  {
    command.append( QString( "username=%1" ).arg( KProcess::quote( auth->user() ) ) );
    m_cifs_login = auth->user();
    
    // We want to allow empty passwords:
    if ( !auth->password().isEmpty() )
    {
      m_proc->setEnvironment( "PASSWD", auth->password() );
    }
  }
  else
  {
    command.append( "guest" );
  }
      
#else

  if ( !run_suid || suid_program.isEmpty() )
  {
    command = QString( "smb4k_mount --no-suid " );
  }
  else if ( run_suid && !suid_program.isEmpty() )
  {
    command = QString( "%1 smb4k_mount --suid " ).arg( suid_program );
  }
      
  QString options = getMountOptions();
  
  if ( !options.stripWhiteSpace().isEmpty() )
  {
    command.append( options );
  }
      
  if ( !m_ip.isEmpty() )
  {
    command.append( " -I "+KProcess::quote( m_ip ) );
  }
      
  if ( !m_workgroup.isEmpty() )
  {
    command.append( " -W "+KProcess::quote( m_workgroup ) );
  }
      
  command.append( " -N" );
    
  if ( command.contains( "-u", true ) == 0 && command.contains( "-g", true ) == 0 && run_suid )
  {
    command.append( QString( " -u %1 -g %2" ).arg( QString( "%1" ).arg( (int)getuid() ) ).arg( QString( "%1" ).arg( (int)getgid() ) ) );
  }
    
  if ( !auth->user().isEmpty() )
  {
    command.append( " //"+KProcess::quote( auth->user() )+"@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( m_path ) );
  }
  else
  {
    command.append( " //guest@"+KProcess::quote( host )+"/"+KProcess::quote( share )+" "+KProcess::quote( m_path ) );
  }
    
#endif

  // Start the mount process:
  *m_proc << command;
  
  startProcess( Mount ); 
}


/****************************************************************************
   Unmounts a share. (Public part)
****************************************************************************/

void Smb4KMounter::unmountShare( const QString &mountpoint, const QString &uid, const QString &gid, bool noMessage )
{
  QString *input = new QString( QString( "%1:%2:%3:%4:%5" ).arg( Unmount ).arg( mountpoint ).arg( uid ).arg( gid ).arg( noMessage ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a share. (Private part)
***************************************************************************/

void Smb4KMounter::unmount( const QString &mountpoint, const QString &uid, const QString &gid, bool noMessage )
{
  m_config->setGroup( "Super User Privileges" );
  bool run_suid = m_config->readBoolEntry( "Run SUID", false );
  QString suid_program = m_config->readEntry( "SUID Program", QString::null );

  m_config->setGroup( "Mount Options" );
  bool allow_unmount_foreign = m_config->readBoolEntry( "Unmount Foreign", false );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  
  if ( !mountpoint.isEmpty() )
  {
    QString command;
    m_path = mountpoint;
    m_path.replace( '\044', "\044" );  
    
    if ( (uid.toInt() == (int)getuid() && gid.toInt() == (int)getgid()) || QDir( mountpoint ).canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || QDir( mountpoint ).canonicalPath().startsWith( QDir::home().canonicalPath() ) )
    {      
      if ( !run_suid || suid_program.isEmpty() )
      {
        command = "smb4k_umount --no-suid ";
      }
      else if ( run_suid && !suid_program.isEmpty() )
      {
        command = QString( "%1 smb4k_umount --suid " ).arg( suid_program );
      }
    }
    else
    {
      if ( allow_unmount_foreign )
      {
        if ( !run_suid || suid_program.isEmpty() )
        {
          command = "smb4k_umount --no-suid ";
        }
        else if ( run_suid && !suid_program.isEmpty() )
        {
          command = QString( "%1 smb4k_umount --suid " ).arg( suid_program );
        }
      }
      else
      {
        if ( !noMessage )
        {
          emit error( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        }
        
        emit running( MOUNTER_STOP, false );
        m_working = false;
        
        return;
      }
    }
    
    command.append( KProcess::quote( m_path ) );
    
    *m_proc << command;
    startProcess( Unmount );
  }
  else
  {
    emit error( ERROR_MOUNTPOINT_EMPTY, QString::null );
    emit running( MOUNTER_STOP, false );
    m_working = false;
  }
}


/****************************************************************************
   Unmounts a dead share. (Public part)
****************************************************************************/

void Smb4KMounter::forceUnmountShare( const QString &mountpoint, const QString &uid, const QString &gid )
{
  QString *input = new QString( QString( "%1:%2:%3:%4" ).arg( ForceUnmount ).arg( mountpoint ).arg( uid ).arg( gid ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts a dead share. (Private part)
***************************************************************************/

void Smb4KMounter::forceUnmount( const QString &mountpoint, const QString &uid, const QString &gid )
{
  m_config->setGroup( "Super User Privileges" );
  bool force_unmount = m_config->readBoolEntry( "Force Unmount", false );
  QString suid_program = m_config->readEntry( "SUID Program", QString::null );

  m_config->setGroup( "Mount Options" );
  bool allow_unmount_foreign = m_config->readBoolEntry( "Unmount Foreign", false );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
  
  QString kernel = ((Smb4KCore *)parent())->getKernelVersion();
  
  if ( !force_unmount )
  {
    emit error( ERROR_FEATURE_NOT_ENABLED, QString::null );
    emit running( MOUNTER_STOP, false );
    m_working = false;
    return;
  }
  
  // This will most likely never be the case, but we have to be prepared:
  if ( suid_program.isEmpty() )
  {
    emit error( ERROR_FORCE_UNMOUNT_SUID_MISSING, QString::null );
    emit running ( MOUNTER_STOP, false );
    m_working = false;
    return;
  }  
  
  // Check if the kernel supports the lazy unmount.  
  int kern_maj = kernel.section( ".", 0, 0 ).toInt();
  int kern_min = kernel.section( ".", 1, 1 ).toInt();
  int kern_rel = kernel.section( ".", 2, 2 ).toInt();
  
  if ( kern_maj < 2 ||
       ( kern_maj <= 2 && kern_min < 4 ) ||
       ( kern_maj <= 2 && kern_min <= 4 && kern_rel < 11 ) )
  {
    emit error( ERROR_WRONG_KERNEL, kernel );
    emit running( MOUNTER_STOP, false );
    m_working = false;
    return;
  }
  
  if ( !mountpoint.isEmpty() )
  {
    QString command;
    m_path = mountpoint;
    m_path.replace( '\044', "\044" );  
    
    bool execute = false;
    
    if ( uid.toInt() == (int)getuid() && gid.toInt() == (int)getgid()|| QDir( mountpoint ).canonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || QDir( mountpoint ).canonicalPath().startsWith( QDir::home().canonicalPath() ) )
    {     
      if ( KMessageBox::questionYesNo( (QWidget *)this, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
      { 
        command = QString( "%1 smb4k_umount --suid -l " ).arg( suid_program );
        execute = true;
      }
    }
    else
    {
      if ( allow_unmount_foreign )
      {
        if ( KMessageBox::questionYesNo( (QWidget *)this, i18n( "Do you really want to force the unmounting of this share?" ), QString::null, KStdGuiItem::yes(), KStdGuiItem::no(), "Dont Ask Forced", KMessageBox::Notify ) == KMessageBox::Yes )
        {
          command = QString( "%1 smb4k_umount --suid -l " ).arg( suid_program );
          execute = true;
        }
      }
      else
      {
        emit error( ERROR_UNMOUNTING_NOT_ALLOWED, QString::null );
        emit running( MOUNTER_STOP, false );
        m_working = false;
      }
    }
    
    command.append( KProcess::quote( m_path ) );
    
    if ( execute )
    {
      *m_proc << command;
      startProcess( Unmount );
    }
  }
  else
  {
    emit error( ERROR_MOUNTPOINT_EMPTY, QString::null );
    emit running( MOUNTER_STOP, false );
    m_working = false;  
  }
}


/***************************************************************************
   Unmounts all shares at once. (Public part)
***************************************************************************/

void Smb4KMounter::unmountAllShares()
{
  QString *input = new QString( QString( "%1" ).arg( UnmountAll ) );
  m_queue.enqueue( input );
}


/***************************************************************************
   Unmounts all shares at once.
***************************************************************************/

void Smb4KMounter::unmountAll()
{
  m_config->setGroup( "Super User Privileges" );
  bool run_suid = m_config->readBoolEntry( "Run SUID", false );
  QString suid_program = m_config->readEntry( "SUID Program", QString::null );

  m_config->setGroup( "Mount Options" );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );
    
  if ( !m_lastJob )
  {
    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      unmountShare( (*it)->getPath(), QString( "%1" ).arg( (*it)->getUID() ), QString( "%1" ).arg( (*it)->getGID() ), true );
    }
    
    m_working = false;
  }
  else
  {
    QString command;
    
    for ( QValueListIterator<Smb4KShare *> it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
    {
      if ( ((*it)->getUID() == (int)getuid() && (*it)->getGID() == (int)getgid()) || (*it)->getCanonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) || (*it)->getCanonicalPath().startsWith( QDir::home().canonicalPath() ) )
      {
        if ( run_suid )
        {
          command.append( QString( "%1 smb4k_umount --suid " ).arg( suid_program ) ).append( KProcess::quote( (*it)->getPath().replace( '\044', "\044" ) ) ).append( " ; " );
        }
        else
        {
          command.append( "smb4k_umount --no-suid " ).append( KProcess::quote( (*it)->getPath().replace( '\044', "\044" ) ) ).append( " ; " );
        }
        
#ifndef __FreeBSD__          
          command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( (*it)->getCanonicalPath().replace( '\044', "\044" ) ) ).append( " ; " );
          command.append( "rmdir --ignore-fail-on-non-empty " ).append( KProcess::quote( (*it)->getCanonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
#else
          command.append( "rmdir " ).append( KProcess::quote( (*it)->getCanonicalPath().replace( '\044', "\044" ) ) ).append( " ; " );
          command.append( "rmdir " ).append( KProcess::quote( (*it)->getCanonicalPath().section( "/", 0, -2 ) ) ).append( " ; " );
#endif        
      }
      else
      {
        continue;
      }
    }

    command.truncate( command.length() - 2 );
    
    *m_proc << command;
  
    startProcess( UnmountAll );
  }
}


/***************************************************************************
   Starts any process.
***************************************************************************/

void Smb4KMounter::startProcess( int state )
{
  m_buffer = QString::null;
  m_state = state;

  if ( m_state != Import )
  {
    QApplication::setOverrideCursor( waitCursor );
  }

  if ( UnmountAll && m_lastJob )
  {
    m_proc->detach();
    m_proc->start( KProcess::DontCare, KProcess::NoCommunication );
  }
  else
  {
    m_proc->start( KProcess::NotifyOnExit, KProcess::AllOutput );
  }
}


/***************************************************************************
   Ends any process. This functions tells the mounter what to do
   afterwards.
***************************************************************************/

void Smb4KMounter::endProcess()
{
  switch ( m_state )
  {
    case Mount:
      processMount();
      break;
    case Unmount:
      processUnmount();
      break;
    case ForceUnmount:
      processUnmount();
      break;
    default:
      break;
  }
  m_state = Idle;
  m_path = QString::null;
  QApplication::restoreOverrideCursor();
  m_proc->clearArguments();

  m_working = false;
  emit running( MOUNTER_STOP, m_working );
}


/***************************************************************************
   Process mounts.
***************************************************************************/

void Smb4KMounter::processMount()
{
#ifndef __FreeBSD__
    
  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "failed", true ) == 0 && 
         m_buffer.contains( "ERR", true ) == 0 && 
         m_buffer.contains( "/bin/sh:" ) == 0 &&
         m_buffer.contains( "mount:", true ) == 0 && 
         m_buffer.contains( "smbmnt" ) == 0 &&
         m_buffer.contains( m_path ) == 0 &&
         m_buffer.contains( "mount error" ) == 0 )
    {
      QString mountpoint;
      
      if ( m_share.contains( "'" ) == 2 )
      {
        m_share = m_share.replace( QRegExp( "'" ), "" );
      }

      QString name = QString( "//%1/%2" ).arg( m_host ).arg( m_share );
      
      bool broken = false;
      
      QDir *d = new QDir( m_path );
      
      if ( !d || !d->exists() || !d->isReadable() )
      {
        broken = true;
      }
      
      delete d;
      
      if ( m_filesystem == "smbfs" )
      {
        m_mounted_shares.append( new Smb4KShare( name, m_path, m_filesystem, (int)getuid(), (int)getgid(), broken ) );
      }
      else if ( m_filesystem == "cifs" )
      {
        m_mounted_shares.append( new Smb4KShare( name, m_path, m_filesystem, m_cifs_login, false, broken ) );
      }
    }
    else
    {
      if ( m_buffer.contains( "ERRbadpw" ) != 0 || m_buffer.contains( "ERRnoaccess" ) != 0 
           || m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
      {
        if ( m_share.contains( "'" ) == 2 )
        {
          m_share = m_share.replace( QRegExp( "'" ), "" );
        }
        
        int state = Smb4KPasswdReader::None;
        
        if ( m_buffer.contains( "ERRbadpw" ) != 0 )
        {
          state = Smb4KPasswdReader::BadPasswd;
        }
        else if ( m_buffer.contains( "ERRnoaccess" ) != 0 )
        {
          state = Smb4KPasswdReader::AccessDenied;
        }
        else if ( m_buffer.contains( "mount error 13 = Permission denied" ) != 0 )
        {
          state = Smb4KPasswdReader::PermDenied;
        }
        
        // If the user supplied auth information, we will retry mounting.
        if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_workgroup, m_host, m_share, state ) )
        {
          mountShare( m_workgroup, m_host, m_ip, m_share );
        }
      }
      else if ( m_buffer.contains( "ERRnosuchshare" ) != 0 && m_share.contains( "_" ) != 0 )
      {
        m_share = m_share.replace( "_", " " );
        mountShare( m_workgroup, m_host, m_ip, m_share );        
      }
      else
        emit error( ERROR_MOUNTING_SHARE, m_buffer );
    }
  }
  
#else
    
  if ( m_proc->normalExit() )
  {
    if ( m_buffer.contains( "Authentication error", true ) == 0 
         && m_buffer.contains( "Connection refused", true ) == 0 
         && m_buffer.contains( "Operation not permitted", true ) == 0 )
    {
      import();  // FIXME: What is this for?
      
      if ( m_share.contains( "'" ) == 2 )
      {
        m_share = m_share.replace( QRegExp( "'" ), "" );
      }

      Smb4KAuthInfo *auth = ((Smb4KCore *)parent())->passwdReader()->getAuth( m_workgroup, m_host, m_share );
     
      QString name = QString( "//%1@%2/%3" ).arg( auth->user().upper(), m_host.upper(), m_share.upper() );
      
      bool broken = false;
      
      QDir *d = new QDir( m_path );
      
      if ( !d || !d->exists() || !d->isReadable() )
      {
        broken = true;
      }
      
      delete d;
      
      m_mounted_shares.append( new Smb4KShare( name, m_path, m_filesystem, (int)getuid(), (int)getgid(), broken ) );
    }
    else
    {
      if ( m_buffer.contains( "Authentication error" ) != 0 )
      {
        if ( m_share.contains( "'" ) == 2 )
        {
          m_share = m_share.replace( QRegExp( "'" ), "" );
        }
      
        // If the user supplied auth information, we will retry mounting.
        if ( ((Smb4KCore *)parent())->passwdReader()->askpass( m_workgroup, m_host, m_share, Smb4KPasswdReader::AuthError ) )
        {
          mountShare( m_workgroup, m_host, m_ip, m_share );
        }
      }
      else
        emit error( ERROR_MOUNTING_SHARE, m_buffer );
    }
  }
  
#endif

  m_workgroup = QString::null;
  m_host = QString::null;
  m_share = QString::null;
  m_ip = QString::null;
  m_filesystem = QString::null;
  m_cifs_login = QString::null;
  
  emit importedShares();
}


/***************************************************************************
   Process unmounts.
***************************************************************************/

void Smb4KMounter::processUnmount()
{
  m_config->setGroup( "Mount Options" );
  QString mount_prefix = m_config->readEntry( "Default Path", QDir::homeDirPath().append( "/smb4k/" ) );

  if ( m_proc->normalExit() )
  {
    // FIXME: This is a potential problem. What if the unmount
    // process returns some debugging info like the mount process
    // under SuSE? Need more info.
    if ( m_buffer.isEmpty() )
    {
      // Since the mount point is unique, the mount point is greped
      // and used to determine the share that has to be removed from
      // m_mountedShares.
      Smb4KShare *share = findShareByPath( m_path );
      
      if ( share->getCanonicalPath().startsWith( QDir( mount_prefix ).canonicalPath() ) )
      {
        QDir *d = new QDir( share->getCanonicalPath() );
                
        if ( d->rmdir( d->canonicalPath(), true ) )
        {
          d->cdUp();
          d->rmdir( d->canonicalPath(), true );
        }
        
        delete d;
      }
        
      m_mounted_shares.remove(share);
    }
    else
    {
      emit error( ERROR_UNMOUNTING_SHARE, m_buffer );
    }
  }
  
  emit importedShares();
}


/***************************************************************************
   Check if a share is already mounted
***************************************************************************/

bool Smb4KMounter::isMounted( const QString &name )
{
  return findShareByName( name ) != NULL ? true : false;
}


/***************************************************************************
   Find a share in the list with its path
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByPath(const QString &path)
{
  if ( path.isEmpty() )
  {
    return NULL;
  }
  
  QValueListIterator<Smb4KShare *> it;
  
  QDir *d = new QDir( path );
  QString cp = d->canonicalPath().append( "/" );
  
  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( (*it)->getPath().upper() == path.upper() || (*it)->getCanonicalPath().upper() == cp.upper() )
    {
      break;
    }
  }
  
  delete d;
  
  return it == m_mounted_shares.end() ? NULL : *it;
}


/***************************************************************************
   Find a share in the list with its name
***************************************************************************/

Smb4KShare* Smb4KMounter::findShareByName(const QString &name)
{
  QString n = name;
  
  QValueListIterator<Smb4KShare *> it;
  
  for ( it = m_mounted_shares.begin(); it != m_mounted_shares.end(); ++it )
  {
    if( (*it)->getName().upper() == name.upper() )
    {
      break;
    }
    else if ( (*it)->getName().upper() == n.replace( " ", "_" ).upper() )
    {
      break;
    }
  }
  
  return it == m_mounted_shares.end() ? NULL : *it;
}


/***************************************************************************
   Returns the advanced Samba options as chosen by the user.
***************************************************************************/

const QString Smb4KMounter::getMountOptions()
{
  m_config->setGroup( "Samba" );
  QString options;
  
#ifndef __FreeBSD__

  QString filesystem = m_config->readEntry( "Mount Filesystem", "smbfs" );

  if ( filesystem == "smbfs" )
  {
    if ( m_config->readBoolEntry( "Use Kerberos", false ) )
      options.append( "krb," );

    if ( !m_config->readEntry( "NetBIOS Name", QString::null ).isEmpty() )
      options.append( QString( "netbiosname='%1'," ).arg( m_config->readEntry( "NetBIOS Name", QString::null ) ) );

    if ( !m_config->readEntry( "Socket Options", QString::null ).isEmpty() )
      options.append( QString( "sockopt=\"%1\"," ).arg( m_config->readEntry( "Socket Options", QString::null ) ) );

    if ( !m_config->readEntry( "NetBIOS Scope", QString::null ).isEmpty() )
      options.append( QString( "scope=\"%1\"," ).arg( m_config->readEntry( "NetBIOS Scope", QString::null ) ) );
      
    if ( !m_config->readEntry( "Mount Charset", QString::null ).isEmpty() )
      options.append( QString( "iocharset=%1," ).arg( m_config->readEntry( "Mount Charset", QString::null ) ) );
      
    if ( !m_config->readEntry( "Mount Codepage", QString::null ).isEmpty() )
      options.append( QString( "codepage=%1," ).arg( m_config->readEntry( "Mount Codepage", QString::null ) ) );
      
    if ( !m_config->readEntry( "Mount Cache", QString::null ).isEmpty() )
      options.append( QString( "ttl=%1," ).arg( m_config->readEntry( "Mount Cache", QString::null ) ) );
      
    if ( m_config->readBoolEntry( "Mount Unicode", false ) )
      options.append( "unicode," );

    if ( m_config->readBoolEntry( "Mount LFS", false ) )
      options.append( "lfs," );
  
    if ( !m_config->readEntry( "Mount FMASK", QString::null ).isEmpty() )
      options.append( QString( "fmask=%1," ).arg( m_config->readEntry( "Mount FMASK", QString::null ) ) );
  
    if ( !m_config->readEntry( "Mount DMASK", QString::null ).isEmpty() )
      options.append( QString( "dmask=%1," ).arg( m_config->readEntry( "Mount DMASK", QString::null ) ) );
  
  }
  else
  {
    if ( !m_config->readEntry( "Mount FMASK", QString::null ).isEmpty() )
      options.append( QString( "file_mode=%1," ).arg( m_config->readEntry( "Mount FMASK", QString::null ) ) );
  
    if ( !m_config->readEntry( "Mount DMASK", QString::null ).isEmpty() )
      options.append( QString( "dir_mode=%1," ).arg( m_config->readEntry( "Mount DMASK", QString::null ) ) );
      
    if ( m_config->readBoolEntry( "Mount RSize", false ) )
      options.append( "rsize," );
      
    if ( m_config->readBoolEntry( "Mount WSize", false ) )
      options.append( "wsize," );
  }
    
  if ( !m_config->readEntry( "Mount UID", QString::null ).isEmpty() )
    options.append( QString( "uid=%1," ).arg( m_config->readEntry( "Mount UID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount GID", QString::null ).isEmpty() )
    options.append( QString( "gid=%1," ).arg( m_config->readEntry( "Mount GID", QString::null ) ) );

  if ( !m_config->readEntry( "Port", QString::null ).isEmpty() )
    options.append( QString( "port=%1," ).arg( m_config->readNumEntry( "Port", 139 ) ) );

  if ( m_config->readBoolEntry( "Mount ReadWrite", true ) )
    options.append( "rw," );
  else
    options.append( "ro," );
    
#else

  if ( !m_config->readEntry( "Mount UID", QString::null ).isEmpty() )
    options.append( QString( "-u %1" ).arg( m_config->readEntry( "Mount UID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount GID", QString::null ).isEmpty() )
    options.append( QString( " -g %1" ).arg( m_config->readEntry( "Mount GID", QString::null ) ) );

  if ( !m_config->readEntry( "Mount FMASK", QString::null ).isEmpty() )
    options.append( QString( " -f %1" ).arg( m_config->readEntry( "Mount FMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Mount DMASK", QString::null ).isEmpty() )
    options.append( QString( " -d %1" ).arg( m_config->readEntry( "Mount DMASK", QString::null ) ) );

  if ( !m_config->readEntry( "Mount Charset", QString::null ).isEmpty() && !m_config->readEntry( "Mount Codepage", QString::null ).isEmpty() )
  {
    options.append( QString( " -E %1:%2" ).arg( m_config->readEntry( "Mount Charset", QString::null ) ).arg( m_config->readEntry( "Mount Codepage", QString::null ) ) );
  }

#endif

  return options;
}


/////////////////////////////////////////////////////////////////////////////
// SLOT IMPLEMENTATIONS
/////////////////////////////////////////////////////////////////////////////

/****************************************************************************
   Internal slots.
****************************************************************************/

void Smb4KMounter::slotProcessExited( KProcess * )
{
  endProcess();
}


void Smb4KMounter::slotReceivedStdout( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::slotReceivedStderr( KProcess *, char *buf, int len )
{
  m_buffer.append( QString::fromLocal8Bit( buf, len ) );
}


void Smb4KMounter::init()
{
  if ( !m_working && !m_queue.isEmpty() )
  {
    // Tell the mounter, that it is busy.
    m_working = true;

    QString *item = m_queue.dequeue();
    int todo = item->section( ":", 0, 0 ).toInt();

    switch ( todo )
    {
      case MountRecent:
        mountRecent();
        break;
      case Import:
        import();
        break;
      case Mount:
        emit running( MOUNTER_MOUNTING, m_working );
        mount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), item->section( ":", 4, 4 ) );
        break;
      case Unmount:
        emit running( MOUNTER_UNMOUNTING, m_working );
        unmount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ), (bool)item->section( ":", 4, 4 ).toInt() );
        break;
      case ForceUnmount:
        emit running( MOUNTER_UNMOUNTING, m_working );
        forceUnmount( item->section( ":", 1, 1 ), item->section( ":", 2, 2 ), item->section( ":", 3, 3 ) );
        break;
      case UnmountAll:
        unmountAll();
        break;
      default:
        break;
    }
 
    delete item;
  }
}


/***************************************************************************
   Initializes periodic checking. At the moment only the detection of 
   mounted shares is supported. (connected to second timer)
***************************************************************************/

void Smb4KMounter::check()
{
  if ( !m_working || m_queue.isEmpty() )
  {
    
    m_queue.enqueue( new QString( QString( "%1:" ).arg( Import ) ) );
  }
}


#include "smb4kmounter.moc"
