/* mod_scgi.c
 *
 * Apache 2 implementation of the SCGI protocol.
 *
 */

#include "ap_config.h"
#include "apr_strings.h"
#include "httpd.h"
#include "http_config.h"
#include "http_core.h"
#include "http_request.h"
#include "http_log.h"
#include "http_protocol.h"
#include "mpm_common.h"
#include "util_script.h"

#define MOD_SCGI_VERSION "1.2"
#define SCGI_PROTOCOL_VERSION "1"
#define DEFAULT_TIMEOUT  60 /* default socket timeout */

#define UNSET 0
#define ENABLED 1
#define DISABLED 2

/*
 * Configuration record.  Used per-directory configuration data.
 */
typedef struct {
    char *addr;
    apr_port_t port;
    int enabled; /* mod_scgi is enabled from this directory */
    int timeout;
} scgi_cfg;

/*
 * Declare ourselves so the configuration routines can find and know us.
 * We'll fill it in at the end of the module.
 */
module AP_MODULE_DECLARE_DATA scgi_module;

/*
 * Locate our directory configuration record for the current request.
 */
static scgi_cfg *
our_dconfig(request_rec *r)
{
    return (scgi_cfg *) ap_get_module_config(r->per_dir_config, &scgi_module);
}

static int scgi_translate(request_rec *r)
{
	scgi_cfg *cfg = our_dconfig(r);

	if (cfg->enabled == ENABLED) {
		r->handler = "scgi-handler";
		return OK;
	}
	return DECLINED;
}

static void log_err(const char *file, int line, request_rec *r,
                    apr_status_t status, const char *msg)
{
    char buf[256] = "";
    apr_strerror(status, buf, sizeof(buf));
    ap_log_rerror(file, line, APLOG_ERR, status, r, "scgi: %s: %s", buf, msg);
}

static void log_debug(const char *file, int line, request_rec *r, const
                      char *msg)
{
    ap_log_rerror(file, line, APLOG_DEBUG, APR_SUCCESS, r, msg);
}


static char *lookup_name(apr_table_t *t, const char *name)
{
    const apr_array_header_t *hdrs_arr = apr_table_elts(t);
    apr_table_entry_t *hdrs = (apr_table_entry_t *) hdrs_arr->elts;
    int i;

    for (i = 0; i < hdrs_arr->nelts; ++i) {
        if (hdrs[i].key == NULL)
            continue;

        if (strcasecmp(hdrs[i].key, name) == 0)
            return hdrs[i].val;
    }
    return NULL;
}


static char *lookup_header(request_rec *r, const char *name)
{
    return lookup_name(r->headers_in, name);
}


static char *lookup_env(request_rec *r, const char *name)
{
    return lookup_name(r->subprocess_env, name);
}


static void add_header(apr_table_t *t, const char *name, const char *value)
{
    if (name != NULL && value != NULL)
        apr_table_addn(t, name, value);
}

/* buffered socket implementation (buckets are overkill) */

#define BUFFER_SIZE 8000

struct sockbuff {
    apr_socket_t *sock;
    char buf[BUFFER_SIZE];
    int used;
};

static void binit(struct sockbuff *s, apr_socket_t *sock)
{
    s->sock = sock;
    s->used = 0;
}

static apr_status_t sendall(apr_socket_t *sock, char *buf, apr_size_t len)
{
    apr_status_t rv;
    apr_size_t n;
    while (len > 0) {
        n = len;
        if ((rv = apr_send(sock, buf, &n))) return rv;
        buf += n;
        len -= n;
    }
    return APR_SUCCESS;
}

static apr_status_t bflush(struct sockbuff *s)
{
    apr_status_t rv;
    ap_assert(s->used >= 0 && s->used <= BUFFER_SIZE);
    if (s->used) {
            if ((rv = sendall(s->sock, s->buf, s->used))) return rv;
            s->used = 0;
    }
    return APR_SUCCESS;
}

static apr_status_t bwrite(struct sockbuff *s, char *buf, apr_size_t len)
{
    apr_status_t rv;
    if (len >= BUFFER_SIZE - s->used) {
        if ((rv = bflush(s))) return rv;
        while (len >= BUFFER_SIZE) {
            if ((rv = sendall(s->sock, buf, BUFFER_SIZE))) return rv;
            buf += BUFFER_SIZE;
            len -= BUFFER_SIZE;
        }
    }
    if (len > 0) {
        ap_assert(len < BUFFER_SIZE - s->used);
        memcpy(s->buf + s->used, buf, len);
        s->used += len;
    }
    return APR_SUCCESS;
}

static apr_status_t bputs(struct sockbuff *s, char *buf)
{
    return bwrite(s, buf, strlen(buf));
}

static apr_status_t bputc(struct sockbuff *s, char c)
{
    char buf[1];
    buf[0] = c;
    return bwrite(s, buf, 1);
}


static apr_status_t
send_headers(request_rec *r, struct sockbuff *s, scgi_cfg *cfg)
{
    apr_table_t *t = apr_table_make(r->pool, 40); /* headers to send */
    const apr_array_header_t *hdrs_arr;
    apr_table_entry_t *hdrs;
    unsigned long int n = 0;
    char *buf;
    int i;
    apr_status_t rv = 0;
    apr_port_t  port = 0;
    apr_sockaddr_port_get(&port, r->connection->remote_addr);

    log_debug(APLOG_MARK,r, "sending headers");
    /* CONTENT_LENGTH must come first and always be present */
    add_header(t, "CONTENT_LENGTH",
               apr_psprintf(r->pool, "%ld", r->remaining));
    add_header(t, "SCGI", SCGI_PROTOCOL_VERSION);
    add_header(t, "SERVER_SOFTWARE", ap_get_server_version());
    add_header(t, "SERVER_PROTOCOL", r->protocol);
    add_header(t, "SERVER_NAME", ap_get_server_name(r));
    add_header(t, "SERVER_ADMIN", r->server->server_admin);
    add_header(t, "SERVER_ADDR", r->connection->local_ip);
    add_header(t, "SERVER_PORT", apr_psprintf(r->pool, "%u",
                                              ap_get_server_port(r)));
    add_header(t, "REMOTE_ADDR", r->connection->remote_ip);
    add_header(t, "REMOTE_PORT", apr_psprintf(r->pool, "%d", port));
    add_header(t, "REMOTE_USER", r->user);
    add_header(t, "REQUEST_METHOD", r->method);
    add_header(t, "REQUEST_URI", r->unparsed_uri);
    add_header(t, "QUERY_STRING", r->args ? r->args : "");
    add_header(t, "SCRIPT_NAME", r->uri);
    /* skip PATH_INFO, the 'uri' contains the full path */
    /* add_header(t, "PATH_INFO", r->path_info); */
    add_header(t, "HTTPS", lookup_env(r, "HTTPS"));
    add_header(t, "CONTENT_TYPE", lookup_header(r, "Content-type"));
    add_header(t, "DOCUMENT_ROOT", ap_document_root(r));
    add_header(t, "HTTP_ACCEPT", lookup_header(r, "Accept"));
    add_header(t, "HTTP_ACCEPT_CHARSET", lookup_header(r, "Accept-Charset"));
    add_header(t, "HTTP_ACCEPT_ENCODING", lookup_header(r, "Accept-Encoding"));
    add_header(t, "HTTP_ACCEPT_LANGUAGE", lookup_header(r, "Accept-Language"));
    add_header(t, "HTTP_AUTHORIZATION", lookup_header(r, "Authorization"));
    add_header(t, "HTTP_COOKIE", lookup_header(r, "Cookie"));
    add_header(t, "HTTP_EXPECT", lookup_header(r, "Expect"));
    add_header(t, "HTTP_FROM", lookup_header(r, "From"));
    add_header(t, "HTTP_HOST", lookup_header(r, "Host"));
    add_header(t, "HTTP_IF_MATCH", lookup_header(r, "If-Match"));
    add_header(t, "HTTP_IF_MODIFIED_SINCE",
               lookup_header(r, "If-Modified-Since"));
    add_header(t, "HTTP_IF_NONE_MATCH", lookup_header(r, "If-None-Match"));
    add_header(t, "HTTP_IF_RANGE", lookup_header(r, "If-Range"));
    add_header(t, "HTTP_IF_UNMODIFIED_SINCE",
               lookup_header(r, "If-Unmodified-Since"));
    add_header(t, "HTTP_RANGE", lookup_header(r, "Range"));
    add_header(t, "HTTP_REFERER", lookup_header(r, "Referer"));
    add_header(t, "HTTP_TE", lookup_header(r, "TE"));
    add_header(t, "HTTP_USER_AGENT", lookup_header(r, "User-Agent"));

    hdrs_arr = apr_table_elts(t);
    hdrs = (apr_table_entry_t*) hdrs_arr->elts;

    /* calculate length of header data (including nulls) */
    for (i = 0; i < hdrs_arr->nelts; ++i) {
        n += strlen(hdrs[i].key) + 1;
        n += strlen(hdrs[i].val) + 1;
    }

    buf = apr_psprintf(r->pool, "%lu:", n);
    if (!buf)
        return APR_ENOMEM;
    rv = bputs(s, buf);
    if (rv)
        return rv;

    for (i = 0; i < hdrs_arr->nelts; ++i) {
        rv = bputs(s, hdrs[i].key);
        if (rv) return rv;
        rv = bputc(s, '\0');
        if (rv) return rv;
        rv = bputs(s, hdrs[i].val);
        if (rv) return rv;
        rv = bputc(s, '\0');
        if (rv) return rv;
    }

    rv = bputc(s, ',');
    if (rv)
        return rv;

    return APR_SUCCESS;
}

static apr_status_t send_request_body(request_rec *r, struct sockbuff *s)
{
    if (ap_should_client_block(r)) {
        char buf[BUFFER_SIZE];
        apr_status_t rv;
        apr_off_t len;

        while ((len = ap_get_client_block(r, buf, sizeof buf)) > 0) {
            if ((rv = bwrite(s, buf, len))) return rv;
        }
        if (len == -1)
            return HTTP_INTERNAL_SERVER_ERROR; /* what to return? */
    }
    return APR_SUCCESS;
}

static apr_status_t
open_socket(apr_socket_t **sock, request_rec *r, scgi_cfg *cfg)
{
    char *laddr         = cfg->addr ? cfg->addr : "localhost";
    apr_port_t lport    = cfg->port ? cfg->port : 4000;
    int ltimeout        = cfg->timeout ? cfg->timeout : DEFAULT_TIMEOUT;
    apr_interval_time_t timeout = apr_time_from_sec(ltimeout);
    int retries = 4;
    int sleeptime = 1;
    apr_status_t rv;
    apr_sockaddr_t *addr;

    rv = apr_sockaddr_info_get(&addr, laddr, AF_INET, lport, 0, r->pool);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "apr_sockaddr_info_get() error");
        return rv;
    }

    *sock = NULL;
    rv = apr_socket_create(sock, AF_INET, SOCK_STREAM, r->pool);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "apr_socket_create() error");
        return rv;
    }

    rv = apr_socket_timeout_set(*sock, timeout);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "apr_socket_timout_set() error");
        return rv;
    }

 restart:
    rv = apr_socket_connect(*sock, addr);
    if (rv) {
        if (APR_STATUS_IS_ECONNREFUSED(rv) && retries > 0) {
            /* server may be temporarily down, retry */
            ap_log_rerror(APLOG_MARK, APLOG_NOERRNO|APLOG_DEBUG, rv, r,
                          "scgi: connection refused, retrying");
            apr_sleep(sleeptime);
            --retries;
            sleeptime *= 2;
            goto restart;
        }
        log_err(APLOG_MARK, r, rv, "scgi: can't connect to server");
        return rv;
    }

    ap_sock_disable_nagle(*sock);
    return APR_SUCCESS;
}



static int scgi_handler(request_rec *r)
{
    apr_status_t rv = 0;
    int http_status = 0;
    scgi_cfg *cfg   = NULL;
    struct sockbuff s;
    apr_socket_t *sock;
    apr_bucket_brigade *bb = NULL;
    apr_bucket *b          = NULL;
    char buf[MAX_STRING_LEN];
    const char *location;

    if (strcmp(r->handler, "scgi-handler"))
        return DECLINED;

    cfg = our_dconfig(r);
    if (!cfg->enabled)
        return DECLINED;

    http_status = ap_setup_client_block(r, REQUEST_CHUNKED_ERROR);
    if (http_status != OK)
        return http_status;

    log_debug(APLOG_MARK, r, "connecting to server");

    rv = open_socket(&sock, r, cfg);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "error connecting to scgi server");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    binit(&s, sock);

    rv = send_headers(r, &s, cfg);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "error sending request headers");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    rv = send_request_body(r, &s);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "error sending request body");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    rv = bflush(&s);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "error sending request");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    log_debug(APLOG_MARK, r, "reading response headers");
    bb = apr_brigade_create(r->connection->pool, r->connection->bucket_alloc);
    b = apr_bucket_socket_create(sock, r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);
    b = apr_bucket_eos_create(r->connection->bucket_alloc);
    APR_BRIGADE_INSERT_TAIL(bb, b);

    rv = ap_scan_script_header_err_brigade(r, bb, buf);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "error reading response headers");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    location = apr_table_get(r->headers_out, "Location");

    if (location && location[0] == '/' &&
        ((r->status == HTTP_OK) || ap_is_HTTP_REDIRECT(r->status))) {

        apr_brigade_destroy(bb);

        /* Internal redirect -- fake-up a pseudo-request */
        r->status = HTTP_OK;

        /* This redirect needs to be a GET no matter what the original
         * method was.
         */
        r->method = apr_pstrdup(r->pool, "GET");
        r->method_number = M_GET;

        ap_internal_redirect_handler(location, r);
        return OK;
    }

    rv = ap_pass_brigade(r->output_filters, bb);
    if (rv) {
        log_err(APLOG_MARK, r, rv, "ap_pass_brigade()");
        return HTTP_INTERNAL_SERVER_ERROR;
    }

    return OK;
}


static int scgi_init(apr_pool_t *p, apr_pool_t *plog, apr_pool_t *ptemp,
                       server_rec *base_server)
{
    ap_add_version_component(p, "mod_scgi/" MOD_SCGI_VERSION);
    return OK;
}


static void *scgi_create_dir_config(apr_pool_t *p, char *dirspec)
{
    scgi_cfg *cfg = apr_pcalloc(p, sizeof(scgi_cfg));

    cfg->enabled = UNSET;
    cfg->addr = UNSET;
    cfg->port = UNSET;
    cfg->timeout = UNSET;

    return cfg;
}

#define MERGE(b, n, a) (n->a == UNSET ? b->a : n->a)

static void *
scgi_merge_dir_config(apr_pool_t *p, void *basev, void *newv)
{
    scgi_cfg* cfg = apr_pcalloc(p, sizeof(scgi_cfg));
    scgi_cfg* base = basev;
    scgi_cfg* new = newv;

    cfg->enabled = MERGE(base, new, enabled);
    cfg->addr = MERGE(base, new, addr);
    cfg->port = MERGE(base, new, port);
    cfg->timeout = MERGE(base, new, timeout);

    return cfg;
}


static const char *
cmd_server(cmd_parms *cmd, void *pcfg, const char *addr_and_port)
{
    apr_status_t rv;
    scgi_cfg *cfg = pcfg;
    char *scope_id = NULL; /* A ip6 parameter - not used here. */

    if (cmd->path == NULL)
        return "not a server command";

    rv = apr_parse_addr_port(&cfg->addr, &scope_id, &cfg->port,
                             addr_and_port, cmd->pool);
    if (rv)
        return "error parsing address:port string";

    return NULL;
}


static const char *
cmd_handler(cmd_parms* cmd, void* pcfg, int flag)
{
    scgi_cfg *cfg = pcfg;

    if (cmd->path == NULL) /* server command */
        return "not a server command";

    if (flag)
        cfg->enabled = ENABLED;
    else
        cfg->enabled = DISABLED;

    return NULL;
}


static const char *
cmd_timeout(cmd_parms *cmd, void* pcfg, const char *strtimeout)
{
    scgi_cfg *cfg = pcfg;

    if (cmd->path == NULL)
        return "not a server command";

    cfg->timeout = atoi(strtimeout);

    return NULL;
}


static const command_rec scgi_cmds[] =
{
    AP_INIT_TAKE1("SCGIServer", cmd_server, NULL, ACCESS_CONF,
                  "Address and port of an SCGI server (e.g. localhost:4000)"),
    AP_INIT_FLAG( "SCGIHandler", cmd_handler, NULL, ACCESS_CONF,
                  "On or Off to enable or disable the SCGI handler"),
    AP_INIT_TAKE1("SCGIServerTimeout", cmd_timeout, NULL, ACCESS_CONF,
                  "Timeout (in seconds) for communication with the SCGI server."),
    {NULL}
};


static void scgi_register_hooks(apr_pool_t *p)
{
    ap_hook_post_config(scgi_init, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_handler(scgi_handler, NULL, NULL, APR_HOOK_MIDDLE);
    ap_hook_translate_name(scgi_translate, NULL, NULL, APR_HOOK_LAST);
}


/* Dispatch list for API hooks */
module AP_MODULE_DECLARE_DATA scgi_module = {
    STANDARD20_MODULE_STUFF,
    scgi_create_dir_config,             /* create per-dir config structs */
    scgi_merge_dir_config,              /* merge per-dir config structs */
    NULL,                               /* create per-server config structs */
    NULL,                               /* merge per-server config structs */
    scgi_cmds,                          /* table of config file commands */
    scgi_register_hooks,                /* register hooks */
};
