
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Timothy J. McBrayer 
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Radharamanan Radhakrishnan

#include "IIRScram_IntegerLiteral.hh"
#include "IIR_IntegerSubtypeDefinition.hh"
#include "IIR_Declaration.hh"
#include "StandardPackage.hh"
#include "symbol_table.hh"
#include "resolution_func.hh"
#include "set.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

IIRScram_IntegerLiteral::~IIRScram_IntegerLiteral() {}

IIR_Boolean
IIRScram_IntegerLiteral::_is_negative()
{
  IIR_Char* man = _get_mantissa();
  if (man[0] == '-') {
    return TRUE;
  }
  return FALSE;
}

void 
IIRScram_IntegerLiteral::_publish_vhdl_range(ostream &_vhdl_out){
  _publish_vhdl( _vhdl_out );
}

void 
IIRScram_IntegerLiteral::_publish_vhdl(ostream &_vhdl_out) {
  register int i;
  IIR_Char* man = _get_mantissa();
  IIR_Char* exp = _get_exponent();
  char force_paranthesis = 0;

  if (_is_negative() == TRUE) {
    force_paranthesis = 1;
    _vhdl_out << "(";
  }
  if (_get_base() != 10) {
    _vhdl_out << _get_base() << "#";
  }
  for (i = 0; i < _get_mantissa_length(); i++) {
    _vhdl_out << man[i];
  }
  if (_get_base() != 10) {
    _vhdl_out << "#";
  }
  if (_get_exponent() != NULL && _get_exponent_length() > 0 ) {
    _vhdl_out << "E";
    for (i = 0; i < _get_exponent_length(); i++) {
      _vhdl_out << exp[i];
    }
  }
  if (force_paranthesis == 1) {
    _vhdl_out << ")";
  }
}

ostream &
IIRScram_IntegerLiteral::_print( ostream &os ) {
  register int i;
  IIR_Char* man = _get_mantissa();
  IIR_Char* exp = _get_exponent();

  if (_get_base() != 10) {
    os << _get_base() << "#";
  }
  for (i = 0; i < _get_mantissa_length(); i++) {
    os << man[i];
  }
  if (_get_base() != 10) {
    os << "#";
  }
  if (_get_exponent() != NULL) {
    os << "E";
    for (i = 0; i < _get_exponent_length(); i++) {
      os << exp[i];
    }
  }
  return os;
}

void 
IIRScram_IntegerLiteral::_publish_cc_data( published_file &_cc_out ) {
  _cc_out <<  print(0);
}

void 
IIRScram_IntegerLiteral::_publish_cc_condition( published_file &_cc_out ) {
  //_current_publiish_node is set during conditions in case statements
  ASSERT( _get_current_publish_node() != NULL );
  IIR_TypeDefinition* type = _get_current_publish_node()->get_subtype();
  if( type->_is_scalar_type() == TRUE &&
      type->_is_kernel_type() == FALSE ){
    type->_publish_cc_kernel_type( _cc_out );
  }
  else {
    _cc_out << type->_get_cc_type_name();
  }
  _cc_out << OS("(ObjectBase::VARIABLE,");
  _publish_cc_universal_value( _cc_out );
  _cc_out << CS(")");
}

void 
IIRScram_IntegerLiteral::_publish_cc_state_object_init( published_file &_cc_out ) {
  _cc_out << "new ";
  _publish_cc_lvalue( _cc_out );
}

void 
IIRScram_IntegerLiteral::_publish_cc_lvalue( published_file &_cc_out ) {
  ASSERT( get_subtype() != NULL );
  ASSERT( get_subtype()->_is_scalar_type() == TRUE );

  if (get_subtype()->_is_scalar_type() == TRUE) {
    get_subtype()->_publish_cc_kernel_type( _cc_out );
  }
  else {
    _cc_out << get_subtype()->_get_cc_type_name();
  }
  _cc_out << OS("(ObjectBase::VARIABLE,")
	  << OS("IntegerType(ObjectBase::VARIABLE,") 
	  << "(int) " << print(0) << "," << NL()
	  << "SavantintegerType_info" << CS("),")
	  << NL();
  
  ASSERT ( get_subtype()->_is_scalar_type() == TRUE );
  
  get_subtype()->_publish_cc_object_type_info( _cc_out, false );
  _cc_out << CS(")");
}

void
IIRScram_IntegerLiteral::_publish_cc_rvalue( published_file &_cc_out ){
  _publish_cc_universal_value( _cc_out );
}

void 
IIRScram_IntegerLiteral::_publish_cc_value( published_file &_cc_out ) {
  _cc_out << print(0);
}

void 
IIRScram_IntegerLiteral::_publish_cc_universal_value( published_file &_cc_out ) {
  _cc_out << "UniversalInteger(";
  _publish_cc_value( _cc_out );
  _cc_out << ")";
}

void 
IIRScram_IntegerLiteral::_publish_cc_initialization_value( published_file &_cc_out ) {
  _publish_cc_universal_value( _cc_out );
}

void 
IIRScram_IntegerLiteral::_publish_cc_elaborate( published_file &_cc_out ){
  _cc_out << print(0);
}

void
IIRScram_IntegerLiteral::_publish_cc_bounds( published_file &_cc_out ) {
  _cc_out << "defaultInfo";
}

void
IIRScram_IntegerLiteral::_publish_cc_wait_data( published_file &_cc_out ){
  _cc_out << "IntegerType(ObjectBase::VARIABLE, ";
  _publish_cc_universal_value( _cc_out );
  _cc_out << ")";
}

void
IIRScram_IntegerLiteral::_publish_cc_headers( published_file &_cc_out ) {
  _publish_cc_include( _cc_out, "tyvis/IntegerType.hh" );
}

const string
IIRScram_IntegerLiteral::_get_cc_type_name(){
  return "IntegerType";
}

void 
IIRScram_IntegerLiteral::_publish_cc_ams_function(published_file &_cc_out) {
  _publish_cc_data(_cc_out);
}

const string
IIRScram_IntegerLiteral::print(IIR_Int32 length) {
  ostringstream valbuf;
  ostringstream retbuf;
  string valstring;

  valbuf << _get_value();
  valstring = valbuf.str();

  if( (unsigned int)length > valstring.length() ) { 
    // pad with spaces if necessary
    for (unsigned int i = 0; i < length - valstring.length(); i++) {
      retbuf << " ";
    }
  }
  retbuf << valstring;

  return retbuf.str();
}

IIR_Int32
IIRScram_IntegerLiteral::_get_value() {
  register int i;
  IIR_Int32 value = 0;
  IIR_Int32 base = _get_base();
  IIR_Char* man = _get_mantissa();
  register int sign = 1;

  ASSERT(_get_mantissa_length() > 0);

  if(man[0] == '-') {
    sign = -1;
    value = sign * _char_to_int(man[1]);
  } else {
    sign = 1;
    value = _char_to_int(man[0]);
    if(_get_mantissa_length() > 1) {
      value = value * base + _char_to_int(man[1]);
    }
  }
  for (i = 2; i < _get_mantissa_length(); i++) {
    value = value * base + sign *_char_to_int(man[i]);
  }
  if (_get_exponent_length() > 0) {
    IIR_Char* exp = _get_exponent();
    IIR_Int32 exp_val;
    if (exp[0] == '+') {
      i = 2;
      exp_val = _char_to_int(exp[1]);
    }
    else if (exp[0] == '-') {
      i = 2;
      exp_val = _char_to_int(exp[1]);
    }
    else {
      i = 1;
      exp_val = _char_to_int(exp[0]);
    }
    for (/* i already set */; i < _get_exponent_length(); i++) {
      exp_val = exp_val * 10 + _char_to_int(exp[i]);
    }
    // exp_val is now the base-10 representation of the exponent.  The
    // value of this node is value * base^exp_val.  Care must be taken if
    // the exponent is < 0.
    if (exp[0] == '-') {
      for (i = 0; i < exp_val; i++) {
	value = value / base;
      }
    }
    else {
      for (i = 0; i < exp_val; i++) {
	value = value * base;
      }
    }
  }
  return value;
}

void
IIRScram_IntegerLiteral::_set_subtype( IIR_TypeDefinition *my_subtype ){
  ASSERT( my_subtype->_is_iir_integer_type_definition() == TRUE ||
	  my_subtype->_is_iir_physical_type_definition() == TRUE );
  IIRBase_IntegerLiteral::_set_subtype( my_subtype );
}

set<IIR_TypeDefinition> *
IIRScram_IntegerLiteral::_get_rval_set(IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *retval = 
    new set<IIR_TypeDefinition>( StandardPackage::get_savant_universal_integer() );
  return retval;
}

IIR_Boolean
IIRScram_IntegerLiteral::_is_integer_literal() {
  return true;
}

IIR_IntegerLiteral *
IIRScram_IntegerLiteral::_get_integer_static_value(){
  return get( _get_base(), _get_mantissa(), _get_mantissa_length(), _get_exponent(), _get_exponent_length() );
}

visitor_return_type *IIRScram_IntegerLiteral::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_IntegerLiteral(this, arg);
};

