
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Timothy J. McBrayer 
//          Krishnan Subramani  
//          Narayanan Thondugulam
//          Radharamanan Radhakrishnan
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram_DeclarationList.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_LibraryClause.hh"
#include "IIR_LibraryDeclaration.hh"
#include "IIR_SelectedName.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_UseClause.hh"
#include "IIR_DisconnectSpecification.hh"
#include "IIR_FileDeclaration.hh"
#include "IIR_TypeDeclaration.hh"
#include "error_func.hh"
#include "symbol_table.hh"
#include "set.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"
#include "IIR_NatureDeclaration.hh"
#include "IIR_TerminalDeclaration.hh"

IIRScram_DeclarationList::~IIRScram_DeclarationList() {}

void
IIRScram_DeclarationList::_publish_vhdl(ostream &_vhdl_out) {
  IIR_Declaration *decl = first();
  while (decl != NULL) {
    decl->_publish_vhdl(_vhdl_out);
    decl = successor(decl);
  }
}

void 
IIRScram_DeclarationList::_publish_vhdl_decl(ostream &_vhdl_out) {
  IIR_Declaration *decl = first();
  
  while (decl != NULL){
    decl->_publish_vhdl_decl(_vhdl_out);
    decl = successor(decl);
  }
}

IIR_Boolean
IIRScram_DeclarationList::_publish_cc_type_info( published_file &_cc_out, 
						 IIR_Boolean firstDeclFlag ){
  IIR_Declaration *decl;
  PublishedUnit oldUnit = _get_currently_publishing_unit();
  IIR_Boolean firstDecl = TRUE, semiColonFlag = FALSE;
  IIR_TypeDeclaration *typeDecl = NULL;
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      typeDecl = (IIR_TypeDeclaration *) decl;
      if (typeDecl->_is_incomplete_type_declaration() == FALSE) {
	// Access types and file types do not have type_info structures yet
	if ((typeDecl->_is_access_type() == FALSE) &&
	    (typeDecl->_is_file_type() ==  FALSE)) {
	  if ( _get_currently_publishing_unit() != FUNCTION &&
	       _get_currently_publishing_unit() != PROCEDURE &&
	       _get_currently_publishing_unit() != PACKAGE_PUB &&
	       _get_currently_publishing_unit() != PACKAGE_BODY ){
	    semiColonFlag = FALSE;
	    if ( firstDecl && firstDeclFlag ){
	      _cc_out << " :" << NL();
	      firstDecl = FALSE;
	    }
	    else {
	      firstDecl = FALSE;
	      _cc_out << "," << NL();
	    }
	  }
	  else {
	    semiColonFlag = TRUE;
	  }
	  
	  decl->_publish_cc_type_info( _cc_out );
	  if (semiColonFlag == TRUE) {
	    _cc_out.end_statement();
	  }
	}
      }
      break;
      
    default:
      break;
    }
    decl = successor(decl); 
  }

  
  _set_currently_publishing_unit(oldUnit);
  
  return firstDecl;
}

void
IIRScram_DeclarationList::_publish_cc_extern_type_info( published_file &_cc_out ){
  IIR_Declaration *decl;
  PublishedUnit oldUnit = _get_currently_publishing_unit();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_extern_type_info" );
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	decl->_publish_cc_extern_type_info( _cc_out );
      }
      break;

    default:
      break;
    }
    decl = successor(decl); 
  }
  
  _set_currently_publishing_unit(oldUnit);
}

void
IIRScram_DeclarationList::_publish_cc_type_info_and_constant_declarations( published_file &_cc_out ){
  IIR_Declaration *decl;
  PublishedUnit oldUnit = _get_currently_publishing_unit();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_type_info_and_constant_declaration" );
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      if (decl->_is_incomplete_type_declaration() == FALSE) {
	decl->_publish_cc_extern_type_info( _cc_out );
      }
      break;

    case IIR_CONSTANT_DECLARATION:
    case IIR_CONSTANT_INTERFACE_DECLARATION:
      ASSERT ( decl->_is_constant() ==  TRUE);
      _cc_out << decl->_get_cc_type_name();
      _cc_out << "  ";
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out.end_statement();
    default:
      break;
    }
    decl = successor(decl); 
  }
  
  _set_currently_publishing_unit(oldUnit);
}

IIR_Boolean
IIRScram_DeclarationList::_publish_cc_type_info_and_constant_definitions( published_file &_cc_out,
						 IIR_Boolean firstDeclFlag ){
  IIR_Declaration *decl;
  PublishedUnit oldUnit = _get_currently_publishing_unit();
  IIR_Boolean localFirstDecl = firstDeclFlag;
  IIR_Boolean semiColonFlag = FALSE;
  IIR_TypeDeclaration *typeDecl = NULL;
  
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      typeDecl = (IIR_TypeDeclaration *) decl;
      if (typeDecl->_is_incomplete_type_declaration() == FALSE) {
	// Access types and file types do not have type_info structures yet
	if ((typeDecl->_is_access_type() == FALSE) &&
	    (typeDecl->_is_file_type() ==  FALSE)) {
	  if ( _get_currently_publishing_unit() != FUNCTION &&
	       _get_currently_publishing_unit() != PROCEDURE &&
	       _get_currently_publishing_unit() != PACKAGE_PUB &&
	       _get_currently_publishing_unit() != PACKAGE_BODY ){
	    semiColonFlag = FALSE;
	    if( localFirstDecl == TRUE ) {
	      _cc_out << " :" << NL();
	    }
	    else {
	      _cc_out << "," << NL();
	    }
	    localFirstDecl = FALSE;
	  }
	  else {
	    semiColonFlag = TRUE;
	  }
	  
	  decl->_publish_cc_type_info( _cc_out );
	  if (semiColonFlag == TRUE) {
	    _cc_out.end_statement();
	  }
	}
      }
      break;

    case IIR_CONSTANT_DECLARATION:
      if( localFirstDecl == TRUE ){
	_cc_out << " : ";
      }
      else {
	_cc_out << "," << NL();
      }
      localFirstDecl = FALSE;
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      decl->_publish_cc_constructor_args( _cc_out );

      break;
      
    default:
      break;
    }
    decl = successor(decl); 
  }
  
  _set_currently_publishing_unit(oldUnit);
  
  return localFirstDecl;
}

void 
IIRScram_DeclarationList::_publish_cc( published_file &_cc_out ) {
  IIR_Declaration *decl;
  IIR_Boolean has_sub_prog = FALSE;

  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc" );

  // First collect all the includes needed for this declartive region
  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      has_sub_prog = TRUE;
      break;
    default:
      break;
    }
    decl = successor(decl);
  }

  // If the declarative region has sub programs then include these stuff
  if (has_sub_prog == TRUE) {
    _publish_cc_include( _cc_out, "tyvis/VHDLProcess.hh" );
  }
  // Walk through the collected list and publish the headers
  decl = first();
  while( decl != NULL ){
    switch( decl->get_kind() ){
    case IIR_COMPONENT_DECLARATION:
      decl->_publish_cc_include( _cc_out );
      break;
    case IIR_ATTRIBUTE_SPECIFICATION:
      _cc_out << "extern ";
      decl->_publish_cc_decl( _cc_out );
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      static_cast<IIR_SubprogramDeclaration *>(decl)->_publish_cc_prototype( _cc_out );
      break;
    case IIR_USE_CLAUSE:
      decl->_publish_cc_headers( _cc_out );
      break;
    case IIR_CONSTANT_DECLARATION:
      _cc_out << "extern ";
      _cc_out << decl->_get_cc_type_name();
      _cc_out << " ";
      decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out.end_statement();
      break;
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      //  if (decl->_is_incomplete_type_declaration() == FALSE) {
      //     decl->_publish_cc_decl( _cc_out );
      //   }
      break;

    case IIR_FILE_DECLARATION:
      IIRScram::_publish_cc_include(_cc_out, "tyvis/SharedFileType.hh");
      break;
      
    default:
      break;
    }
    decl = successor(decl);
  }
}

void 
IIRScram_DeclarationList::_publish_cc_decl( published_file &_cc_out ) {
  IIR_Declaration *decl;
  IIR_Int32 num_decls_in_file = 0;

  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_decl" );

  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
    case IIR_VARIABLE_DECLARATION:
    case IIR_SIGNAL_INTERFACE_DECLARATION:
    case IIR_VARIABLE_INTERFACE_DECLARATION:
    case IIR_FILE_DECLARATION:
    case IIR_CONFIGURATION_SPECIFICATION:
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
    case IIR_ALIAS_DECLARATION:
    case IIR_ATTRIBUTE_DECLARATION:
    case IIR_USE_CLAUSE:
    case IIR_TERMINAL_DECLARATION:
      break;

     case IIR_CONSTANT_DECLARATION:
       _cc_out << decl->_get_cc_type_name() << " ";
       decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
       _cc_out << "(ObjectBase::VARIABLE);" << NL();
       break;
       
    case IIR_COMPONENT_DECLARATION:
      decl->_publish_cc_elaborate( _cc_out );
      SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_decl" );
      break;
      
    case IIR_ATTRIBUTE_SPECIFICATION:
      static_cast<IIR_AttributeSpecification*>(decl)->_publish_cc_decl_with_constructor( _cc_out );
      break;

    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      decl->_publish_cc_decl( _cc_out );
      num_decls_in_file++;
      break;

    case IIR_QUANTITY_DECLARATION:
    case IIR_FREE_QUANTITY_DECLARATION:
    case IIR_BRANCH_QUANTITY_DECLARATION:
    case IIR_SOURCE_QUANTITY_DECLARATION:
      num_decls_in_file++;
      break;
      
    case IIR_NATURE_DECLARATION: {
      if ( static_cast<IIR_NatureDeclaration *>(decl)->get_nature()->get_kind() != 
	   IIR_ARRAY_SUBNATURE_DEFINITION ){
        _cc_out << "AMSType ";
        static_cast<IIR_NatureDeclaration*>(decl)->_get_reference_terminal()->_publish_cc_lvalue(_cc_out);
        _cc_out << "(ObjectBase::TERMINAL, \"";
        static_cast<IIR_NatureDeclaration*>(decl)->_get_reference_terminal()->_publish_cc_lvalue(_cc_out);
        _cc_out << "\")";
	_cc_out.end_statement();
        num_decls_in_file++;
      }
      else { 
	// found an array nature declaration.
	ostringstream err;                                                                 
	err << "IIRScram_DeclarationList::_publish_cc_decl. Unimplemented declaration type: " << decl->get_kind_text();
	report_error( decl, err.str() );
	abort();
      }
    }
      break;
    default:{
      ostringstream err;                                                                 
      err << "IIRScram_DeclarationList::_publish_cc_decl. Unimplemented declaration type: " << decl->get_kind_text();
      report_error( decl, err.str() );
      abort();
    }
    }
    decl = successor(decl);
  }
}

void 
IIRScram_DeclarationList::_publish_cc_package_declarations( published_file &_cc_out ) {
  IIR_Declaration *decl;
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_package_declarations" );

  decl = first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_COMPONENT_DECLARATION:
      decl->_publish_cc_elaborate( _cc_out );
      break;
    case IIR_TYPE_DECLARATION:
    case IIR_SUBTYPE_DECLARATION:
      decl->_publish_cc_extern_type_info( _cc_out );
      break;
    case IIR_PROCEDURE_DECLARATION:
    case IIR_FUNCTION_DECLARATION:
      ((IIR_SubprogramDeclaration *)decl)->_publish_cc_prototype( _cc_out );
      break;
    case IIR_ALIAS_DECLARATION:
      decl->_publish_cc_decl( _cc_out );
      break;
    case IIR_USE_CLAUSE:
    case IIR_ATTRIBUTE_DECLARATION:
    case IIR_TERMINAL_DECLARATION:
      break;
    case IIR_ATTRIBUTE_SPECIFICATION:
      _cc_out << "extern ";
      decl->_publish_cc_decl( _cc_out );
      break;      
    case IIR_CONSTANT_DECLARATION:
      _cc_out << "extern ";
      _cc_out << decl->get_subtype()->_get_cc_type_name();
      _cc_out << " ";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out.end_statement();
      break;
    case IIR_SIGNAL_DECLARATION:
      _cc_out << "extern ";
      _cc_out << decl->get_subtype()->_get_cc_type_name();
      _cc_out << " ";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out << "_info";
      _cc_out.end_statement();
      break;
    case IIR_FILE_DECLARATION:
      // A shared file declaration...
      _cc_out << "extern ";
      _cc_out << "SharedFileType* ";
      decl->_publish_cc_lvalue( _cc_out );
      _cc_out.end_statement();
      break;

    case IIR_NATURE_DECLARATION:
      if( static_cast<IIR_NatureDeclaration *>(decl)->get_nature()->get_kind() != IIR_ARRAY_NATURE_DEFINITION && 
          static_cast<IIR_NatureDeclaration *>(decl)->get_nature()->get_kind() != IIR_ARRAY_SUBNATURE_DEFINITION ){
	_cc_out << "extern AMSType ";
	static_cast<IIR_NatureDeclaration*>(decl)->_get_reference_terminal()->_get_declarator()->_publish_cc_lvalue(_cc_out);
	_cc_out.end_statement();
      }
      else {
	// code for array nature declaration. will be added later.
	ostringstream err;                                                                
	err << "IIRScram_DeclarationList::_publish_cc_package_declarations. Unimplemented declaration type: " << decl->get_kind_text();
	report_error( decl, err.str() );
	abort();
       }
       break;	

    default:
	ostringstream err;                                                                 
	err << "IIRScram_DeclarationList::_publish_cc_package_declarations. Unimplemented declaration type: " << decl->get_kind_text();
	report_error( decl, err.str() );
	abort();
    }
    decl = successor(decl);
  }
}

void 
IIRScram_DeclarationList::_publish_cc_decl_subprograms_and_types( published_file &_cc_out ) {
  IIR_Declaration *decl;

  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_decl_subprograms_and_types" );

  decl = first();
  while( decl != NULL ){
    if( decl->_is_subprogram() == TRUE ){
      decl->_publish_cc_decl( _cc_out );
    }
    if( decl->_is_type() == TRUE ){
      decl->_publish_cc_type_info( _cc_out );
    }
    decl = successor(decl);
  }
}  

void 
IIRScram_DeclarationList::_publish_cc_prototype( published_file &_cc_out ) {
  IIR_Declaration *decl;

  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_prototype" );

  decl = first();
  while( decl != NULL ){
    if( decl->_is_subprogram() == TRUE ){
      ((IIR_SubprogramDeclaration *) decl)->_publish_cc_prototype( _cc_out );
    }
    decl = successor(decl);
  }
}  

set<IIR_Declaration> *
IIRScram_DeclarationList::_find_declarations( IIR_Name *to_find ){
  ASSERT( to_find != NULL );
  set<IIR_Declaration> *retval = NULL;
  // The following assertion holds true because an indexed name doesn't generally
  // map into a declaration.  So, we can't expect to look at a list of declarations,
  // searching for an indexed name, and really find anything.
  ASSERT( to_find->get_kind() != IIR_INDEXED_NAME );
  
  switch( to_find->get_kind() ){
  case IIR_SIMPLE_NAME:{
    ASSERT( to_find->get_prefix() != NULL );
    ASSERT( to_find->get_prefix()->_is_iir_text_literal() == TRUE );
    IIR_TextLiteral *prefix = (IIR_TextLiteral *)to_find->get_prefix();
    retval = _find_declarations( prefix );
    break;
  }
  case IIR_SELECTED_NAME:{
    IIR_SelectedName *as_selected_name = (IIR_SelectedName *)to_find;
    ASSERT( to_find->get_prefix() != NULL );
    ASSERT( to_find->get_prefix()->_is_iir_name() == TRUE );
    
    set<IIR_Declaration> *found = _find_declarations( (IIR_Name *)to_find->get_prefix() );
    if( found == NULL ){
      retval = NULL;
    }
    else{
      ASSERT( as_selected_name->get_suffix() != NULL );
      ASSERT( as_selected_name->get_suffix()->_is_iir_name() == TRUE );
      set<IIR_Declaration> *current_set;
      retval = new set<IIR_Declaration>;

      IIR_Declaration *current_decl = found->get_element();
      while( current_decl != NULL ){
	current_set = 
	  current_decl->_find_declarations( (IIR_Name *)as_selected_name->get_suffix() );
	if( current_set != NULL ){
	  retval->add( current_set );
	  delete current_set;
	}
	current_decl = found->get_next_element();
      }
      if( retval->num_elements() == 0 ){
	delete retval;
	retval = NULL;
      }
    }
    delete found;
    break;
  }
  default:{
    ostringstream err;
    err << "Internal error in IIRScram_DeclarationList::_find_declarations - "
	<< " don't know how to find a " << to_find->get_kind_text();
    report_error( to_find, err.str() );
    abort();
  }
  }
  return retval;
}

set<IIR_Declaration> *
IIRScram_DeclarationList::_find_declarations( IIR_TextLiteral *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  ASSERT( to_find != NULL );
  
  IIR_Declaration *current = first();
  while( current != NULL ){
    if( IIR_TextLiteral::_cmp( current->get_declarator(), to_find ) == 0 ){
      retval->add( current );
    }
    else{
      if( current->get_kind() == IIR_TYPE_DECLARATION ){
	// We need to search the implicit declarations for whatever we're
	// looking for.
	IIR_TypeDeclaration *as_type_decl = (IIR_TypeDeclaration *)current;
	ASSERT( as_type_decl->_get_implicit_declarations() != NULL );
	IIR_Declaration *current_implicit;
	current_implicit = as_type_decl->_get_implicit_declarations()->get_element();
	while( current_implicit != NULL ){
	  if( IIR_TextLiteral::_cmp( current_implicit->get_declarator(), to_find ) == 0 ){
	    retval->add( current_implicit );
	  }
	  current_implicit = as_type_decl->_get_implicit_declarations()->get_next_element();
	}
	// If we have an enumeration type, we need to search for the our
	// culprit in the literal list.
	if( current->_is_enumeration_type() == TRUE ){
	  set<IIR_Declaration> *enumeration_literals = current->_find_declarations( to_find );
	  if( enumeration_literals != NULL ){
	    ASSERT( enumeration_literals->num_elements() == 1 );
	    retval->add( enumeration_literals );
	  }
	}
      }
    }
    current = successor( current );
  }
  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }
  return retval;
}

void
IIRScram_DeclarationList::_publish_cc_constants( published_file &_cc_out ) {
  IIR_Declaration *current_declaration = first();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_constants" );
  
  while( current_declaration != NULL ){
    if( current_declaration->_is_constant() == TRUE ){
      _cc_out << current_declaration->_get_cc_type_name();
      _cc_out << " ";
      current_declaration->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out.end_statement();
    }
    current_declaration = successor(current_declaration);
  }
}

void
IIRScram_DeclarationList::_publish_cc_global_constants_assignments( published_file &_cc_out ) {
  IIR_Declaration *current_declaration = first();

  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_global_constants_assignments" );
  
  while( current_declaration != NULL ){
    if( current_declaration->get_kind() == IIR_CONSTANT_DECLARATION ){
      _cc_out << "::";
      current_declaration->_get_declarator()->_publish_cc_lvalue( _cc_out );
      _cc_out << " = ";
      _cc_out << current_declaration->_get_cc_type_name();
      current_declaration->_publish_cc_constructor_args( _cc_out );
      _cc_out << ";" << NL();
    }
    current_declaration = successor(current_declaration);
  }
}

void
IIRScram_DeclarationList::_publish_cc_file_objects( published_file &_cc_out ) {
  IIR_Declaration *decl = first();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_file_objects" );
  
  while( decl != NULL ){
    if( decl->get_kind() == IIR_FILE_DECLARATION ){
      if( (decl->_get_declarative_region()->get_kind() == IIR_ARCHITECTURE_DECLARATION) ||
	  (decl->_get_declarative_region()->_is_iir_concurrent_statement() == TRUE) ){
	// A shared file declaration is necessary here...
	_cc_out << "  SharedFileType *";
	decl->_get_declarator()->_publish_cc_elaborate( _cc_out );
	_cc_out.end_statement();
      }
      else {
	_cc_out << decl->get_subtype()->_get_cc_type_name();
	_cc_out << " ";
	decl->_get_declarator()->_publish_cc_elaborate( _cc_out );
	_cc_out.end_statement();
      }
    }
    decl = successor(decl);
  }
}

void
IIRScram_DeclarationList::_publish_cc_file_objects_init( published_file &_cc_out ) {
  IIR_Declaration *decl = first();
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_file_objects_init" );
  
  while( decl != NULL ){
    if( decl->get_kind() == IIR_FILE_DECLARATION ){
      IIR_FileDeclaration *file_decl = (IIR_FileDeclaration *) decl;
      ASSERT ( file_decl->get_kind() == IIR_FILE_DECLARATION );
      
      if( (decl->_get_declarative_region()->get_kind() == IIR_ARCHITECTURE_DECLARATION) ||
	  (decl->_get_declarative_region()->_is_iir_concurrent_statement() == TRUE) ){
	// A shared file declaration...
	file_decl->_publish_cc_shared_file_decl( _cc_out );
      }      
      else  {
	decl->_get_declarator()->_publish_cc_lvalue( _cc_out );
	_cc_out << ".reset(";
	file_decl->get_file_logical_name()->_publish_cc_lvalue( _cc_out );
	_cc_out << ")";
	_cc_out.end_statement();
      }
    }
    decl = successor(decl);
  }
}

IIR_Boolean
IIRScram_DeclarationList::_publish_cc_constants_init( published_file &_cc_out, 
						      IIR_Boolean firstDecl) {
  IIR_Declaration *current_declaration = first();
  IIR_Boolean     flag = FALSE;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_DeclarationList::_publish_cc_constants_init" );

  while( current_declaration != NULL ){
    switch( current_declaration->get_kind() ){
    case IIR_CONSTANT_DECLARATION:
      if( firstDecl == TRUE ){
	_cc_out << " : ";
	firstDecl = FALSE;
      }
      else {
	_cc_out << "," << NL();
      }
      current_declaration->_get_declarator()->_publish_cc_lvalue( _cc_out );
      current_declaration->_publish_cc_constructor_args( _cc_out );
      flag = TRUE;
      break;
    default:
      break;
    }
    current_declaration = successor(current_declaration);
  }
  return firstDecl;
}

void 
IIRScram_DeclarationList::_type_check_configuration_specifications( IIR_List &statement_list ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_configuration_specification() == TRUE ){
      IIR_ConfigurationSpecification *as_specification;
      as_specification = (IIR_ConfigurationSpecification *)current_declaration;
      as_specification->_type_check( statement_list, 
				     IIRScram_ConfigurationSpecification::CONFIG_SPEC );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_type_check_disconnection_specifications( ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_disconnect_specification() == TRUE ){
      IIR_DisconnectSpecification *as_specification;
      as_specification = (IIR_DisconnectSpecification *)current_declaration;
      as_specification->_type_check( *(IIR_DeclarationList *)this );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_type_check_attribute_specifications( IIR_List &statement_list ){
  IIR_Declaration *current_declaration = first();
  while( current_declaration != NULL ){
    if( current_declaration->_is_iir_attribute_specification() == TRUE ){
      IIR_AttributeSpecification *as_specification;
      as_specification = (IIR_AttributeSpecification *)current_declaration;
      as_specification->_type_check( *((IIR_DeclarationList *)this) );
    }
    current_declaration = successor( current_declaration );
  }
}

void 
IIRScram_DeclarationList::_make_visible_as_context_list( symbol_table *sym_tab ){
  IIR_Declaration *current = first();
  while( current != NULL ){
    ASSERT( current->get_kind() == IIR_LIBRARY_CLAUSE || 
	    current->get_kind() == IIR_USE_CLAUSE );
    // Don't do anything with library clauses, we don't need them in the
    // symbol table and putting them there causes homograph problems.
    
//     if( current->get_kind() == IIR_LIBRARY_CLAUSE ){
//       IIR_LibraryClause *as_clause = (IIR_LibraryClause *)current;
//       if( as_clause->_get_library_declaration() != NULL ){
// 	sym_tab->make_visible( as_clause->_get_library_declaration() );
//       }
//     }
//     else{
    if( current->get_kind() == IIR_USE_CLAUSE ){
      IIR_UseClause *as_use_clause = (IIR_UseClause *)current;
      ASSERT( as_use_clause->get_selected_name() != NULL );
      current->_make_interface_visible( sym_tab );
    }
    current = successor( current );
  }
}

IIR_DeclarationList *
IIRScram_DeclarationList::_build_constrained_list( IIR_Boolean (IIR::*constraint_function)() ){
  ASSERT ( constraint_function != 0 );
  
  // Not using clone because clone copies the elements in the list.
  // We want a new list that contains the exact same elements.
  IIR_DeclarationList *cloned_list = new IIR_DeclarationList();
  //  copy_location( this, cloned_list );
  _listcopy( *cloned_list, *this );

  ASSERT( cloned_list->get_kind() == get_kind() );
  ASSERT( cloned_list->num_elements() == num_elements() );

  IIR_Declaration *next = NULL;
  IIR_Declaration *current = cloned_list->first();
  while( current != NULL ){
    next = cloned_list->successor( current );    
    if( (current->*constraint_function)() == FALSE ){
      cloned_list->remove( current );
    }
    current = next;
  }
  return cloned_list;
}

IIR *
IIRScram_DeclarationList::_clone(){
  IIR_DeclarationList *my_clone = new IIR_DeclarationList();
  IIR_List::_clone( my_clone );
  
  ASSERT( get_kind() == my_clone->get_kind() );
  
  return my_clone;
}

bool
IIRScram_DeclarationList::_contains_foreign_attribute(){
  bool retval = false;
  set<IIR_AttributeSpecification> *foreign_attributes = _find_foreign_attributes();
  if( foreign_attributes != 0 ){
    retval = true;
    delete foreign_attributes;
  }
  return retval;
}

void
IIRScram_DeclarationList::_set_declarative_region( IIR *new_declarative_region ){
  IIR_Declaration *current = first();
  while( current != 0 ){
    current->_set_declarative_region( new_declarative_region );
    current = successor( current );
  }
}

set<IIR_AttributeSpecification> *
IIRScram_DeclarationList::_find_foreign_attributes(){
  set<IIR_AttributeSpecification> *retval = 0;

  IIR_Declaration *current = first();
  while( current != 0 ){
    if( current->_is_foreign_attribute_specification() == TRUE ){
      if( retval == 0 ){
	retval = new set<IIR_AttributeSpecification>;
      }
      ASSERT( current->get_kind() == IIR_ATTRIBUTE_SPECIFICATION );
      retval->add( (IIR_AttributeSpecification *)current );
    }
    current = successor( current );
  }
  return retval;
}

visitor_return_type *
IIRScram_DeclarationList::_accept_visitor( node_visitor *visitor, 
					   visitor_argument_type *arg ) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_DeclarationList(this, arg);
}
