
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur    
//          Timothy J. McBrayer
//          Krishnan Subramani 
//          Narayanan Thondugulam
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram.hh"
#include "IIRScram_ComponentInstantiationStatement.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_Identifier.hh"
#include "IIR_IndexedName.hh"
#include "IIR_SelectedName.hh"
#include "IIR_Label.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_SimpleName.hh"
#include "IIR_TerminalInterfaceDeclaration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_ComponentConfiguration.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_DesignatorExplicit.hh"
#include "error_func.hh"
#include "set.hh"
#include "library_manager.hh"
#include "published_file.hh"
#include "sstream-wrap.hh"

using std::cerr;
using std::endl;

extern bool parse_error;

void 
IIRScram_ComponentInstantiationStatement::_publish_vhdl(ostream &_vhdl_out) {

  ASSERT( get_instantiated_unit() != NULL );
  
  _publish_vhdl_stmt_label(_vhdl_out);

  if( get_instantiated_unit()->_is_entity_decl() == TRUE ){
    _vhdl_out << "entity ";
  }
  else if( get_instantiated_unit()->_is_configuration_decl() == TRUE ){
    _vhdl_out << "configuration ";
  }
  else if( get_instantiated_unit()->_is_component_decl() == TRUE ){
    _vhdl_out << "component ";
  }
  
  get_instantiated_unit()->_publish_vhdl(_vhdl_out);
  if(generic_map_aspect.num_elements() != 0) {
    _vhdl_out << " generic map ( ";
    generic_map_aspect._publish_vhdl(_vhdl_out);
    _vhdl_out << " )";    
  }
  if(port_map_aspect.num_elements() != 0) {
    _vhdl_out << "\n port map ( ";
    port_map_aspect._publish_vhdl(_vhdl_out);
    _vhdl_out << " )";
  }
  _vhdl_out << ";\n";
}

IIRScram_ComponentInstantiationStatement::IIRScram_ComponentInstantiationStatement() : 
  my_instantiation_type( IIR_COMPONENT_DECLARATION ),
  my_configuration( NULL ){}

IIRScram_ComponentInstantiationStatement::~IIRScram_ComponentInstantiationStatement(){}

void 
IIRScram_ComponentInstantiationStatement::_set_instantiation_type( IIR_Kind instantiation_type ){
  my_instantiation_type = instantiation_type;
}

IIR_TypeDefinition * 
IIRScram_ComponentInstantiationStatement::_get_port_type( int port_num ){
  return get_instantiated_unit()->_get_port_type( port_num );
}

IIR_Kind 
IIRScram_ComponentInstantiationStatement::_get_instantiation_type(){
  return my_instantiation_type;
}

void 
IIRScram_ComponentInstantiationStatement::_resolve_instantiated_unit_for_entity(){
  IIR *entity_name = 0;
  IIR *arch_name = 0;
  IIR_EntityDeclaration *entity_decl = 0;

  if( get_instantiated_unit()->get_kind() == IIR_INDEXED_NAME ){
    // The prefix of the indexed name is the entity.  The suffix is the
    // architecture.
    IIRScram_IndexedName *as_indexed_name = (IIR_IndexedName *)get_instantiated_unit();
    entity_name = as_indexed_name->get_prefix();
    arch_name = as_indexed_name->get_suffix();
      
    ASSERT( entity_name->_is_iir_name() == TRUE );
    entity_decl = library_manager::instance()->lookup_entity( TRUE, 
							      (IIR_Name *)entity_name,
							      _get_work_library() );
    if( entity_decl != NULL ){
      ASSERT( arch_name->_is_iir_name() == TRUE );
      IIR_ArchitectureDeclaration *arch_decl = 
	library_manager::instance()->lookup_architecture( TRUE,
							  entity_decl,
							  (IIR_Name *)arch_name );

      if( arch_decl == 0 ){
	report_undefined_symbol( arch_name );
      }
      else{
	as_indexed_name->set_prefix( entity_name->_decl_to_decl( entity_decl ) );
	as_indexed_name->set_suffix( arch_name->_decl_to_decl( arch_decl ) );
	
	ASSERT( as_indexed_name->_is_resolved() == TRUE );
	set_instantiated_unit( as_indexed_name );
      }
    }
    else{
      report_undefined_symbol( entity_name );
    }
  }
  else{
    entity_name = get_instantiated_unit();
    ASSERT( entity_name->_is_iir_name() == TRUE );
    entity_decl = library_manager::instance()->lookup_entity( TRUE, 
							      (IIR_Name *)entity_name,
							      _get_work_library() );
      
    set_instantiated_unit( get_instantiated_unit()->_decl_to_decl( entity_decl ) );
  }
}

void
IIRScram_ComponentInstantiationStatement::_resolve_instantiated_unit_for_component(){
  set<IIR_Declaration> *instantiated_unit_decls =
    get_instantiated_unit()->_symbol_lookup(&IIR::_is_iir_component_declaration);

  if( instantiated_unit_decls == NULL ){
    report_undefined_symbol( get_instantiated_unit() );
  }
  else{
    // This needs to get pushed into a separate method.
    switch( instantiated_unit_decls->num_elements() ){
    case 0:{
      ostringstream err;
      err << "|" << *get_instantiated_unit() << "| was not declared as a";
	
      switch( _get_instantiation_type() ){
      case IIR_ENTITY_DECLARATION:
	err << "n entity";
	break;
      case IIR_COMPONENT_DECLARATION:
	err << " component";
	break;
      case IIR_CONFIGURATION_DECLARATION:
	err << " configuration";
	break;
      default:
	err << " (unknown type)";
      }
      err << " in this scope.";
	
      report_error( this, err.str() );
      break;
    }
    case 1:{
      IIR_Declaration *instantiated_decl = instantiated_unit_decls->get_element();
      set_instantiated_unit( get_instantiated_unit()->_decl_to_decl( instantiated_decl ) );
	
      break;
    }
    default:{
      report_ambiguous_error( get_instantiated_unit(), instantiated_unit_decls );
      break;
    }
    }
  }
  delete instantiated_unit_decls;
}

void 
IIRScram_ComponentInstantiationStatement::_type_check_instantiated_unit(){
  ASSERT( get_instantiated_unit() != NULL );
    
  switch( _get_instantiation_type() ){
  case IIR_ENTITY_DECLARATION:{
    _resolve_instantiated_unit_for_entity();
    break;
  }
  case IIR_COMPONENT_DECLARATION:{
    _resolve_instantiated_unit_for_component();
    // We're also going to set up a default configuration here.  If there
    // is a configuration specification, it will come back in and set the
    // proper configuration.
    _set_configuration( _get_default_binding_indication() );
    break;
  }
  case IIR_CONFIGURATION_DECLARATION:{
    ostringstream err;
    err << "IIRScram_ComponentInstantiationStatement::_type_check() :ERROR " 
	<< "\n - configurations in instantiation stmt not handled yet\n";
    report_error( this, err.str() );
    break;
  }
  default:{
    ostringstream err;
    err << "IIRScram_ComponentInstantiationStatement::_type_check() :ERROR " 
	<< "\n - Illegal declaration type\n";
    report_error( this, err.str() );
    break;
  }
  }
}

void 
IIRScram_ComponentInstantiationStatement::_type_check(){
  // So, the instantiated unit should be resolved, unless there was an error.

  if( get_instantiated_unit() != NULL && get_instantiated_unit()->_is_resolved() == TRUE ){
    // The locals are in "get_instantiated_unit()->_get_generic_list()"
    generic_map_aspect._resolve_and_order( get_instantiated_unit()->_get_generic_list(), 0, this );
    // The locals are in "get_instantiated_unit()->_get_port_list()"
    port_map_aspect._resolve_and_order( get_instantiated_unit()->_get_port_list(), 0, this );
  }
}

void
IIRScram_ComponentInstantiationStatement::_publish_cc_binding_name(ostream& outstream) {
  ASSERT(get_instantiated_unit() != NULL);
  get_instantiated_unit()->_publish_cc_binding_name(outstream);
}

void
IIRScram_ComponentInstantiationStatement::_publish_createNetInfo( published_file &_cc_out ){
  ASSERT(get_instantiated_unit()->_is_iir_declaration());
  
  IIR_Declaration *decl = (IIR_Declaration *) get_instantiated_unit();
  IIR_AssociationElement *actual_clause = NULL;
  IIR *formal = NULL;
  IIR_Label *label = _get_label();

  if(decl->get_kind() == IIR_COMPONENT_DECLARATION) {
    actual_clause = port_map_aspect.first();
    if(actual_clause == NULL){
      return;
    }
    label->_publish_cc_elaborate( _cc_out.get_stream() );

    if (_get_currently_publishing_unit() == GENERATE_FOR) {
      _cc_out << "_elab_obj[i - generateLeft].buildSignalTree();" << NL();
    }
    else {
      _cc_out << "_elab_obj->buildSignalTree();" << NL();
    }
    
    while(actual_clause != NULL) {
      formal = actual_clause->get_formal();
      
      if (actual_clause->_get_actual() == NULL) {
	// This could be an open oprt. So flag an error and bail out
	cerr << "Warning - IIRScram_ComponentInstantiationStatement::"
	     << "_publish_createNetInfo() - open ports have not yet been"
	     << " handled.\n";
      }
      
      if (formal != NULL) {
	ASSERT(formal->_is_resolved() == TRUE);
	ASSERT(actual_clause->_get_actual() != NULL);
	
	switch(formal->_get_mode()) {
	case IIR_IN_MODE:
	  _publish_cc_elaborate_Add( _cc_out, actual_clause, label );
	  if (actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL)
	    _publish_cc_elaborate_UpType( _cc_out, actual_clause, label );
	  break;
	case IIR_OUT_MODE:
	  // Code for adding the resolution information tree
	  _publish_cc_elaborate_addChild( _cc_out, actual_clause, label );
	  if (formal->get_kind() == IIR_FUNCTION_CALL)
	    _publish_cc_elaborate_UpType( _cc_out, actual_clause, label );
	  break;
	case IIR_INOUT_MODE:
	  _publish_cc_elaborate_Add( _cc_out, actual_clause, label );
	  // Code for adding the resolution information tree
	  _publish_cc_elaborate_addChild( _cc_out, actual_clause, label );
	  if (formal->get_kind() == IIR_FUNCTION_CALL)
	    _publish_cc_elaborate_UpType( _cc_out, actual_clause, label );
	  break;
	default:
	  if(formal->get_kind() == IIR_TERMINAL_INTERFACE_DECLARATION) {
	    //do nothing
	  }
	  else {
	    ostringstream err;
	    err << "Buffer and linkage mode not yet supported in "
		<< "Component instantiation statements\n";
	    report_error( this, err.str() );
	    break;
	  }
	}
      }
      actual_clause = port_map_aspect.successor(actual_clause);
    }
  }
}

void 
IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_Add( published_file &_cc_out,
								     IIR_AssociationElement *actual_clause,
								     IIR_Label *label ){
  IIR* decl = actual_clause->get_formal();
  IIR_FunctionCall *upTypeConversionFn = NULL;
  IIR_FunctionCall *downTypeConversionFn = NULL;
  IIR_Declaration* tempDeclaration = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_Add" );
  
  if( actual_clause->_get_actual() != NULL &&
      actual_clause->_get_actual()->get_kind() != IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind() != IIR_FUNCTION_CALL ){
    _cc_out << OS("Add(");
    ASSERT(actual_clause->_get_actual() != NULL);
    if(actual_clause->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
      downTypeConversionFn = (IIR_FunctionCall *)actual_clause->_get_actual();
      downTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
    } 
    else {
      if( actual_clause->_get_actual()->_is_iir_declaration() == TRUE ){
	tempDeclaration = (IIR_Declaration *) actual_clause->_get_actual();
	_publish_cc_scoping_prefix( _cc_out.get_stream(),
				    tempDeclaration->_get_declarative_region(),
				    this);
      }
      else if( actual_clause->_get_actual()->_is_name() == TRUE ){
	tempDeclaration = actual_clause->_get_actual()->_get_prefix_declaration();
	_publish_cc_scoping_prefix( _cc_out.get_stream(), 
				    tempDeclaration->_get_declarative_region(),
				    this);
      }
      else {
	cerr << "Error : IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_addChild( published_file &_cc_out ) - declaration kind not handled yet.\n";
	abort();
      }
      actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
    }
    
    _cc_out << "," << NL();
    label->_publish_cc_elaborate( _cc_out.get_stream() );
    
    if (_get_currently_publishing_unit() == GENERATE_FOR) {
      _cc_out << "_elab_obj[i - generateLeft].";
    }
    else {
      _cc_out << "_elab_obj->";
    }
    
    if(decl->get_kind() == IIR_FUNCTION_CALL) {
      upTypeConversionFn = (IIR_FunctionCall *)decl;
      upTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
    } else {
      decl->_publish_cc_elaborate( _cc_out );
    }    
    _cc_out << CS(");");
    
    if(upTypeConversionFn != NULL) {
      _cc_out << OS("setUpConversionFunctionId(");
      label->_publish_cc_elaborate( _cc_out.get_stream() );
      if (_get_currently_publishing_unit() == GENERATE_FOR) {
	_cc_out << "_elab_obj[i - generateLeft].";
      }
      else {
	_cc_out << "_elab_obj->";
      }
      
      upTypeConversionFn->_publish_cc_elaborate_arg( _cc_out );
      _cc_out << "," << NL();
      upTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name( _cc_out );
      _cc_out << CS(");");
    }
  }
}

void 
IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_addChild( published_file &_cc_out,
									  IIR_AssociationElement *actual_clause,
									  IIR_Label *label) {
  IIR* decl = actual_clause->get_formal();
  IIR_Declaration* tempDeclaration;

  if (actual_clause->_get_actual() == NULL) {
    cerr << "Warning: the actual in the association was NULL. Assuming to be"
	 << "open clause and ignoring.\n";
    return;
  }
 
  if( actual_clause->_get_actual()->get_kind() != IIR_FUNCTION_CALL &&
      actual_clause->get_formal()->get_kind() != IIR_FUNCTION_CALL ){
    SCRAM_CC_REF( _cc_out, "IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_addChild" );

    _cc_out << OS("addChild(");

    if(actual_clause->_get_actual()->_is_iir_declaration() ){
	_publish_cc_scoping_prefix( _cc_out.get_stream(),
				    static_cast<IIR_Declaration *>(actual_clause->_get_actual())->_get_declarative_region(),
				    this );
    }
    else if( actual_clause->_get_actual()->_is_name() ){
      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
				  actual_clause->_get_actual()->_get_prefix_declaration(),
				  this );
    }
    else {
      cerr << "Error : IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_addChild( published_file &_cc_out ) - declaration kind not handled yet.\n";
      abort();
    }
    actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
  
    _cc_out << "," << NL();
    label->_publish_cc_elaborate( _cc_out.get_stream() );

    if (_get_currently_publishing_unit() == GENERATE_FOR) {
      _cc_out << "_elab_obj[i - generateLeft].";
    }
    else {
      _cc_out << "_elab_obj->";
    }
  
    if ( decl->_is_iir_declaration() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl;
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
    }
    else if ( decl->_is_name() == TRUE ) {
      tempDeclaration = (IIR_Declaration *) decl->_get_prefix_declaration();
      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
      _publish_cc_scoping_prefix( _cc_out.get_stream(), tempDeclaration->_get_declarative_region(),this );
    }
    else {
      cerr << "Error : IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_addChild( published_file &_cc_out ) - declaration kind not handled yet.\n";
      abort();
    }
    
    decl->_publish_cc_elaborate( _cc_out );
    _cc_out << CS(");");
  }
}

void
IIRScram_ComponentInstantiationStatement::_publish_cc_connect_call( published_file &_cc_out ){
  ostringstream objectName;

  objectName << *_get_label() << "_elab_obj";
  _set_current_elab_name( objectName.str() );
  
  _publish_connect( _cc_out );
}

void
IIRScram_ComponentInstantiationStatement::_publish_connect( published_file &_cc_out ){
  IIR_Label *label = _get_label();
  int noofinputsignals =0;
  int noofoutputsignals =0;
  IIR_Boolean first = false;
  IIR_Declaration* tempDeclaration = NULL;
  ASSERT(get_instantiated_unit()->_is_iir_declaration() == TRUE);

  IIR_AssociationElement *association = port_map_aspect.first();
  while(association != NULL) {
    if (association->get_formal() != NULL) {
      switch (association->get_formal()->_get_mode()) {
      case IIR_OUT_MODE:
	if( association->get_formal()->get_kind() == IIR_FUNCTION_CALL ){
	  _publish_cc_elaborate_DownType( _cc_out, association,label);
	}
	break;
      case IIR_IN_MODE:
      case IIR_INOUT_MODE:
	if (association->_get_actual() != NULL) {
	  if( association->_get_actual()->get_kind()==IIR_FUNCTION_CALL ){
	    _publish_cc_elaborate_DownType( _cc_out, association, label );
	  }
	}
	break;
      default:
	break;
      }
    }
    association = port_map_aspect.successor(association);
  }

  association = port_map_aspect.first(); 
  if(association == NULL){
    label->_publish_cc_elaborate( _cc_out.get_stream() );
    if (_get_currently_publishing_unit() == GENERATE_FOR) {
      _cc_out << "_elab_obj[i - generateLeft].connect(0, 0);" << NL();
    }
    else {
      _cc_out << "_elab_obj->connect(0,0);" << NL();
    }
    return;
  }
  label->_publish_cc_elaborate( _cc_out.get_stream() );
  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << OS("_elab_obj[i - generateLeft].connect(");
  }
  else {
    _cc_out << OS("_elab_obj->connect(");
  }

  //Find out if the formals and actuals of port_map_aspect are present
  //If they are present use them otherwise get the order from component decl
  while(association != NULL) {
    if (association->get_formal() != NULL) {
      switch(association->get_formal()->_get_mode()) {
      case IIR_IN_MODE:
	noofinputsignals++;
	first = true;
	break;
      case IIR_OUT_MODE:
      case IIR_INOUT_MODE:
	noofoutputsignals++;
	first = true;
	break;

      default:
	break;
      }
    }
    association = port_map_aspect.successor(association);
  }
  
  _cc_out << noofinputsignals << "," << NL() << noofoutputsignals;
  
  if(first == true) {
    association = port_map_aspect.first();  
    while(association != NULL) {
      if (association->get_formal() != NULL) {
	switch(association->get_formal()->_get_mode()) {
	case IIR_IN_MODE:
	  _cc_out << "," << NL();
	  if (association->_get_actual()->get_kind()!=IIR_FUNCTION_CALL){
	    _cc_out << "&";
	  }
	  
	  ASSERT(association->_get_actual() != NULL);
	  if (association->_get_actual()->get_kind() == IIR_FUNCTION_CALL) {
	    if (association->get_formal()->get_kind()==IIR_FUNCTION_CALL){
	      ((IIR_FunctionCall *)association->get_formal())->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
	    }
	    else{
	      association->get_formal()->_publish_cc_elaborate( _cc_out );
	    }
	    _cc_out << "_tmpSigNetInfo";
	  }
	  else if (association->_get_actual()->get_kind() == IIR_INDEXED_NAME){
	    IIR_IndexedName *tempIdxName = (IIR_IndexedName *) association->_get_actual();
	    ASSERT ( tempIdxName->get_kind() == IIR_INDEXED_NAME );
	    if (tempIdxName->get_prefix()->_is_iir_declaration() == TRUE) {
	      tempDeclaration = (IIR_Declaration *) tempIdxName->get_prefix();
	      ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
	      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
					  tempDeclaration->_get_declarative_region(),
					  _get_enclosing_scope());
	    }
	    association->_get_actual()->_publish_cc_elaborate( _cc_out );
	  }
	  else {
	    ASSERT ( association->_get_actual()->_is_iir_declaration() == TRUE );
	    IIR_Declaration *decl = (IIR_Declaration *) association->_get_actual();
	    
	    _publish_cc_scoping_prefix( _cc_out.get_stream(), decl->_get_declarative_region(), 
					_get_enclosing_scope());
	    decl->_publish_cc_elaborate( _cc_out );
	  }
	  break;

	default:
	  break;
	}
      }
      association = port_map_aspect.successor(association);
    }
    
    association = port_map_aspect.first();  
    for(;association != NULL; ) {
      if (association->_get_actual() == NULL) {
	cerr << "Warning: actual in port map was NULL. Assuming open clause"
	     << " and generating defaults.\n";
	ASSERT ( association->get_formal() != NULL );
	ASSERT ( association->get_formal()->get_subtype() != NULL );
	
	_cc_out << "," << NL() << "new " 
		<< association->get_formal()->get_subtype()->_get_cc_type_name();
	_cc_out << "(ObjectBase::SIGNAL_NETINFO)";
	association = port_map_aspect.successor(association);
	continue;
      }

      if (association->get_formal() != NULL) {
	switch(association->get_formal()->_get_mode()) {
	case IIR_OUT_MODE:
	case IIR_INOUT_MODE:
	  _cc_out << "," << NL();
	  if (association->get_formal()->get_kind() != IIR_FUNCTION_CALL) {
	    _cc_out << "&";
	  }
	  
	  if (( association->_get_actual() &&
		association->_get_actual()->get_kind() == IIR_FUNCTION_CALL ) ||
	      association->get_formal()->get_kind() == IIR_FUNCTION_CALL ) {
	    if( association->get_formal()->_get_mode() == IIR_INOUT_MODE ){
	      if (association->get_formal()->get_kind()==IIR_FUNCTION_CALL){
		((IIR_FunctionCall *)association->get_formal())->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
	      }
	      else{
		association->get_formal()->_publish_cc_elaborate( _cc_out );
	      }
	      _cc_out << "_tmpSigNetInfo";
	    }
	    if( association->get_formal()->_get_mode()==IIR_OUT_MODE ){
	      if( association->get_formal()->get_kind()==IIR_FUNCTION_CALL ){
		((IIR_FunctionCall *)association->get_formal())->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
	      }
	      else {
		association->get_formal()->_publish_cc_elaborate( _cc_out );
	      }
	      _cc_out << "_tmpSigNetInfo";
	    }  
	  }
	  else {
	    if (association->_get_actual()->_is_iir_declaration() == TRUE) {
	      tempDeclaration = (IIR_Declaration *) association->_get_actual();
	      ASSERT (tempDeclaration->_is_iir_declaration() == TRUE );
	      _publish_cc_scoping_prefix( _cc_out.get_stream(), 
					  tempDeclaration->_get_declarative_region(),
					  _get_enclosing_scope());
	    }
	    else if (association->_get_actual()->get_kind() == IIR_INDEXED_NAME) {
	      IIR_IndexedName *tempIdxName = (IIR_IndexedName *) association->_get_actual();
	      ASSERT ( tempIdxName->get_kind() == IIR_INDEXED_NAME );
	      if (tempIdxName->get_prefix()->_is_iir_declaration() == TRUE) {
		tempDeclaration = (IIR_Declaration *) tempIdxName->get_prefix();
		ASSERT ( tempDeclaration->_is_iir_declaration() == TRUE );
		_publish_cc_scoping_prefix( _cc_out.get_stream(),
					    tempDeclaration->_get_declarative_region(),
					    _get_enclosing_scope());
	      }
	    }
	    
	    association->_get_actual()->_publish_cc_elaborate( _cc_out );
	  }
	  
	  break;

	default:
	  break;
	}
      }
      association = port_map_aspect.successor(association);
    }
  }
  _cc_out << CS(");");
}

void
IIRScram_ComponentInstantiationStatement::_publish_connect_terminals(published_file &_cc_out) {
  // local variables needed for publishing this node ...
  //  ASSERT(get_instantiated_unit()->_is_terminal() == TRUE);
  ASSERT(get_instantiated_unit()->_is_iir_declaration() == TRUE);
      
  // The above assert uses a method that's no longer used.  The new version
  // ssems to use is_attribute and that's what I change it to - jkm3.6
  //The above method cannot use is_terminal and has to use is_iir_declarations
  // and so I am going to change it to that - jkm3.19
   
  IIR_Boolean first = false;
  IIR_AssociationElement* actual_clause = port_map_aspect.first();
  IIR* current_formal = actual_clause->get_formal();
  IIR_Label *label = _get_label();
  IIR_Int32 noofterminals = 0;
  IIR *port_map_element = NULL;
      
  label->_publish_cc_elaborate(_cc_out);
  _cc_out << "_elab_obj->connectTerminals(";
  //Find out if the formals and actuals of port_map_aspect are present
  //If they are present use them otherwise get the order from component decl
  ASSERT(current_formal != NULL);

  while(actual_clause != NULL) {
    if(actual_clause->get_formal()->get_kind() ==
       IIR_TERMINAL_INTERFACE_DECLARATION) {
      noofterminals++;
      first = true;
    }
  
    actual_clause = port_map_aspect.successor(actual_clause);
  }
  _cc_out << noofterminals; 
  
  if(first == true) {
    actual_clause = port_map_aspect.first();
    while(actual_clause != NULL) {
      port_map_element = actual_clause->_get_actual();
      ASSERT(port_map_element != NULL);
      switch(port_map_element->get_kind()) {
      case IIR_TERMINAL_INTERFACE_DECLARATION: {
        _cc_out << ", ";
        port_map_element->_publish_cc_lvalue(_cc_out);
      }
      break;
  // have to look if this condition will ever occur!
      case IIR_TERMINAL_DECLARATION: {
        _cc_out << ", &";
        port_map_element->_publish_cc_lvalue(_cc_out);
      }
      break;
      // this case is for the electrical'reference case !!
      /*case IIR_REFERENCE_ATTRIBUTE: {
    
        IIR_ReferenceAttribute *ref_attribute = ((IIR_ReferenceAttribute*)port_map_element);
  
        ASSERT(ref_attribute->get_prefix()->get_kind() == IIR_NATURE_DECLARATION);
  
        IIR_NatureDeclaration *nature_decl =
    
          ((IIR_NatureDeclaration*)ref_attribute->get_prefix());
      
        _cc_out << ", &";
        nature_decl->_get_reference_terminal()->_publish_cc(_cc_out);
      }
      break;*/
      default:
        // nothing to be done for the default case !!
      // Kat, dec 26, 1997
        break;
      }
      actual_clause = port_map_aspect.successor(actual_clause);
    }  
  }
  _cc_out << ", \"" ;
  get_label()->_print(_cc_out.get_stream());
  _cc_out << "\");" << NL();
          
}   

#ifdef PROCESS_GRAPH
void
IIRScram_ComponentInstantiationStatement::_publish_cc_driver_info( published_file &_cc_out ) {
  IIR_AssociationElement* actual_clause = port_map_aspect.first();
  while(actual_clause != NULL) {
    if (actual_clause->get_formal() != NULL) {
      if (actual_clause->_get_actual()->get_kind()!=IIR_FUNCTION_CALL &&
	  actual_clause->_get_actual()->get_kind()!= IIR_INDEXED_NAME &&
	  actual_clause->_get_actual()->get_subtype()->_is_scalar_type() == TRUE) {
	_cc_out << "fp << \"Signal:";
	actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
	_cc_out << " \"<< endl;" << endl;
	_publish_cc_scoping_prefix(actual_clause->_get_actual( _cc_out ), _get_enclosing_scope( _cc_out ));
	actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
	_cc_out << ".dump_connectivity_info(fp);" << endl;
	_publish_cc_scoping_prefix(actual_clause->_get_actual( _cc_out ), _get_enclosing_scope( _cc_out ));
      }
      actual_clause = port_map_aspect.successor(actual_clause);
    }
  }
}
#endif

#ifdef PROCESS_COMBINATION
void
IIRScram_ComponentInstantiationStatement::
_static_elaborate(IIR_ArchitectureDeclaration *arch,
		  IIR_DeclarationList *cfglist,
		  char *hier_location) {
  IIR_ComponentDeclaration *compDecl;
  IIR_ConfigurationSpecification *configSpec;
  
  char *comp_name, *cfg_name, *hier_ptr;
  IIR_Declaration *decl;
  int hier_length = 0;

  // Add to hierarchy string
  hier_length = strlen( hier_location );
  hier_ptr = hier_location + hier_length;
  for (int i = 0; i < get_label()->get_declarator()->get_text_length(); i++) {
    hier_ptr[i] = get_label()->get_declarator()->get_text()[i];
  }
  hier_ptr[get_label()->get_declarator()->get_text_length()] = '_';
  hier_ptr[get_label()->get_declarator()->get_text_length() + 1] = '\0';

  // get component being instantiated
  compDecl = (IIR_ComponentDeclaration*)get_instantiated_unit();
  ASSERT(compDecl->get_kind() == IIR_COMPONENT_DECLARATION);
  comp_name = compDecl->get_declarator()->_convert_to_string();
#ifdef DEBUG_ELAB
  cout << "Elaborating instance " << hier_location << " of component |"
       << comp_name << "|\n";
#endif
  // walk through arch_decl_part to find the correct configSpec
  decl = arch->architecture_declarative_part.first();
  while (decl != NULL) {
    if (decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      configSpec = (IIR_ConfigurationSpecification*)decl;
      IIR_ComponentDeclaration *name;
      name = (IIR_ComponentDeclaration*)configSpec->get_component_name();
      ASSERT (name->get_kind() == IIR_COMPONENT_DECLARATION);
      cfg_name = name->get_declarator()->_convert_to_string();
      if (!strcmp(cfg_name, comp_name)) {
	break;
      }
    }
    decl = arch->architecture_declarative_part.successor(decl);
  }
  if (decl == NULL) {
    cout << "ERROR: could not find configuration for component " 
	 << *((IIR_Declaration*)compDecl)->get_declarator() << " instantiated at "
	 << hier_location << "\n";
    exit(1);
  }
  else {
#ifdef DEBUG_ELAB
    cout << "Found configuration for component |" << comp_name << "|\n";
#endif
  }

  // OK, I've got everything I need now.  I have:
  //       compInstStmt (this)--has g&p maps (local->actual)
  //       compDecl (compDecl)--defines locals
  //       configSpec (configSpec)--has entity_aspect, g&p maps (formal->local)
  if (generic_map_aspect.num_elements() != 0) {
    cout << "ERROR:  Sorry, generics in component instantiation statements "
	 << "not supported during static elaboration yet\n";
    exit(1);
  }

  // This checks the limitations on statically elaborated components
  compDecl->_static_elaborate(arch, cfglist, hier_location);

  // make the actual signal of the port map the clone of the formal.  The
  // local is needed to connect the two port maps together. The configSpec
  // maps formal to local, compInsStmt maps local to actual.
#define IIR_AEBX IIR_AssociationElementByExpression
  IIR_AEBX *formal_assoc, *actual_assoc;
  IIR_SignalInterfaceDeclaration *formal, *flocal, *alocal;
  IIR_Declaration *actual;
  
  formal_assoc = (IIR_AEBX*)configSpec->port_map_aspect.first();
  while (formal_assoc != NULL) {
    ASSERT(formal_assoc->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION);
    formal = (IIR_SignalInterfaceDeclaration*)formal_assoc->get_formal();
    flocal = (IIR_SignalInterfaceDeclaration*)formal_assoc->get_actual();
    ASSERT(formal->get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
    ASSERT(flocal->get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);

    actual_assoc = (IIR_AEBX*)port_map_aspect.first();
    while (actual_assoc != NULL) {
      ASSERT(actual_assoc->get_kind() ==IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION);
      alocal = (IIR_SignalInterfaceDeclaration*)actual_assoc->get_formal();
      ASSERT(alocal->get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION);
      if (0 == IIRScram_TextLiteral::_cmp(alocal->_get_declarator(), 
					  flocal->get_declarator() )) {
	break; // found the matching association
      }
      actual_assoc = (IIR_AEBX*)port_map_aspect.successor(actual_assoc);
    }
    if (actual_assoc == NULL) {
      ostringstream err;
      err << "Internal scram error: unable to locate actual for |"
	  << *formal->_get_declarator() << "| in port map for component |" 
	  << *compDecl->_get_declarator() << "|";
      report_error(this, err.str() );
      ASSERT(actual_assoc != NULL);
    }

    actual = (IIR_Declaration*)actual_assoc->get_actual();

    ASSERT( actual->get_kind() == IIR_SIGNAL_INTERFACE_DECLARATION
	    || actual->get_kind() == IIR_SIGNAL_DECLARATION
	    || actual->get_kind() == IIR_INDEXED_NAME );

    formal->_my_clone = (IIR_SignalInterfaceDeclaration*)actual->_clone();
#ifdef DEBUG_ELAB
    cout << "formal: " << *formal->_get_declarator() << "\tlocal: "
	 << *flocal->_get_declarator() << "\tactual: "
	 << *actual->_get_declarator() << "\n";
#endif

    IIR_Declaration *impclone, *impsig;

    if( formal->_get_implicit_declarations() != NULL ){
      impsig = formal->_get_implicit_declarations()->get_element();
      while (impsig != NULL) {
	impclone = (IIR_Declaration*)impsig->_clone();
	// figure out the correct name for the elaborated implicit signal
	IIR_TextLiteral *signame;
	signame = ((IIR_SignalDeclaration*)actual)->_clone()->_get_declarator();
	char *oldname = impclone->_get_declarator()->_convert_to_string();
	char *attrname = strrchr(oldname, '_');
	ASSERT(attrname != NULL);
	ostringstream str;
	str << *signame << attrname;
	char *newname = str.str();
	str.rdbuf()->freeze(0);
	impclone->set_declarator(IIR_Identifier::get(newname, strlen(newname)));
	if ( NULL == formal->_my_clone->_get_implicit_declarations() ) {
	  formal->_my_clone->_set_implicit_declarations( new set<IIR_Declaration> );
	}
	formal->_my_clone->_get_implicit_declarations()->add(impclone);
	impsig = formal->_get_implicit_declarations()->get_next_element();
      }
    }

    formal_assoc = 
      (IIR_AEBX*)configSpec->port_map_aspect.successor(formal_assoc);
  }
#undef IIR_AEBX

  // Elaborate the specified architecture
  configSpec->_static_elaborate(arch, cfglist, hier_location);

  hier_location[hier_length] = '\0';

#ifdef DEBUG_ELAB
  cout << "Completed elaboration of component instance; ascending to \"" 
       << hier_location << "\"\n";
#endif
}
#endif

void
IIRScram_ComponentInstantiationStatement::_publish_cc_concurrent_stmt_init( published_file &_cc_out,
									    IIR_DeclarationList *){

  IIR_ComponentDeclaration* component_name = NULL;

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentInstantiationStatement::_publish_cc_concurrent_stmt_init" );
  _cc_out << OS("{");
  ASSERT( get_instantiated_unit()->get_kind() == IIR_COMPONENT_DECLARATION );
  component_name = static_cast<IIR_ComponentDeclaration *>(get_instantiated_unit());

  // This is an overkill! this statement is required to handle component
  // instantiations that do not have component declarations.  The analyzer should/will
  // create a dummy component_declaration and hook it up.  However, this component
  // declaration will not appear in the list of component declarations in this
  // design unit.  So, we have to explicity publish the dummy component declaration
  // here.  This is brute force because currently there is no way to distinguish or
  // check if this component declaration has already been published or not.  Have to
  // discuss with Dale and resolve this issue..... -DJ
  component_name->_publish_cc_elaborate( _cc_out );
  
  if (_get_currently_publishing_unit() != GENERATE_FOR) {
    component_name->_publish_cc_include( _cc_out );
    _get_label()->_publish_cc_elaborate( _cc_out.get_stream() );
    _cc_out << "_elab_obj = new "
	    << component_name->_get_cc_elaboration_class_name()
	    << OS("(");
    _get_generic_map_aspect()->_publish_cc_lvalue( _cc_out );
    _cc_out << CS(");");
  }
  
  _cc_out << _get_current_publish_node()->_get_cc_elaboration_class_name()
	  << "* enclosingArch = this;" << NL();
  _get_label()->_get_declarator()->_publish_cc_lvalue( _cc_out.get_stream() );
  _cc_out << "_elab_obj";
  
  if (_get_currently_publishing_unit() == GENERATE_FOR) {
    _cc_out << "[i].boundedEntity = ";
  }
  else {
    _cc_out << "->boundedEntity = ";
  }
  
  if (_get_configuration() != NULL) {
    if ( _get_configuration()->get_kind() == IIR_CONFIGURATION_SPECIFICATION ){
      IIR_ConfigurationSpecification *config_spec_decl = 
	(IIR_ConfigurationSpecification*) _get_configuration();
      
      if(config_spec_decl == NULL) {
	_cc_out << "NULL;";	
      } 
      else if(config_spec_decl->get_entity_aspect() != NULL) {
	// Add an include for the entity aspect's elab file...
	config_spec_decl->get_entity_aspect()->_publish_cc_include_elab( _cc_out );
	_cc_out << "new "
		<< config_spec_decl->get_entity_aspect()->_get_cc_elaboration_class_name()
		<< "(";
	IIR* tempNode = _get_current_publish_node();
	_set_current_publish_node( this );
	// Set and resent current_published_unit to print the generic
	// constants based on the fact that a component_instantiation
	// is being generated...
	PublishedUnit oldUnit = _get_currently_publishing_unit();
	_set_currently_publishing_unit(NONE);
	config_spec_decl->generic_map_aspect._publish_cc_generic_map_aspect( _cc_out );
	
	IIR_Boolean commaFlag = FALSE;
	if (config_spec_decl->generic_map_aspect.first() != NULL) {
	  commaFlag = TRUE;
	}
	
	port_map_aspect._publish_cc_unconstrained_ports( _cc_out, commaFlag );
	_cc_out << ");" << NL();
	
	_set_currently_publishing_unit(oldUnit);
	config_spec_decl->_publish_cc_port_map_aspect( _cc_out );
	_set_current_publish_node( tempNode );
      } 
      else {
	_cc_out << " NULL;";
      }
    }
    else if (_get_configuration()->get_kind() == IIR_COMPONENT_CONFIGURATION ){
      IIR_ComponentConfiguration* config_spec_decl = (IIR_ComponentConfiguration*) _get_configuration();
      if(config_spec_decl == NULL) {
	_cc_out << " NULL;";	
      } 
      else if(config_spec_decl->get_entity_aspect() != NULL) {
	_cc_out << " new "
		<< config_spec_decl->get_entity_aspect()->_get_cc_elaboration_class_name()
		<< "(";
	IIR* tempNode = _get_current_publish_node();
	_set_current_publish_node( this );
	PublishedUnit oldUnit = _get_currently_publishing_unit();
	_set_currently_publishing_unit(NONE);
	config_spec_decl->generic_map_aspect._publish_cc_generic_map_aspect( _cc_out );
	_set_currently_publishing_unit(oldUnit);
	_cc_out << ");" << NL();
	if (config_spec_decl->get_entity_aspect()->_get_port_list() != NULL) {
	  const string old_elab_name = _get_current_elab_name();
	  ostringstream elabStream;
	  elabStream << *_get_declarator() << "_elab_obj->";
	  _set_current_elab_name( elabStream.str() );
	  _set_current_publish_node( get_instantiated_unit() );
	  config_spec_decl->get_entity_aspect()->_get_port_list()->_publish_cc_port_map_aspect( _cc_out );
	  _set_current_elab_name( old_elab_name );
	}
	_set_current_publish_node( tempNode );
      } 
      else {
	_cc_out << " NULL;";
      }
      _cc_out << NL();
    }
  }
  else {
    _cc_out << " NULL;" << NL();
    _cc_out << "cerr << \"Warning: Unbounded component.\" << endl;";
  }
 
  _cc_out << CS("}");
  
  port_map_aspect._set_passed_through_out_port(TRUE);
}

IIR_ConfigurationSpecification*
IIRScram_ComponentInstantiationStatement::_get_configuration_specification(IIR_DeclarationList* decl_list) {
  
  IIR_ComponentDeclaration* componentname = NULL;
  IIR_ConfigurationSpecification* config_spec_decl = NULL;

  ASSERT( get_instantiated_unit()->get_kind() == IIR_COMPONENT_DECLARATION );  
  componentname = (IIR_ComponentDeclaration *)get_instantiated_unit();

  // Searching for configuration specification
  IIR_Declaration* decl = decl_list->first();
  while(decl != NULL) {
    if(decl->_is_iir_configuration_specification() == TRUE) {
      config_spec_decl = (IIR_ConfigurationSpecification*)decl;
      if(config_spec_decl->_has_same_component((IIR_ComponentInstantiationStatement*)this) == TRUE) {
	if (_get_enclosing_scope()->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
	  return config_spec_decl;
	}
	if(config_spec_decl->_configures_this_component((IIR_ComponentInstantiationStatement*)this) == TRUE) {
	  return config_spec_decl;
	}
      }
    }
    decl = decl_list->successor(decl);
  }
  return NULL;
}

IIR_ConfigurationSpecification*
IIRScram_ComponentInstantiationStatement::_get_configuration_specification_from_any_scope(IIR_DeclarationList *decl_list) {
  // This method iteratively searches the outer scope(s) to locate a
  // configuration specification if one was not found in the inner scope
  IIR_ConfigurationSpecification *config_spec = NULL;
  IIR* current_scope                          = _get_enclosing_scope();
  
  do {
    config_spec = _get_configuration_specification(decl_list);
    if (config_spec != NULL) {
      return config_spec;
    }
    
    if (current_scope != NULL) {
      decl_list     = current_scope->_get_declaration_list();
      ASSERT ( decl_list != NULL );
      current_scope = current_scope->_get_enclosing_scope();
    }
    else {
      decl_list = NULL;
    }
  } while (decl_list != NULL);

  return NULL;
}

void
IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_UpType( published_file &_cc_out,
									IIR_AssociationElement *actual_clause, 
									IIR_Label *label ){
  IIR *formal = actual_clause->get_formal();
  IIR *actual = actual_clause->_get_actual();
  IIR_Mode mode = formal->_get_mode();
  IIR *formal_publish_target = formal;
  IIR_FunctionCall *upTypeConversionFn = NULL;
  if( formal->get_kind() == IIR_FUNCTION_CALL ){
    formal_publish_target = 
      static_cast<IIR_FunctionCall *>(formal)->parameter_association_list.first();
    upTypeConversionFn = static_cast<IIR_FunctionCall *>(formal);
  }

  IIR_FunctionCall *actual_as_function_call = NULL;
  if( actual->get_kind() == IIR_FUNCTION_CALL ){
    actual_as_function_call = static_cast<IIR_FunctionCall *>(actual);
  } 

  SCRAM_CC_REF(_cc_out, "IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_UpType");
  if( mode == IIR_IN_MODE || mode == IIR_INOUT_MODE ){
    _cc_out << OS("Addall(");
    actual_as_function_call->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
    _cc_out << "," << NL();
    label->_publish_cc_elaborate( _cc_out.get_stream() );
    _cc_out << "_elab_obj->";
    formal_publish_target->_publish_cc_elaborate( _cc_out );    
    _cc_out << CS(");");
  }

  if( mode == IIR_OUT_MODE || mode == IIR_INOUT_MODE  ){
    IIR_TypeDefinition *target_actual_subtype = NULL;
    if( mode == IIR_OUT_MODE ){
      target_actual_subtype = actual->get_subtype();
    }
    else if( mode == IIR_INOUT_MODE ){
      target_actual_subtype = actual_as_function_call->parameter_association_list.first()->get_formal()->get_subtype();
    }
    ASSERT( target_actual_subtype != NULL );

    // New TypeConvert ptr
    _cc_out << "TypeConvert *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr = new TypeConvert(this);" << NL();
      
    _cc_out << target_actual_subtype->_get_cc_type_name()
	    << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo = new " << target_actual_subtype->_get_cc_type_name();
    _cc_out << OS("(ObjectBase::SIGNAL_NETINFO");
    target_actual_subtype->_publish_cc_object_type_info( _cc_out );

    if( target_actual_subtype->_is_array_type() == TRUE ||
	 target_actual_subtype->_is_record_type() == TRUE  ){
      _cc_out << "," << NL();
      target_actual_subtype->_publish_cc_resolution_function_id(_cc_out);
    }
    
    if( target_actual_subtype->_is_unconstrained_array_type() == TRUE ){
      target_actual_subtype->_publish_cc_range( _cc_out );
    }
    _cc_out << CS(");");
    
    // Code added for handling arrays
    //----------------------------------------------
    
    //New variable of actual
    _cc_out << target_actual_subtype->_get_cc_type_name();
    _cc_out << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable = new ";

    target_actual_subtype = NULL;
    
    if( mode == IIR_OUT_MODE  ){
      target_actual_subtype = actual->get_subtype();
    }
    else if( mode == IIR_INOUT_MODE  ){
      target_actual_subtype = 
	actual_as_function_call->parameter_association_list.first()->get_formal()->get_subtype();
    }
    ASSERT (target_actual_subtype != NULL);

    _cc_out << target_actual_subtype->_get_cc_type_name();
    _cc_out << OS("(ObjectBase::VARIABLE");
    target_actual_subtype->_publish_cc_object_type_info( _cc_out );

    if( target_actual_subtype->_is_array_type() ||
	target_actual_subtype->_is_record_type() ){
      _cc_out << "," << NL();
      target_actual_subtype->_publish_cc_resolution_function_id(_cc_out);
    }
    
    if( target_actual_subtype->_is_unconstrained_array_type() == TRUE ){
      target_actual_subtype->_publish_cc_range( _cc_out );
    }
    _cc_out << CS(");");
    
    
    //New signal of formal
    if( mode == IIR_INOUT_MODE ){
      _cc_out << actual->get_subtype()->_get_cc_type_name();
    }
    else if( mode == IIR_OUT_MODE ){
      _cc_out << upTypeConversionFn->parameter_association_list.first()->get_formal()->get_subtype()->_get_cc_type_name();
    }
    _cc_out << "* ";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal = new ";

    target_actual_subtype = NULL;
    if( mode == IIR_INOUT_MODE ){
      target_actual_subtype = actual->get_subtype();
    }
    else if( mode == IIR_OUT_MODE ){
      target_actual_subtype = upTypeConversionFn->parameter_association_list.first()->get_formal()->get_subtype();
    }
    ASSERT ( target_actual_subtype != NULL );

    _cc_out << target_actual_subtype->_get_cc_type_name();
    _cc_out << OS("(ObjectBase::SIGNAL");
    target_actual_subtype->_publish_cc_object_type_info( _cc_out );

    if( target_actual_subtype->_is_array_type() ||
	target_actual_subtype->_is_record_type() ){
      _cc_out << "," << NL();
      target_actual_subtype->_publish_cc_resolution_function_id(_cc_out);
      
      if( target_actual_subtype->_is_unconstrained_array_type() == TRUE ){
	target_actual_subtype->_publish_cc_range( _cc_out );
      }
    }
    _cc_out << CS(");");
    
    //addTypeConversionFn(UptypeFn)
    _cc_out << "int ";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_id = addTypeConversionFn("; 
    upTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name( _cc_out );
    _cc_out << ");" << NL();

    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->setSignal(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal);" << NL();
    
    //TypeConvertptr->setVariable(actual_variable)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->setVariable(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable);" << NL();
 
    //TypeConvertptr->setSignalNetinfo(actual_signalnetinfo)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->setSignalNetinfo(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo);" << NL();
    
    //TypeConvertptr->setConversionFunctionId(formal_fnID)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->setConversionFunctionId(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_id);" << NL();

    //formal_signal->setElaborationinfo(formal)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal->setElaborationInfo(";

    label->_publish_cc_elaborate( _cc_out.get_stream() );
    _cc_out << "_elab_obj->";
    ((IIR_FunctionCall*)formal)->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
    _cc_out << ");" << NL();
    
    //addUpConvertDriver(*actual_signetinfo, *actual_variable, TCptr)
    _cc_out << "addUpConvertDriver(*";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo, *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable, ";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr);" << NL();
    
    SCRAM_CC_REF( _cc_out, "IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_UpType" );

    //addChild(actual, *actual_signetinfo)
    _cc_out << OS("addChild(");
    if( actual->get_kind() == IIR_FUNCTION_CALL ){
      actual_as_function_call->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
    }
    else {
      actual->_publish_cc_elaborate( _cc_out );
    }
    _cc_out << ", " << NL() <<  "*";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo" << CS(");");

    //addDriver(actual, *formal_signal)
    _cc_out << OS("addDriver(");
    if( actual->get_kind() == IIR_FUNCTION_CALL ){
      actual_as_function_call->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
    }
    else {
      actual_clause->_get_actual()->_publish_cc_elaborate( _cc_out );
    }
    _cc_out << "," << NL() << "*";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal" << CS(");");
  }
}

void 
IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_DownType( published_file &_cc_out,
									  IIR_AssociationElement *actual_clause,
									  IIR_Label * ){

  IIR *formal = actual_clause->get_formal();
  IIR *actual = actual_clause->_get_actual();
  IIR_Mode mode = formal->_get_mode();
  IIR *formal_publish_target = formal;
  if( formal->get_kind() == IIR_FUNCTION_CALL ){
    formal_publish_target = static_cast<IIR_FunctionCall *>(formal)->parameter_association_list.first();
  }
  //  ASSERT( actual->get_kind() == IIR_FUNCTION_CALL );
  IIR_FunctionCall *downTypeConversionFn = 0;
  if( actual->get_kind() == IIR_FUNCTION_CALL ){
    downTypeConversionFn = static_cast<IIR_FunctionCall *>(actual);
  }
  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ComponentInstantiationStatement::_publish_cc_elaborate_DownType" );
  if( mode == IIR_OUT_MODE  ){
    _cc_out << static_cast<IIR_FunctionCall *>(formal)->parameter_association_list.first()->get_formal()->get_subtype()->_get_cc_type_name();
    _cc_out << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo = new ";
    _cc_out << static_cast<IIR_FunctionCall *>(formal)->parameter_association_list.first()->get_formal()->get_subtype()->_get_cc_type_name();
    _cc_out << "(ObjectBase::SIGNAL_NETINFO);" << NL();
    
    _cc_out << OS("Addall(*");
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo," << NL();
    actual->_publish_cc_elaborate( _cc_out );
    _cc_out << CS(");");

  }

  if( mode == IIR_IN_MODE || mode == IIR_INOUT_MODE ){
    IIR_TypeDefinition *formal_subtype = NULL;
    if (formal_publish_target->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION)  {
      formal_subtype = static_cast<IIR_AssociationElementByExpression *>(formal_publish_target)->get_formal()->get_subtype();
    }
    else {
      formal_subtype = formal_publish_target->get_subtype();
    }
    
    ASSERT (formal_subtype != NULL);
    
    //New TypeConvert 
    _cc_out << "TypeConvert *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr = new TypeConvert(this);" << NL();
    
    //New SignalNetinfo of formal
    _cc_out << formal_subtype->_get_cc_type_name();
    _cc_out << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo = new ";
    _cc_out << formal_subtype->_get_cc_type_name();
    _cc_out << OS("(ObjectBase::SIGNAL_NETINFO");
    formal_subtype->_publish_cc_object_type_info(_cc_out, TRUE, "", TRUE);
    
    if ((formal_subtype->_is_scalar_type() == FALSE) &&
	(formal_subtype->_is_access_type() == FALSE)) {
      _cc_out << "," << NL();
      formal_subtype->_publish_cc_resolution_function_id(_cc_out);
    }
    
    _cc_out << CS(");");
    
    //New Variable of formal
    _cc_out << formal_subtype->_get_cc_type_name();
    _cc_out << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable = new ";
    _cc_out << formal_subtype->_get_cc_type_name();
    _cc_out << OS("(ObjectBase::VARIABLE");
    formal_subtype->_publish_cc_object_type_info(_cc_out, TRUE, "", TRUE);

    if ((formal_subtype->_is_scalar_type() == FALSE) &&
	(formal_subtype->_is_access_type() == FALSE)) {
      _cc_out << "," << NL();
      formal_subtype->_publish_cc_resolution_function_id(_cc_out);
    }
    _cc_out << CS(");");
      
    
    //New Signal of actual
    if( actual->get_kind() == IIR_FUNCTION_CALL)  {
      _cc_out << static_cast<IIR_FunctionCall *>(actual)->parameter_association_list.first()->get_formal()->get_subtype()->_get_cc_type_name();
    }
    else {
      _cc_out << actual->get_subtype()->_get_cc_type_name();  
    }
    _cc_out << " *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    IIR_TypeDefinition *target_actual_subtype = NULL;
    _cc_out << "_tmpSignal = new ";
    if( actual->get_kind()==IIR_FUNCTION_CALL)  {
      target_actual_subtype = static_cast<IIR_FunctionCall *>(actual)->parameter_association_list.first()->get_formal()->get_subtype();
      _cc_out << target_actual_subtype->_get_cc_type_name();
    }
    else {
      target_actual_subtype = actual->get_subtype();
      _cc_out << target_actual_subtype->_get_cc_type_name(); 
    }

    ASSERT ( target_actual_subtype != NULL );
    _cc_out << OS("(ObjectBase::SIGNAL");

    target_actual_subtype->_publish_cc_object_type_info( _cc_out );

    if( target_actual_subtype->_is_array_type() == TRUE ||
	target_actual_subtype->_is_record_type() == TRUE ){
      _cc_out << "," << NL();
      target_actual_subtype->_publish_cc_resolution_function_id( _cc_out );
      
      if( target_actual_subtype->_is_unconstrained_array_type() == TRUE ){
	_cc_out << "," << NL();
	target_actual_subtype->_publish_cc_range( _cc_out );
      }
    }
    
    _cc_out << CS(");");

    //addTypeConversionFn(DownTypeFn) 
    _cc_out << "int ";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_id";
    _cc_out << " = addTypeConversionFn("; 
    downTypeConversionFn->get_implementation()->_publish_cc_type_conversion_function_name( _cc_out );
    _cc_out << ");" << NL();

    //TCptr->setSignal(actual_signal)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->";
    _cc_out << "setSignal(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal);" << NL();
    
    //TCptr->setVariable(formal_variable)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->";
    _cc_out << "setVariable(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable);" << NL();
    
    //TCptr->setSignalNetinfo(formal_signetinfo)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->";
    _cc_out << "setSignalNetinfo(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo);" << NL();
    
    //TCptr->setConversionFunctionId(actual_fn_id)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr->";
    _cc_out << "setConversionFunctionId(";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_id);" << NL();

    //actual_signal->setElaborationInfo(actual)
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal->setElaborationInfo(";
    
    ((IIR_FunctionCall*)actual)->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
    _cc_out << ");" << NL();
    
    //addUpConvertDriver(*formal_signetinfo, *formal_variable, TCptr)
    _cc_out << "addUpConvertDriver(*";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo, *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigVariable, ";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_typeConvertptr);" << NL();
    
    //addDriver(formal, *actual_signal)
    _cc_out << "addDriver(*";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSigNetInfo, *";
    formal_publish_target->_publish_cc_elaborate( _cc_out );
    _cc_out << "_tmpSignal);" << NL();

    if( mode == IIR_INOUT_MODE ){
      _cc_out << OS("Addall( *");
      formal_publish_target->_publish_cc_elaborate( _cc_out );
      _cc_out << "_tmpSigNetInfo," << NL();
      
      if( actual->get_kind()==IIR_FUNCTION_CALL ){
	((IIR_FunctionCall*)actual)->parameter_association_list.first()->_publish_cc_elaborate( _cc_out );
      }
      else {
	actual->_publish_cc_elaborate( _cc_out );
      }
      _cc_out << CS(");");
    }
  }
}

void
IIRScram_ComponentInstantiationStatement::_publish_form_characteristic_expressions(published_file &_cc_out) {
  IIR_Label *label = _get_label();
  label->_publish_cc_elaborate(_cc_out.get_stream());
  _cc_out << "_elab_obj->formCharacteristicExpressions();" << NL();

}
IIR_Label *
IIRScram_ComponentInstantiationStatement::_find_instantiate_label( IIR_SimpleName *to_find ){
  IIR_Label *retval = NULL;
  if( get_label() != NULL ){
    if( IIR_TextLiteral::_cmp( get_label()->get_declarator(), to_find ) == 0 ){
      retval = get_label();
    }
  }

  return retval;
}

IIR_ComponentDeclaration*
IIRScram_ComponentInstantiationStatement::_build_implicit_component_declaration(IIR_EntityDeclaration *entity,
										const string &compName) {
  IIR_ComponentDeclaration *componentDecl = new IIR_ComponentDeclaration;
  
  copy_location(this, componentDecl);
  componentDecl->local_generic_clause = entity->generic_clause;
  componentDecl->local_port_clause    = entity->port_clause;
  componentDecl->set_declarator(IIR_Identifier::get( compName ));

  return componentDecl;
}

IIR_ConfigurationSpecification*
IIRScram_ComponentInstantiationStatement::_build_implicit_configuration_specification( IIR_LibraryUnit *instantiatedUnit, 
										       IIR_ComponentDeclaration *compDecl) {
  IIR_ConfigurationSpecification *config_spec = new IIR_ConfigurationSpecification;
  copy_location(this, config_spec);
  config_spec->set_component_name(compDecl);
  config_spec->set_entity_aspect(instantiatedUnit);

  IIR_DesignatorExplicit *designator = new IIR_DesignatorExplicit();
  designator->set_name( get_label() );
  config_spec->instantiation_list.append( designator );


  return config_spec;
}

void
IIRScram_ComponentInstantiationStatement::_add_to_configuration_specification(IIR_DeclarationList *searchList, IIR_ComponentDeclaration *compDecl, IIR_ComponentInstantiationStatement *newStmt) {
  IIR_ConfigurationSpecification *configSpec = NULL;
  IIR_Declaration                *decl;
  
  decl = searchList->first();
  while (decl != NULL) {
    if (decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      configSpec = (IIR_ConfigurationSpecification *) decl;
      if (configSpec->get_component_name() == compDecl) {
	break;
      }
      configSpec = NULL;
    }
    decl = searchList->successor(decl);
  }
  
  if (configSpec == NULL) {
    cerr << "Error :- Unable to locate configuration specification for"
	 << " anonymous component declaration.\n";
    abort();
  }
  
  ASSERT (configSpec != NULL);
  
  IIR_DesignatorExplicit *newDesignator = new IIR_DesignatorExplicit;
  copy_location(this, newDesignator);
  newDesignator->set_name(newStmt->get_label());
  configSpec->instantiation_list.append(newDesignator);
}
  
IIR *
IIRScram_ComponentInstantiationStatement::_transmute() {
  // Code to transmute direct component instantiations to a component
  // declaration, configuration specification and a new instantiation.

  ASSERT ( get_instantiated_unit() != 0 );  
  if( get_instantiated_unit()->_is_component_decl() ){
    return this;
  }

  IIR_ArchitectureDeclaration *arch = 0;
  IIR_EntityDeclaration *entity = 0;
  if( get_instantiated_unit() != 0 ){
    arch = get_instantiated_unit()->_get_cc_instantiated_architecture();
    entity = get_instantiated_unit()->_get_cc_instantiated_entity();
  }
  ASSERT( entity != 0 );

  IIR_ConfigurationSpecification *config_spec = 0;  
  if( arch == 0 ){
    // No architecture was specified, let's see if there was an explicit
    // configuration specification.
    if( _get_configuration() != 0 ){
      ASSERT( _get_configuration()->_is_iir_configuration_specification() );
      config_spec = static_cast<IIR_ConfigurationSpecification *>(_get_configuration());
    }
    else{
      // We'll go find the default arch for the entity then.
      arch = library_manager::instance()->lookup_default_architecture( entity );
      if( arch == 0 ){
	string err( "Could not find a default architecture for entity |" +
		    entity->get_declarator()->_to_string() + "| and no architecure was " +
		    "explicitly specified." );
	report_error( this, err );
	return this;
      }
    }
  }

  ASSERT (entity != 0);

  ostringstream component_stream;
  string componentName;
  component_stream << "AC" << entity << "_" << arch;
  componentName = component_stream.str();
  
  IIR_ComponentDeclaration *component_decl = 0;
  IIR_DeclarationList *enclosingDecls = 0;
  set<IIR_Declaration> *compSet = 0;
  
  enclosingDecls = _get_enclosing_scope()->_get_declaration_list();
  ASSERT (enclosingDecls != 0);
  
  compSet = enclosingDecls->_find_declarations( IIR_Identifier::get(componentName) );
  if( compSet == 0 ){
    // A component declaration and cofiguration specification does not exist
    // so create a new one and add them to the list.
    component_decl = _build_implicit_component_declaration( entity, componentName );
    enclosingDecls->append(component_decl);

    if( config_spec == 0 ){
      ASSERT( arch != 0 );
      config_spec = _build_implicit_configuration_specification( arch, component_decl );
      config_spec->generic_map_aspect = generic_map_aspect;
      config_spec->port_map_aspect    = port_map_aspect;
      enclosingDecls->append( config_spec );
    }
  }
  else {
    ASSERT ( compSet->get_element()->get_kind() == IIR_COMPONENT_DECLARATION );
    component_decl = (IIR_ComponentDeclaration *) compSet->get_element();
  }
  
  IIR_ComponentInstantiationStatement *newStmt = new IIR_ComponentInstantiationStatement;
  
  copy_location(this, newStmt);
  newStmt->set_label(get_label());
  newStmt->generic_map_aspect    = generic_map_aspect;
  newStmt->port_map_aspect       = port_map_aspect;
  newStmt->set_instantiated_unit( component_decl );
  newStmt->_set_configuration( config_spec );
  
  _add_to_configuration_specification( enclosingDecls, component_decl, newStmt );

  return newStmt;
}

IIR_ConfigurationSpecification *
IIRScram_ComponentInstantiationStatement::_get_default_binding_indication(){
  // This method attempts to implement the rules found on page 79 of the
  // '93 LRM.
  IIR_ConfigurationSpecification *retval = 0;
  ASSERT( get_instantiated_unit() != 0 );
  ASSERT( _get_instantiation_type() == IIR_COMPONENT_DECLARATION );
  
  IIR *component_name = get_instantiated_unit();

  IIR_ArchitectureDeclaration *arch = 0;
  IIR_EntityDeclaration *entity = library_manager::instance()->lookup_entity( FALSE, 
									      component_name,
									      _get_work_library() );
  if( entity != 0 ){
    arch = library_manager::instance()->lookup_default_architecture( entity );
    // So, we found an entity/arch that matched.  Now we'll build an
    // IIR_ConfigurationSpec to return
    if( arch != 0 ){
      retval = _build_default_entity_aspect( component_name, arch );
    }
    else{
      ostringstream err;
      err << "Component |" << *component_name << "| has no explicit binding indication,"
	  << " and the default binding is broken.  While entity |" << *entity << "| "
	  << "is a match by name, there is no cooresponding architecture";
      report_error( this, err.str() );
    }
  }
  

  return retval;
}

IIR *
IIRScram_ComponentInstantiationStatement::_get_configuration(){ return my_configuration; }

void 
IIRScram_ComponentInstantiationStatement::_set_configuration( IIR *new_config ){ 
  if( my_configuration != 0 ){
    if( my_configuration->_is_iir_declaration() == TRUE &&
	((IIR_Declaration *)my_configuration)->_is_implicit_declaration() == true ){
      // Then we generated it, we need to delete it.
      delete my_configuration;
    }
  }
  my_configuration = new_config; 
}

IIR_ConfigurationSpecification *
IIRScram_ComponentInstantiationStatement::_build_default_entity_aspect(IIR *component_name,
								       IIR_ArchitectureDeclaration *arch){
  ASSERT( arch->_get_entity() != 0 );

  IIR_ConfigurationSpecification *retval = new IIR_ConfigurationSpecification();
  copy_location( this, retval );
  copy_location( this, &retval->port_map_aspect );
  copy_location( this, &retval->generic_map_aspect );
  retval->_set_design_file( _get_design_file() );
  retval->set_component_name( get_instantiated_unit() );
  retval->set_entity_aspect( arch );
  retval->_set_is_implicit( true );
  retval->port_map_aspect._build_default_map( static_cast<IIR_ComponentInstantiationStatement *>(this),
					      &arch->_get_entity()->port_clause,
					      component_name->_get_port_list() );
  retval->generic_map_aspect._build_default_map( static_cast<IIR_ComponentInstantiationStatement *>(this),
						 &arch->_get_entity()->generic_clause,
						 component_name->_get_generic_list() );  
  
  return retval;
}

visitor_return_type *
IIRScram_ComponentInstantiationStatement::_accept_visitor( node_visitor *visitor, 
							   visitor_argument_type *arg ){
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ComponentInstantiationStatement(this, arg);
}

