
// Copyright (c) 1996-2003 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com
//          Malolan Chetlur     
//          Krishnan Subramani  
//          Timothy J. McBrayer 
//          Narayanan Thondugulam
//	    Magnus Danielson	cfmd@swipnet.se

#include "IIRScram_ComponentDeclaration.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_TerminalInterfaceDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_InterfaceList.hh"
#include "IIR_ConstantInterfaceDeclaration.hh"
#include "IIR_AttributeSpecificationList.hh"
#include "IIR_AttributeSpecification.hh"
#include "IIR_LibraryDeclaration.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"
#include "published_header_file.hh"
#include "published_cc_file.hh"
#include "sstream-wrap.hh"
#include "language_processing_control.hh"

extern language_processing_control *lang_proc;

IIRScram_ComponentDeclaration::~IIRScram_ComponentDeclaration() {}

set<IIR_Declaration> *
IIRScram_ComponentDeclaration::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;

  set<IIR_Declaration> *current_set = local_generic_clause._find_declarations( to_find );
  if ( current_set != NULL ){
    retval->add( current_set );
  }
  current_set = local_port_clause._find_declarations( to_find );
  if ( current_set != NULL ){
    retval->add( current_set );
  }
  if ( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }
  return retval;
}

void 
IIRScram_ComponentDeclaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  _vhdl_out << " component ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " is\n";

  if (local_generic_clause.num_elements() != 0) {
    _vhdl_out << " generic( ";
    local_generic_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";
  }
  if (local_port_clause.num_elements() != 0) {
    _vhdl_out << " port( ";
    local_port_clause._publish_vhdl(_vhdl_out);
    _vhdl_out << " );\n";
  }
  _vhdl_out << " end component ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n";
}

IIR_PortList *
IIRScram_ComponentDeclaration::_get_port_list(){
  return &local_port_clause;
}

IIR_TypeDefinition *
IIRScram_ComponentDeclaration::_get_port_type( int port_num ){
  ASSERT( port_num < local_port_clause.num_elements() );
  IIR_Declaration *port_element 
    = (IIR_Declaration *)local_port_clause.get_nth_element( port_num );

  ASSERT( port_element->_is_iir_declaration() == TRUE );

  return port_element->get_subtype();
}

IIR_GenericList *
IIRScram_ComponentDeclaration::_get_generic_list(){
  return &local_generic_clause;
}

void 
IIRScram_ComponentDeclaration::_make_interface_visible( symbol_table *sym_tab ){
  sym_tab->make_visible( &local_generic_clause );
  sym_tab->make_visible( &local_port_clause );
}

IIRScram_Declaration::declaration_type 
IIRScram_ComponentDeclaration::_get_type(){
   return COMPONENT;
}

void
IIRScram_ComponentDeclaration::_publish_cc_binding_name(ostream& outstream) {
  outstream << *_get_declarator();
}

void
IIRScram_ComponentDeclaration::_publish_cc_elaborate( published_file & ){
  _publish_cc_headerfile();
  _publish_cc_ccfile();
}

void
IIRScram_ComponentDeclaration::_publish_cc_headerfile(){
  const string filename = _get_cc_elaboration_class_name();

  PublishedUnit oldUnit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(DUMMY_ENTITY_DECL);
  
  published_header_file header_file( _get_work_library()->_get_path_to_directory(), 
				     filename,
				     this );
  SCRAM_CC_REF( header_file, "IIRScram_ComponentDeclaration::_publish_cc_headerfile" );

  IIRScram::_publish_cc_include( header_file, "tyvis/_savant_entity_elab.hh" );
  if ( lang_proc->processing_vhdl_ams() ) {
    IIRScram::_publish_cc_include( header_file, "tyvis/_savant_entity_ams_elab.hh" );
  }
  IIRScram::_publish_cc_include( header_file, "tyvis/SignalNetinfo.hh" );

  // Request the Hierarchy.hh for our hierarchy stuff
  IIRScram::_publish_cc_include( header_file, "tyvis/Hierarchy.hh" );

  if( lang_proc->processing_vhdl_ams() ){
    IIRScram::_publish_cc_include( header_file, "tyvis/_savant_entity_ams_elab.hh" );
  }

  if(local_generic_clause.num_elements() > 0) {
    local_generic_clause._publish_cc_headers( header_file );
  }
  if (local_port_clause.num_elements() > 0) {
    local_port_clause._publish_cc_headers( header_file );
  }

  _set_currently_publishing_unit(oldUnit);
  
  _publish_cc_class( header_file );
}

void
IIRScram_ComponentDeclaration::_publish_cc_class( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_class" );

  ostringstream elabStream;
  if( lang_proc->processing_vhdl_ams() ){  
    elabStream << " _savant_entity_ams_elab";
  }
  else {
    elabStream << " _savant_entity_elab";
  }
  string elabString = elabStream.str();
  
  _cc_out << "class " << _get_cc_elaboration_class_name()
          << OS(": public" + elabString + "{");
  _cc_out << "public:" << NL();
  _cc_out << _get_cc_elaboration_class_name() << "();" << NL();

  if ( local_generic_clause.num_elements() > 0 ){
    _cc_out << OS( _get_cc_elaboration_class_name() + "(" ) << NL();
    local_generic_clause._publish_generic_parameter_list( _cc_out );
    _cc_out << CS(");");
  }

  _cc_out << "~" << _get_cc_elaboration_class_name() << "();" << NL();

  local_generic_clause._publish_cc_elaborate( _cc_out );
  local_port_clause._publish_cc_elaborate( _cc_out );
  _cc_out << elabString << " *boundedEntity;" << NL()
	  << "void instantiate( Hierarchy *hier );" << NL()
	  << "void createNetInfo();" << NL()
	  << "void connect(int, int, ...);" << NL()
	  << "void partition(){}" << NL()
	  << "void buildSignalTree();" << NL();
  if( lang_proc->processing_vhdl_ams() ){
    _cc_out << "void connectTerminals(int, ...);"<<NL()
            << "void formCharacteristicExpressions();"<<NL();
  }  
  _cc_out << CS("};");
}

void
IIRScram_ComponentDeclaration::_publish_cc_ccfile(){
  const string filename = _get_cc_elaboration_class_name();
  PublishedUnit oldUnit = _get_currently_publishing_unit();
  _set_currently_publishing_unit(DUMMY_ENTITY_DECL);

  published_cc_file cc_file( _get_work_library()->_get_path_to_directory(),
			     filename,
			     this );

  SCRAM_CC_REF( cc_file, "IIRScram_ComponentDeclaration::_publish_cc_ccfile" );

  _publish_cc_include( cc_file );
  _publish_cc_constructor( cc_file );
  _publish_cc_destructor( cc_file );
  _publish_cc_instantiate( cc_file );
  _publish_cc_createNetInfo( cc_file );
  _publish_cc_connect( cc_file );
  _publish_cc_getboundentityinfo( cc_file );
  if( lang_proc->processing_vhdl_ams() ) {
    _publish_cc_ams_connect_terminals( cc_file );
    _publish_cc_ams_form_characteristic_expressions( cc_file );
  }
  _set_currently_publishing_unit(oldUnit);
}

void
IIRScram_ComponentDeclaration::_publish_cc_constructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_constructor" );

  _publish_cc_constructor_with_no_arguments( _cc_out );
  if (local_generic_clause.num_elements() > 0) {
    _publish_cc_constructor_with_arguments( _cc_out );
  }
}

void 
IIRScram_ComponentDeclaration::_publish_cc_constructor_with_no_arguments(published_file &_cc_out){
  int numPortClause = local_port_clause.num_elements();
  
  bool published_initializers = false;
  
  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_constructor_with_no_arguments" );

  _cc_out << _get_cc_elaboration_class_name() + "::" +
    _get_cc_elaboration_class_name() << "()";
  if ( numPortClause > 0 || local_generic_clause.num_elements() > 0 ) {
    _cc_out << OS(":") << NL();
    published_initializers = true;
  }
  local_generic_clause._publish_generic_init( _cc_out );
  if (local_generic_clause.num_elements() > 0 ) {
    if (local_port_clause.num_elements() > 0){
      _cc_out << "," << NL();
    }
  }

  if ( published_initializers ){
    _cc_out << CS("");
  }

  local_port_clause._publish_cc_port_init( _cc_out );
  _cc_out << OS("{")
	  << "boundedEntity = NULL;"
	  << CS("}");
}

void 
IIRScram_ComponentDeclaration::_publish_cc_constructor_with_arguments( published_file &_cc_out ){
  int numPortClause = local_port_clause.num_elements();

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_constructor_with_arguments" );

  _cc_out << _get_cc_elaboration_class_name() << "::"
	  << _get_cc_elaboration_class_name() << "(" << NL();
  local_generic_clause._publish_generic_parameter_list( _cc_out );
  _cc_out << " )";

  _cc_out << ":" << NL();
  local_generic_clause._publish_generic_init_by_arguments( _cc_out );
  if ( numPortClause > 0 ){
    _cc_out << "," << NL();
    local_port_clause._publish_cc_port_init( _cc_out );
  }

  _cc_out << OS("{")
	  << "boundedEntity = NULL;\n"
	  << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_destructor( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_destructor" );

  _cc_out << OS(_get_cc_elaboration_class_name() + "::~"
		+ _get_cc_elaboration_class_name() + "(){")
	  << "delete boundedEntity;" << NL()
	  << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_instantiate( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_instantiate" );

  _cc_out << "void" << NL() 
	  << OS(_get_cc_elaboration_class_name() +
		"::instantiate( Hierarchy *hier ){")
	  << OS("if (boundedEntity ){")
	  << "boundedEntity->instantiate(hier);" << NL()
	  << CS("}")
	  << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_createNetInfo( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_createNetInfo" );

  _cc_out << "void" << NL()
	  << OS( _get_cc_elaboration_class_name() +
		 "::createNetInfo(){");
  _cc_out << OS("if (boundedEntity != NULL) {")
	  << "boundedEntity->createNetInfo();" << NL()
	  << CS("}")
	  << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_connect( published_file &_cc_out ) {
   ostringstream connectparams;

   SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_connect" );

   _cc_out << "void" << NL()
         << OS( _get_cc_elaboration_class_name() +
                "::connect(int inputsignals, int outputsignals, ...){")
         << "int NoofSignals = inputsignals + outputsignals;" << NL()
         << "va_list ap;" << NL()
         << "int currentSignal = 0;" << NL()
         << "va_start(ap, outputsignals);" << NL()
         << "for(int i=0; i <NoofSignals; i++) {" << NL()
         << "addToFanOut(va_arg(ap, VHDLType*) );" << NL()
         << "}" << NL()
         << "va_end(ap);" << NL();

   IIR_InterfaceDeclaration* portelement = NULL;
   IIR_Mode mode;
   int index = 0;
   int first = 0;

   _cc_out << OS("if( inputsignals > 0 ){");
   portelement = local_port_clause.first();
   for(; portelement != NULL; ) {
     mode = portelement->get_mode();
     if((mode == IIR_IN_MODE)) {
       first = 1;
       _cc_out << OS("currentSignal = checkSetSourceInfo(");
       portelement->_publish_cc_elaborate( _cc_out );
       _cc_out << "," << NL()
             << "currentSignal,"
             << CS("fanOutInfo);");
       index++;
     }
     portelement = local_port_clause.successor(portelement);
   }

   _cc_out << CS("}");

   _cc_out << OS("if( outputsignals > 0 ){");
   portelement = local_port_clause.first();
   for(; portelement != NULL; ) {
    if (portelement->get_kind() != IIR_TERMINAL_INTERFACE_DECLARATION) {
     mode = portelement->get_mode();
     if (mode == IIR_INOUT_MODE) {
       first = 1;
       _cc_out << OS("checkAdd( ");
       portelement->_publish_cc_elaborate( _cc_out );
       _cc_out << "," << NL()   
             << "currentSignal," << NL()
             << CS("fanOutInfo );");
       
       _cc_out << OS("currentSignal = checkSetSourceInfo(");
       portelement->_publish_cc_elaborate( _cc_out );
       _cc_out << ","
             << NL() << "currentSignal,"
             << CS("fanOutInfo);");
       index++;
     }
     else if ( mode == IIR_OUT_MODE ){
       first = 1;
       _cc_out << OS("currentSignal = checkAdd(");
       portelement->_publish_cc_elaborate( _cc_out );
       _cc_out << "," << NL()
             << "currentSignal," 
             << CS("fanOutInfo);");
       index++;
     }
     else if ( mode != IIR_IN_MODE ){
       cerr << "IIRScramComponentDeclaration::_publish_cc_connect Invalid mode in the port "
          << endl;
       abort();
     }
    }
    portelement = local_port_clause.successor(portelement);
   }
   _cc_out << CS("}");
       
   //###Assumes binding entity and component has same # of o/io ports
             
   _cc_out << OS("if( boundedEntity != NULL ){");
   _cc_out << OS("boundedEntity->connect(inputsignals, outputsignals");
   if(first == 1) {
     portelement = local_port_clause.first();
     for(; portelement != NULL; ) {
       mode = portelement->get_mode();
       if((mode == IIR_IN_MODE )) {
       _cc_out << "," << NL() << "&";
       portelement->_publish_cc_elaborate( _cc_out );
       }
       portelement = local_port_clause.successor(portelement);
     }
       
     portelement = local_port_clause.first();
     for(; portelement != NULL; ) {
       mode = portelement->get_mode();
       if((mode == IIR_OUT_MODE ) || (mode == IIR_INOUT_MODE)) {
       _cc_out << "," << NL() << "&";
       portelement->_publish_cc_elaborate( _cc_out );
       }
       portelement = local_port_clause.successor(portelement);
     }
   }         
   _cc_out << CS(");");
   _cc_out << CS("}");
   _cc_out << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_getboundentityinfo( published_file &_cc_out ) {

  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_getboundentityinfo" );

  _cc_out << "void" << NL()
	  << OS( _get_cc_elaboration_class_name() +
		 "::buildSignalTree(){")
	  << OS("if (boundedEntity != NULL) {")
	  << "boundedEntity->buildSignalTree();" << NL()
	  << CS("}")
	  << CS("}");
}

void
IIRScram_ComponentDeclaration::_publish_cc_include( published_file &_cc_out ){
  SCRAM_CC_REF( _cc_out, 
		"IIRScram_ComponentDeclaration::_publish_cc_include( published_file &_cc_out )");
  const string filename = _get_cc_elaboration_class_name() + ".hh";
  IIRScram::_publish_cc_include(_cc_out, filename );
}

#ifdef PROCESS_COMBINATION
void
IIRScram_ComponentDeclaration::
_static_elaborate(IIR_ArchitectureDeclaration *arch,
		  IIR_DeclarationList *cfglist,
		  char *hier_location) {

  IIR_InterfaceDeclaration *port = NULL;

#ifdef DEBUG_ELAB
  cout << "Now checking component declaration at \"" << hier_location 
       << "\"\n";
#endif

  // component declaration generics aren't yet supported.
  if (local_generic_clause.num_elements() != 0) {
    cout << "ERROR:  Sorry, generics in component declarations "
	 << "not supported durign static elaboration yet\n";
    exit(1);
  }
  // attributes of components aren't yet supported.
  if (attributes.num_elements() != 0) {
    cout << "ERROR:  Sorry, attributes of component declarations "
	 << "not supported during static elaboration yet\n";
    abort();
  }

  // resolved signals aren't yet supported.
  port = local_port_clause.first();
  while (port != NULL) {
    IIR_Kind type = port->get_subtype()->get_kind();
    if (type == IIR_ENUMERATION_SUBTYPE_DEFINITION ||
	type == IIR_INTEGER_SUBTYPE_DEFINITION ||
	type == IIR_FLOATING_SUBTYPE_DEFINITION ||
	type == IIR_PHYSICAL_SUBTYPE_DEFINITION ||
	type == IIR_ARRAY_SUBTYPE_DEFINITION ||
	type == IIR_ACCESS_SUBTYPE_DEFINITION) {
      
      IIRScram_TypeDefinition *port_subtype;
      port_subtype = port->get_subtype();
      if (port_subtype->_get_resolution_function() != NULL) {
	cout << "ERROR: resolved signals in component declarations "
	     << "not supported during static elaboration yet\n";
	abort();
      }
    }
    port = local_port_clause.successor(port);
  }

#ifdef DEBUG_ELAB
  cout << "component declaration is clean--no generics, attributes, or resolved ports.\n";
#endif
}
#endif

visitor_return_type *IIRScram_ComponentDeclaration::_accept_visitor(node_visitor *visitor, visitor_argument_type *arg) {
  ASSERT(visitor != NULL);
  return visitor->visit_IIR_ComponentDeclaration(this, arg);
};

void
IIRScram_ComponentDeclaration::_publish_cc_ams_connect_terminals(published_file &_cc_out) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_connect_terminals" );
  ostringstream objectname;
  IIR_InterfaceDeclaration *portelement = NULL;
  int terminal_index = 0;
  _cc_out << "void\n";
  this->_publish_cc_binding_name(_cc_out.get_stream());
  _cc_out << "_elab::connectTerminals(int numberOfTerminals, ...) {\n";
  _cc_out << "va_list ap;\n";
  _cc_out << "terminalInfo = (AMSType**)new "
          << "char[numberOfTerminals * sizeof(AMSType*)];\n";
  _cc_out << "va_start(ap, numberOfTerminals);\n";
  _cc_out << "for(int i=0; i <numberOfTerminals; i++) {\n";
  _cc_out << "  terminalInfo[i] = va_arg(ap, AMSType*);\n";
  _cc_out << "}\n";
  
  _cc_out << "char *labelInfo = va_arg(ap, char *);\n" ;
  _cc_out << "va_end(ap);\n"; 
  
  portelement = local_port_clause.first();
  
  for(; portelement != NULL;
      portelement = local_port_clause.successor(portelement) ) {
  
    switch(portelement->get_kind()) {
      case IIR_TERMINAL_INTERFACE_DECLARATION: {  
        _cc_out << "  setTerminalInfo(";
        (static_cast<IIR_TerminalInterfaceDeclaration *>(portelement))->_publish_cc_lvalue(_cc_out);
        _cc_out << " , *(terminalInfo[" << terminal_index << "]));\n";
        terminal_index++;
      }
      break;
      default:
        break;
    } 
  }
  _cc_out << "boundedEntity->connectTerminals(numberOfTerminals";
  
  portelement = local_port_clause.first();
  
  for(; portelement != NULL;
      portelement = local_port_clause.successor(portelement) ) {
  
    switch(portelement->get_kind()) {
    case IIR_TERMINAL_INTERFACE_DECLARATION: {
      _cc_out << ", &";
     (static_cast<IIR_TerminalInterfaceDeclaration *>(portelement))->_publish_cc_lvalue(_cc_out);
    }
    break;
    default:
      break;
    }
  }
  _cc_out << ", labelInfo);\n";
  _cc_out << "}\n";
}

void  
IIRScram_ComponentDeclaration::_publish_cc_ams_form_characteristic_expressions(published_file &_cc_out) {
  SCRAM_CC_REF( _cc_out, "IIRScram_ComponentDeclaration::_publish_cc_ams_form_characteristic_expressions" );
  _cc_out << "void\n";
  this->_publish_cc_binding_name(_cc_out.get_stream());
  _cc_out << "_elab::formCharacteristicExpressions() {\n";
  _cc_out << "  boundedEntity->formCharacteristicExpressions();\n}\n";
}
