/*
 *  fields-list.c        -
 *
 *  Created: 20020317
 *
 *  Copyright (c) 2001, 2002 by Tomasz Trojanowski
 *  All rights reserved
 *
 *  $Id: fields-list.c,v 1.8.2.1 2002/03/26 18:09:15 tomek Exp $
 *
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <fields-list.h>

#include <intl.h>
#include <globals.h>
#include <database.h>
#include <appointment-list.h>
#include <contact-list.h>
#include <appointment-window.h>
#include <contact-window.h>
#include <main-window.h>

fields_list_item_type *fields_list_items[2] =
{
    appointment_list_items,
    contact_list_items
};

/* Funkcja obsugujca kliknicie myszk na licie terminw lub kontaktw. */
static void list_button_press(GtkCList *clist, GdkEventButton *event, gpointer data)
{
    gint row;
    gint item_id;
    
    g_assert(clist != NULL);
    
    if (!event)
        return;
    
    /* Obsuga podwjnego kliknicia pierwszym przyciskiem myszki. */
    if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
        if (gtk_clist_get_selection_info(clist, event->x, event->y, &row, NULL))
	{
	    item_id = (gint)gtk_clist_get_row_data(clist, row);
	    
	    switch (current_list)
	    {
	        case LT_APPOINTMENT:
	            create_appointment_window(item_id);
		    break;
		case LT_CONTACT:
		    create_contact_window(item_id);
		    break;
	    }
	}
}

/* Funkcja obsugujca sygna "click-column" listy terminw lub kontaktw.. */
static void list_click_column(GtkCList *clist, gint column, gpointer data)
{
    gint id;
    
    /* Ustalenie identyfikatora pola w kliknitej kolumnie. */
    id = (gint)g_list_nth_data(list_params[current_list].id_list, column - 1);
    
    /* Ustalanie parametrw sortowania. */
    if (list_params[current_list].sort_field != id)
    {
        list_params[current_list].sort_field = id;
	list_params[current_list].sort_growly = TRUE;
    }
    else
        list_params[current_list].sort_growly = !list_params[current_list].sort_growly;
    
    /* Uaktualnianie listy kontaktw. */
    show_fields_list(current_list, FALSE);
}

/* Funkcja obsugujca sygna "resize_column" listy terminw. */
static void list_resize_column(GtkCList *list, gint column, gint width, gpointer data)
{
    /* Zapisywanie szerokoci kolumny do struktury fields_list_items. */
    /* tm - doda zapisywanie szerokoci kolumn w pliku konfiguracyjnym. */
    if (column != 0)
        fields_list_items[current_list]
	    [(gint)g_list_nth_data(list_params[current_list].id_list, column - 1)].width = width;
}

/* tm - poczy funkcje insert_appointment_item_callback i insert_contact_item_callback. */

/* Funkcja suca do utworzenia i wstawienia elementw do listy terminw. */
static void insert_appointment_item_callback(appointment_data *item, void *clist)
{
    if (list_params[LT_APPOINTMENT].id_list != NULL)
    {
        gchar **list_item;
	gint row;
	gint i;
	gint id_list_length;
	
	id_list_length = g_list_length(list_params[LT_APPOINTMENT].id_list);
	
	/* Przydzielenie pamici dla zmiennej przechowujcej pola elementu listy terminw. */
	list_item = g_malloc(sizeof(gchar *) * (id_list_length + 1));
	
	/* Utworzenie pola elementu zawierajacego dane wykorzystywane przy sortowaniu. */
	if (list_params[LT_APPOINTMENT].sort_field != -1)
	    list_item[0] = get_appointment_field_value(item, list_params[LT_APPOINTMENT].sort_field);
	else
	    list_item[0] = g_strdup_printf("sortowanie");
	
	/* Utworzenie elementu listy. */
	for (i = 0; i < id_list_length; i++)
	    list_item[i+1] = get_appointment_field_value(item, (gint)g_list_nth_data(list_params[LT_APPOINTMENT].id_list, i));
	
	/* Wstawienie elementu do listy. */
	row = gtk_clist_append(GTK_CLIST(clist), list_item);
	
	/* Przypisanie identyfikatora rekordu tabeli terminw do terminu wstawianego do listy. */
	gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer)item->id);
	
	/* Zwalnianie pamici zajmowanej przez elementy listy. */
	for (i = 0; i <= id_list_length; i++)
	    g_free(list_item[i]);
	
	/* Zwalnianie pamici zajmowanej przez zmienn przechowujac pola elementu listy. */
	g_free(list_item);
    }
}

/* Funkcja suca do utworzenia i wstawienia elementw do listy kontaktw. */
static void insert_contact_item_callback(contact_data *item, void *clist)
{
    if (list_params[LT_CONTACT].id_list != NULL)
    {
        gchar **list_item;
        gint row;
        int i;
	int id_list_length;
	
	id_list_length = g_list_length(list_params[LT_CONTACT].id_list);

        /* Przydzielanie pamici dla zmiennej przechowujcej pola elementu listy kontaktw. */
        list_item = g_malloc(sizeof(gchar *) * (id_list_length + 1));

        /* Utworzenie pola elementu zawierajcego dane wykorzystywane przy sortowaniu. */
	if (list_params[LT_CONTACT].sort_field != -1)
	    list_item[0] = get_contact_field_value(item, list_params[LT_CONTACT].sort_field);
	else
            list_item[0] = g_strdup_printf("sortowanie");
    
        /* Utworzenie elementu listy.*/
        for (i = 0; i < id_list_length; i++)
	    list_item[i+1] = get_contact_field_value(item, (gint)g_list_nth_data(list_params[LT_CONTACT].id_list, i));

        /* Wstawianie elementu do listy. */
        row = gtk_clist_append(GTK_CLIST(clist), list_item);
    
        /* Przypisanie identyfikatora rekordu tabeli kontaktw do wstawianego
           do listy elementu. */
        gtk_clist_set_row_data(GTK_CLIST(clist), row, (gpointer)item->id);
    
        /* Zwalnianie pamici zajmowanej przez elementy listy. */
	for (i = 0; i <= id_list_length; i++)
            g_free(list_item[i]);
	
	/* Zwalnianie pamici zajmowanej przez zmienn przechowujc pola elementu listy. */
	g_free(list_item);
    }
}

/* Tworzenie kolumn listy terminw lub kontaktw. */
static void create_list_columns(GtkWidget *clist)
{
    GtkWidget *label;
    int i;

    /* Tworzenie kolumny zawierajacej dane wykorzystywane przy sortowaniu. */
    label = gtk_label_new("Sort");
    gtk_widget_show(label);

    gtk_clist_set_column_widget(GTK_CLIST(clist), 0, label);

    /* Tworzenie pozostaych kolumn listy terminw lub kontaktw. */
    for (i = 0; i < g_list_length(list_params[current_list].id_list); i++)
    {
        gtk_clist_set_column_width(GTK_CLIST(clist), i + 1,
            fields_list_items[current_list]
	        [(gint)g_list_nth_data(list_params[current_list].id_list, i)].width);

        label = gtk_label_new(gettext(fields_list_items[current_list]
	            [(gint)g_list_nth_data(list_params[current_list].id_list, i)].name));

	gtk_widget_show(label);
	
	gtk_clist_set_column_widget(GTK_CLIST(clist), i + 1, label);
    }
}

/* Funkcja tworzca list terminw lub kontaktw. */
static GtkWidget *create_fields_list()
{
    GtkWidget *scrolled;
    GtkWidget *clist;
    
    /* Obszar z paskami przewijania. */
    scrolled = gtk_scrolled_window_new(NULL, NULL);
    gtk_widget_show(scrolled);
    
    gtk_container_set_border_width(GTK_CONTAINER(scrolled), 5);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
                                   GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
    
        /* Lista terminw lub kontaktw. */
	if (g_list_length(list_params[current_list].id_list) > 0)
        {
	    clist = gtk_clist_new(g_list_length(list_params[current_list].id_list) + 1);
	    gtk_widget_show(clist);

            gtk_signal_connect(GTK_OBJECT(clist), "button_press_event",
                               GTK_SIGNAL_FUNC(list_button_press), NULL);
	    gtk_signal_connect(GTK_OBJECT(clist), "click_column",
	                       GTK_SIGNAL_FUNC(list_click_column), NULL);
	    gtk_signal_connect(GTK_OBJECT(clist), "resize_column",
	                       GTK_SIGNAL_FUNC(list_resize_column), NULL);
	
	    gtk_container_add(GTK_CONTAINER(scrolled), clist);
	    gtk_clist_column_titles_show(GTK_CLIST(clist));
	
	    /* Tworzenie kolumn listy terminw lub kontaktw. */
	    create_list_columns(clist);
	
	    /* Wstawianie danych do listy terminw lub kontaktw. */
	    switch (current_list)
	    {
	        case LT_APPOINTMENT:
	            db_appointment_foreach(insert_appointment_item_callback, (void *)clist);
		    break;
	        case LT_CONTACT:
	            db_contact_foreach(insert_contact_item_callback, (void *)clist);
		    break;
	    }
	
	    /* Sortowanie listy terminw. */
	    if (list_params[current_list].sort_growly)
	        gtk_clist_set_sort_type(GTK_CLIST(clist), GTK_SORT_ASCENDING);
	    else
	        gtk_clist_set_sort_type(GTK_CLIST(clist), GTK_SORT_DESCENDING);
	    gtk_clist_set_sort_column(GTK_CLIST(clist), 0);
	    gtk_clist_sort(GTK_CLIST(clist));
	
	    /* Ukrywanie kolumny zawierajcej dane wykorzystywane przy sortowaniu. */
	    gtk_clist_set_column_visibility(GTK_CLIST(clist), 0, FALSE);
	}
	else
	{
	    clist = gtk_clist_new(1);
	    gtk_widget_show(clist);

	    gtk_container_add(GTK_CONTAINER(scrolled), clist);
	}
	
    return scrolled;
}

/* Funkcja zwracajca ilo elementw okrelonej listy pl (kontaktw lub terminw). */
guint list_items_length(fields_list_type list)
{
    switch (list)
    {
        case LT_APPOINTMENT:
	    return IDAF_END_DATE + 1;
	case LT_CONTACT:
	    return IDCF_PHONE_WORK + 1;
	default:
	    g_assert_not_reached();
	    return 0;
    }
}

/* Funkcja tworzca i uaktualniajca list terminw lub kontaktw. */
void show_fields_list(fields_list_type fields_list, gboolean force)
{
    GtkWidget *hbox;
    gpointer list;
    
    if (force || current_list == fields_list)
    {
        current_list = fields_list;
        
        /* Pobranie wskanika do gwnej czci okna. */
        hbox = gtk_object_get_data(GTK_OBJECT(main_window), "hbox");
        g_assert(hbox != NULL);
    
        /* Usunicie starej listy. */
        list = g_list_nth_data(gtk_container_children(GTK_CONTAINER(hbox)), 1);
	if (list != NULL)
            gtk_container_remove(GTK_CONTAINER(hbox), GTK_WIDGET(list));
    
        /* Utworzenie nowej listy. */
        list = create_fields_list();
        gtk_box_pack_start(GTK_BOX(hbox), GTK_WIDGET(list), TRUE, TRUE, 0);
	
	/* Aktualizacja menu okienka gwnego. */
	update_main_menu();
    }
}
