/*
 * Resource Manager
 *
 * Copyright (C) 2001-2002, Olaf Kirch <okir@lst.de>
 */

#ifndef RESMGRD_H
#define RESMGRD_H

#include <sys/socket.h>
#include <sys/un.h>
#include <stdarg.h>
#include "protocol.h"
#include "utils.h"

#define _PATH_RESMGR_CONF	"/etc/resmgr.conf"
#ifndef _PATH_LOCKS
# define _PATH_LOCKS		"/var/lock";
#endif
#ifndef _PATH_PROC_BUS_USB
# define _PATH_PROC_BUS_USB	"/proc/bus/usb"
#endif

struct svc {
	struct sockaddr_un sun;
	int		fd;
};

typedef struct res_device res_device_t;
struct res_device {
	res_device_t *	next;
	char *		name;
	char *		lock;
	int		flags;
};

typedef struct res_acl res_acl_t;

typedef struct res_class res_class_t;
struct res_class {
	res_class_t *	next;
	char *		name;
	unsigned int	refcnt;
	res_device_t *	devices;
	res_acl_t *	acl;
};

typedef struct res_user res_user_t;
struct res_user {
	res_user_t *	next;
	char *		name;
	unsigned int	refcnt;
	unsigned int	nconnections;
	unsigned int	nsessions;

	unsigned int	nclasses;
	res_class_t **	classes;
};

typedef struct res_session res_session_t;
struct res_session {
	res_session_t *	next;
	char *		id;
	res_user_t *	user;
};

typedef struct res_name res_name_t;
typedef struct res_family res_family_t;
struct res_family {
	const char *	name;
	unsigned int	access_flags;
	res_name_t *	(*parse_name)(const char *);
	const char *	(*print_name)(res_name_t *);
	void		(*free_name)(res_name_t *);
	int		(*match)(res_name_t *, res_device_t *);
	int		(*open)(res_name_t *, int);
};

struct res_name {
	res_family_t *	ops;
};

extern res_session_t *	res_sessions;
extern res_user_t *	res_users;
extern res_class_t *	res_classes;
extern const char *	res_config_lock_dir;
extern res_family_t	res_family_file;
extern res_family_t	res_family_scsi;
extern res_family_t	res_family_usb;
extern res_family_t	res_family_paride;

/* Possible device flag values */
#define DEV_FLAGS_RO	0x0001
#define DEV_FLAGS_SCSI	0x0002
#define DEV_FLAGS_PARIDE 0x0004

extern void		svc_start(struct svc *, const char *);
extern struct conn *	svc_accept(struct svc *);

extern void		respond(struct conn *, int, const char *, ...);
extern void		message(int, const char *, ...);

extern void		res_config_set_filename(const char *);
extern void		res_config_reload(void);

extern int		res_user_login(res_user_t *, const char *);
extern int		res_user_logout(res_user_t *, const char *);
extern res_user_t *	res_user_get(const char *);
extern res_user_t *	res_user_create(const char *);
extern void		res_user_resolve_acls(res_user_t *, const char *);
extern res_device_t *	res_user_get_device(res_user_t *, res_name_t *, int);
extern int		res_user_open(res_user_t *, res_name_t *, int);
extern void		res_user_grant(res_user_t *, res_class_t *);
extern void		res_user_revoke(res_user_t *, res_class_t *);
extern void		res_user_revoke_all(res_user_t *);
extern void		res_user_free(res_user_t *);

extern res_session_t *	res_session_create(const char *, const char *);
extern void		res_session_free(res_session_t *);

extern res_class_t *	res_class_get(const char *);
extern res_class_t *	res_class_create(const char *);
extern res_device_t *	res_class_add(res_class_t *, const char *, int);
extern void		res_class_remove(res_class_t *, const char *);
extern res_device_t *	res_class_get_device(res_class_t *, res_name_t *);
extern void		res_class_revoke(res_class_t *, res_user_t *);
extern void		res_class_free(res_class_t *);

extern int		res_device_sane(const char *);
extern res_device_t *	res_device_create(const char *, int);
extern int		res_device_lock(res_device_t *, uid_t, pid_t);
extern int		res_device_unlock(res_device_t *, uid_t);
extern void		res_device_free(res_device_t *);

extern res_name_t *	res_name_parse(const char *);
extern const char *	res_name_print(res_name_t *);
extern void		res_name_free(res_name_t *);
extern int		res_name_match(res_name_t *, res_device_t *);
extern int		res_name_open(res_name_t *, int);
extern res_family_t *	res_family_get(const char *);

extern res_acl_t *	res_acl_parse(unsigned int, char **);
extern void		res_acl_add(res_acl_t **, res_acl_t *, int);
extern int		res_acl_match(res_acl_t *,
				const char *, const char *,
				const char **);
extern void		res_acl_free(res_acl_t *);

extern const char *	res_find_scsi_generic(res_device_t *);

extern int		opt_debug;
extern time_t		current_time;

#endif /* RESMGRD_H */
