/*
 * This source code is a product of Sun Microsystems, Inc. and is provided
 * for unrestricted use.  Users may copy or modify this source code without
 * charge.
 *
 * SUN SOURCE CODE IS PROVIDED AS IS WITH NO WARRANTIES OF ANY KIND INCLUDING
 * THE WARRANTIES OF DESIGN, MERCHANTIBILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE, OR ARISING FROM A COURSE OF DEALING, USAGE OR TRADE PRACTICE.
 *
 * Sun source code is provided with no support and without any obligation on
 * the part of Sun Microsystems, Inc. to assist in its use, correction,
 * modification or enhancement.
 *
 * SUN MICROSYSTEMS, INC. SHALL HAVE NO LIABILITY WITH RESPECT TO THE
 * INFRINGEMENT OF COPYRIGHTS, TRADE SECRETS OR ANY PATENTS BY THIS SOFTWARE
 * OR ANY PART THEREOF.
 *
 * In no event will Sun Microsystems, Inc. be liable for any lost revenue
 * or profits or other special, indirect and consequential damages, even if
 * Sun has been advised of the possibility of such damages.
 *
 * Sun Microsystems, Inc.
 * 2550 Garcia Avenue
 * Mountain View, California  94043
 */
/*
 * December 30, 1994:
 * Functions linear2alaw, linear2ulaw have been updated to correctly
 * convert unquantized 16 bit values.
 * Tables for direct u- to A-law and A- to u-law conversions have been
 * corrected.
 * Borge Lindberg, Center for PersonKommunikation, Aalborg University.
 * bli@cpk.auc.dk
 *
 */
/*
 * Downloaded from comp.speech site in Cambridge.
 *
 */

#include "config_unix.h"
#include "config_win32.h"
#include "audio_types.h"
#include "codec_g711.h"

/*
 * g711.c
 *
 * u-law, A-law and linear PCM conversions.
 */
#define	SIGN_BIT	(0x80)		/* Sign bit for a A-law byte. */
#define	QUANT_MASK	(0xf)		/* Quantization field mask. */
#define	NSEGS		(8)		/* Number of A-law segments. */
#define	SEG_SHIFT	(4)		/* Left shift for segment number. */
#define	SEG_MASK	(0x70)		/* Segment field mask. */

static short seg_aend[8] = {0x1F, 0x3F, 0x7F, 0xFF,
			    0x1FF, 0x3FF, 0x7FF, 0xFFF};
static short seg_uend[8] = {0x3F, 0x7F, 0xFF, 0x1FF,
			    0x3FF, 0x7FF, 0xFFF, 0x1FFF};

/* copy from CCITT G.711 specifications */
unsigned char _u2a[128] = {			/* u- to A-law conversions */
	1,	1,	2,	2,	3,	3,	4,	4,
	5,	5,	6,	6,	7,	7,	8,	8,
	9,	10,	11,	12,	13,	14,	15,	16,
	17,	18,	19,	20,	21,	22,	23,	24,
	25,	27,	29,	31,	33,	34,	35,	36,
	37,	38,	39,	40,	41,	42,	43,	44,
	46,	48,	49,	50,	51,	52,	53,	54,
	55,	56,	57,	58,	59,	60,	61,	62,
	64,	65,	66,	67,	68,	69,	70,	71,
	72,	73,	74,	75,	76,	77,	78,	79,
/* corrected:
	81,	82,	83,	84,	85,	86,	87,	88, 
   should be: */
	80,	82,	83,	84,	85,	86,	87,	88,
	89,	90,	91,	92,	93,	94,	95,	96,
	97,	98,	99,	100,	101,	102,	103,	104,
	105,	106,	107,	108,	109,	110,	111,	112,
	113,	114,	115,	116,	117,	118,	119,	120,
	121,	122,	123,	124,	125,	126,	127,	128};

unsigned char _a2u[128] = {			/* A- to u-law conversions */
	1,	3,	5,	7,	9,	11,	13,	15,
	16,	17,	18,	19,	20,	21,	22,	23,
	24,	25,	26,	27,	28,	29,	30,	31,
	32,	32,	33,	33,	34,	34,	35,	35,
	36,	37,	38,	39,	40,	41,	42,	43,
	44,	45,	46,	47,	48,	48,	49,	49,
	50,	51,	52,	53,	54,	55,	56,	57,
	58,	59,	60,	61,	62,	63,	64,	64,
	65,	66,	67,	68,	69,	70,	71,	72,
/* corrected:
	73,	74,	75,	76,	77,	78,	79,	79,
   should be: */
	73,	74,	75,	76,	77,	78,	79,	80,
	80,	81,	82,	83,	84,	85,	86,	87,
	88,	89,	90,	91,	92,	93,	94,	95,
	96,	97,	98,	99,	100,	101,	102,	103,
	104,	105,	106,	107,	108,	109,	110,	111,
	112,	113,	114,	115,	116,	117,	118,	119,
	120,	121,	122,	123,	124,	125,	126,	127};

static short search(
   short val,
   short *table,
   short size)
{
   short i;
   
   for (i = 0; i < size; i++) {
      if (val <= *table++)
	 return (i);
   }
   return (size);
}

/*
 * linear2alaw() - Convert a 16-bit linear PCM value to 8-bit A-law
 *
 * linear2alaw() accepts an 16-bit integer and encodes it as A-law data.
 *
 *		Linear Input Code	Compressed Code
 *	------------------------	---------------
 *	0000000wxyza			000wxyz
 *	0000001wxyza			001wxyz
 *	000001wxyzab			010wxyz
 *	00001wxyzabc			011wxyz
 *	0001wxyzabcd			100wxyz
 *	001wxyzabcde			101wxyz
 *	01wxyzabcdef			110wxyz
 *	1wxyzabcdefg			111wxyz
 *
 * For further information see John C. Bellamy's Digital Telephony, 1982,
 * John Wiley & Sons, pps 98-111 and 472-476.
 */
static unsigned char
linear2alaw(short pcm_val)	/* 2's complement (16-bit range) */
{
   short	 mask;
   short	 seg;
   unsigned char aval;
   
   pcm_val = pcm_val >> 3;

   if (pcm_val >= 0) {
      mask = 0xD5;		/* sign (7th) bit = 1 */
   } else {
      mask = 0x55;		/* sign bit = 0 */
      pcm_val = -pcm_val - 1;
   }
   
   /* Convert the scaled magnitude to segment number. */
   seg = search(pcm_val, seg_aend, 8);
   
   /* Combine the sign, segment, and quantization bits. */
   
   if (seg >= 8)		/* out of range, return maximum value. */
      return (unsigned char) (0x7F ^ mask);
   else {
      aval = (unsigned char) seg << SEG_SHIFT;
      if (seg < 2)
	 aval |= (pcm_val >> 1) & QUANT_MASK;
      else
	 aval |= (pcm_val >> seg) & QUANT_MASK;
      return (aval ^ mask);
   }
}

/*
 * alaw2linear() - Convert an A-law value to 16-bit linear PCM
 *
 */
static short
alaw2linear(
   unsigned char	a_val)
{
   short t;
   short seg;
   
   a_val ^= 0x55;
   
   t = (a_val & QUANT_MASK) << 4;
   seg = ((unsigned)a_val & SEG_MASK) >> SEG_SHIFT;
   switch (seg) {
   case 0:
      t += 8;
      break;
   case 1:
      t += 0x108;
      break;
   default:
      t += 0x108;
      t <<= seg - 1;
   }
   return ((a_val & SIGN_BIT) ? t : -t);
}

#define	BIAS		(0x84)		/* Bias for linear code. */
#define CLIP            8159

/*
* linear2ulaw() - Convert a linear PCM value to u-law
*
* In order to simplify the encoding process, the original linear magnitude
* is biased by adding 33 which shifts the encoding range from (0 - 8158) to
* (33 - 8191). The result can be seen in the following encoding table:
*
*	Biased Linear Input Code	Compressed Code
*	------------------------	---------------
*	00000001wxyza			000wxyz
*	0000001wxyzab			001wxyz
*	000001wxyzabc			010wxyz
*	00001wxyzabcd			011wxyz
*	0001wxyzabcde			100wxyz
*	001wxyzabcdef			101wxyz
*	01wxyzabcdefg			110wxyz
*	1wxyzabcdefgh			111wxyz
*
* Each biased linear code has a leading 1 which identifies the segment
* number. The value of the segment number is equal to 7 minus the number
* of leading 0's. The quantization interval is directly available as the
* four bits wxyz.  * The trailing bits (a - h) are ignored.
*
* Ordinarily the complement of the resulting code word is used for
* transmission, and so the code word is complemented before it is returned.
*
* For further information see John C. Bellamy's Digital Telephony, 1982,
* John Wiley & Sons, pps 98-111 and 472-476.
*/
static unsigned char
linear2ulaw(
   short pcm_val)	/* 2's complement (16-bit range) */
{
   short         mask;
   short	 seg;
   unsigned char uval;
   
   /* Get the sign and the magnitude of the value. */
   pcm_val = pcm_val >> 2;
   if (pcm_val < 0) {
      pcm_val = -pcm_val;
      mask = 0x7F;
   } else {
      mask = 0xFF;
   }
   if ( pcm_val > CLIP ) pcm_val = CLIP;		/* clip the magnitude */
   pcm_val += (BIAS >> 2);
   
   /* Convert the scaled magnitude to segment number. */
   seg = search(pcm_val, seg_uend, 8);
   
   /*
   * Combine the sign, segment, quantization bits;
   * and complement the code word.
   */
   if (seg >= 8)		/* out of range, return maximum value. */
      return (unsigned char) (0x7F ^ mask);
   else {
      uval = (unsigned char) (seg << 4) | ((pcm_val >> (seg + 1)) & 0xF);
      return (uval ^ mask);
   }
   
}


/*
 * ulaw2linear() - Convert a u-law value to 16-bit linear PCM
 *
 * First, a biased linear code is derived from the code word. An unbiased
 * output can then be obtained by subtracting 33 from the biased code.
 *
 * Note that this function expects to be passed the complement of the
 * original code word. This is in keeping with ISDN conventions.
 */
static short
ulaw2linear(
   unsigned char	u_val)
{
   short t;
   
   /* Complement to obtain normal u-law value. */
   u_val = ~u_val;
   
   /*
    * Extract and bias the quantization bits. Then
    * shift up by the segment number and subtract out the bias.
    */
   t = ((u_val & QUANT_MASK) << 3) + BIAS;
   t <<= ((unsigned)u_val & SEG_MASK) >> SEG_SHIFT;
   
   return ((u_val & SIGN_BIT) ? (BIAS - t) : (t - BIAS));
}

#ifdef INTERLAW_CONVERSION

/* A-law to u-law conversion */
static unsigned char
alaw2ulaw(
   unsigned char	aval)
{
   aval &= 0xff;
   return (unsigned char) ((aval & 0x80) ? (0xFF ^ _a2u[aval ^ 0xD5]) :
	   (0x7F ^ _a2u[aval ^ 0x55]));
}

/* u-law to A-law conversion */
static unsigned char
ulaw2alaw(
   unsigned char	uval)
{
   uval &= 0xff;
   return (unsigned char) ((uval & 0x80) ? (0xD5 ^ (_u2a[0xFF ^ uval] - 1)) :
			   (0x55 ^ (_u2a[0x7F ^ uval] - 1)));
}

#endif /* INTERLAW_CONVERSION */

/* ---------- end of g711.c ----------------------------------------------------- */

/****************************************************************************/
/* RAT specific interface (Orion Hodson, November 1998)                     */
/****************************************************************************/

#include "config_unix.h"
#include "config_win32.h"
#include "memory.h"
#include "util.h"
#include "debug.h"
#include "audio_types.h"
#include "codec_types.h"
#include "codec_g711.h"

short         mulawtolin[256];
unsigned char lintomulaw[65536];

short         alawtolin[256];
unsigned char lintoalaw[8192]; 

void 
g711_init()
{
        int32_t i;
        
        for(i = 0; i < 256; i++)
                mulawtolin[i] = ulaw2linear((unsigned char)i);

        for(i = -32767; i < 32768; i++) 
                lintomulaw[(unsigned short)i] = linear2ulaw((sample)i);

        for(i = 0; i < 256; i++) 
                alawtolin[i] = alaw2linear((unsigned char)i);

        for(i = -32767; i < 32768; i+= 8) 
                lintoalaw[(unsigned short)i>>3] = linear2alaw((sample)i);

}

#define PAYLOAD(x)    (x)
#define STATE_SIZE(x) (x)
#define FRAME_SIZE(x) (x)

static codec_format_t cs[] = {
/* 8kHz */
        {"-law", "PCMU-8K-Mono",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(0), STATE_SIZE(0), FRAME_SIZE(160), /* 20 ms */
         {DEV_S16, 8000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"-law", "PCMU-8K-Stereo",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(83), STATE_SIZE(0), FRAME_SIZE(320), /* 20 ms */
         {DEV_S16, 8000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 16kHz */
        {"-law", "PCMU-16K-Mono",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(84), STATE_SIZE(0), FRAME_SIZE(160), /* 10 ms */
         {DEV_S16, 16000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"-law", "PCMU-16K-Stereo",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(85), STATE_SIZE(0), FRAME_SIZE(320), /* 10 ms */
         {DEV_S16, 16000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 32kHz */
        {"-law", "PCMU-32K-Mono",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(86), STATE_SIZE(0), FRAME_SIZE(160), /* 5 ms */
         {DEV_S16, 32000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"-law", "PCMU-32K-Stereo",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(87), STATE_SIZE(0), FRAME_SIZE(320), /* 5 ms */
         {DEV_S16, 32000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 48kHz */                
        {"-law", "PCMU-48K-Mono",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(88), STATE_SIZE(0), FRAME_SIZE(160), /* 3.333... ms */
         {DEV_S16, 48000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"-law", "PCMU-48K-Stereo",
         "ITU G.711 -law codec.  Sun Microsystems public implementation.",
         PAYLOAD(89), STATE_SIZE(0), FRAME_SIZE(320), /* 3.333... ms */
         {DEV_S16, 48000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 8kHz */
        {"A-law", "PCMA-8K-Mono",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(8), STATE_SIZE(0), FRAME_SIZE(160), /* 20 ms */
         {DEV_S16, 8000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"A-law", "PCMA-8K-Stereo",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(90), STATE_SIZE(0), FRAME_SIZE(320), /* 20 ms */
         {DEV_S16, 8000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 16kHz */
        {"A-law", "PCMA-16K-Mono",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(91), STATE_SIZE(0), FRAME_SIZE(160), /* 10 ms */
         {DEV_S16, 16000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"A-law", "PCMA-16K-Stereo",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(124), STATE_SIZE(0), FRAME_SIZE(320), /* 10 ms */
         {DEV_S16, 16000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 32kHz */
        {"A-law", "PCMA-32K-Mono",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(92), STATE_SIZE(0), FRAME_SIZE(160), /* 5 ms */
         {DEV_S16, 32000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"A-law", "PCMA-32K-Stereo",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(93), STATE_SIZE(0), FRAME_SIZE(320), /* 5 ms */
         {DEV_S16, 32000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
/* 48kHz */
        {"A-law", "PCMA-48K-Mono",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(94), STATE_SIZE(0), FRAME_SIZE(160), /* 3.333... ms */
         {DEV_S16, 48000, 16, 1, 160 * BYTES_PER_SAMPLE}},
        {"A-law", "PCMA-48K-Stereo",
         "ITU G.711 A-law codec.  Sun Microsystems public implementation.",
         PAYLOAD(95), STATE_SIZE(0), FRAME_SIZE(320), /* 3.333... ms */
         {DEV_S16, 48000, 16, 2, 2 * 160 * BYTES_PER_SAMPLE}},
};

#define G711_NUM_FORMATS (sizeof(cs) / sizeof (codec_format_t))
#define G711_IS_MULAW(idx)  (idx < (G711_NUM_FORMATS / 2))

uint16_t
g711_get_formats_count()
{
        return (uint16_t) G711_NUM_FORMATS;
}

const codec_format_t*
g711_get_format(uint16_t idx)
{
        assert(idx < G711_NUM_FORMATS);
        return &cs[idx];
}

int
g711_encode (uint16_t idx, u_char *state, sample *in, coded_unit *out)
{
        int len;
        u_char *p, *pe;

        assert(idx < G711_NUM_FORMATS);
        UNUSED(state);

        /* No difference between A-law and U-law sizes */
        len = cs[idx].mean_coded_frame_size;
        
        out->state     = NULL;
        out->state_len = 0;
        out->data      = (u_char*)block_alloc(len);
        out->data_len  = len;

        p  = out->data;
        pe = out->data + len;
        if (G711_IS_MULAW(idx)) {
                while(p != pe) {
                        *p = s2u(*in);
                        p++; in++;
                }
        } else {
                while(p != pe) {
                        *p = s2a(*in);
                        p++; in++;
                }
        }
        
        return len;
}

int
g711_decode(uint16_t idx, u_char *state, coded_unit *in, sample *out)
{
        int len;
        u_char *p, *pe;

        assert(idx < G711_NUM_FORMATS);
        UNUSED(state);

        len = cs[idx].mean_coded_frame_size;

        p  = in->data;
        pe = in->data + len;
        if (G711_IS_MULAW(idx)) {
                while(p != pe) {
                        *out = u2s(*p);
                        out++; p++;
                }
        } else {
                while(p != pe) {
                        *out = a2s(*p);
                        out++; p++;
                }
        }

        return len;
}

