//  Original version of this file ripped from
//
//  Pingus - A free Lemmings clone
//  Copyright (C) 1999 Ingo Ruhnke <grumbel@gmx.de>
//
//  Some small modifications & additions made by Harry Storbacka to 
//  make it work perfectly in Race.  - Thanks Ingo!

#include <ClanLib/display.h>

#include <algorithm>
#include <cstdio>
#include <assert.h>
// #include "PingusError.hh"
#include "blitter.h"
#include "color.h"
#include "caimagemanipulation.h"

// For some weird reason, min and max doesn't work under windows.
// I guess the win32 algorithmn header file is broken or something.
// -- mbn
/*#ifdef WIN32
#define min(a, b) (a<b?a:b)
#define max(a, b) (a>b?a:b)
#endif
*/
void
Blitter::put_surface(CL_Canvas* canvas, CL_Surface* sur,
		     int x, int y)
{
	Blitter::put_surface(canvas, sur->get_provider(), x, y);
}

void 
Blitter::put_surface(CL_Canvas* canvas, CL_SurfaceProvider* provider,
		     int x, int y)
{
  char str[32];
  
  switch(provider->get_depth())
    {
    case  8:
      put_surface_8bit(canvas, provider, x, y);
      break;
    case 32:
      put_surface_32bit(canvas, provider, x, y);
      break;
    default: 
      sprintf(str, "%d", provider->get_depth());
       std::cout << "Blitter:put_surface:Unknow color depth: " << std::string(str);
      break;
    }    
}

void
Blitter::put_surface_8bit(CL_Canvas* provider, CL_SurfaceProvider* sprovider,
			  int x, int y)
{
  int start_i;
  unsigned char* tbuffer; // Target buffer
  int twidth, theight, tpitch;
  
  unsigned char* sbuffer; // Source buffer
  int swidth, sheight, spitch;

  CL_Palette* palette;
  int x_offset, y_offset;

  provider->lock();
  sprovider->lock();

  tbuffer = static_cast<unsigned char*>(provider->get_data());
  sbuffer = static_cast<unsigned char*>(sprovider->get_data());

  //std::cout << "Colorkey: " << sprovider->get_src_colorkey() << std::endl;
  
  palette = sprovider->get_palette();

  if (!palette)
    {
      char str[1024];
      sprintf(str, "Couldn't find palette: %d", sprovider->get_depth());
      // throw PingusError(str);
      std::cout << std::string(str);
    }

  twidth = provider->get_width();
  theight = provider->get_height();
  tpitch = provider->get_pitch();

  swidth = sprovider->get_width();
  sheight = sprovider->get_height();
  spitch = sprovider->get_pitch();

  if (y < 0)
    y_offset = 0-y;
  else 
    y_offset = 0;
  
  if (x < 0) 
    x_offset = -x;
  else
    x_offset = 0;

  for(int line=y_offset; line < sheight && (line + y) < theight; ++line) 
    {
      start_i = ((line + y) * tpitch) + (x*4);
      
      for(int i=start_i+(4*x_offset),j=line*spitch+x_offset; 
	  i < start_i + (4*swidth) && (i-start_i+(x*4)) < (4*twidth);
	  i += 4, ++j) 
	{
	  if (sbuffer[j] != sprovider->get_src_colorkey()) 
	    {
	      tbuffer[i + 0] = 255;                                  // alpha
	      tbuffer[i + 1] = palette->palette[sbuffer[j] * 3 + 2]; // blue
	      tbuffer[i + 2] = palette->palette[sbuffer[j] * 3 + 1]; // green
	      tbuffer[i + 3] = palette->palette[sbuffer[j] * 3 + 0]; // red
	    }
	}
    }

  sprovider->unlock();
  provider->unlock();  
}

void
Blitter::put_surface_32bit(CL_Canvas* canvas, CL_SurfaceProvider* provider,
			   int x_pos, int y_pos)
{
  float red, green, blue, alpha;
  float tred, tgreen, tblue, talpha;

  int swidth = provider->get_width();
  int sheight = provider->get_height();

  int twidth = canvas->get_width();
  int theight = canvas->get_height();

	canvas->lock();
	provider->lock();

	for(int y=max(0, -y_pos); y < sheight && (y + y_pos) < theight ; y++)
	{
		for(int x=max(0,-x_pos); x < swidth && (x + x_pos) < twidth; x++)
		{
			provider->get_pixel(x, y, &red, &green, &blue, &alpha);
	
			if( alpha > 0 ) 
			{
				canvas->get_pixel(x + x_pos, y + y_pos, &tred, &tgreen, &tblue, &talpha);
				canvas->draw_pixel(x + x_pos, y + y_pos,
				 max(0.0, min(1.0, (red * alpha) + (tred * (1.0-alpha)))),
				 max(0.0, min(1.0, (green * alpha) +(tgreen * (1.0-alpha)))),
				 max(0.0, min(1.0, (blue * alpha)  + (tblue * (1.0-alpha)))),
				 1.0 ); // max(0.0, min(1.0, alpha * alpha + (talpha*(1.0-alpha)))));
			}
		}
	}
	provider->unlock();
	canvas->unlock();
}

void
Blitter::put_alpha_surface(CL_Canvas* provider, CL_SurfaceProvider* sprovider,
	    int x, int y)
{
  int start_i;
  unsigned char* tbuffer; // Target buffer
  int twidth, theight, tpitch;
  
  unsigned char* sbuffer; // Source buffer
  int swidth, sheight, spitch;

  CL_Palette* palette;
  int x_offset, y_offset;

  //  assert(sprovider->get_depth() == 8);
  if (sprovider->get_depth() != 8)
    {
      char str[1024];
      sprintf(str, "Image has wrong color depth: %d", sprovider->get_depth());
//      throw PingusError(str);
      std::cout << std::string(str);
    }
  //  assert(provider->get_pixel_format() == RGBA8888);

  provider->lock();
  sprovider->lock();

  tbuffer = static_cast<unsigned char*>(provider->get_data());
  sbuffer = static_cast<unsigned char*>(sprovider->get_data());
  
  palette = sprovider->get_palette();
  assert(palette);

  twidth = provider->get_width();
  theight = provider->get_height();
  tpitch = provider->get_pitch();

  swidth = sprovider->get_width();
  sheight = sprovider->get_height();
  spitch = sprovider->get_pitch();

  if (y < 0)
    y_offset = 0-y;
  else 
    y_offset = 0;
  
  if (x < 0) 
    x_offset = -x;
  else
    x_offset = 0;

  for(int line=y_offset; line < sheight && (line + y) < theight; ++line) {
    start_i = ((line + y) * tpitch) + (x*4);

    for(int i=start_i+(4*x_offset),j=line*spitch+x_offset; 
	i < start_i + (4*swidth) && (i-start_i+(x*4)) < (4*twidth); i+=4,++j) {
      if (sbuffer[j]) {
	tbuffer[i + 0] = 0;                                  // alpha
      }
    }
  }

  sprovider->unlock();
  provider->unlock();  
}

CL_Canvas*
Blitter::clear_canvas(CL_Canvas* canvas)
{
  unsigned char* buffer;
  
  canvas->lock();
  buffer = static_cast<unsigned char*>(canvas->get_data());
  memset(buffer, 0, sizeof(unsigned char) * canvas->get_pitch() * canvas->get_height());
  canvas->unlock();

  return canvas;
}

CL_Canvas*
Blitter::create_canvas(CL_Surface* sur)
{
  return create_canvas(sur->get_provider());
}

CL_Canvas*
Blitter::create_canvas(CL_SurfaceProvider* prov)
{
  CL_Canvas* canvas = new CL_Canvas(prov->get_width(), prov->get_height());

  switch (prov->get_bytes_per_pixel())
    {
    case 3:
      {
	canvas->lock();
	prov->lock();
	
	int buffer_size = prov->get_pitch () * prov->get_height ();
	unsigned char* sbuffer = static_cast<unsigned char*>(prov->get_data ());
	unsigned char* tbuffer = static_cast<unsigned char*>(canvas->get_data ());
	
	for (int si = 0, ti = 0; si < buffer_size; si += 3, ti += 4)
	  {
	    tbuffer[ti + 0] = 255;
	    tbuffer[ti + 1] = sbuffer[si + 0];
	    tbuffer[ti + 2] = sbuffer[si + 1];
	    tbuffer[ti + 3] = sbuffer[si + 2];
	  }
	  
	prov->unlock();
	canvas->unlock();
      }

    case 4:
      canvas->lock();
      prov->lock();
      memcpy(canvas->get_data(), prov->get_data(),
 	     sizeof(unsigned char) * prov->get_height() * prov->get_pitch());
      prov->unlock();
      canvas->unlock();
      break;

    default:
      put_surface(canvas, prov, 0, 0);
      break;
    }
  return canvas;
}

CL_Surface* 
Blitter::scale_surface (CL_Surface* sur, int width, int height)
{
  return CL_Surface::create (Blitter::scale_surface_to_canvas(sur, width, height), true);
}

CL_Canvas* 
Blitter::scale_surface_to_canvas (CL_Surface* sur, int width, int height)
{
  Color color;
  CL_SurfaceProvider* provider = sur->get_provider ();
  CL_Canvas* canvas = new CL_Canvas (width, height);

  provider->lock ();
  canvas->lock ();

  unsigned char* sbuffer = static_cast<unsigned char*>(provider->get_data ());
  unsigned char* tbuffer = static_cast<unsigned char*>(canvas->get_data ());
  int pwidth = provider->get_width ();
  int pheight = provider->get_height ();
  
  switch (provider->get_bytes_per_pixel ())
    {
    case 3:
      {
	// We assume that we have the data in RGB888, which might not be
	// the case
	for (int y = 0; y < height; y++)
	  for (int x = 0; x < width; x++)
	    {
	      int ti = (y * width + x) * 4;
	      int si = ((y * pheight / height) * pwidth
			+ (x * pwidth / width)) * 3;
		
	      tbuffer[ti + 0] = 255; // alpha
	      tbuffer[ti + 1] = sbuffer[(si + 0)]; // blue
	      tbuffer[ti + 2] = sbuffer[(si + 1)]; // green
	      tbuffer[ti + 3] = sbuffer[(si + 2)]; // red
	    }
      }
      break;
    case 4:
      {
	// We assume that we have the data in RGBA8888, which might not be
	// the case
	for (int y = 0; y < height; y++)
	  for (int x = 0; x < width; x++)
	    {
	      int ti = (y * width + x) * 4;
	      int si = ((y * pheight / height) * pwidth
			+ (x * pwidth / width)) * 4;
		
	      tbuffer[ti + 0] = sbuffer[(si + 0)]; // alpha
	      tbuffer[ti + 1] = sbuffer[(si + 1)]; // blue
	      tbuffer[ti + 2] = sbuffer[(si + 2)]; // green
	      tbuffer[ti + 3] = sbuffer[(si + 3)]; // red
	    }
      }
      break;
    default:
      // Slow but generic, using get_data () would be better, but would
      // require quite a bit of work
      for (int y = 0; y < height; y++)
	for (int x = 0; x < width; x++)
	  {
	    // std::cout << "X: " << x << " Y: " << y << std::endl;
	    provider->get_pixel (x * provider->get_width () / width ,
				 y * provider->get_height () / height,
				 &color.red, &color.green, &color.blue, &color.alpha);
	    // ignoring the source alpha due to get_pixel brokeness... no time to test the patch
//	    canvas->draw_pixel (x, y, color.red, color.green, color.blue, color.alpha);
	    canvas->draw_pixel (x, y, color.red, color.green, color.blue, 0);
	  }
      break;
    }

  canvas->unlock ();
  provider->unlock ();

  return canvas;
}

/*
// Converts a SurfaceProvider based surface, to a Canvas
// based one. The old one will not be deleted.
CL_Surface*
Blitter::convert_to_emptyprovider(CL_Surface* ssurf)
{
  CL_Canvas* tprov = convert_to_emptyprovider(ssurf->get_provider());
  return CL_Surface::create(tprov, true);
}

// Converts a SurfaceProvider, to an Canvas and returns
// the newly allocated provider, you need to delete it yourself.
CL_Canvas*
Blitter::convert_to_emptyprovider(CL_SurfaceProvider* sprov)
{
  CL_Canvas* tprov;
  CL_Palette* palette;
  unsigned char* sbuffer;
  unsigned char* tbuffer;
  int i;

  sprov->lock();
  switch(sprov->get_depth()) 
    {
    case 32:
      tprov = new CL_Canvas(sprov->get_width(),
			    sprov->get_height());
      tprov->lock();

      sbuffer = static_cast<unsigned char*>(sprov->get_data());
      tbuffer = static_cast<unsigned char*>(tprov->get_data());

      for(i=0; i < (tprov->get_height() * tprov->get_pitch()); ++i)
	{
	  tbuffer[i] = sbuffer[i];
	}

      tprov->unlock();
      break;
    case 8:
      tprov = new CL_Canvas(sprov->get_width(),
			    sprov->get_height());
      palette = sprov->get_palette();
      tprov->lock();
      
      sbuffer = static_cast<unsigned char*>(sprov->get_data());
      tbuffer = static_cast<unsigned char*>(tprov->get_data());    

      for(i=0; i < (sprov->get_height() * sprov->get_pitch()); ++i)
	{
	  tbuffer[i * 4 + 0] = 255;
	  tbuffer[i * 4 + 1] = palette->palette[sbuffer[i] * 3 + 2];
	  tbuffer[i * 4 + 2] = palette->palette[sbuffer[i] * 3 + 1];
	  tbuffer[i * 4 + 3] = palette->palette[sbuffer[i] * 3 + 0];
	}
      
      tprov->unlock();      
      break;
    default:
      std::cout << "convert_to_emptyprovider(): Wrong source format: " 
	   << static_cast<int>(sprov->get_depth()) << std::endl;
      assert(false);
      break;
    }
  sprov->unlock();
  
  return tprov;
}
*/ 







// ----------------------------------
// Harry's additions...




void Blitter::put_surface_no_blending(CL_Canvas* canvas, CL_Surface* sur, int x, int y)
{
	Blitter::put_surface_32bit_no_blending(canvas, sur->get_provider(), x, y);
}



void Blitter::put_surface_32bit_no_blending(CL_Canvas* canvas, CL_SurfaceProvider* provider,
			   int x_pos, int y_pos)
{
	int s_width = provider->get_width();
	int s_height = provider->get_height();
	int s_pitch = provider->get_pitch();
	int s_bpp = provider->get_depth();

	if( s_bpp != 32)
	{
		std::cout << "Blitter::put_surface_32bit_no_blending() " << std::endl;
		std::cout << "source bpp: " << provider->get_depth() << std::endl;
		std::cout << "convert to 32bpp first!" << std::endl;
		return;
	}

	int d_width = canvas->get_width();
	int d_height = canvas->get_height();
	int d_pitch = d_width * 4;

	if( x_pos > d_width ) return; // outside 
	if( y_pos > d_height ) return; // outside 

	canvas->lock();
	provider->lock();


	for(int y=max(0, -y_pos); y < s_height && (y + y_pos) < d_height ; y++)
	{
		unsigned char *src_data = static_cast<unsigned char*>( provider->get_data() );
		src_data += (y*s_pitch);
		
		unsigned char *dest_data = static_cast<unsigned char*>( canvas->get_data() );
		dest_data += ( (y_pos+y)*d_pitch + x_pos*4 );
	
		// copy line.
		// works only on 32 bpp
		if( (x_pos + s_width) < d_width ) // line fits on canvas
		{
			memcpy( dest_data, src_data, sizeof(unsigned char) * s_width * 4 );
		}
		else  // line doesn't fit on canvas.
		{
			int length = d_width - x_pos; // amount of pixels in line that fit on the canvas

			memcpy( dest_data, src_data, sizeof(unsigned char) * length * 4 );		
		}
	}

	provider->unlock();
	canvas->unlock();
}

CL_Surface* Blitter::convert_to_32_bpp(CL_Surface *sur)
{
	// surface must be 32 bpp so that it can be memcpyed to the canvas.

	CL_Canvas *dest = Blitter::convert_to_32_bpp( sur->get_provider());
	return CL_Surface::create(dest);
}

CL_Canvas* Blitter::convert_to_32_bpp(CL_SurfaceProvider* provider)
{
	int s_width = provider->get_width();
	int s_height = provider->get_height();
	int s_pitch = provider->get_pitch();
	int s_bpp = provider->get_depth();

	if( s_bpp == 32)
	{
		std::cout << "Blitter::convert_to_32_bpp" << std::endl;
		std::cout << "source bpp: " << provider->get_depth() << std::endl;
		std::cout << "surface allready 32 bpp" << std::endl;
		return NULL;
	}

	if( s_bpp != 24 )
	{
		std::cout << "Blitter::convert_to_32_bpp" << std::endl;
		std::cout << "source bpp: " << provider->get_depth() << std::endl;
		std::cout << "only 24 bpp and 32 bpp bg textures supported!" << std::endl;
		return NULL;
	}

	CL_Canvas *canvas = new CL_Canvas( s_width, s_height );

	int d_pitch = canvas->get_pitch();

	canvas->lock();
	provider->lock();

	for(int y=0; y < s_height; y++)
	{
		unsigned char *src_data = static_cast<unsigned char*>( provider->get_data() );
		unsigned char *dest_data = static_cast<unsigned char*>( canvas->get_data() );

		src_data += y*s_pitch;
		dest_data += y*d_pitch;
		
		for(int x=0; x < s_width; x++)
		{
			dest_data[0] = 255;             // a
			dest_data[1] = src_data[0];     // r
			dest_data[2] = src_data[1];     // g
			dest_data[3] = src_data[2];     // b
			dest_data += 4; // 32 bpp
			src_data += 3;  // 24 bpp
		}
	}
	provider->unlock();
	canvas->unlock();

	return canvas;
}




 
 
 
 
void Blitter::texture_fill_mirror( CL_Canvas *canvas, CL_Surface *surface )
{
	CL_Surface *flipped_x = CAImageManipulation::flip( surface, true );
	CL_Surface *flipped_y = CAImageManipulation::flip( surface, false );
	CL_Surface *flipped_xy = CAImageManipulation::flip( flipped_y, true );

	int width = surface->get_width();
	int height = surface->get_height();

	int times_x = canvas->get_width() / width + 1;
  	int times_y = canvas->get_height() / height + 1;

	for( int y = 0; y < times_y; y++ )
	{
		for( int x = 0; x < times_x; x++ )
		{
			if( y % 2 == 0 )
			{
				if( x % 2 == 0 )
				{
					Blitter::put_surface_no_blending( canvas, surface, x*width, y*height );
				}
				else
				{
					Blitter::put_surface_no_blending( canvas, flipped_x, x*width, y*height );
				}
			}
			else
			{
				if( x % 2 == 0 )
				{
					Blitter::put_surface_no_blending( canvas, flipped_y, x*width, y*height );
				}
				else
				{
					Blitter::put_surface_no_blending( canvas, flipped_xy, x*width, y*height );
				}
			}
		}
	}
}
 
 
 
void Blitter::texture_fill( CL_Canvas *canvas, CL_Surface *surface )
{
	int width = surface->get_width();
	int height = surface->get_height();
  
	int times_x = canvas->get_width() / width + 1;
  	int times_y = canvas->get_height() / height + 1;

	for( int y = 0; y < times_y; y++ )
	{
		for( int x = 0; x < times_x; x++ )
		{
			Blitter::put_surface_no_blending( canvas, surface, x*width, y*height );
		}
	}
}


// color key fill that ignores source alpha. It keeps the alpha of the canvas.

void Blitter::ckey_texture_fill( CL_Canvas *canvas, CL_Surface *surface,
								float cR, float cG, float cB )
{
	CL_SurfaceProvider *provider = surface->get_provider();
	
	int s_width = provider->get_width();
	int s_height = provider->get_height();
	
	float r,g,b,a, bogus_a;
	
	canvas->lock();
	provider->lock();
	
	for( unsigned int y=0; y < canvas->get_height(); y++ )
	{
		for( unsigned int x=0; x < canvas->get_height(); x++ )
		{
			int times_x = (int)(x / s_width);
		    int src_x = x - (times_x * s_width);
			
			int times_y = (int)(y / s_height);
		    int src_y = y - (times_y * s_height);
			
		    canvas->get_pixel( x, y, &r, &g, &b, &a );
			
			if( r == cR && g == cG && b == cB )
			{
				provider->get_pixel( src_x, src_y, &r, &g, &b, &bogus_a );
			    canvas->draw_pixel( x, y, r, g, b, a ); // doesn't change canvas alpha
			}
		}
	}

	canvas->unlock();
	provider->unlock();
}


/* EOF */
