/***************************************************************************
  view.cpp
  -------------------
  View class for QBrew
  -------------------
  Copyright 1999-2004, David Johnson
  Please see the header file for copyright and license information
 ***************************************************************************/

#include <qcheckbox.h>
#include <qcombobox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qstringlist.h>
#include <qtabwidget.h>
#include <qwhatsthis.h>

#include "controller.h"
#include "fixedspinbox.h"
#include "grainpage.h"
#include "hopspage.h"
#include "miscpage.h"
#include "notepage.h"
#include "model.h"
#include "resource.h"
#include "style.h"

#include "view.h"

using AppResource::CHAR_NBSP;
using AppResource::CHAR_LATIN_DEGREE;

//////////////////////////////////////////////////////////////////////////////
// Construction, Destruction                                                //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// View()
// ------
// Constructor

View::View(Controller *controller, Model *model)
    : QWidget(controller, "View"), controller_(controller), model_(model)
{
    initWidgets();
    initLayouts();

    mainlayout_->activate();
    // the following should allow enough room to display all widgets
    ingredienttab_->setMinimumHeight(192);

    // connections
    connect(model_,  SIGNAL(recipeChanged()), this, SLOT(flushView()));

    // start with new view
    flushView();
}

View::~View() { ; }

//////////////////////////////////////////////////////////////////////////////
// initWidget()
// ------------
// Initialize the widgets

// TODO: consider using UI file for layout

void View::initWidgets()
{
    // title
    titlelabel_ = new QLabel(this, "titlelabel_");
    titlelabel_->setText("Title");
    titleedit_ = new QLineEdit(this, "titleedit_");
    QWhatsThis::add(titleedit_,
                    "Use this field to enter the title of the recipe");
    connect(titleedit_, SIGNAL(textChanged(const QString &)),
            this, SLOT(setTitle(const QString &)));
    // style
    stylelabel_ = new QLabel(this, "stylelabel_");
    stylelabel_->setText("Style");
    stylecombo_ = new QComboBox(false, this, "stylecombo_");
    stylecombo_->insertStringList(model_->stylesList());
    QWhatsThis::add(stylecombo_,
                    "Use this control to select the style of recipe"
                    "<p>This only affects the minimum and maximum values"
                    "in the <em>characteristics</em> section");
    connect(stylecombo_, SIGNAL(activated(const QString &)),
            this, SLOT(setStyle(const QString &)));

    // brewer
    brewerlabel_ = new QLabel(this, "brewerlabel_");
    brewerlabel_->setText("Brewer");
    breweredit_ = new QLineEdit(this, "breweredit_");
    QWhatsThis::add(breweredit_,
                    "Use this field to enter the name of the brewer");
    connect(breweredit_, SIGNAL(textChanged(const QString &)),
            this, SLOT(setBrewer(const QString &)));

    // size
    sizelabel_ = new QLabel(this, "sizelabel_");
    sizelabel_->setText("Size");
    sizespin_ = new FixedSpinBox(this, "sizespin_", 0.00, 100.00, 0.25, 2);
    QWhatsThis::add(sizespin_,
                    "Use this control to enter the size of the recipe batch");
    connect(sizespin_, SIGNAL(valueChanged(double)),
            this, SLOT(setSize(double)));

     // mash
    mashcheck_ = new QCheckBox("Mash", this, "mashcheck_");
    QWhatsThis::add(mashcheck_,
                    "Check this box to indicate that your recipe is mashed");
    connect(mashcheck_, SIGNAL(toggled(bool)),
            this, SLOT(setMash(bool)));

    // style layout widgets
    stylebox_ = new QGroupBox(this, "stylebox_");
    ogminimum_ = new QLabel(stylebox_, "ogminimum_");
    ogmaximum_ = new QLabel(stylebox_, "ogmaximum_");
    ogrecipe_ = new QLabel(stylebox_, "ogrecipe_");
    ibuminimum_ = new QLabel(stylebox_, "ibuminimum_");
    ibumaximum_ = new QLabel(stylebox_, "ibumaximum_");
    iburecipe_ = new QLabel(stylebox_, "iburecipe_");
    srmminimum_ = new QLabel(stylebox_, "srmminimum_");
    srmmaximum_ = new QLabel(stylebox_, "srmmaximum_");
    srmrecipe_ = new QLabel(stylebox_, "srmrecipe_");
    vertseparator_ = new QFrame(stylebox_, "vertseparator_");
    vertseparator_->setFrameStyle(QFrame::VLine | QFrame::Sunken);
    finalgravity_ = new QLabel(stylebox_, "finalgravity_");
    alcoholbyvol_ =  new QLabel(stylebox_, "alcoholbyvol_");
    alcoholbywgt_ =  new QLabel(stylebox_, "alcoholbywgt_");

    // ingredient widgets
    ingredienttab_ = new QTabWidget(this, "ingredienttab_");
    grainpage_ = new GrainPage(model_, this, controller_, "grainpage_");
    hoppage_ = new HopsPage(model_, this, controller_, "hoppage_");
    miscpage_ = new MiscPage(model_, this, controller_, "miscpage_");
    notepage_ = new NotePage(model_, this, controller_, "notepage_");
    ingredienttab_->addTab(grainpage_, "&Grains");
    ingredienttab_->addTab(hoppage_, "Ho&ps");
    ingredienttab_->addTab(miscpage_, "&Miscellaneous");
    ingredienttab_->addTab(notepage_, "&Notes");
}

//////////////////////////////////////////////////////////////////////////////
// initLayouts()
// -------------
// Initialize the layouts

void View::initLayouts()
{
    mainlayout_ = new QVBoxLayout(this, 6, 6, "mainlayout_");
    recipelayout_ = new QGridLayout(2, 10, 3, "recipelayout_");
    stylelayout_ = new QGridLayout(stylebox_, 4, 9, 9, 3, "stylelayout_");

    // mainlayout_
    mainlayout_->addLayout(recipelayout_, 0);
    mainlayout_->addWidget(stylebox_, 0);
    mainlayout_->addWidget(ingredienttab_, 1);

    // recipelayout_
    recipelayout_->addColSpacing(1, 3);
    recipelayout_->addColSpacing(3, 6);
    recipelayout_->addColSpacing(5, 3);
    recipelayout_->addColSpacing(7, 3);
    recipelayout_->setColStretch(2, 3);
    recipelayout_->setColStretch(6, 1);
    recipelayout_->setColStretch(8, 1);
    recipelayout_->addWidget(titlelabel_, 0, 0, Qt::AlignRight);
    recipelayout_->addWidget(titleedit_, 0, 2);
    recipelayout_->addWidget(stylelabel_, 0, 4, Qt::AlignRight);
    recipelayout_->addMultiCellWidget(stylecombo_, 0, 0, 6, 8); //span 4 cols
    recipelayout_->addWidget(brewerlabel_, 1, 0, Qt::AlignRight);
    recipelayout_->addWidget(breweredit_, 1, 2);
    recipelayout_->addWidget(sizelabel_, 1, 4, Qt::AlignRight);
    recipelayout_->addWidget(sizespin_, 1, 6);
    recipelayout_->addWidget(mashcheck_, 1, 8, Qt::AlignRight);

    // stylelayout_
    stylelayout_->addRowSpacing(0, 9);
    stylelayout_->addColSpacing(1, 3);
    stylelayout_->addColSpacing(3, 3);
    stylelayout_->addColSpacing(5, 3);
    stylelayout_->addColSpacing(7, 3);
    stylelayout_->setColStretch(0, 1);
    stylelayout_->setColStretch(2, 1);
    stylelayout_->setColStretch(4, 1);
    stylelayout_->setColStretch(8, 1);
    stylelayout_->addWidget(ogminimum_, 1, 0);
    stylelayout_->addWidget(ibuminimum_, 1, 2);
    stylelayout_->addWidget(srmminimum_, 1, 4);
    stylelayout_->addWidget(ogmaximum_, 2, 0);
    stylelayout_->addWidget(ibumaximum_, 2, 2);
    stylelayout_->addWidget(srmmaximum_, 2, 4);
    stylelayout_->addWidget(ogrecipe_, 3, 0);
    stylelayout_->addWidget(iburecipe_, 3, 2);
    stylelayout_->addWidget(srmrecipe_, 3, 4);
    stylelayout_->addMultiCellWidget(vertseparator_, 1, 3, 6, 6);
    stylelayout_->addWidget(finalgravity_, 1, 8);
    stylelayout_->addWidget(alcoholbyvol_, 2, 8);
    stylelayout_->addWidget(alcoholbywgt_, 3, 8);
}

//////////////////////////////////////////////////////////////////////////////
// Slot Implementations                                                     //
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// setTitle()
// ---------------------
// set the recipe title

void View::setTitle(const QString &title)
{
    model_->recipe()->setTitle(title);
    controller_->documentModified();
}

//////////////////////////////////////////////////////////////////////////////
// setStyle()
// ---------------------
// set the recipe style

void View::setStyle(const QString &style)
{
    model_->recipe()->setStyle(model_->style(style));
    updateStyle();
    controller_->documentModified();
}

//////////////////////////////////////////////////////////////////////////////
// setBrewer()
// ---------------------
// set the recipe brewer

void View::setBrewer(const QString &brewer)
{
    model_->recipe()->setBrewer(brewer);
    controller_->documentModified();
}

//////////////////////////////////////////////////////////////////////////////
// setSize()
// ---------------------
// set the recipe size

void View::setSize(double size)
{
    model_->recipe()->setSize(Volume(size, model_->recipe()->size().unit()));
    updateStyle();
    controller_->documentModified();
}

//////////////////////////////////////////////////////////////////////////////
// setMash()
// ---------------------
// set mashed / unmashed

void View::setMash(bool mash)
{
    model_->recipe()->setMashed(mash);
    flushView();
    controller_->documentModified();
}

///////////////////////////////////////////////////////////////////////////////
// Miscellaneous                                                             //
///////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// flushView()
// -----------
// reset the view

void View::flushView()
{
    // save the modified flag to prevent side effects of setting widgets
    bool oldmod = model_->modified();
    // update recipe widgets
    titleedit_->setText(model_->recipe()->title());
    // TODO: need some way to shortcircuit this, so not every widget is updated
    breweredit_->setText(model_->recipe()->brewer());
    stylecombo_->clear();
    stylecombo_->insertStringList(model_->stylesList());
    setComboItem(stylecombo_, model_->recipe()->style().name());
    sizespin_->setValue(model_->recipe()->size().amount());
    sizespin_->setSuffix(" " + model_->recipe()->size().unit().name());
    mashcheck_->setChecked(model_->recipe()->mashed());
    // update page widgets
    grainpage_->documentChanged();
    hoppage_->documentChanged();
    miscpage_->documentChanged();
    notepage_->documentChanged();
    // update style widgets
    updateStyle();
    // restore modified flag
    model_->setModified(oldmod);
    controller_->documentModified();
}

//////////////////////////////////////////////////////////////////////////////
// updateStyle()
// ------------------
// Update the style when recipe changes

// TODO: clean up the formatting
void View::updateStyle()
{
    Style style = model_->recipe()->style();

    // beer style has changed, so update labels in stylelayout_
    stylebox_->setTitle(stylecombo_->currentText() + " Characteristics");
    ogminimum_->setText("Minimum Gravity: " +
                        QString::number(style.OGLow(), 'f', 3));
    ogmaximum_->setText("Maximum Gravity: " +
                        QString::number(style.OGHi(), 'f', 3));
    ogrecipe_->setText(QString("<b>Recipe") + CHAR_NBSP + "Gravity:" + CHAR_NBSP +
                       QString::number(Calc::OG(model_->recipe()), 'f', 3) + "</b>");
    ibuminimum_->setText("Minimum Bitterness: " +
                         QString::number(style.IBULow()));
    ibumaximum_->setText("Maximum Bitterness: " +
                         QString::number(style.IBUHi()));
    iburecipe_->setText(QString("<b>Recipe") + CHAR_NBSP + "Bitterness:" + CHAR_NBSP +
                        QString::number(Calc::IBU(model_->recipe()), 'f', 0) + "</b>");
    srmminimum_->setText("Minimum Color: " +
                         QString::number(style.SRMLow()) + CHAR_LATIN_DEGREE);
    srmmaximum_->setText("Maximum Color: " +
                         QString::number(style.SRMHi()) + CHAR_LATIN_DEGREE);
    srmrecipe_->setText(QString("<b>Recipe") + CHAR_NBSP + "Color:" + CHAR_NBSP +
                        QString::number(Calc::SRM(model_->recipe()), 'f', 0) +
                        CHAR_LATIN_DEGREE + "</b>");
    finalgravity_->setText("Estimated FG: " +
                           QString::number(Calc::FGEstimate(model_->recipe()), 'f', 3));
    alcoholbyvol_->setText(QString("<b>Alcohol") + CHAR_NBSP + "by" +
                           CHAR_NBSP + "Volume:" +
                           CHAR_NBSP + QString::number(Calc::ABV(model_->recipe()) * 100.0,
                                                       'f', 1) + "%</b>");
    alcoholbywgt_->setText(QString("<b>Alcohol") + CHAR_NBSP + "by" +
                           CHAR_NBSP + "Weight:" +
                           CHAR_NBSP + QString::number(Calc::ABW(model_->recipe()) * 100.0,
                                                       'f', 1) + "%</b>");
}

///////////////////////////////////////////////////////////////////////////////
// setComboItem()
// --------------
// Set a given combo box to the given entry.

void View::setComboItem(QComboBox *combo, const QString &txt)
{
    QListBox *box = combo->listBox();
    QListBoxItem *item = box->findItem(txt, Qt::ExactMatch);
    if (item) {
        combo->setCurrentItem(box->index(item));
    } else {
        combo->insertItem(txt);
    }
}
