package HTMLIO::CheckBoxGroup;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: CheckBoxGroup.pm,v 1.2 2001/02/10 22:21:50 muaddie Exp $

use strict;

=head1 NAME

HTMLIO::CheckBoxGroup - View class for a group of checkboxes.

=head1 SYNOPSIS

see HTMLIO

=head1 DESCRIPTION

see HTMLIO

=cut

use HTMLIO;
use PSP::HTML::Entities;
use HTMLIO::Utils;

@HTMLIO::CheckBoxGroup::ISA = qw(HTMLIO);

=head1 METHODS

see HTMLIO for further methods

=head2 html_input

 instance
 (string $html) html_input ([\@keys])

DESCRIPTION:

Will return the appropriate HTML necessary to set up user input through
a check box group.  C<@keys> specifies which keys from the hash received 
by C<set_possible_hash> to display in the list. If not passed, all the 
keys will be displayed.

=cut

sub html_input {
  my ($this, $name, $data_obj, $slice, $join) = @_;

  my ($possible,$ordering) = $data_obj->possible_hash();
  $slice ||= $ordering;
  $slice ||= [ sort keys %$possible ];
  $join ||= "\n";
  my $orig_attr = $this->attributes();
  my %selected = map { $_ => 1 } $data_obj->value();

  my @html;

  for my $key (@$slice) {
    my %attr;
    $attr{value} = $key;
    $selected{$key} and
      $attr{checked} = "true";
    push @html, (html_tag('checkbox',\%attr,[qw(value)]).
		 (defined $possible->{$key} ? $possible->{$key} : $key));
  }

  return join $join, @html;
}

=head2 html_view

 instance
 (string $data) html_view ([string \@slice, string $join])

DESCRIPTION:

Will display the values entered, or the values of the hash passed in
with C<set_possible_hash> (see C<set_display>). C<@slice> is a list of keys
for the same hash. If passed in, only that data will be displayed,
otherwise, all data received will all be displayed, joined on C<$join>
if recieved.

=cut

sub html_view {
  my ($this, $data_obj, $raw_p, $slice, $join) = @_;
  $join ||= "\n";
 
  my ($possible,$ordering) = $data_obj->possible_hash();
  my @data = $raw_p ? ($data_obj->raw_value()) : ($data_obj->value());
  $slice or $slice = [0...$#data];
  my @html;
  my $value;
  foreach (@$slice) {
    $value = $data[$_];
    $value = $possible->{$value} if $this->{_display};
    push @html, encode_entities($value);
  }

  return join $join, @html;
}

sub erf {
  my ($this, $data_obj, $raw_p, $slice, $join) = @_;
 
  my ($possible,$ordering) = $data_obj->possible_hash();
  my @data = @{$this->{_data}};
  $slice or $slice = [0...(scalar(@data) - 1)];
  my $value;

  my @html;
  foreach (@$slice) {
    $value = $raw_p ? $data[$_]->raw_value() : $data[$_]->value();
    $value = $possible->{$value} if $this->{_display};
    push @html, encode_entities($value);
  }

  return join $join, @html;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
