package AtomicData::Date;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Date.pm,v 1.1 2000/11/23 23:36:15 muaddib Exp $

use strict;

=head1 NAME

AtomicData::Date - date.

=head1 SYNOPSIS

see AtomicData.pm

=head1 DESCRIPTION

Implements data encapsulation for dates. This package is only good
till year 9,999.

=cut

use POSIX qw(strftime);
use AtomicData::Time;
@AtomicData::Date::ISA = qw (AtomicData::Time);

=head1 METHODS

see AtomicData.pm for further methods.

=head2 _canonicalize

 instance/private
 (int $value) _canonicalize (int $raw_value)

DESCRIPTION:

Will accept a value and return the cannonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _canonicalize {
  my ($this, $val) = @_;

  # perform text canonicalization.
  $val = $this->AtomicData::Text::_canonicalize($val);

  # blank is already canonical.
  $val eq "" and return $val;

  # if it is similar to mm-dd-[yy]yy and then [yy]yy-mm-dd.
  if ($val =~ /^(\d+)[ \/\-](\d+)[ \/\-](\d+)$/) {
    my ($mon,$day,$year);
    if ($1 > 12) {
      ($year,$mon,$day) = ($1,$2,$3);
    } else {
      ($mon,$day,$year) = ($1,$2,$3);
    }
    if (length($year) == 2) {
      $this->{_two_digit_year}++;
      $year = ($year > 50) ? 1900+$year : 2000+$year;
    } elsif (length($year) != 4) {
      $this->{_can_failed} = ['Unrecognized date string.'];
      return undef;
    } elsif ($year =~ /^0+$/) {
      $this->{_can_failed} = ['Year cannot be 0.'];
      return undef;
    }
    # month, day, year is all we need.
    return sprintf "%02d/%02d/%04d", $mon, $day, $year;
  }

  # see if it is any kind of time string.
  my $time = $this->AtomicData::Time::_canonicalize($val);
  defined $time or return;
  
  # convert seconds since epoch to canonical form.
  return strftime("%m/%d/%Y", localtime($time));
}

=head2 _verify

 instance/private
 (int $value) _verify ()

DESCRIPTION:

Will accept a value and return the canonical form or undef and set
C<$this->{_can_failed}>.

=cut

sub _verify {
  my ($this) = @_;

  my ($bool,$problems) = $this->SUPER::_verify();
  $bool or return (0, $problems);

  my $value = $this->canonical_value();
  my $parameters = $this->parameters();

  if ($parameters->{four_digit_year} and $this->{_two_digit_year}) {
    push @$problems, "'$value' contains a two-digit year.";
  }

  @$problems and return (0, $problems);
  return (1, []);
}

=head2 _format

  instance
  (string $zip_code) _format (int $can_value)

DESCRIPTION:

Returns a formatted date.

=cut

sub _format {
  my ($this, $value) = @_;

  if ($this->{_format}->{no_pad}) {
    $value =~ s|([\A/])0*|$1|g;
  }

  if ($this->{_format}->{hyphens}) {
    $value =~ s|/|-|g;
  }

  return $value;
}

=head2 day

 instance
 (int $date) day ()

DESCRIPTION:

Will return the day portion of the date.

=cut

sub day {
  my ($this) = @_;
  (my $date = $this->canonicalize()) =~ /^(\d+)\/(\d+)\/(\d+)$/ or return;
  return $2;
}

=head2 sday

 instance
 (string $day) sday ()

DESCRIPTION:

Will return the string day (e.g., Monday) for the date being
encapsulated.

=cut

sub sday {
  my ($this) = @_;
  (my $date = $this->canonicalize()) =~ /^(\d+)\/(\d+)\/(\d+)$/ or return;
  my $time = mktime(1,0,12,$2,$1-1,$3-1900);
  return strftime("%A",localtime($time));
}

=head2 month

 instance
 (int $date) month ()

DESCRIPTION:

Will return the month portion of the date.

=cut

sub month {
  my ($this) = @_;
  (my $date = $this->canonicalize()) =~ /^(\d+)\/(\d+)\/(\d+)$/ or return;
  return $1;
}

=head2 smonth

 instance
 (string $month) smonth ()

DESCRIPTION:

Will return the string name of the month (e.g., December).

=cut

sub smonth {
  my ($this) = @_;
  (my $date = $this->canonicalize()) =~ /^(\d+)\/(\d+)\/(\d+)$/ or return;
  my $time = mktime(1,0,12,$2,$1-1,$3-1900);
  return strftime("%B",localtime($time));
}

=head2 year

 instance
 (int $date) year ()

DESCRIPTION:

Will return the year portion of the date.

=cut

sub year {
  my ($this) = @_;
  (my $date = $this->canonicalize()) =~ /^(\d+)\/(\d+)\/(\d+)$/ or return;
  return $3;
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
