// ------------------------------------------------------------------------- //
// $Id: mainwindow.h,v 1.46 2004/01/06 19:45:45 weismann Exp $
// ------------------------------------------------------------------------- //

/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

// ------------------------------------------------------------------------- //

#ifndef _MAINWINDOW_H_ 
#define _MAINWINDOW_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#endif

#ifndef __GNUC__
# ifdef _WIN32
#  include <windows.h>
# endif
#endif

#if HAVE_GL_GL_H
# include <GL/gl.h> 
#endif

#if HAVE_OPENGL_GL_H	// MAC OS X
# include <OpenGL/gl.h>
#endif

#ifndef HAVE_SDL
# include "keycodes.h"
# include "mousecodes.h"
#else
# ifdef _WIN32
#  include <SDL_keysym.h>
#  include <SDL_mouse.h>
# else
#  include <SDL/SDL_keysym.h>
#  include <SDL/SDL_mouse.h>
# endif
# define SDL_BUTTON_WHEEL_UP     4
# define SDL_BUTTON_WHEEL_DOWN   5
#endif

#include "common.h"
#include "ftdraw.h"
#include "image.h"
#include "node.h"
#include "nodecontroller.h"
#include "group.h"

// ------------------------------------------------------------------------- //

//! A SDL-inspired key-event
class KeyEvent
{
public:
	SDLKey sym;
	uint mod;
	enum {DOWN, UP} type;
};

//! A mouse-button-event
class MouseButtonEvent
{
public:
	uint button;
};

//! A SDL-inspired resize-event
class ResizeEvent
{
public:
	uint new_width;
	uint new_height;
};

//! A SDL-inspired event
class Event
{
public:
	typedef enum {Null, Key, MouseButton, Resize} _type_;
	_type_ get_type() const { return _type; }
	Event(_type_ type = Null) : _type(type) { }
	const KeyEvent& key() const {
		assert(_type == Key);
		return _key_event;
	}
	KeyEvent& key() {
		assert(_type == Key);
		return _key_event;
	}
	const MouseButtonEvent& mousebutton() const {
		assert(_type == MouseButton);
		return _mouse_button_event;
	}
	MouseButtonEvent& mousebutton() {
		assert(_type == MouseButton);
		return _mouse_button_event;
	}
	const ResizeEvent& resize() const {
		assert(_type == Resize);
		return _resize_event;
	}
	ResizeEvent& resize() {
		assert(_type == Resize);
		return _resize_event;
	}
private:
	_type_ _type;
	union {
		KeyEvent         _key_event;
		MouseButtonEvent  _mouse_button_event;
		ResizeEvent      _resize_event;
	};
};

//! A circular buffer of keyboard events
class EventBuffer
{
public:
	EventBuffer(int size) : _size(size), _head(0), _tail(0) { 
		assert(size>0); _eventbuf = new Event[size];
	}
	~EventBuffer() {
		delete[] _eventbuf;
	}
	void insert(const Event &event);
	const Event *peek() const { 
		return (_head == _tail) ? 0 : &_eventbuf[_tail]; 
	}
	void next() {
		assert(_head != _tail); // buffer empty
		_tail = (_tail + 1) % _size;
	}
private:
	int _size;
	int  _head, _tail;
	Event *_eventbuf;
};

// ------------------------------------------------------------------------- //

//! Describes the size and (optionally) placement of a window
class WindowGeometry
{
public:
	WindowGeometry(uint w, uint h)
		: _width(w), _height(h), _using_placement(false) { };
	void set_placement(int x, int x_sign, int y, int y_sign) {
		_x               = x;
		_x_sign          = x_sign;
		_y               = y;
		_y_sign          = y_sign;
		_using_placement = true;
	};
	uint _width, _height;
	bool _using_placement;
	int  _x, _x_sign;
	int  _y, _y_sign;
};

//! A singleton (DP, pp. 127ff) implementing the presentation window
class MainWindow {
public:
	MainWindow(); 
	virtual ~MainWindow();
	// Pure virtual methods to be overloaded and implemented by 
	// child class specific to the graphical window api used by
	// that child class.
	virtual void create_window(WindowGeometry &g, bool fullscreen)=0;
	virtual void swap_buffers()=0;
	virtual void make_rasterfont()=0;
	virtual bool event(bool block)=0;
	virtual void toggle_fullscreen()=0;
	virtual std::string get_resolutions()=0;
	virtual std::string get_resolution()=0;
	virtual void set_resolution(std::string r)=0;

	uint    get_width() { return _width; };
	uint    get_height() { return _height; };
	const   Event* getevent(bool block);
	void    printString(int x, int y, const char *format, ... ); // Needed?
	void    init_gfx();
	void    frame();
	void    reset_time();
	bool    get_fullscreen() const { return _fullscreen; }
	bool    get_wireframe() const { return _wireframe; }
	void    set_wireframe(bool w) { _wireframe = w;    }
	bool    get_debug_osd() const { return _debug_osd; }
	void    set_debug_osd(bool d) { _debug_osd = d;    }
	void    load_textures();
	void    print_at(uint x, uint y, const char *fmt, ... );
	Group*  get_root() const { return _root; }
	void    new_root();
	void    init_gl();
	bool	read_framebuffer(byte *buf, unsigned int buf_size);
	bool    make_screenshot(const char * filename, float scale=1.0);
	float   _frame_time;     ///< Time at start of frame
	float   _frame_duration; ///< Time passed between last frame and this frame
protected:
	EventBuffer *_keybuf;
	GLuint  m_font_offset;
	uint    _width, _height;  ///< window dimension 
	bool    _fullscreen;  ///< flag telling whether fullscreen is on/off
private:
//	class ParticleSystem *_particle_system;
	void display (void);
	void draw_osd();       ///< Overlay printing (for debugging)
	struct _overlay_item
	{
		int x, y;
		char str[128];
	};
	std::vector<_overlay_item> _overlay_items;
	typedef std::vector<_overlay_item>::iterator _overlay_item_it;
	static MainWindow* _instance;
	bool    _wireframe;      ///< flag telling whether wireframe mode is on/off
	bool    _debug_osd;      ///< flag telling whether debug osd is drawn
	uint    _frames;
	uint    _frames_since_time;
	float   _frames_time;
	float   _fps;
	float   _last_idle;
	Group*  _root;
	Slider* _gui_slider;
};

#endif	// _MAINWINDOW_H_

// ------------------------------------------------------------------------- //

