// ------------------------------------------------------------------------- //
// $Id: common.h,v 1.44 2003/12/07 13:29:59 weismann Exp $
// ------------------------------------------------------------------------- //
/*
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _COMMON_H_
#define _COMMON_H_

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_ASSERT_H
# include <assert.h>
#endif

#if STDC_HEADERS
# include <stdlib.h>
# include <stdio.h>
#endif

#if HAVE_IOSTREAM
# include <iostream>
#endif

#if HAVE_SSTREAM
# include <sstream>
#endif

#if HAVE_CMATH
# include <cmath>
#endif

#include "logging.h"
#include "clock.h"

// ------------------------------------------------------------------------- //
// Globals

extern class Clock g_clock;
extern class MainWindow* g_renderer;
//extern class GuiFactory* g_fact;

#define ONE_SIXTYFOURTH 0.015625f

// ------------------------------------------------------------------------- //
// Platform dependent typedefs

typedef unsigned short uint16, ushort;
typedef unsigned int   uint32, uint;
typedef unsigned char  byte,   uchar;

#ifndef M_PI
# define M_PI 3.14159265358979323846  /* pi */
#endif

inline float deg2rad(float a) { return a*M_PI/180.0f; }
inline float rad2deg(float a) { return a*180.0f/M_PI; }

// ------------------------------------------------------------------------- //
//! two-dimensional float vector

template <class T>
class _v2 {
public:
	typedef _v2<T> v2type;

	T _f[2];
	
	_v2() {};
	_v2(T x, T y) { _f[0] = x; _f[1] = y; }

	inline void set(T x, T y) { _f[0] = x; _f[1] = y; };

	inline T x() const { return _f[0]; }
	inline T y() const { return _f[1]; }

	inline v2type& operator += (const v2type& v)
	{
		_f[0] += v._f[0];
		_f[1] += v._f[1];
		return *this;
	}

	inline v2type& operator *= (const v2type& v)
	{
		_f[0] *= v._f[0];
		_f[1] *= v._f[1];
		return *this;
	}

	inline v2type& operator /= (const v2type& v)
	{
		_f[0] /= v._f[0];
		_f[1] /= v._f[1];
		return *this;
	}
};

template <class T>
inline std::ostream& operator<< (std::ostream& os, const _v2<T>& v)
{
	os << v.x() << " " << v.y() << std::endl;
	return os;
}
#if HAVE_SSTREAM
template <typename T> std::string to_string(const T &thing) 
{
	// convert a thing to a string
	std::ostringstream oss; 
	oss << thing; 
	return std::string(oss.str()); 
}
#endif

template <class T>
inline _v2<T> operator + (const _v2<T>& v, const _v2<T>& w)
{
	return _v2<T>(v._f[0] + w._f[0], v._f[1] + w._f[1]);
}


typedef _v2<float> v2;
typedef _v2<uint>  iv2;

// ------------------------------------------------------------------------- //
//! A representation of a rectangle by lower-left and upper-right corner

class Rect {
public:
	iv2 _lower_left, _upper_right;
	Rect() { };
	Rect(const iv2& ll, const iv2& ur) : _lower_left(ll), _upper_right(ur) {};
	Rect(uint llx, uint lly, uint urx, uint ury)
		: _lower_left(llx, lly), _upper_right(urx, ury) { };

	inline void set(uint llx, uint lly, uint urx, uint ury)
	{
		_lower_left.set (llx, lly);
		_upper_right.set(urx, ury);
	}

	bool can_hold(iv2 size) const { 
		return size.x() <= width() && size.y() <= height(); }

	inline uint top()    const { return _upper_right.y(); }
	inline uint bottom() const { return _lower_left.y();  }
	inline uint left()   const { return _lower_left.x();  }
	inline uint right()  const { return _upper_right.x(); }

	inline uint width()  const { return right() - left();   }
	inline uint height() const { return top()   - bottom(); }
};

// ------------------------------------------------------------------------- //
//! 3 dimensional float vector

class v3 {
public:
	float _f[3];
	
	v3() {}
	v3(float x, float y, float z) { _f[0] = x; _f[1] = y; _f[2] = z; }

	inline void set(float x, float y, float z) { _f[0]=x; _f[1]=y; _f[2]=z; };

	inline float x() const { return _f[0]; }
	inline float y() const { return _f[1]; }
	inline float z() const { return _f[2]; }

	inline v3& operator += (const v3& v)
	{
		_f[0] += v._f[0];
		_f[1] += v._f[1];
		_f[2] += v._f[2];
		return *this;
	}

	inline v3& operator /= (const float a)
	{
		_f[0] /= a;
		_f[1] /= a;
		_f[2] /= a;
		return *this;
	}


	inline float length() const {
		return sqrt(_f[0]*_f[0] + _f[1]*_f[1] + _f[2]*_f[2]);
	}

	inline void make_norm() {
		*this /= length();
	}
};

inline v3 operator - (const v3& v, const v3& w)
{
	return v3(v._f[0] - w._f[0], v._f[1] - w._f[1], v._f[2] - w._f[2]);
}

inline v3 operator * (const v3& v, const float a)
{
	return v3(v._f[0]*a, v._f[1]*a, v._f[2]*a);
}

inline v3 operator * (const v3& v, const v3& u)
{
	return v3(v._f[0]*u._f[0], v._f[1]*u._f[1], v._f[2]*u._f[2]);
}

inline v3 operator / (const v3& v, const float a)
{
	return v3(v._f[0]/a, v._f[1]/a, v._f[2]/a);
}

inline v3 operator + (const v3& v, const v3& w)
{
	return v3(v._f[0] + w._f[0], v._f[1] + w._f[1], v._f[2] + w._f[2]);
}

inline v3 vcross(const v3& u, const v3& v)
{
	return v3(
		u.y()*v.z() - v.y()*u.z(),
		-(u.x()*v.z() - v.x()*u.z()),
		u.x()*v.y() - v.x()*u.y());
}

inline std::ostream& operator<< (std::ostream& os, const v3& v)
{
	os << "(" << v.x() << ", " << v.y() << ", " << v.z() << ")" << std::endl;
	return os;
}

// ------------------------------------------------------------------------- //
//! 3 dim size

class size3 {
public:
	struct {
		float min, max;
	} x, y, z;

	size3() {};
	size3(float xmin, float xmax,
	      float ymin, float ymax,
	      float zmin, float zmax) {
		x.min = xmin;
		x.max = xmax;
		y.min = ymin;
		y.max = ymax;
		z.min = zmin;
		z.max = zmax;
	};
};

std::ostream& operator<< (std::ostream& os, const size3& s);

// ------------------------------------------------------------------------- //
//! 4 dimensional byte vector

class bv4 {
public:
	byte _b[4];

	bv4() {};
	bv4(uint32 v) {
		_b[0] = (v & 0xff000000) >> 24;
		_b[1] = (v & 0x00ff0000) >> 16;
		_b[2] = (v & 0x0000ff00) >> 8;
		_b[3] = (v & 0x000000ff);
	}
	bv4(byte r, byte g, byte b, byte a) { _b[0]=r; _b[1]=g; _b[2]=b; _b[3]=a;}
	bv4(float r, float g, float b, float a) {
		_b[0] = byte(r*255.0f);
		_b[1] = byte(g*255.0f);
		_b[2] = byte(b*255.0f);
		_b[3] = byte(a*255.0f);
	}

	inline void set(byte r, byte g, byte b, byte a) { 
		_b[0]=r; _b[1]=g; _b[2]=b; _b[3]=a; };

	inline byte& r() { return _b[0]; }
	inline byte& g() { return _b[1]; }
	inline byte& b() { return _b[2]; }
	inline byte& a() { return _b[3]; }

};

typedef class bv4 rgba;

inline bv4 operator * (const bv4& v, const float a)
{
	return bv4(v._b[0]*a, v._b[1]*a, v._b[2]*a, v._b[3]*a);
}


// ------------------------------------------------------------------------- //
//! 3x3 float matrix

class m33 {
public:
	float _f[3][3];

	m33() {}
	m33(float m00, float m01, float m02,
	    float m10, float m11, float m12,
	    float m20, float m21, float m22)
	{
		_f[0][0] = m00; _f[0][1] = m01; _f[0][2] = m02;
		_f[1][0] = m10; _f[1][1] = m11; _f[1][2] = m12;
		_f[2][0] = m20; _f[2][1] = m21; _f[2][2] = m22;
	}

	inline void set(float m00, float m01, float m02,
	                float m10, float m11, float m12,
	                float m20, float m21, float m22)
	{
		_f[0][0] = m00; _f[0][1] = m01; _f[0][2] = m02;
		_f[1][0] = m10; _f[1][1] = m11; _f[1][2] = m12;
		_f[2][0] = m20; _f[2][1] = m21; _f[2][2] = m22;
	}

	inline v3 xcol() { return v3(_f[0][0], _f[1][0], _f[2][0]); }
	inline v3 ycol() { return v3(_f[0][1], _f[1][1], _f[2][1]); }
	inline v3 zcol() { return v3(_f[0][2], _f[1][2], _f[2][2]); }

	float *ptr() { return (float*) _f; }
	const float *ptr() const {return (const float*)_f; }

	void makeRotate(float xangle, float yangle, float zangle);
	void makeIdentity() { set(1.0f, 0.0f, 0.0f, 
	                          0.0f, 1.0f, 0.0f,
	                          0.0f, 0.0f, 1.0f); }
	void create_from_dir(const v3& xdir, const v3& up = v3(0.0f, 1.0f, 0.0f));
};

std::ostream& operator<< (std::ostream& os, const m33& m);

// ------------------------------------------------------------------------- //
//! 4x4 float matrix

class m44 {
public:
	float _f[4][4];

	m44() {}
	m44(float m00, float m01, float m02, float m03,
	    float m10, float m11, float m12, float m13,
	    float m20, float m21, float m22, float m23,
	    float m30, float m31, float m32, float m33)
	{
		_f[0][0] = m00; _f[0][1] = m01; _f[0][2] = m02; _f[0][3] = m03;
		_f[1][0] = m10; _f[1][1] = m11; _f[1][2] = m12; _f[1][3] = m13;
		_f[2][0] = m20; _f[2][1] = m21; _f[2][2] = m22; _f[2][3] = m23;
		_f[3][0] = m30; _f[3][1] = m31; _f[3][2] = m32; _f[3][3] = m33;
	}

	inline void set(float m00, float m01, float m02, float m03,
	                float m10, float m11, float m12, float m13,
	                float m20, float m21, float m22, float m23,
	                float m30, float m31, float m32, float m33)
	{
		_f[0][0] = m00; _f[0][1] = m01; _f[0][2] = m02; _f[0][3] = m03;
		_f[1][0] = m10; _f[1][1] = m11; _f[1][2] = m12; _f[1][3] = m13;
		_f[2][0] = m20; _f[2][1] = m21; _f[2][2] = m22; _f[2][3] = m23;
		_f[3][0] = m30; _f[3][1] = m31; _f[3][2] = m32; _f[3][3] = m33;
	}

	// These were stolen from osg. Need to understand the difference...
	float *ptr() { return (float*) _f; }
	const float *ptr() const {return (const float*)_f; }

	void makeRotate(v3 dir, float angle);
};

std::ostream& operator<< (std::ostream& os, const m44& m);

// is_pow_two(x) returns true if x is a power of two
inline bool is_pow_two   (uint x) { return (x-1 & x) == 0; }

// next_pow_two(x) returns smallest power of two not less than x
inline uint next_pow_two (uint x)
{
	uint r = 0x80000000;
	while (r && r >= x) {
		r >>= 1;
	}
	return r ? r << 1 : 1;
}

#if !defined(_WIN32) || (defined __GNUC__)
inline int min(int a, int b) { return a < b ? a : b; }
inline int max(int a, int b) { return a > b ? a : b; }
#endif

#if !defined(_WIN32) 
inline float fmin(float a, float b) { return a < b ? a : b; }
inline float fmax(float a, float b) { return a > b ? a : b; }
#endif

inline float frand() { return (float)rand() / RAND_MAX; }

// ------------------------------------------------------------------------- //
//! proper handling of memory 

// allocate memory before you try to strdup
char *xstrdup (const char *str);

// ------------------------------------------------------------------------- //
//! locale stuff 

#ifdef ENABLE_NLS
# include <libintl.h>
# include <locale.h>
# define _(x) gettext(x) 
#else
# define _(x) x
#endif

// ------------------------------------------------------------------------- //
//! OpenGL 

#define PrintGLError { \
	GLenum errCode; \
	const GLubyte *errString; \
	if ((errCode = glGetError()) != GL_NO_ERROR) { \
		errString = gluErrorString(errCode); \
		log_warning("OpenGL error: %s\n", errString); \
	} \
}

// ------------------------------------------------------------------------- //
//! Macros... they're hacks, but quite useful

#define DISPLAYBOX(g, pos, color) \
	{ \
	static Leaf* l = 0; \
	if (!l) { \
		l = g->add_box(); \
	} \
	l->set_color((color)); \
	l->set_pos((pos)); \
	}

#define DISPLAYLINE(g, pos1, pos2, color) \
	{ \
	static Leaf* l = 0; \
	v3 v = (pos2) - (pos1); \
	if (!l) { \
		l = g->add_line(); \
	} \
	l->get_primitive(0)->line_prim(v); \
	l->set_color(color); \
	l->set_pos(pos1); \
	}


extern "C" void glMultMatrixf(const float *p);
extern float glmat[16];
inline void load_GL_matrix(m33& mat, v3& pos)
{
	float* m = mat.ptr();
	float x = pos.x();
	float y = pos.y();
	float z = pos.z();
	// C:  mat[row][col] = mat[row*3+col]
	// GL: mat[col][row] = mat[col*4+row]
	glmat[ 0] = m[0]; glmat[ 1] = m[3]; glmat[ 2] = m[ 6]; glmat[ 3] = 0.0f;
	glmat[ 4] = m[1]; glmat[ 5] = m[4]; glmat[ 6] = m[ 7]; glmat[ 7] = 0.0f;
	glmat[ 8] = m[2]; glmat[ 9] = m[5]; glmat[10] = m[ 8]; glmat[11] = 0.0f;
	glmat[12] = x;    glmat[13] = y;    glmat[14] = z;     glmat[15] = 1.0f;
	glMultMatrixf(&glmat[0]);
}

#endif	// _COMMON_H_

// ------------------------------------------------------------------------- //

