/************************************************************************/
/* sccw - Soundcard CW for Linux   Steven J. Merrifield			*/
/*			 	   VK3ESM  sjm@ee.latrobe.edu.au	*/
/*				   http://livewire.ee.latrobe.edu.au	*/
/*									*/
/* Feb 92 - First MS-DOS release using PC speaker (Turbo Pascal)	*/
/* Oct 92 - Added punctuation support, varying character delay		*/
/* Aug 95 - Added mouse support for sending practice			*/
/* Jul 96 - Added soundcard support using FM chip			*/
/* Aug 96 - Complete rewrite in ANSI-C, first Linux release	 	*/
/* Dec 96 - Modified by Richard Everitt G4ZFE for use as a Morse Code	*/
/*          pileup program.						*/
/* Apr 97 - Added a competition mode. Implemented each voice as a	*/
/*          Kernel thread using LinuxThreads (by Xavier Leroy,		*/
/*          Xavier.Leroy@inria.fr)					*/
/*          Richard Everitt, G4ZFE. For score results see:		*/
/*		http://www.babbage.demon.co.uk/pileup.html		*/
/* Jul 01 - Fixed scanf() buffer overflows				*/
/************************************************************************/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <unistd.h> 
#include <signal.h>
#include <sys/io.h>
#include <sys/types.h>
#include <time.h>
#include "pileup.h"
#include <pthread.h>

extern int AdLib_found();
extern int Play_sound();
extern void Cleanup ();

/* Function prototypes	*/
void stop_thread (void*);
void Code_2_snd(int,int,int,int);
int Read_data ();
int send_callsign (int,int,int,int);
void controlc_handler (int);
void *start_thread (void*);
void stop_thread (void*);
void decrement_count (int);
void *keyboard_thread (void*);
int near_character (char*,char);
int check_call (char*);

pthread_mutex_t global_data_mutex = PTHREAD_MUTEX_INITIALIZER;

char *Code[] = 
		{".-","-...","-.-.","-..",".","..-.","--.",
		"....","..",".---","-.-",".-..","--","-.","---",
		".--.","--.-",".-.","...","-","..-","...-",
		".--","-..-","-.--","--..",  /* A..Z */
		"-----",".----","..---","...--","....-",
		".....","-....","--...","---..","----.", /* 0..9 */
		"..--..","-..-.",""}; /*  q-mark, slant, space */

char *callsign[MAX_CALLS];		/* Calls to be sent		*/

char *calls_tx[MAX_CALLS];		/* Calls sent in this session	*/
char *calls_rx[MAX_CALLS];		/* Calls copied in this session	*/

int  number_of_calls;			/* Max. number to be sent	*/

int  max_speed = 0;			/* Maximum speed (WPM) reached	*/
int  rx_count = 0;			/* Number of calls sent in sess	*/
int  tx_count = 0;			/* Number of calls rx'ed in ses	*/
int  score=0;				/* Cumulative score		*/

int  finished = FALSE;			/* Flag to indicate if user     */
					/* wants to give up!		*/

int  channel = 0;			/* Voice to send callsign on	*/
int  Morse_gap = 3;
int  AdLib_vol = 48;

int  competition_mode = FALSE;		/* Competition mode/Practice 	*/
					/* mode				*/
int  number_session = NUMBER_COMPETITION_CALLSIGNS;
					/* Number of callsigns sent in	*/
					/* Competition mode		*/
int  number_voices;			/* Maximum number of voices	*/
int  base_speed;			/* Slowest CW speed to be used	*/

pthread_t threads [9];			/* One thread for each voice	*/

/************************************************************************/
/* Convert the string into sound					*/
/************************************************************************/
void Code_2_snd(int a, int voice, int speed, int tone)
{
	int i;
	double dit,dah,wait;
	/* Voice 0 is the loudest, 8 the quietist			*/
	int volume_decrease = voice * 4;

	dit = (double)(60.0/(speed*50.0)) * 1000000.0; 	/* delay in us (PARIS method) */
	dah = 3 * dit;
	wait = Morse_gap * dit;

	for (i=0;i<strlen(Code[a]);i++)
	{
		if (Code[a][i] == '.') 
		{
			Play_sound(voice,tone,dit,AdLib_vol - volume_decrease);
		}
		if (Code[a][i] == '-')
		{
			Play_sound(voice,tone,dah,AdLib_vol-volume_decrease);
		}
		usleep(dit);	/* delay between each dit or dah */
	}
	usleep(wait);	 	/* delay between each each characte4 */
}


/************************************************************************/
/* Read a file, and store all the callsigns read			*/
/************************************************************************/
int Read_data ()
{
	FILE *f;
	int i=0;

	if ((f = fopen("calls.dat","rt")) == NULL)
	{
		fprintf(stderr,"Error opening file.\n");
		return(1);
	}

	while (!feof(f) && i < MAX_CALLS)
	{
		callsign[i] = malloc (12);
		fgets (callsign[i], 12, f);

		/* Remove the \n (if any)				*/
		if (callsign [i] [strlen(callsign[i]) - 1] == 0x0A)
			callsign [i] [strlen(callsign[i]) - 1] = '\0'; 

		i++;
	}

	number_of_calls = i - 1;

	fclose(f);
	return(0);
}

/************************************************************************/
/* Send a callsign on a given voice at specified speed and tone		*/
/************************************************************************/
int send_callsign (int number, int voice, int speed,int tone)
{
	int i, ch;

	Code_2_snd (38,voice,speed,tone);

	/* Keep a track of the fastest speed reached		*/
	if (speed > max_speed)
		max_speed = speed;

	/* Display the callsign about to be sent		*/
	if (!competition_mode)
	{
		printf ("%s ",callsign[number]);
	 	fflush (stdout);	
	}

	/* send callsign					*/
	for (i=0; i < strlen (callsign[number]) ; i++)
	{
		ch = callsign [number] [i];
		if (ch == ' ')
			continue;	

		if (isalpha(ch)) ch = toupper(ch); 
		if (((ch >= 'A') && (ch <= 'Z')) ||
			((ch >= '0' ) && (ch <= '9')) ||
			(ch == '?') || (ch == '/') || (ch == ' ') ||
			(ch == 0x0A))
		{
			if ((ch >= 'A') && (ch <= 'Z')) 
				ch = ch - 65;
			else if ((ch >= '0') && (ch <= '9')) 
				ch = ch - 22;
			else if (ch == '?') 
				ch = 36;
			else if (ch == '/') 
				ch = 37;
			else if (ch == ' ') 
				ch = 38;
			Code_2_snd(ch,voice,speed,tone);
		}
	}

	number_of_calls--;

	return (0);
}


/************************************************************************/
/* On a control C event stop all SoundBlaster voices              	*/
/************************************************************************/
void controlc_handler (int sig)
{
	int i;
	signal (sig, SIG_IGN);

	for (i=0; i < number_voices; i++)
		pthread_cancel (threads[i]);

	for (i=0; i < number_voices; i++)
		pthread_join (threads[i], NULL);

	printf ("\n");

	exit (1);
}

/************************************************************************/
/* Called on a control C event for each thread  			*/
/************************************************************************/
void stop_thread (void *arg)
{
	Cleanup ();

	ioperm(reg_port,17,0);
	ioperm(data_port,17,0);
}

/************************************************************************/
/* Store the callsign being sent for checking at the end		*/
/************************************************************************/
void decrement_count (int call)
{
	pthread_mutex_lock (&global_data_mutex);

	number_session--;

	/* Save the callsign just sent.			*/
	calls_tx[tx_count++] = strdup (callsign [call]);

	if (finished == TRUE)
		number_session = 0;

	pthread_mutex_unlock (&global_data_mutex);
}

/************************************************************************/
/* (Competition mode only). A thrad to read user input. The callsign	*/
/* entered is stored for checking at the end.				*/
/************************************************************************/
void *keyboard_thread (void *arg)
{
	char call[15];

	while (finished == FALSE)
	{
		fgets (call, sizeof(call), stdin);
		call[strlen(call)-1]='\0';

		/* Store the callsign entered from the	*/
		/* keyboard				*/
		calls_rx[rx_count++] = strdup (call);

		/* '0' is the abort now flag		*/
		if (!strcmp (call,"0"))
			finished = TRUE;
	}
	return NULL;
}

/************************************************************************/
/* Start a thread for each voice.                                   	*/
/************************************************************************/
void *start_thread (void *arg)
{
	int call;
	int r_speed, r_tone, k;
	pthread_t tid;

	/* Use the thread ID and process ID as seed	*/
	tid = pthread_self();

	/* Seed the random generator			*/
	srand((int)tid+getpid());	

	ioperm(reg_port,17,1);
	ioperm(data_port,17,1);

	/* Allow the thread to be cancelled immediatedly	*/
	pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
	pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL);

	/* On thread cancellation, clear the soundblaster voice	*/	
	pthread_cleanup_push ((void*) stop_thread, (void*) NULL);

	while (number_session > 0)          
	{
		call = rand () % number_of_calls;
		r_speed = rand () % 20;
		r_tone = (rand () % 300) + r_speed;
		k = rand () % 3;

		decrement_count(call);

		send_callsign (call,(int)arg,r_speed+base_speed,r_tone+BASE_TONE);

		sleep (k);
	}

	pthread_cleanup_pop (0);

	return NULL;
}

/************************************************************************/
/* Attempt to match partials in a string, e.g call sent is g4zfe	*/
/* copied as g4zfi.							*/
/************************************************************************/
int near_character (char *call, char letter)
{
	char	c;
	int	i;

	for (i=0; i < strlen(call); i++)
	{
		c = call[i];

		if (c == 'E' && letter == 'I')
			return (TRUE);

		if (c == 'I' && letter == 'E')
			return (TRUE);

		if (c == 'S' && letter == 'H')
			return (TRUE);

		if (c == 'H' && letter == 'S')
			return (TRUE);

		if (c == 'D' && letter == 'B')
			return (TRUE);

		if (c == 'B' && letter == 'D')
			return (TRUE);

		if (c >= '0' && c <= '9')
		{
			if (letter >= '0' && letter <= '9')
			{
				if (c > letter)
				{
					if (c - letter == 1)
						return (TRUE);
				}
				else if (letter - c == 1)
				{
					return (TRUE);
				}
			}
		}
	}

	return (FALSE);
}

/************************************************************************/
/* Find the best match to the received callsign.			*/
/************************************************************************/
int check_call (char *rx_call)
{
	int i,j;		/* Local counters		*/
	int match;		/* Characters match		*/
	int partial;		/* Partial character match	*/
	char *p;
	char *ptr;
	char temp_call [15];
	int possible_match = -1;
	int possible_match_score = 0;
	float possible_match_rating = 0;
	float rating = 0;

	/* Check the received call against each of the 		*/
	/* transmitted calls in order to find the best match	*/
	for (i=0; i < tx_count; i++)
	{
		strcpy (temp_call, calls_tx[i]);
		ptr = rx_call;
		match = 0;
		partial = 0;

		/* Ignore empty callsigns			*/
		if (strlen (temp_call) == 0 || strlen (rx_call) == 0)
			continue;
		
		/* Check for correct callsign match		*/
		if (!strcmp (rx_call, temp_call))
		{
			score += strlen(rx_call);
			return (i);
		}

		/* Find exact character matches			*/
		for (j=0; j < strlen(temp_call); j++)
		{
			if (calls_tx[i][j] != '\0')
			{
				p = strchr (ptr, temp_call[j]);

				if (p != NULL)
				{
					/* Increment pointer 	*/
					/* to search rest of str*/
					ptr = p++;
					match++;

					/* Mark characeters so	*/
					/* it will not be 	*/
					/* checked again	*/
					temp_call[j] = '.';
				}
			}
		}

		if (match > 0)
		{
			/* Find partial matches			*/
			for (j=0; j < strlen(temp_call); j++)
				if (near_character (rx_call, temp_call[j]))
					partial++;
		}

		/* Rate this match. Partial matches are only 	*/
		/* worth half as much as a complete match	*/
		rating = match + (partial/2);

		/* Determine if this match is the "best one"	*/
		if (rating > possible_match_rating)
		{
			possible_match = i;
			possible_match_rating = match + (partial/2);
			possible_match_score = match;
		}
	}

	/* If we've found a match then return it		*/
	if (possible_match != -1)
	{
		score += possible_match_score;
		return (possible_match);
	}

	return (-1);
}


/************************************************************************/
/* Setup the IO ports, and start sending, then reset the IO ports on	*/ 
/* exit. Note that this requires the setuid bit to be set, since we are	*/
/* playing with the hardware directly.					*/
/************************************************************************/
int main(int argc, char *argv[])
{
	pthread_t keyb_thread;
	int i;
	int j;
	int retcode;
	int max_score = 0;
	int results [MAX_CALLS];
	char *tx_string[MAX_CALLS];
	int mode;

	/* Check that we're running as root otherwise we'll SEGV later	*/
	if (geteuid() != 0)
	{
		printf ("Sorry. This program must be run as root \n");
		printf ("or made a setuid root program. This is needed in\n");
		printf ("order to access hardware registers.\n\n");
		exit (1);
	}

	if (!(AdLib_found))
	{
		fprintf(stderr,"\nAdLib/SoundBlaster card not found.\n");
		exit(1);
	}

	Read_data ();

	printf ("How many voices (1 to 9)\n");
	fscanf (stdin,"%d",&number_voices);

	if (number_voices < 0 || number_voices > 9)
	{
		printf ("Only 9 voices allowed ...\n");
		exit (1);
	}

	printf ("Starting speed (wpm)\n");
	fscanf (stdin,"%d",&base_speed);

	printf ("Enter '0' for competion mode or '1' for practice mode\n");
	fscanf (stdin,"%d",&mode);

	if (mode == 0)
	{
		number_session = NUMBER_COMPETITION_CALLSIGNS;
		competition_mode = TRUE;
	}
	else
	{
		number_session = number_of_calls;
		competition_mode = FALSE;
	}

	signal (SIGINT, controlc_handler);

	/* Create a thread to handle the keyboard		*/
	if (competition_mode)
	{
		retcode = pthread_create (&keyb_thread, NULL, keyboard_thread, (void*) NULL );
		printf ("Enter '0' to abort the session! GL..\n");
		sleep (1);
	}

	/* Create a thread for each voice requested		*/
	for (i=0; i< number_voices; i++)
	{
		retcode = pthread_create (&threads[i], 
					  NULL, 
					  start_thread,
					  (void*) i);
		if (retcode != 0)
			fprintf (stderr,"Creation of thread %d failed \n", i);
	}

	/* Wait for all the threads to finish			*/
	for (i=0; i< number_voices; i++)
	{
		pthread_join (threads[i], NULL);
	}

	if (competition_mode)
	{
		/* wait for the last call to be copied			*/
		sleep (2);

		/* Convert everything to upper case			*/
		for (i=0; i < tx_count; i++)
		{
			for (j=0; j < strlen (calls_tx[i]); j++)
				calls_tx [i][j] = toupper (calls_tx[i][j]);

			tx_string [i] = strdup (calls_tx[i]);
		}

		for (i=0; i < rx_count; i++)
			for (j=0; j < strlen (calls_rx[i]); j++)
				calls_rx [i][j] = toupper (calls_rx[i][j]);

		/* Pad out rx array					*/
		for (i=rx_count; i< tx_count; i++)
			calls_rx [i] = strdup ("");

		/* Determine maximum score				*/
		for (i=0; i < tx_count; i++)
		{
			max_score += strlen (calls_tx[i]);
			results[i] = -1;
		}
	
		/* Check calls received against those sent		*/
		/* and store the position of any matches		*/
		for (i=0; i < rx_count; i++)
		{
			retcode = check_call (calls_rx[i]);
			if (retcode != -1)
			{
				results [retcode] = i;
				strcpy (calls_tx[retcode],"");
			}
		}

		/* Display the results					*/
		printf ("\n");
		printf ("TX\t\tRX\t   TX\t\tRX\n");
		printf ("--\t\t--\t   --\t\t--\n");

		for (j=0; j < tx_count/2; j++) 
		{
			printf ("%-12s ",tx_string[j]);
			if (results[j] != -1)	
				printf ("%-12s",calls_rx[results[j]]);
			else
				printf ("%-12s"," ");

			printf ("%-12s ",tx_string[j+tx_count/2]);
			if (results[j+tx_count/2] != -1)
				printf ("%-12s\n",calls_rx[results[j+tx_count/2]]);
			else
				printf ("%-12s\n"," ");
		}


		printf ("\n");
		printf ("Accuracy: %d/%d. Max speed: %d\n",score,max_score,max_speed);
		printf ("Score: %d\n", score * max_speed * number_voices);

		/* Free up as we're about to exit		*/
		for (i=0; i < tx_count; i++)
		{
			free (tx_string[i]);
		}

		for (i=0; i < rx_count; i++)
			free (calls_rx[i]);

	}

	for (i=0; i < tx_count; i++)
	{
		free (calls_tx[i]);
	}

	/* Free up as we're about to exit		*/
	for (i=0; i < number_of_calls; i++)
		free (callsign [i]);

	ioperm(reg_port,17,0);
	ioperm(data_port,17,0);

	return(0);
}


